------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             E X P _ U T I L                              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.48 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  Package containing utility procedures used throughout the expander

with Rtsfind; use Rtsfind;
with Types;   use Types;

package Exp_Util is

   package Multi_Use is

   --  It is often the case that an expression appears only once in the code
   --  but must be referenced many times by the expanded code without being
   --  evaluated more than once. This package provides an abstraction that
   --  satisfies this requirement. Such an expression is represented by an
   --  object of type Exp_Id that is initialized with Prepare or New_Exp_Id
   --  then each reference to the expression is done with New_Ref.

      type Exp_Id is private;

      procedure Prepare
        (Exp  : Node_Id;
         Res  : out Exp_Id;
         Code : out List_Id);
      --  given an Exp, create an new Id (Res) that can be used later for
      --  new references to this expression without forcing a re-evaluation.
      --  If the expression is an entity_name, no special code is generated,
      --  but for more complex expressions a temporary is created that is
      --  either the value of the expression or a pointer to the value (for
      --  composite types). Code returns the declaration that have to be
      --  inserted before any use of the expression, it is No_List if no
      --  code is generated.

      function New_Exp_Id (Exp : Node_Id; N : Node_Id) return Exp_Id;
      --  Same as Prepare but the code is inserted before N and analyzed.
      --  Typically, N is the instruction containing Exp.

      function New_Exp_Id (Exp : Node_Id; L : List_Id) return Exp_Id;
      --  Same as Prepare but the code is inserted at the end of L and not
      --  analyzed. Typically L is the Actions of an expression_actions.

      function New_Ref (E : Exp_Id; Loc : Source_Ptr) return  Node_Id;
      --  Return a new reference to an expression referenced by E

      function Wrap (Code : List_Id; N : Node_Id) return Node_Id;
      --  Given a code list returned by a call to Prepare, and a node N
      --  for an expression typically containing references constructed
      --  using New_Ref, returns the node unchanged if the code list is
      --  empty, and otherwise returns an expression actions node which
      --  contains the required code.

   private
      type Exp_Id is record
         Id        : Entity_Id;
         Is_Access : Boolean;

         --  Some flags have to be inherited by the different copies

         Assignment_OK : Boolean;
      end record;
   end Multi_Use;

   function Build_Runtime_Call (Loc : Source_Ptr; RE : RE_Id) return Node_Id;
   --  Build an N_Procedure_Call_Statement calling the given runtime entity.
   --  The call has no parameters. The first argument provides the location
   --  information for the tree and for error messages. The call node is not
   --  analyzed on return, the caller is responsible for analyzing it.

   type Compare_Result is (LT, LE, EQ, NE, GT, GE, Unknown);
   function Compile_Time_Compare (L, R : Node_Id) return Compare_Result;
   --  Given two expression nodes, finds out whether it can be determined
   --  at compile time how the runtime values will compare. An Unknown
   --  result means that the result of a comparison cannot be determined at
   --  compile time, otherwise the returned result indicates the known result
   --  of the comparison, given as tightly as possible (i.e. EQ or LT is a
   --  preferred returned value to LE). This routine is only used in cases
   --  where the caller knows that the expression types are scalar.

   function Convert_To (T : Entity_Id; Exp : Node_Id) return Node_Id;
   --  Returns an expression that represents the result of a checked convert
   --  of expression Exp to type T. If the base type of Exp is T, then no
   --  conversion is required, and Exp is returned unchanged. Otherwise an
   --  N_Type_Conversion node is constructed to convert the expression.

   function Unchecked_Convert_To
     (T    : Entity_Id;
      Exp  : Node_Id)
      return Node_Id;
   --  Like Convert_To, but if a conversion is actually needed, constructs
   --  an N_Unchecked_Type_Conversion node to do the required conversion.

   function New_Class_Wide_Subtype
     (CW_Typ : Entity_Id;
      N      : Node_Id)
      return   Entity_Id;
   --  Create an implicit subtype of CW_Typ attached to node N.

   function Expand_Class_Wide_Subtype
     (N          : Node_Id;
      Class_Type : Entity_Id;
      E          : Multi_Use.Exp_Id)
      return       List_Id;
   --  When a Class_Wide_Subtype value is encountered, that is to say in such
   --  cases as:
   --
   --    X: T'Class := Exp
   --    new T'Class'(Exp)
   --
   --  This function generates the list of declarations defining a record
   --  of the following form :
   --
   --    type anon is record
   --       _parent : Root_Type_Of (Typ) constrained with Exp discriminants;
   --       Extension : String (1 .. expr to match size of Exp);
   --    end record;
   --
   --  This record is compatible with any value of T'Class thanks to the
   --  first field and has the same size as Exp thanks to the second field.
   --  This record is attached to the subclass by its Equivalent_Type field.
   --  The declarations are not analyzed, the caller is responsible for
   --  analyzing them after they have been inserted into the tree.

   procedure Expand_Subtype_From_Expr
     (N             : Node_Id;
      Unc_Type      : Entity_Id;
      Subtype_Indic : Node_Id;
      Exp           : Node_Id);
   --  Build a constrained subtype from the initial value in object
   --  declarations and/or allocations when the type is indefinite (including
   --  class-wide).

   function Find_Prim_Op (T : Entity_Id; Name : Name_Id) return Entity_Id;
   --  Find the first primitive operation of type T whose name is 'Name'.
   --  this function allows the use of a primitive operation which is not
   --  directly visible

   procedure Expand_Tagged_Copy
     (N   : Node_Id;
      Lhs : Multi_Use.Exp_Id;
      Rhs : Multi_Use.Exp_Id;
      Typ : Entity_Id);
   --  Expand the code for copying the value of Rhs to Lhs. The arguments are
   --  entities rather than expressions because this procedure is
   --  used in conjunction with prepare_multi_use_of_Exp. The value of a
   --  tagged type excludes its tag. If the type is controlled it also excludes
   --  the finalization pointers

   procedure Append_Freeze_Action (T : Entity_Id; N : Node_Id);
   --  Add a new freeze action for the given type. The freeze action is
   --  attached to the freeze node for the type. Actions will be elaborated
   --  in the order in which they are added. Note that the added node is not
   --  analyzed. The analyze call is found in Sem_Ch13.Expand_N_Freeze_Entity.

   procedure Append_Freeze_Actions (T : Entity_Id; L : List_Id);
   --  Adds the given list of freeze actions (declarations or statements)
   --  for the given type. The freeze actions are attached to the freeze
   --  node for the type. Actions will be elaborated in the order in which
   --  they are added, and the actions within the list will be elaborated in
   --  list order. Note that the added nodes are not analyzed. The analyze
   --  call is found in Sem_Ch13.Expand_N_Freeze_Entity.

   procedure Wrap_Cleanup_Procedure (N : Node_Id);
   --  Given an N_Subprogram_Body node, this procedure adds an Abort_Defer
   --  call at the start of the statement sequence, and an Abort_Undefer call
   --  at the end of the statement sequence. All cleanup routines (i.e. those
   --  that are called from "at end" handlers) must defer abort on entry and
   --  undefer abort on exit. Note that it is assumed that the code for the
   --  procedure does not contain any return statements which would allow the
   --  flow of control to escape doing the undefer call.

end Exp_Util;
