/*======================================================================

    A driver for the Adaptec/Trantor T460 SCSI Adapter

    Written by David Hinds, dhinds@allegro.stanford.edu

    The SCSI code is essentially straight from Drew Eckhardt's
    generic NCR5380 driver, with 53c400 patches by Kevin Lentin
    (kevinl@cs.monash.edu.au)

    Copyright 1993, Drew Eckhardt
        Visionary Computing
	drew@colorado.edu

======================================================================*/

#include <linux/config.h>

#ifdef MODULE
#include <linux/module.h>
#include <linux/version.h>
#endif

#define AUTOSENSE
#define PSEUDO_DMA

#include <asm/system.h>
#include <asm/io.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/ioport.h>

#include "drivers/block/blk.h"
#include "drivers/scsi/scsi.h"
#include "drivers/scsi/hosts.h"
#include "t460_cs.h"
#include "drivers/scsi/NCR5380.h"
#include "drivers/scsi/constants.h"
#include "drivers/scsi/sd.h"

#include "version.h"
#include "cs_types.h"
#include "cs.h"
#include "cistpl.h"
#include "ds.h"

#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
static char *version =
    "t460_cs.c $Revision$ $Date$ (David Hinds)\n";
#endif

/*====================================================================*/

/* Parameters that can be set with 'insmod' */

/* Bit map of interrupts to choose from */
static u_long irq_mask = 0xdeb8;

/*====================================================================*/

static void t460_release(u_long arg);
static int t460_event(event_t event, int priority,
		      event_callback_args_t *args);

static dev_link_t *t460_attach(void);
static void t460_detach(dev_link_t *);

int t460_abort(Scsi_Cmnd *);
int t460_queue_command(Scsi_Cmnd *, void (*done)(Scsi_Cmnd *));
int t460_reset(Scsi_Cmnd *);

static int t460_detect(Scsi_Host_Template *);
static int t460_biosparam(Disk * disk, int dev, int *ip);

static Scsi_Host_Template driver_template = {
    NULL, NULL, "Adaptec AHA460 / Trantor T460",
    t460_detect, NULL, NULL, NULL,
    t460_queue_command, t460_abort, t460_reset, NULL,
    t460_biosparam,
    /* can queue */ CAN_QUEUE, /* id */ 7, SG_ALL,
    /* cmd per lun */ CMD_PER_LUN , 0, 0, DISABLE_CLUSTERING
};

static dev_link_t *dev_list = NULL;

static dev_info_t dev_info = "t460_cs";

static int t460_port, t460_irq;

/*====================================================================*/

static void cs_error(int func, int ret)
{
    CardServices(ReportError, dev_info, (void *)func, (void *)ret);
}

/*====================================================================*/

static dev_link_t *t460_attach(void)
{
    client_reg_t client_reg;
    dev_link_t *link;
    int ret;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("t460_attach()\n");
#endif

    /* Create new SCSI device */
    link = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);
    memset(link, 0, sizeof(struct dev_link_t));
    link->release.function = &t460_release;
    link->release.data = (u_long)link;

    link->io.NumPorts1 = 0x10;
    link->io.Attributes1 = IO_DATA_PATH_WIDTH_16;
    link->io.IOAddrLines = 10;
    link->irq.Attributes = IRQ_TYPE_EXCLUSIVE;
    link->irq.IRQInfo1 = IRQ_INFO2_VALID|IRQ_LEVEL_ID;
    link->irq.IRQInfo2 = irq_mask;
    link->conf.Attributes = CONF_ENABLE_IRQ;
    link->conf.Vcc = 50;
    link->conf.IntType = INT_MEMORY_AND_IO;
    link->conf.Present = PRESENT_OPTION;

    /* Register with Card Services */
    link->next = dev_list;
    dev_list = link;
    client_reg.dev_info = &dev_info;
    client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
    client_reg.event_handler = &t460_event;
    client_reg.EventMask =
	CS_EVENT_RESET_REQUEST | CS_EVENT_CARD_RESET |
	CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
	CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
    client_reg.Version = 0x0210;
    client_reg.event_callback_args.client_data = link;
    ret = CardServices(RegisterClient, &link->handle, &client_reg);
    if (ret != 0) {
	cs_error(RegisterClient, ret);
	t460_detach(link);
	return NULL;
    }
    
    return link;
} /* t460_attach */

/*====================================================================*/

static void t460_detach(dev_link_t *link)
{
    dev_link_t **linkp;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("t460_detach(0x%p)\n", link);
#endif
    
    /* Locate device structure */
    for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
	if (*linkp == link) break;
    if (*linkp == NULL)
	return;

    if (link->state & DEV_CONFIG) {
	t460_release((u_long)link);
	if (link->state & DEV_STALE_CONFIG) {
	    link->state |= DEV_STALE_LINK;
	    return;
	}
    }

    if (link->handle)
	CardServices(DeregisterClient, link->handle);
    
    /* Unlink device structure, free bits */
    *linkp = link->next;
    if (link->priv) {
	/* What do we do with the host structure ??? */
    }
    kfree_s(link, sizeof(struct dev_link_t));
    
} /* t460_detach */

/*====================================================================*/

static void t460_config(dev_link_t *link)
{
    client_handle_t handle;
    tuple_t tuple;
    cisparse_t parse;
    struct Scsi_Host *host;
    int i, j;
    u_char tuple_data[64];
    
    handle = link->handle;
    host = link->priv;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("t460_config(0x%p)\n", link);
#endif

    do {
	tuple.DesiredTuple = CISTPL_CONFIG;
	i = CardServices(GetFirstTuple, handle, &tuple);
	if (i != CS_SUCCESS) break;
	tuple.TupleData = tuple_data;
	tuple.TupleDataMax = 64;
	tuple.TupleOffset = 0;
	i = CardServices(GetTupleData, handle, &tuple);
	if (i != CS_SUCCESS) break;
	i = CardServices(ParseTuple, handle, &tuple, &parse);
	if (i != CS_SUCCESS) break;
	link->conf.ConfigBase = (caddr_t)parse.config.base;
    } while (0);
    if (i != CS_SUCCESS) {
	cs_error(ParseTuple, i);
	link->state &= ~DEV_CONFIG_PENDING;
	return;
    }

    /* Configure card */
    driver_template.usage_count = &mod_use_count_;
    link->state |= DEV_CONFIG;

    do {

	for (j = 0; j < 4; j++) {
	    link->conf.ConfigIndex = 8 + (j&1) + 4*(j>>1);
	    link->io.BasePort1 = 0x360 - 0x10*(j&1) - 0x100*(j>>1);
	    i = CardServices(RequestIO, link->handle, &link->io);
	    if (i == CS_SUCCESS) break;
	}
	if (i != CS_SUCCESS) {
	    cs_error(RequestIO, i);
	    break;
	}
	
	i = CardServices(RequestIRQ, link->handle, &link->irq);
	if (i != CS_SUCCESS) {
	    cs_error(RequestIRQ, i);
	    break;
	}
	i = CardServices(RequestConfiguration, link->handle, &link->conf);
	if (i != CS_SUCCESS) {
	    cs_error(RequestConfiguration, i);
	    break;
	}

    } while (0);

    if (i != 0) {
	t460_release((u_long)link);
	return;
    }

    sprintf(link->dev_name, "t460");
    link->state &= ~DEV_CONFIG_PENDING;

    /* Set configuration options for t460_detect() */
    t460_port = link->io.BasePort1;
    t460_irq = link->irq.AssignedIRQ;
    
    scsi_register_module(MODULE_SCSI_HA, &driver_template);
    
} /* t460_config */

/*====================================================================*/

static void t460_release(u_long arg)
{
    dev_link_t *link = (dev_link_t *)arg;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("t460_release(0x%p)\n", link);
#endif

    if (*driver_template.usage_count != 0) {
	printk("t460_cs: release postponed, '%s' still open\n",
	       link->dev_name);
	link->state |= DEV_STALE_CONFIG;
	return;
    }

    if (link->dev_name[0] != '\0')
	scsi_unregister_module(MODULE_SCSI_HA, &driver_template);
    
    CardServices(ReleaseConfiguration, link->handle);
    CardServices(ReleaseIO, link->handle, &link->io);
    CardServices(ReleaseIRQ, link->handle, &link->irq);
    
    link->state &= ~DEV_CONFIG;
    if (link->state & DEV_STALE_LINK)
	t460_detach(link);
    
} /* t460_release */

/*====================================================================*/

static int t460_event(event_t event, int priority,
		      event_callback_args_t *args)
{
    dev_link_t *link = args->client_data;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("t460_event()\n");
#endif
    
    switch (event) {
#ifdef PCMCIA_DEBUG
    case CS_EVENT_REGISTRATION_COMPLETE:
	if (pc_debug)
	    printk("t460_cs: registration complete\n");
	break;
#endif
    case CS_EVENT_CARD_REMOVAL:
	link->state &= ~DEV_PRESENT;
	if (link->state & DEV_CONFIG) {
	    link->release.expires = 5;
	    add_timer(&link->release);
	}
	break;
    case CS_EVENT_CARD_INSERTION:
	link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
	t460_config(link);
	break;
    case CS_EVENT_PM_SUSPEND:
	link->state |= DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_RESET_PHYSICAL:
	if (link->state & DEV_CONFIG)
	    CardServices(ReleaseConfiguration, link->handle);
	break;
    case CS_EVENT_PM_RESUME:
	link->state &= ~DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_CARD_RESET:
	if (link->state & DEV_CONFIG) {
	    CardServices(RequestConfiguration, link->handle, &link->conf);
	    NCR5380_reset(NULL);
	}
	break;
    }
    return 0;
} /* t460_event */

/*====================================================================*/

#ifdef MODULE

char kernel_version[] = UTS_RELEASE;

int init_module(void) {
    servinfo_t serv;
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(version);
#endif
    CardServices(GetCardServicesInfo, &serv);
    if (serv.Revision != CS_RELEASE_CODE) {
	printk("t460_cs: Card Services release does not match!\n");
	return -1;
    }
    register_pcmcia_driver(&dev_info, &t460_attach, &t460_detach);
    return 0;
}

void cleanup_module(void) {
    printk("t460_cs: unloading\n");
    unregister_pcmcia_driver(&dev_info);
    while (dev_list != NULL) {
	if (dev_list->state & DEV_CONFIG)
	    panic("t460_cs: device still configured!\n");
	else
	    t460_detach(dev_list);
    }
}

#endif

/*====================================================================*/

static int t460_detect(Scsi_Host_Template * tpnt)
{
    struct Scsi_Host *instance;

    instance = scsi_register(tpnt, sizeof(struct NCR5380_hostdata));
    instance->io_port = t460_port + 8;

    NCR5380_init(instance, FLAG_NCR53C400);

    instance->irq = t460_irq;
    if (request_irq(instance->irq, t460_intr, SA_INTERRUPT, "t460_cs")) {
	printk("scsi%d : irq %d not free, interrupts disabled\n", 
	       instance->host_no, instance->irq);
	instance->irq = IRQ_NONE;
    } 

    printk("scsi%d : at port %d", instance->host_no, instance->io_port);
    if (instance->irq == IRQ_NONE)
	printk (" interrupts disabled");
    else 
	printk (" irq %d", instance->irq);
    printk(" options CAN_QUEUE=%d  CMD_PER_LUN=%d",
	   CAN_QUEUE, CMD_PER_LUN);
    NCR5380_print_options(instance);
    printk("\n");

    return 1;
}

/*====================================================================*/

const char *generic_NCR5380_info(void)
{
    static const char string[] = "Trantor T460 Info";
    return string;
}

/*====================================================================*/

static int t460_biosparam(Disk * disk, int dev, int *ip)
{
  int size = disk->capacity;
  ip[0] = 64;
  ip[1] = 32;
  ip[2] = size >> 11;
  return 0;
}

/*====================================================================*/

static inline int t460_pread(struct Scsi_Host *instance,
			     unsigned char *dst, int len)
{
    int blocks = len / 128;
    int start = 0;
    int i;
    int bl;
    NCR5380_local_declare();

    NCR5380_setup(instance);

#if (NDEBUG & NDEBUG_C400_PREAD)
    printk("53C400r: About to read %d blocks for %d bytes\n", blocks, len);
#endif

    NCR5380_write(C400_CONTROL_STATUS_REG, CSR_BASE | CSR_TRANS_DIR);
    NCR5380_write(C400_BLOCK_COUNTER_REG, blocks);
    while (1) {
    
#if (NDEBUG & NDEBUG_C400_PREAD)
	printk("53C400r: %d blocks left\n", blocks);
#endif

	if ((bl=NCR5380_read(C400_BLOCK_COUNTER_REG)) == 0) {
#if (NDEBUG & NDEBUG_C400_PREAD)
	    if (blocks)
		printk("53C400r: blocks still == %d\n", blocks);
	    else
		printk("53C400r: Exiting loop\n");
#endif
	    break;
	}

#if 1
	if (NCR5380_read(C400_CONTROL_STATUS_REG) & CSR_GATED_53C80_IRQ) {
	    printk("53C400r: Got 53C80_IRQ start=%d, blocks=%d\n", start, blocks);
	    return -1;
	}
#endif

#if (NDEBUG & NDEBUG_C400_PREAD)
	printk("53C400r: Waiting for buffer, bl=%d\n", bl);
#endif

	while (NCR5380_read(C400_CONTROL_STATUS_REG) & CSR_HOST_BUF_NOT_RDY)
	    ;
#if (NDEBUG & NDEBUG_C400_PREAD)
	printk("53C400r: Transferring 128 bytes\n");
#endif

	for (i=0; i<128; i++)
	    dst[start+i] = NCR5380_read(C400_HOST_BUFFER);
	start+=128;
	blocks--;
    }

#if (NDEBUG & NDEBUG_C400_PREAD)
    printk("53C400r: EXTRA: Waiting for buffer\n");
#endif
    while (NCR5380_read(C400_CONTROL_STATUS_REG) & CSR_HOST_BUF_NOT_RDY)
	;

#if (NDEBUG & NDEBUG_C400_PREAD)
    printk("53C400r: Transferring EXTRA 128 bytes\n");
#endif
    for (i=0; i<128; i++)
	dst[start+i] = NCR5380_read(C400_HOST_BUFFER);
    start+=128;
    blocks--;

#if (NDEBUG & NDEBUG_C400_PREAD)
    printk("53C400r: Final values: blocks=%d   start=%d\n", blocks, start);
#endif

    if (!(NCR5380_read(C400_CONTROL_STATUS_REG) & CSR_GATED_53C80_IRQ))
	printk("53C400r: no 53C80 gated irq after transfer");
#if (NDEBUG & NDEBUG_C400_PREAD)
    else
	printk("53C400r: Got 53C80 interupt and tried to clear it\n");
#endif

/* DON'T DO THIS - THEY NEVER ARRIVE!
    printk("53C400r: Waiting for 53C80 registers\n");
    while (NCR5380_read(C400_CONTROL_STATUS_REG) & CSR_53C80_REG)
	;
*/

    if (!(NCR5380_read(BUS_AND_STATUS_REG) & BASR_END_DMA_TRANSFER))
	printk("53C400r: no end dma signal\n");
#if (NDEBUG & NDEBUG_C400_PREAD)
    else
	printk("53C400r: end dma as expected\n");
#endif

    NCR5380_write(MODE_REG, MR_BASE);
    NCR5380_read(RESET_PARITY_INTERRUPT_REG);
    return 0;
}

static inline int t460_pwrite(struct Scsi_Host *instance,
			      unsigned char *src, int len)
{
    int blocks = len / 128;
    int start = 0;
    int i;
    int bl;
    NCR5380_local_declare();

    NCR5380_setup(instance);

#if (NDEBUG & NDEBUG_C400_PWRITE)
    printk("53C400w: About to write %d blocks for %d bytes\n", blocks, len);
#endif

    NCR5380_write(C400_CONTROL_STATUS_REG, CSR_BASE);
    NCR5380_write(C400_BLOCK_COUNTER_REG, blocks);
    while (1) {
	if (NCR5380_read(C400_CONTROL_STATUS_REG) & CSR_GATED_53C80_IRQ) {
	    printk("53C400w: Got 53C80_IRQ start=%d, blocks=%d\n", start, blocks);
	    return -1;
	}

	if ((bl=NCR5380_read(C400_BLOCK_COUNTER_REG)) == 0) {
#if (NDEBUG & NDEBUG_C400_PWRITE)
	    if (blocks)
		printk("53C400w: exiting loop, blocks still == %d\n", blocks);
	    else
		printk("53C400w: exiting loop\n");
#endif
	    break;
	}

#if (NDEBUG & NDEBUG_C400_PWRITE)
	printk("53C400w: %d blocks left\n", blocks);

	printk("53C400w: waiting for buffer, bl=%d\n", bl);
#endif
	while (NCR5380_read(C400_CONTROL_STATUS_REG) & CSR_HOST_BUF_NOT_RDY)
	    ;

#if (NDEBUG & NDEBUG_C400_PWRITE)
	printk("53C400w: transferring 128 bytes\n");
#endif
	for (i=0; i<128; i++)
	    NCR5380_write(C400_HOST_BUFFER, src[start+i]);
	start+=128;
	blocks--;
    }
    if (blocks) {
#if (NDEBUG & NDEBUG_C400_PWRITE)
	printk("53C400w: EXTRA waiting for buffer\n");
#endif
	while (NCR5380_read(C400_CONTROL_STATUS_REG) & CSR_HOST_BUF_NOT_RDY)
	    ;

#if (NDEBUG & NDEBUG_C400_PWRITE)
	printk("53C400w: transferring EXTRA 128 bytes\n");
#endif
	for (i=0; i<128; i++)
	    NCR5380_write(C400_HOST_BUFFER, src[start+1]);
	start+=128;
	blocks--;
    }
#if (NDEBUG & NDEBUG_C400_PWRITE)
    else
	printk("53C400w: No EXTRA required\n");
#endif
    
#if (NDEBUG & NDEBUG_C400_PWRITE)
    printk("53C400w: Final values: blocks=%d   start=%d\n", blocks, start);
#endif

#if 0
    printk("53C400w: waiting for registers to be available\n");
    THEY NEVER DO!
    while (NCR5380_read(C400_CONTROL_STATUS_REG) & CSR_53C80_REG)
	;
    printk("53C400w: Got em\n");
#endif

    /* Let's wait for this instead - could be ugly */
    /* All documentation says to check for this. Maybe my hardware is too
     * fast. Waiting for it seems to work fine! KLL
     */
    while (!(i = NCR5380_read(C400_CONTROL_STATUS_REG) & CSR_GATED_53C80_IRQ))
	;

    /*
     * I know. i is certainly != 0 here but the loop is new. See previous
     * comment.
     */
    if (i) {
#if (NDEBUG & NDEBUG_C400_PWRITE)
	prink("53C400w: got 53C80 gated irq (last block)\n");
#endif
	if (!((i=NCR5380_read(BUS_AND_STATUS_REG)) & BASR_END_DMA_TRANSFER))
	    printk("53C400w: No END OF DMA bit - WHOOPS! BASR=%0x\n",i);
#if (NDEBUG & NDEBUG_C400_PWRITE)
	else
	    printk("53C400w: Got END OF DMA\n");
#endif
    }
    else
	printk("53C400w: no 53C80 gated irq after transfer (last block)\n");

#if 0
    if (!(NCR5380_read(BUS_AND_STATUS_REG) & BASR_END_DMA_TRANSFER)) {
	printk("53C400w: no end dma signal\n");
    }
#endif

#if (NDEBUG & NDEBUG_C400_PWRITE)
    printk("53C400w: waiting for last byte...\n");
#endif
    while (!(NCR5380_read(TARGET_COMMAND_REG) & TCR_LAST_BYTE_SENT))
    	;

#if (NDEBUG & NDEBUG_C400_PWRITE)
    printk("53C400w:     got last byte.\n");
    printk("53C400w: pwrite exiting with status 0, whoopee!\n");
#endif
    return 0;
}

/*====================================================================*/

#include "drivers/scsi/NCR5380.c"
