%
%  Fortran mode for jed
%
%   Loading this file, then executing 'fortran_mode' will start fortran mode
%   on the current buffer.  The only keys that get remapped are:
%     ^M (return)   which runs the fortan newline command
%     ^I (tab) which indents the current line
%     ^[; (escape semicolon) runs the fortran comment command
%     ^[: (escape colon) runs fortran uncomment command
%     ^[^M : start a continuation line

!if (is_defined ("Fortran_Continue_Char"))
{
   variable Fortran_Continue_Char = "&";      % default continuation char
}

!if (is_defined ("Fortran_Comment_String"))
{
   variable Fortran_Comment_String = "C ";
}

!if (is_defined ("Fortran_Indent_Amount"))
{
   variable Fortran_Indent_Amount = 2;
}

% goto beginning of line and skip past continuation char
define fortran_skip_label ()
{
   bol ();
   skip_chars ("0-9 \t");
   if (looking_at(Fortran_Continue_Char)) go_right(1);
   skip_white ();
}

% computes fortran indent
define fortran_get_indent ()
{
   variable col = 7;    %/*  at top of buffer it should be 7 n'est pas? */
   variable cse = CASE_SEARCH;
   CASE_SEARCH = 0;

   push_spot ();
   
   while (up_1 ())
     {
	bol_skip_white();
	if (eolp() or looking_at(Fortran_Continue_Char)) continue;
	fortran_skip_label ();
	col = what_column ();
     
	if (col == 1) continue;
	
	if (looking_at("do ") 
	    or looking_at("else"))
	  col += Fortran_Indent_Amount;
	else if (looking_at("if ") or looking_at("if("))
	  {
	     if (ffind ("then")) col += Fortran_Indent_Amount;
	  }
	break;
     }
   
  % now check current line
   pop_spot ();
   push_spot ();
   fortran_skip_label ();
  
   if (looking_at("end") or
       looking_at("continue") or
       looking_at("else")) col -= Fortran_Indent_Amount;
  
   if (col < 7) col = 7;
   pop_spot ();
   CASE_SEARCH = cse;
   col;
}

% used below-- skips range 
% assumes we are after the label or continuation char 
% and indents the rest to goal
define fortran_tough_indent (goal)
{
   skip_chars ("0-9");
   trim ();
   if (looking_at (Fortran_Continue_Char))
     {
	insert_spaces (6 - what_column());
	go_right(1); trim();
	goal += Fortran_Indent_Amount;
     }
   
   insert_spaces (goal - what_column());
}



% fortran indent routine
define fortran_indent ()
{
   variable col, ch, n, goal;
   push_spot ();
   goal = fortran_get_indent ();
   bol_skip_white ();
   col = what_column ();
   
   switch (char(what_char()))
     {
	isdigit (()) :     %  label

	if (col >= 6)
	  {
	     bol (); trim ();
	     insert_single_space ();
	  }
	
	fortran_tough_indent (goal);
     }
     {
	case Fortran_Continue_Char :  % continuation character
	bol (); trim (); insert ("     ");
	fortran_tough_indent (goal);
     }
     {
	pop (); not (bolp()) or eolp ():                  % general case
	bol (); trim ();
	goal--; insert_spaces (goal);
     }
   pop_spot ();   
   skip_white ();
}

define fortran_is_comment ()
{
   bol ();
   skip_chars (" \t0-9");
   bolp () and not (eolp());
}


define fortran_newline ()
{
   variable p, cont;
   
   if (bolp ())
     {
	newline ();
	return;
     }

   fortran_indent ();
   push_spot ();
   bskip_white (); trim ();
   
   if (what_column () > 72)
     {
	push_spot ();
	bol_skip_white();
	!if (bolp()) message ("Line exceeds 72 columns.");
	pop_spot ();
     }
   
   p = POINT;
   bskip_chars("-+*=/,(");
   
   cont = (p != POINT);
   
   if (fortran_is_comment ()) cont = 0;
   
   bol_skip_white ();
   if (looking_at("data ")) cont = 0;
   
   pop_spot ();
   
   newline ();
   insert_single_space ();
   if (cont) insert(Fortran_Continue_Char);
   fortran_indent ();
}


define fortran_continue_newline ()
{
   fortran_newline ();
   
   push_spot ();
   bol_skip_white ();
   if (looking_at(Fortran_Continue_Char)) pop_spot ();
   else
     {
	insert (Fortran_Continue_Char);
	pop_spot ();
	fortran_indent ();
	go_right(1);
	skip_white ();
     }
}

%
%   electric labels
%
define fortran_electric_label ()
{
   variable ch, col;
   insert_char (LAST_CHAR);
   push_spot ();
   bol_skip_white (); % test for comment
   if (bolp ()) pop_spot ();
   else
     {
	skip_chars ("0-9"); trim ();
	pop_spot ();
	fortran_indent ();
     }
}


% fortran comment/uncomment functions

define fortran_uncomment ()
{
   push_spot ();
   if (fortran_is_comment ())
     {
	bol ();
	if (looking_at (Fortran_Comment_String)) 
	  deln (strlen (Fortran_Comment_String));
	else del ();
     }
   
   fortran_indent ();
   pop_spot ();
   go_down_1 ();
}

define fortran_comment ()
{
   !if (fortran_is_comment ())
     {
	push_spot ();
	bol ();
	insert (Fortran_Comment_String);
     }
   pop_spot ();
   go_down_1 ();
}

 
%
% Look for beginning of current subroutine/function
%
define fortran_beg_of_subprogram ()
{
   variable cas = CASE_SEARCH;
 
   CASE_SEARCH = 0;
   do
     {
	bol_skip_white ();
	if (POINT)
	  {
	     if (looking_at ("program")
		 or looking_at ("function")
		 or looking_at ("subroutine")) break;
	  }
     }
   while (up_1 ());
   CASE_SEARCH = cas;
}
 
%
% Look for end of current subroutine/function
%
define fortran_end_of_subprogram ()
{
   variable cas = CASE_SEARCH;
   CASE_SEARCH = 0;
   
   do 
     {
	bol_skip_white ();
	if (looking_at ("end"))
	  {
	     go_right (3);
	     skip_white (); 
	     if (eolp ()) break;
	  }
     }
   while (down_1 ());
   CASE_SEARCH = cas;                       %  restore search mode
}
 
%
% shows a ruler for FORTRAN source. Press any key to get rid of
%
define fortran_ruler ()
{
   variable c = what_column ();
   variable r = window_line ();
   
   bol ();
   push_mark ();
   insert ("    5 7 10   15   20   25   30   35   40   45   50   55   60   65   70\n");
   insert ("{    }|{ |    |    |    |    |    |    |    |    |    |    |    |    | }\n");
   
   goto_column (c);
   if (r <= 2) r = 3;
   recenter (r);
   message ("Press SPACE to get rid of the ruler.");
   update (1);
   getkey (); pop ();
   bol ();
   del_region ();
   goto_column (c);
   flush_input ();
   recenter (r);
}
 
define fortran_prev_next_statement (dirfun)
{
   while (dirfun(1))
     {
	bol ();
	skip_chars ("^0-9 \t");
	!if (POINT) break;
     }
    goto_column_best_try (7); pop ();
}
%
% moves cursor to the next statement, skipping comment lines
%
define fortran_next_statement ()
{
   fortran_prev_next_statement (&down);
}

 
%
% moves cursor to the previous fortran statement, skipping comments
%
define fortran_previous_statement ()
{
   fortran_prev_next_statement (&up);
}


%
% main entry point into the fortran mode
%

!if (is_defined("Fortran_Mode"))
{
   variable Fortran_Mode = "Fortran";
   make_keymap (Fortran_Mode);
   definekey ("fortran_newline", "^M",  Fortran_Mode);
   definekey ("indent_line",	"\t",	Fortran_Mode);
   definekey ("fortran_comment",	"^[;",	Fortran_Mode);
   definekey ("fortran_uncomment",	"^[:",	Fortran_Mode);
   definekey ("fortran_continue_newline",	"^[^M",	Fortran_Mode);
   definekey ("self_insert_cmd",	"'",	Fortran_Mode);
   definekey ("self_insert_cmd",	"\"",	Fortran_Mode);
   definekey ("fortran_beg_of_subprogram", "^[^A", Fortran_Mode);
   definekey ("fortran_end_of_subprogram", "^[^E", Fortran_Mode);
   definekey ("fortran_ruler", "^C^R", Fortran_Mode);
   definekey ("fortran_next_statement", "^C^N", Fortran_Mode);
   definekey ("fortran_previous_statement", "^C^P", Fortran_Mode);
   _for (0, 9, 1)
     {
	=$1;
	definekey ("fortran_electric_label", string($1), Fortran_Mode);
     }
}

% Set up syntax table
$1 = "FORTRAN";
create_syntax_table ($1);
define_syntax ("!", "", '%', $1);
define_syntax ("([", ")]", '(', $1);
define_syntax ('"', '"', $1);
define_syntax ('\'', '\'', $1);
% define_syntax ('\\', '\\', $1);
define_syntax ("0-9a-zA-Z_", 'w', $1);        % words
define_syntax ("-+0-9eEdD", '0', $1);   % Numbers
define_syntax (",.", ',', $1);
define_syntax ('D', '#', $1);
define_syntax ("-+/*=", '+', $1);
set_syntax_flags ($1, 1 | 2);


% Fortran 77 keywords + include, record, structure, while:
% backspace block
% call character common complex continue
% data dimension do double
% else end enddo endfile endif entry equivalence exit external
% format function
% goto 
% if implicit include inquire integer intrinsic
% logical
% parameter pause precision program
% real return rewind
% save stop subroutine
% then
% while

() = define_keywords ($1, "dogoifto", 2);
() = define_keywords ($1, "end", 3);
() = define_keywords ($1, "calldataelseexitgotoopenreadrealsavestopthen", 4);
() = define_keywords ($1, "blockcloseenddoendifentrypauseprintwhilewrite", 5);
() = define_keywords ($1, "commondoubleformatrecordreturnrewind", 6);
() = define_keywords ($1, "complexendfileincludeinquireintegerlogicalprogram", 7);
() = define_keywords ($1, "continueexternalfunctionimplicit", 8);
() = define_keywords ($1, "backspacecharacterdimensionintrinsicparameterprecisionstructure", 9);
() = define_keywords ($1, "subroutine", 10);
() = define_keywords ($1, "equivalence", 11);





%!% Mode designed for the purpose of editing FORTRAN files.
%!% After the mode is loaded, the hook 'fortran_hook' is called.
%!% Useful functions include
%!% 
%!%  Function:                    Default Binding:
%!%   indent_line                       TAB
%!%   fortran_newline                   RETURN  
%!%     indents current line, inserts newline and indents it.
%!%   fortran_continue_newline          ESC RETURN
%!%     indents current line, and creates a continuation line on next line.
%!%   fortran_comment                   ESC ;
%!%     comments out current line
%!%   fortran_uncomment                 ESC :
%!%     uncomments current line
%!%   fortran_electric_label            0-9
%!%     Generates a label for current line or simply inserts a digit.
%!%   fortran_next_statement            ^C^N
%!%     moves to next fortran statementm skips comment lines
%!%   fortran_previous_statement        ^C^P
%!%     moves to previous fortran statement, skips comment lines
%!%   fortran_ruler                     ^C^R
%!%     inserts a ruler above the current line. Press any key to continue
%!%   fortran_beg_of_subprogram         ESC ^A
%!%     moves cursor to beginning of current subroutine/function
%!%   fortran_end_of_subprogram         ESC ^E
%!%     moves cursor to end of current subroutine/function
%!%  
%!% Variables include:
%!%   Fortran_Continue_Char   --- character used as a continuation character.  
%!%     By default, its value is "&"
%!%   Fortran_Comment_String  --- string used by 'fortran_comment' to 
%!%     comment out a line.  The default string is "C ";
%!%   Fortran_Indent_Amount   --- number of spaces to indent statements in 
%!%                               a block.  The default is 2.
define fortran_mode ()
{
   use_keymap (Fortran_Mode);
   setmode (Fortran_Mode, 0x4 | 0x10);
   use_syntax_table ("FORTRAN");
   set_buffer_hook ("indent_hook", "fortran_indent");
   runhooks ("fortran_hook");
}
