////////////////////////////////////////////////////////////////////////////////
//  Scene server implementation.                                              //  
//  LAST EDIT: Fri Dec 16 09:04:07 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART VR extension. Copying, distribution and     //
//  legal info is in the file COPYRGHT which should be distributed with this  //
//  file. If COPYRGHT is not available or for more info please contact:       //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright '94 '95 Ekkehard 'Ekki' Beier                                //
////////////////////////////////////////////////////////////////////////////////

#include "vr_yart.h"
#include "vr_scene.h"

const char *VRN_USER = "VR_User";
const char *VRN_SCENE = "VR_Scene";

RT_Vector vr_Position( 0, 0, 0 );
double vr_Angle = 3.14;
int vr_MaxUsers = 10;
RT_String vr_Msg = "";

int VR_Scene::loF, VR_Scene::loV;
int VR_Scene::rpF, VR_Scene::rpV;

RT_ParseEntry VR_Scene::table[] = {
    { "-login", RTP_SPECIAL, 0, 0, "A new {ARG 1 User} trys to enter the scene. The return value is user ID AND user number, or zero, if it is not possible. Second argument is the YART code of its {ARG 2 Representation}.", "String String" },
    { "-logout", RTP_INTEGER, (char*)&loV, &loF, "The user with this {ARG 1 ID} is goimg to leave the scene.", RTPS_INTEGER },
    { "-send", RTP_SPECIAL, 0, 0, "The {ARG 1 User} sends this {ARG 2 Text} to all users in the scene.", "Integer String" },
    { "-command", RTP_SPECIAL, 0, 0, "The {ARG 1 User} sends a {ARG 2 Command} to its representation. The name of the representation is hidden and will be appended automatically.", "Integer String" },
    { "-forward", RTP_SPECIAL, 0, 0, "The {ARG 1 User} wants to move forward about this {ARG 2 Distance}. A negative distance is meaning a backward movement.", "Integer Double" },
    { "-upward", RTP_SPECIAL, 0, 0, "The {ARG 1 User} wants to move upward about this {ARG 2 Distance}. A negative distance is meaning a downward movement.", "Integer Double" },
    { "-rotate", RTP_SPECIAL, 0, 0, "The {ARG 1 User} wants to rotate to the left around the global y axis by the {ARG 2 Angle}. A negative angle is meaning a rotation to the right.", "Integer Double" },
    { "-pick", RTP_SPECIAL, 0, 0, "The {ARG 1 User} picked an {ARG 2 Object}. The result is scene specific. The standard scene returns the complete name of a user if his representation was selected.", "Integer String" },
    { "-resetPosition", RTP_INTEGER, (char*)&rpV, &rpF, "The position of this {ARG 1 User} should be reset to the scene start position.", RTPS_INTEGER },
    { 0, RTP_END, 0, 0, 0, 0 }
};

// append additional scene commands:

int getUserByIDCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 2) {
	Tcl_AppendResult( ip, argv[0],": wrong # args: should be \"",
			 argv[0]," <uid>!\"",
			 0 );
	return TCL_ERROR;
    }
    int uid; int fl;
    if (RT_getInt( &argv[1], uid, fl )) {
	VR_User *us = vr_Scene->find(uid);
	if (us) Tcl_SetResult( ip, us->getName(), TCL_VOLATILE );
	return TCL_OK;
    }
    Tcl_AppendResult( ip, argv[0],": bad argument. Should be an integer.", 0 );
    return TCL_ERROR;
}

int getIDOfPrimitiveCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 2) {
	Tcl_AppendResult( ip, argv[0],": wrong # args: should be \"",
			 argv[0]," <prim>!\"",
			 0 );
	return TCL_ERROR;
    }
    VR_User *us = vr_Scene->find(argv[1]);
    if (us) {
	char tmp[10];
	RT_int2string( us->getID(), tmp );
	Tcl_SetResult( ip, tmp, TCL_VOLATILE );
    }
    return TCL_OK;
}

int sendText2userCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 3) {
	Tcl_AppendResult( ip, argv[0],": wrong # args: should be \"",
			 argv[0]," <uid> <msg>!\"",
			 0 );
	return TCL_ERROR;
    }
    int uid; int fl;
    if (RT_getInt( &argv[1], uid, fl )) {
	VR_User *us = vr_Scene->find(uid);
	if (us) vr_SceneCommunicator->sendText2user(us, argv[2]);
	return TCL_OK;
    }
    Tcl_AppendResult( ip, argv[0],": bad argument 1. Should be an integer.", 0 );
    return TCL_ERROR;
}

int sendCmd2userCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 3) {
	Tcl_AppendResult( ip, argv[0],": wrong # args: should be \"",
			 argv[0]," <uid> <msg>!\"",
			 0 );
	return TCL_ERROR;
    }
    int uid; int fl;
    if (RT_getInt( &argv[1], uid, fl )) {
	VR_User *us = vr_Scene->find(uid);
	if (us) vr_SceneCommunicator->sendCmd2user(us, argv[2]);
	return TCL_OK;
    }
    Tcl_AppendResult( ip, argv[0],": bad argument 1. Should be an integer.", 0 );
    return TCL_ERROR;
}

int sendBlob2userCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 5) {
	Tcl_AppendResult( ip, argv[0],": wrong # args: should be \"",
			 argv[0]," <uid> <local_file> <remote_file> <flag>!\"",
			 0 );
	return TCL_ERROR;
    }
    int uid; int fl;
    if (RT_getInt( &argv[1], uid, fl )) {
	VR_User *us = vr_Scene->find(uid);
	if (us) vr_SceneCommunicator->sendBlob2user(us, argv[2], argv[3], strcmp( argv[4], "0") );
	return TCL_OK;
    }
    Tcl_AppendResult( ip, argv[0],": bad argument 1. Should be an integer.", 0 );
    return TCL_ERROR;
}


int pickCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 3) {
	Tcl_AppendResult( ip, argv[0],": wrong # args: should be \"",
			 argv[0]," <uid> <prim>!\"",
			 0 );
	return TCL_ERROR;
    }
    int uid; int fl;
    if (RT_getInt( &argv[1], uid, fl )) {
	VR_User *us1 = vr_Scene->find(uid);
	VR_User *us2 = vr_Scene->find(argv[2]);
	if (!us1) return TCL_ERROR;
	if (us2) {
	    RT_String s = "You picked ";
	    s += us2->getName();
	    vr_SceneCommunicator->sendText2user( us1, s );
	    s = "You were picked by ";
	    s += us1->getName();
	    vr_SceneCommunicator->sendText2user( us2, s );
	}
	else 
	    vr_SceneCommunicator->sendText2user( us1, "You didn't hit a user." );
	
	return TCL_OK;
    }
    Tcl_AppendResult( ip, argv[0],": bad argument 1. Should be an integer.", 0 );
    return TCL_ERROR;
}

int evalAndBroadcastCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 2) {
	Tcl_AppendResult( ip, argv[0],": wrong # args: should be \"",
			 argv[0]," <cmd>!\"",
			 0 );
	return TCL_ERROR;
    }
    vr_Scene->evalAndBroadcast( RT_String( argv[1] ) );
    return TCL_OK;
}

// Tcl commands to setup the scene:

int positionCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 2) {
	Tcl_AppendResult( ip, argv[0],": wrong # args: should be \"",
			 argv[0]," <double[3]>!\"",
			 0 );
	return TCL_ERROR;
    }
    RT_Vector v; int fl;
    if (RT_getVector( &argv[1], v, fl )) {
	vr_Position = v;
	return TCL_OK;
    }
    Tcl_AppendResult( ip, argv[0],": bad argument. Should be a double vector.", 0 );
    return TCL_ERROR;
}

int angleCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 2) {
	Tcl_AppendResult( ip, argv[0],": wrong # args: should be \"",
			 argv[0]," <double>!\"",
			 0 );
	return TCL_ERROR;
    }
    double d; int fl;
    if (RT_getDouble( &argv[1], d, fl)) {
	vr_Angle = d;
	return TCL_OK;
    }
    Tcl_AppendResult( ip, argv[0],": bad argument. Should be a double.", 0 );
    return TCL_ERROR;
}

int maxUsersCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 2) {
	Tcl_AppendResult( ip, argv[0],": wrong # args: should be \"",
			 argv[0]," <int>!\"",
			 0 );
	return TCL_ERROR;
    }
    int i; int fl;
    if (RT_getInt( &argv[1], i, fl )) {
	vr_MaxUsers = i;
	return TCL_OK;
    }
    Tcl_AppendResult( ip, argv[0],": bad argument. Should be an integer.", 0 );
    return TCL_ERROR;
}

int msgCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 2) {
	Tcl_AppendResult( ip, argv[0],": wrong # args: should be \"",
			 argv[0]," <msg>!\"",
			 0 );
	return TCL_ERROR;
    }
    vr_Msg = argv[1];
    return TCL_OK;
}

// the scene:

VR_Scene::VR_Scene(const char *scd): RT_Object( "server" ), users( 0 ), dumpFile(20) {
    // extract scene name from scd:
    char *tscd = (char*)&scd[ strlen( scd )];
    while (--tscd > scd) 
	if (*tscd == '/' ) {
	    tscd++;
	    break;
	}
    // in case the first char was a '/':
    if (*tscd == '/' ) tscd++;
    xscene = tscd;

    sc = (RT_Scene*)RT_Object::getObject( "sc" );
    if (!sc) sc = new RT_Scene( "sc" );

    RT_String ds(100);
    ds = "source "; 
    ds += scd; 
    
    // overload some commands:
    RT_eval( "foreach cmd {exit cd pwd exec} {proc $cmd {args} \"puts stdout {VR_SCENE message: Try to call $cmd}\"" );

    // append additional scene commands:
    RTM_command( "VR_getUserByID", getUserByIDCMD );
    RTM_command( "VR_getIDOfPrimitive", getIDOfPrimitiveCMD );
    RTM_command( "VR_sendText2user", sendText2userCMD );
    RTM_command( "VR_sendCmd2user", sendCmd2userCMD );
    RTM_command( "VR_sendBlob2user", sendBlob2userCMD );
    RTM_command( "VR_sendBlob2user", sendBlob2userCMD );
    RTM_command( "VR_evalAndBroadcast", evalAndBroadcastCMD );
    RTM_command( "VR_pick", pickCMD );

    // append the setup commands:
    RTM_command( "VR_position", positionCMD );
    RTM_command( "VR_angle", angleCMD );
    RTM_command( "VR_maxUsers", maxUsersCMD );
    RTM_command( "VR_msg", msgCMD );

    // load scene description:
    if ( Tcl_GlobalEval( rt_Ip, (char*)ds ) != TCL_OK ) {
	fprintf( stderr, "Error: %s\n", rt_Ip->result );
	exit( -1 );
    }

    // remove the setup commands:
    Tcl_DeleteCommand( rt_Ip, "VR_position" );
    Tcl_DeleteCommand( rt_Ip, "VR_angle" );
    Tcl_DeleteCommand( rt_Ip, "VR_maxUsers" );
    Tcl_DeleteCommand( rt_Ip, "VR_msg" );

    rt_Output->messageVar( (char*)getSceneName(), "@", (char*)vr_Hostname, 
			  " initialized. Ready for visitors.", 0 );
}

void VR_Scene::dump(const char *fname) {
    FILE *fp = fopen( fname, "w" );
    if (!fp) rt_Output->fatalVar( "Couldn't open temporary file ", fname, "!", 0 ); 
    fprintf( fp, "#### YART/VR dump, Release %s ####\n", RT_GOOD_RELEASE );

    RT_TclObject::classes.print( fp );

    class LFunctoid: public RT_GeneralListFunctoid {
	FILE *fp;
	void exec( RT_GeneralListEntry *e, void * = 0 ) {
	    if (e->isA( RTN_PRIMITIVE )) ((RT_Primitive*)e)->printHierarchical( fp );
	    else if (e->isA( RTN_LIGHT )) ((RT_Light*)e)->print( fp );
	}
      public:
	LFunctoid( FILE *_fp ): fp( _fp ) {}
    } func( fp );
    sc->get_objects().doWithElements( &func );

    fprintf( fp, "# skip constructor: " );
    sc->RT_Scene::print( fp );
    fclose( fp );
}

VR_User *VR_Scene::find(int id) {
    class LFunctoid: public RT_GeneralListFunctoid {
	VR_User *xuser;
	int xid;
	void exec( RT_GeneralListEntry *e, void* = 0 ) {
	    if (((VR_User*)e)->getID() == xid ) xuser = (VR_User*)e;
	}
      public:
	LFunctoid( int id ): xuser( 0 ), xid( id ) {}
	VR_User *user() const { return xuser; }
    } func( id );
    doWithElements( &func );
    return func.user();
}

VR_User *VR_Scene::find(const char *pname) {
    if ( pname[0] == 'u' && pname[1] == 's' ) {
	int no;
	if ( RT_string2int( &pname[2], no )) {
	    class LFunctoid2: public RT_GeneralListFunctoid {
		// this class is named "LFunctoid2" instead of
		// "LFuntoid" to avoid name conflicts on some
		// compilers with the other find() method
		int no;
		VR_User *us; 
		void exec( RT_GeneralListEntry *e, void * = 0 ) {
		    if (((VR_User*)e)->getNumber() == no) us = (VR_User*)e;
		}
	      public:
		LFunctoid2( int _no): no( _no ), us( 0 ) {}
		VR_User *getUser() { return us; }
	    } func( no );
	    doWithElements( &func );
	    return func.getUser();
	}
	return 0;
    }
    return 0;
}
	
void VR_Scene::broadcast( const RT_String &s, VRE_Broadcast mode ) const {
    class LFunctoid: public RT_GeneralListFunctoid {
	VRE_Broadcast mode;
	void exec( RT_GeneralListEntry *e, void *txt) {
	    if (mode == VR_MESSAGE)
		vr_SceneCommunicator->sendText2user( (VR_User*)e, (char*)txt );
	    else 
		vr_SceneCommunicator->sendCmd2user( (VR_User*)e, (char*)txt );
	}
      public:
	LFunctoid( VRE_Broadcast _mode ): mode( _mode ) {}
    } func( mode );
    doWithElements( &func, (void*)(char*)s );
    if (mode == VR_MESSAGE) rt_Output->message( (char*)s );
}

VR_User *VR_Scene::login( const char *name, const char *desc ) {
    rt_Output->messageVar( name, " wants to login.", 0 );

    if ( users == vr_MaxUsers ) {
	rt_Output->error( "Sorry, there are too many users in the scene. Try later again." );
	return 0;
    } 

    // look if this user is already in the scene:
    RT_GeneralListElem *tmp = root;
    if (tmp) do if (((VR_User*)tmp->elem)->getName() == name ) {
	name = 0;
	break;
    }
    while (tmp = tmp->next);
    if ( !name ) {
	rt_Output->error( "You are already in the scene!" );
	return 0;
    } 

    // get the next free user number:
    class LFunctoid: public RT_GeneralListFunctoid {
	int fl, no;
	void exec(RT_GeneralListEntry *e, void * = 0 ) {
	    if (((VR_User*)e)->getNumber() == no ) fl = 1;
	}
      public:
	LFunctoid(): no( 0 ), fl( 0 ) {}
	void incr() { no++; fl = 0;}
	int get() const { return fl; }
	// return 1 if this number is already used 
    } func;

    int no = 0;
    doWithElements( &func );
    while ( func.get() ) {
	no++;
	func.incr();
	doWithElements( &func );
    }

    // create a dump:
    char *tmpName = new char[ strlen( VR_TMPLATE ) + 1];
    strcpy( tmpName, VR_TMPLATE );
    mktemp( tmpName );
    dumpFile = tmpName;
    delete tmpName;

    dump( dumpFile );

    // eval the class description:
    cdesc = desc;
    char str[5];
    RT_int2string( no, str );
    RT_String str2 = "US";
    str2 += str;
    cdesc.replace( "<YART_VR>", (char*)str2, 100 );
    if (Tcl_GlobalEval( rt_Ip, (char*)cdesc) != TCL_OK ) {
	rt_Output->errorVar( "Bad syntax of class description: ", rt_Ip->result, 0 );  
	return 0;
    }

    // create the representation of the user:
    str2 = "US";
    str2 += str; 
    str2 += " us";
    str2 += str;
    if (Tcl_GlobalEval( rt_Ip, (char*)str2) != TCL_OK ) {
	rt_Output->errorVar( "Bad syntax in class description: ", rt_Ip->result, 0 );
	return 0;
    }

    // get a unique uid (but not 0): 
    int uid = (int)random();
    while ( !uid || find( uid )) uid++;
    VR_User *us;
    append( us = new VR_User( name, uid, no ) );
    users++;

    return us;
}

void VR_Scene::newUser(VR_User *us) {
    RT_String hello(150);
    hello = "vr_User -message \"This is ";
    hello += getSceneName();
    hello += '@';
    hello += vr_Hostname;
    hello += ".\"; set vr_UID ";
    char is[10];
    sprintf( is, "%i", us->getID() );
    hello += is;
    
    hello += "; set vr_NO ";
    sprintf( is, "%i", us->getNumber() );
    hello += is;
    
    vr_SceneCommunicator->sendCmd2user( us, hello );
    vr_SceneCommunicator->sendBlob2user( us, dumpFile, "init.dmp", 1 );
    vr_SceneCommunicator->sendCmd2user( us, "source $env(HOME)/tmp/init.dmp" );
     
    // remove the dump file:
    RT_String s(100);
    s = "rm -f ";
    s += dumpFile;
    system( s );

    s = "User ";
    s += us->getName(); 
    s += " logged in.";
    broadcast( s, VR_MESSAGE );
    
    if (vr_Msg != "" ) vr_SceneCommunicator->sendText2user( us, vr_Msg );

    char str[5];
    RT_int2string( us->getNumber(), str );
    RT_String str2 = cdesc;
    str2 += "\nUS";
    str2 += str; 
    str2 += " us";
    str2 += str;
    broadcast( str2, VR_COMMAND );
    
    // insert the new primitive into the scene:
    str2 = "sc -insert us";
    str2 += str; 

    // translate the primitive to the scene specific origin:
    str2 += "; us";
    str2 += str;
    str2 += " -translate ";
    static char tmps[50];
    RT_vector2string( vr_Position, tmps );
    str2 += tmps;
    us->position = vr_Position;

    // rotate the primitive:
    str2 += "; us";
    str2 += str;
    str2 += " -rotate { 0 ";
    RT_double2string( vr_Angle, tmps );
    str2 += tmps;
    str2 += " 0}\n";
    us->angle = vr_Angle;

    evalAndBroadcast( str2 );
}

void VR_Scene::logout( int uid ) {
    VR_User *user = find( uid );
    if (!user) {
	rt_Output->error( "Bad user ID!" );
	return;
    }
    users--;

    killUser( user );
}

void VR_Scene::killUser(VR_User *user) {
    remove( user );    

    vr_SceneCommunicator->killUser( user );

    // broadcast a message:
    RT_String s( 30 );
    s = "User ";
    s += user->getName(); 
    s += " logged out.";
    broadcast( s, VR_MESSAGE );

    // delete the representation:
    s = "delete us";
    char tmp[5];
    sprintf( tmp, "%i", user->getNumber() );
    s += tmp;

    evalAndBroadcast( s );

    delete user;
}

void VR_Scene::send( int uid, char *txt ) {
    VR_User *user = find( uid );
    if (!user) {
	rt_Output->error( "Bad user ID!" );
	return;
    }
    RT_String s = user->getName(); 
    s += ": ";
    s += txt;
    broadcast( s, VR_MESSAGE );
}

void VR_Scene::command( int uid, char *txt ) {
    VR_User *user = find( uid );
    if (!user) {
	rt_Output->error( "Bad user ID!" );
	return;
    }

    RT_String s = "us";
    char tmp[10];
    sprintf( tmp, "%i", user->getNumber() );
    s += tmp;
    s += ' ';
    s += txt;

    evalAndBroadcast( s );
}

void VR_Scene::pick(int uid, char *objName) {
    char tmp[10];
    RT_int2string( uid, tmp );
    Tcl_VarEval( rt_Ip, "VR_pick ", tmp, " ", objName, 0 ); 
}

void VR_Scene::resetPosition(int uid) {
    VR_User *us = find( uid );
    if (!us) {
	rt_Output->error( "Bad user ID!" );
	return;
    }
    char str[5];
    RT_int2string( us->getNumber(), str );

    // translate the primitive to the scene specific origin:
    RT_String str2 = "us";
    str2 += str;
    str2 += " -matrix {1 0 0 0  0 1 0 0  0 0 1 0  0 0 0 1}; us";
    str2 += str;
    str2 += " -translate ";
    static char tmps[50];
    RT_vector2string( vr_Position, tmps );
    str2 += tmps;
    us->position = vr_Position;

    // rotate the primitive:
    str2 += "; us";
    str2 += str;
    str2 += " -rotate { 0 ";
    RT_double2string( vr_Angle, tmps );
    str2 += tmps;
    str2 += " 0}";
    us->angle = vr_Angle;

    evalAndBroadcast( str2 );
}

void VR_Scene::forward( int uid, double val ) {
    VR_User *user = find( uid );
    if (!user) {
	rt_Output->error( "Bad user ID!" );
	return;
    }

    char nstr[5];
    RT_int2string( user->getNumber(), nstr );

    static char tmps[10];

    static RT_String s( 100 );
    // reset the rotation:
    s = "us";
    s += nstr;
    s += " -rotate { 0 ";
    RT_double2string( -user->angle, tmps ); 
    s += tmps;
    s += " 0}; us";
    s += nstr;

    // set new translation:
    double dx = val * sin( user->angle );
    double dz = val * cos( user->angle );
    s += " -translate {";
    RT_double2string( dx, tmps ); s += tmps;
    s += " 0 ";
    RT_double2string( dz, tmps ); s += tmps;
    s += "}; us";

    // write rotation back:
    s += nstr;
    s += " -rotate { 0 ";
    RT_double2string( user->angle, tmps ); s += tmps;
    s += " 0}";

    evalAndBroadcast( s );
}

void VR_Scene::upward( int uid, double val ) {
    VR_User *user = find( uid );
    if (!user) {
	rt_Output->error( "Bad user ID!" );
	return;
    }

    char nstr[5];
    RT_int2string( user->getNumber(), nstr );

    static char tmps[10];

    RT_String s( 20 );
    // add height distance:
    s = "us";
    s += nstr;
    s += " -translate { 0 ";
    user->position.y += val;
    RT_double2string( user->position.y, tmps ); s += tmps;
    s += " 0}";

    evalAndBroadcast( s );
}

void VR_Scene::rotate( int uid, double val ) {
    VR_User *user = find( uid );
    if (!user) {
	rt_Output->error( "Bad user ID!" );
	return;
    }

    char nstr[5];
    RT_int2string( user->getNumber(), nstr );

    static char tmps[10];

    static RT_String s( 100 );

    // reset the rotation:
    s = "us";
    s += nstr;
    s += " -rotate { 0 ";
    RT_double2string( -user->angle, tmps ); 
    s += tmps;
    s += " 0}; us";
    s += nstr;

    // set new rotation:
    s += " -rotate { 0 ";
    user->angle += val;
    RT_double2string( user->angle, tmps ); s += tmps;
    s += " 0}";

    evalAndBroadcast( s );
}

int VR_Scene::objectCMD(char *argv[]) {
    RT_parseTable( argv, table );

    int ret = 0;
    { 	
	int start, diff1 = 0, diff2 = 0;
	char *uname, *desc;
	if ((start = RT_findString( argv, "-login" )) >= 0) 
	    if ( RT_getString( &argv[ start + 1 ], uname, diff1) ) 
		if ( RT_getString( &argv[ start + 2 ], desc, diff2) ) {
		    static char tmp[10];
		    VR_User *u = login( uname, desc );
		    if (u) {
			RT_int2string( u->getID(), tmp ); 
			RT_Object::result( tmp ); 
			RT_int2string( u->getNumber(), tmp ); 
			RT_Object::result( tmp ); 
		    } 
		    else RT_Object::result( "0" );
		    ret++;
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    if (loF) {
	logout( loV );
	ret++;
    }

    if (rpF) {
	resetPosition( rpV );
	ret++;
    }

    { 	
	int start, diff1 = 0, diff2 = 0;
	int uid; char *txt;
	if ((start = RT_findString( argv, "-send" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], uid, diff1) ) 
		if ( RT_getString( &argv[ start + 2 ], txt, diff2) ) {
		    send( uid, txt ); 
		    ret++;
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	
	int start, diff1 = 0, diff2 = 0;
	int uid; char *txt;
	if ((start = RT_findString( argv, "-command" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], uid, diff1) ) 
		if ( RT_getString( &argv[ start + 2 ], txt, diff2) ) {
		    command( uid, txt ); 
		    ret++;
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	
	int start, diff1 = 0, diff2 = 0;
	int uid; double val;
	if ((start = RT_findString( argv, "-forward" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], uid, diff1) ) 
		if ( RT_getDouble( &argv[ start + 2 ], val, diff2) ) {
		    forward( uid, val ); 
		    ret++;
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	
	int start, diff1 = 0, diff2 = 0;
	int uid; double val;
	if ((start = RT_findString( argv, "-upward" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], uid, diff1) ) 
		if ( RT_getDouble( &argv[ start + 2 ], val, diff2) ) {
		    upward( uid, val ); 
		    ret++;
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	
	int start, diff1 = 0, diff2 = 0;
	int uid; double val;
	if ((start = RT_findString( argv, "-rotate" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], uid, diff1) ) 
		if ( RT_getDouble( &argv[ start + 2 ], val, diff2) ) {
		    rotate( uid, val ); 
		    ret++;
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	
	int start, diff1 = 0, diff2 = 0;
	int uid; char *sel;
	if ((start = RT_findString( argv, "-pick" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], uid, diff1) ) 
		if ( RT_getString( &argv[ start + 2 ], sel, diff2) ) {
		    pick( uid, sel ); 
		    ret++;
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    return ret;
};

VR_User *VR_Scene::findInvalidUser() {
    RT_GeneralListElem *tmp = root;
    if (!tmp) return 0;
    do if ( !((VR_User*)tmp->elem)->isValid() ) {
	if (tmp->prev) tmp->prev->next = tmp->next;
	else root = tmp->next;
	if (tmp->next) tmp->next->prev = tmp->prev;
	else last = tmp->prev;
	return (VR_User*)tmp->elem;
    }
    while (tmp = tmp->next);
    return 0;
}

VR_Scene *vr_Scene;


