#ifndef __MANIPULATOR_H__
#define __MANIPULATOR_H__

////////////////////////////////////////////////////////////////////////////////
//  Sample Implementation of a manipulator device  that offers a lot of inter-//
//  action possibilities.                                                     //  
//  LAST EDIT: Thu Feb  9 15:55:39 1995 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRGHT which should be distributed with this  //
//  file. If COPYRGHT is not available or for more info please contact:       //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "device.h"
#include "../high/lookat.h"
#include "../high/usefprim.h"

typedef enum RT_ManipulatorMode {
    RTE_OBJECT_MODE = 0, // manipulate the object
    RTE_CURSOR_MODE, // manipulate the cursor
    RTE_CAMERA_MODE // manipulate the camera (currently not supported)
};

typedef enum RT_ManipulatorXform {
    RTE_SCALE = 0,
    RTE_ROTATE, 
    RTE_TRANSLATE, 
    RTE_ROT_TRANS     
};

typedef enum RT_ManipulatorAxis {
    RTE_NO_AXIS = 0,
    RTE_X_AXIS,
    RTE_Y_AXIS,
    RTE_XY_AXIS,
    RTE_Z_AXIS,
    RTE_XZ_AXIS,
    RTE_YZ_AXIS,
    RTE_XYZ_AXIS
};

typedef enum RT_ManipulatorStyle {
    RTE_SOLID = 0,
    RTE_WIREFRAME,
    RTE_INVISIBLE
};

#define RTD_MANIPULATOR_TRANSFORMATION_ROTATION_MASK 1
#define RTD_MANIPULATOR_TRANSFORMATION_TRANSLATION_MASK 2
#define RTD_MANIPULATOR_AXIS_X_MASK 1
#define RTD_MANIPULATOR_AXIS_Y_MASK 2
#define RTD_MANIPULATOR_AXIS_Z_MASK 4

// defines to correct the sensibiliy of the transformations for the spaceball
#define RTD_MANIPULATOR_SP_ROT_SENS 0.008
#define RTD_MANIPULATOR_SP_TRANS_SENS 0.008
#define RTD_MANIPULATOR_SP_SCALE_SENS 0.008

// defines to correct the sensibiliy of the transformations for the mouse
#define RTD_MANIPULATOR_M_ROT_SENS   4
#define RTD_MANIPULATOR_M_TRANS_SENS 4
#define RTD_MANIPULATOR_M_SCALE_SENS 2

extern const char *RTN_MANIPULATOR;

class RT_Manipulator: public RT_InputDevice {
    static RT_ParseEntry table[];

    //#### the statics for parameter parsing:
    static int modeF,modeG,axisF,axisG,transF,transG,curObjF,curObjG;
    static int boxF,boxV,boxG;
    static int drawStyleF,drawStyleG,drawStyleV;
    static int modeV, axisV, transV;
    static int autoUpdateF,autoUpdateV,autoUpdateG,rayF,resetMatrixF,sensF,sensG;
    static double sensV;
    static char *curObjV;
    static int mtF, mtG;
    static RT_Matrix mtV;

  protected:
    static void checkAxis(RT_Vector &, double, double, double, RT_ManipulatorAxis);
    
    // the cursor object
    RT_Primitive *xcursor;

    // autoUpdate: 
    // 0 = no camera update
    // 1 = camera rendering on each event
    int xautoUpdate;

    void makeAutoUpdate() {
	if (xautoUpdate && get_camera()) 
	    get_camera()->refresh();
    }
    virtual void updateFeedback();
    
    // the sensibility set the speed of the rotation, and translation
    // the initial value is 1:
    float xsensibility;
    // the active axis 
    
    RT_ManipulatorAxis xaxis;
    // the active draw style 

    RT_ManipulatorStyle xdrawStyle;
    // the current box type of the manipulator

    RT_BoxMode xbox; 
    // the current mode of the manipulator

    RT_ManipulatorMode xmode; 
    // the active transformation

    RT_ManipulatorXform xtransformation;

    RT_Primitive *xcurrentObject, *oldCurrentObject;
    // the old mouse positon to calculate the difference
    double oldx, oldy;
    
    int xchanged;
    int get_changed() { return xchanged; }
    void changed(int val) {xchanged = val; } 
    
  public:
    void event(RT_Event*); 

    void objectKilled(RT_Object *o) {
	if ( o == get_currentObject()) currentObject( (RT_Primitive*)-1 );
	RT_InputDevice::objectKilled( o );
    }

    //#### Tcl/C++ methods:
    RT_Manipulator(char *);
    ~RT_Manipulator() { 
	currentObject( (RT_Primitive*)0 );

	if (get_camera()) {
	    get_camera()->feedback->remove( xcursor );
	    get_camera()->refresh();
	}

	delete xcursor; 
    }

    const char *get_class() const { return RTN_MANIPULATOR; } 
    const char *get_description() const { return "A manipulator object can be used for translating, scaling, rotating an object. Furthermore it has to be coupled to a camera object. The manipulator object is driven by spaceball and mouse events."; }
    int isA(const char *c) const { return RT_InputDevice::isA( c ) || RTM_isA( RTN_MANIPULATOR, c ); }
    void father(RT_Object *);
    void print(FILE *) const;

    void mode( RT_ManipulatorMode);
    RT_ManipulatorMode get_mode() const {return xmode;}

    void box(RT_BoxMode); 
    RT_BoxMode get_box() const { return xbox; }

    void transformation( RT_ManipulatorXform t) { xtransformation= t; changed(1);}
    RT_ManipulatorXform get_transformation() const {return xtransformation;}

    void axis(RT_ManipulatorAxis a) {xaxis = a; changed(1);}
    RT_ManipulatorAxis get_axis() const {return xaxis;}

    void drawStyle(RT_ManipulatorStyle s); 
    RT_ManipulatorStyle get_drawStyle() const {return xdrawStyle;}

    void sensibility(float s) {xsensibility = s; changed(1);}
    float get_sensibility() const {return xsensibility;}

    void raytrace() {
  	if (get_camera())  {
  	    if ( get_camera()->isA( RTN_LOOKAT_CAMERA )) {
  		RT_LookatCamera *cam = ( RT_LookatCamera *)get_camera();
 		RT_LookatCameraMode m= cam->get_mode();
  		cam->mode( RTE_LC_RAY_TRACING );
  		cam->rendering();
 		cam->mode(m);
 	    }
 	}
    }

    void resetMatrix() {
	if (xcurrentObject) {
	    RT_Matrix id;
	    xcurrentObject->matrix(id);
	    updateFeedback();
	}
	changed(1);
    }
    
    void autoUpdate(int u) { if (u) xautoUpdate = 1; else xautoUpdate =0; changed(1); }
    int  get_autoUpdate() const {return xautoUpdate;}
    
    virtual void currentObject(RT_Primitive *); 
    RT_Primitive *get_currentObject() const {return xcurrentObject;} 
    
    void matrix(const RT_Matrix &m) { xcursor->matrix( m ); }
    const RT_Matrix &get_matrix() const { return xcursor->get_matrix(); }

    //#### the tcl commands:
    static int classCMD(ClientData, Tcl_Interp *, int, char *[]);
    int objectCMD(char *[]);
};

#endif




