/******************************************************************************
 * --- bt_io.h --- Copyright (C) 1994 by Thomas Weidenfeller
 *
 * File ID:
 *	$Id: bt_io.h,v 1.2 1994/10/26 18:23:46 root Exp $
 *
 * Purpose:
 *	Macros for operating on the printer status and control port
 *
 * Copyright:
 *	Copyright (C) 1994 by T. Weidenfeller. All rights reserved
 *	
 *	You are free to copy, modify and distribute this software as you
 *	see fit, and to use it for any purpose, provided this copyright
 *	notice and the warranty disclaimer are included without any
 *	modification in all copies and modifications. You have to mark all
 *	modifications clearly and you are also commited to give propper
 *	acknowledgment of the original source if you include this software
 *	or parts of it in another software.
 *	
 * Warranty:
 *	The BiTronics device driver and the accompaning material is a free
 *	sample of no commercial value. It comes without any warranty:
 *	
 *	THERE IS NO WARRANTY OF ANY KIND WITH REGARDS TO THIS MATERIAL,
 *	INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *	MECHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. The author
 *	shall not be liable for errors contained in the material or for
 *	incidental or consequental damage in connection with the
 *	furnishing, performance, or use of this material.
 *	
 * Revisions: $Log: bt_io.h,v $
 *	Revision 1.2  1994/10/26  18:23:46  root
 *	General harmonisation (prep 1. ALPHA release).
 *
 * Revision 1.1  1994/10/23  19:37:43  root
 * Initial revision
 *
 *****************************************************************************/

#ifndef BT_IO_H_
#define BT_IO_H_

/******************************************************************************
 * Access to the IO ports
 *****************************************************************************/

#define BT_STAT_OFF	(1)	/* offset of status port from port base addr */
#define BT_CTRL_OFF	(2)	/* offset of control port	*/

#define BT_DATA_IN(minor)	inb_p(      bt_data[(minor)].base_addr)
#define BT_DATA_OUT(minor, b)	outb_p((b), bt_data[(minor)].base_addr)
#define BT_STAT_IN(minor)	inb_p(      bt_data[(minor)].base_addr + BT_STAT_OFF)
#define BT_CTRL_IN(minor)	inb_p(      bt_data[(minor)].base_addr + BT_CTRL_OFF)
#define BT_CTRL_OUT(minor, b)	outb_p((b), bt_data[(minor)].base_addr + BT_CTRL_OFF)


/******************************************************************************
 * The official status port bits
 *****************************************************************************/

#define BT_STAT_RSVD1	(0x01)		/* Reserved			*/
#define BT_STAT_RSVD2	(0x02)		/* Reserved			*/
#define BT_STAT_nIRQST	(0x04)		/* PS/2 only: Low IRQ ACK	*/
#define BT_STAT_nERROR	(0x08)		/* Low:  Printer error		*/
#define BT_STAT_SLCT	(0x10)		/* High: Printer selected	*/
#define BT_STAT_PE	(0x20)		/* High: Paper emptry		*/
#define BT_STAT_nACK	(0x40)		/* Low (0.5us): Char accepted	*/
#define BT_STAT_nBUSY	(0x80)		/* Low:  Printer busy		*/


/******************************************************************************
 * Normal printer status bit definitions
 *****************************************************************************/

/*
 * printer ready to accept printing data
 * (no error, selected, no paper empty, no acknowledging, not busy)
 */
#define BT_PRN_READY		(BT_STAT_nERROR | BT_STAT_SLCT |              BT_STAT_nACK | BT_STAT_nBUSY)
#define BT_PRN_READY_MASK	(BT_STAT_nERROR | BT_STAT_SLCT | BT_STAT_PE | BT_STAT_nACK | BT_STAT_nBUSY)
#define BT_PRN_READY_MSG	"ready"

/*
 * No printer at the port	
 * (no error, selected, paper empty (but busy), no acknowledging) 
 * NOTE: Most PCs are not capable of detecting this status because
 *       if there is no printer at the port there status lines
 *	 float (randomly) around...
 */
#define BT_PRN_NO_PRN		(BT_STAT_nERROR	| BT_STAT_SLCT | BT_STAT_PE | BT_STAT_nACK)
#define BT_PRN_NO_PRN_MASK	(BT_STAT_nERROR | BT_STAT_SLCT | BT_STAT_PE | BT_STAT_nACK | BT_STAT_nBUSY)
#define	BT_PRN_NO_PRN_MSG	"no printer at this port"

/*
 * Paper out
 */
#define BT_PRN_PE		(BT_STAT_PE)
#define BT_PRN_PE_MASK		(BT_STAT_PE)
#define BT_PRN_PE_MSG		"out of paper"

/*
 * off-line
 */
#define BT_PRN_OFFLINE		(0x00)
#define BT_PRN_OFFLINE_MASK	(BT_STAT_SLCT)
#define BT_PRN_OFFLINE_MSG	"off-line"

/*
 * The traditional "on fire?" :-)
 */
#define BT_PRN_UNSPEC		(0x00)
#define BT_PRN_UNSPEC_MASK	(BT_STAT_nERROR)
#define BT_PRN_UNSPEC_MSG	"not ascertainable (on fire, eh?)"

/*
 * printer busy
 */
#define BT_PRN_BUSY		(0x00)
#define BT_PRN_BUSY_MASK	(BT_STAT_nBUSY)
#define BT_PRN_BUSY_MSG		"busy"

/*
 * printer simply timed out
 */
#define BT_PRN_ONLY_BUSY	(BT_STAT_nERROR	| BT_STAT_SLCT ) 
#define BT_PRN_ONLY_BUSY_MASK	(BT_STAT_nERROR | BT_STAT_SLCT | BT_STAT_PE |               BT_STAT_nBUSY)


/******************************************************************************
 * BiTronics printer status bit definitions
 *****************************************************************************/

/*
 * BT acknowledgment
 * (no error, selected, paper empty, acknowledging)
 */ 
#define BT_PRN_ACK		(BT_STAT_nERROR	| BT_STAT_SLCT | BT_STAT_PE)
#define BT_PRN_ACK_MASK		(BT_STAT_nERROR	| BT_STAT_SLCT | BT_STAT_PE | BT_STAT_nACK)
#define BT_PRN_ACK_MSG		"mode acknowledgment"

/*
 * BT adressing
 * (error, but not paper empty)
 */ 
#define BT_PRN_ADR		(0x00)
#define BT_PRN_ADR_MASK		(BT_STAT_nERROR	|                BT_STAT_PE )
#define BT_PRN_ADR_MSG		"ready for reading"

/* the following two are questionable in BT_HOST2PRN mode! */

/*
 * BT clock rises high (nibble ready to be read)
 */
#define BT_PRN_CLK_H		(BT_STAT_nACK)
#define BT_PRN_CLK_H_MASK	(BT_STAT_nACK)
#define BT_PRN_CLK_H_MSG	"data clock high"

/*
 * BT clock went low (still dta available)
 */
#define BT_PRN_CLK_L		(0x00)
#define BT_PRN_CLK_L_MASK	(BT_STAT_nACK)
#define BT_PRN_CLK_L_MSG	"data clock low"


/* the following should never be reached, but who knows with BiTronics??? */
/*
 * BT termination signalling start
 */
#define BT_PRN_TERM		(0x00)
#define BT_PRN_TERM_BITS	(BT_STAT_SLCT)
/* termination request response */
#define BT_PRN_TERM_MASK	(BT_STAT_SLCT | BT_STAT_nACK)
#define BT_PRN_TERM_MSG		"termination in progress (should never be visable to user)"

/*
 * the following two are used in special modes to detect
 * certain proceedings
 */
#define BT_PRN_DATA_MASK	(BT_STAT_nERROR)
#define BT_PRN_XFLAG_MASK	(BT_STAT_SLCT)


/******************************************************************************
 * Macro for the lookup tables
 *****************************************************************************/

/*
 * Macro to create table entry
 */
#define BT_PRN_STE(x)	{ BT_PRN_##x, BT_PRN_##x##_MASK, BT_PRN_##x##_MSG }

#if 0
#define BT_PRN_N(x)	(sizeof(x) / sizeof(BT_PRN_STAT))
#endif


/******************************************************************************
 *
 *****************************************************************************/

/*
 * test if specific status reached
 */
#define BT_STAT_TST(s, a, x)	(!(((s) & (a)) ^ (x)))

/*
 * Status data type
 */
typedef struct bt_status_ {
	int	xor;	/* Value	*/
	int	and;	/* Mask		*/
	char	*msg;	/* description	*/
	/* int	errcode; */ /* error code	*/
} BT_PRN_STAT;




/******************************************************************************
 * The official control port bits
 *****************************************************************************/

#define	BT_CTRL_STROBE	(0x01)		/* High (0.5us): Data vaild	*/
#define	BT_CTRL_AUTOFD	(0x02)		/* High: autom. line feed	*/
#define	BT_CTRL_nRESET	(0x04)		/* Low:  reset printer		*/
#define	BT_CTRL_SLCTIN	(0x08)		/* High: select printer		*/
#define	BT_CTRL_IRQEN	(0x10)		/* High: IRQ enabled		*/
#define	BT_CTRL_PRNDRV	(0x20)		/* High: 8 bit read (not used!)	*/
#define	BT_CTRL_RSVD1	(0x40)		/* Reserved			*/
#define	BT_CTRL_RSVD2	(0x80)		/* Reserved			*/


/******************************************************************************
 * control port bit combinations used to set specific behaviour of port
 *****************************************************************************/

/*
 * switch to normal HOST2PRN mode 
 */
#define BT_HOST_NORMAL		(BT_CTRL_nRESET | BT_CTRL_SLCTIN)

/*
 * drive STROBE low in normal HOST2PRN mode 
 */
#define BT_HOST_STROBE		(BT_CTRL_nRESET | BT_CTRL_SLCTIN | BT_CTRL_STROBE)

/*
 * host requests BiTronics termination	
 */
#define BT_HOST_TERM_REQ	(BT_CTRL_nRESET | BT_CTRL_SLCTIN)

/*
 * host executes termination 
 */
#define BT_HOST_TERM_DO		(BT_CTRL_nRESET | BT_CTRL_SLCTIN | BT_CTRL_AUTOFD)

/*
 * host requests BiTronics operation		
 */
#define BT_HOST_REQ 		(BT_CTRL_nRESET | BT_CTRL_AUTOFD)

/*
 * drive STROBE low for BiTronics negotiation	
 */
#define BT_HOST_BT_STROBE 	(BT_CTRL_nRESET | BT_CTRL_STROBE | BT_CTRL_AUTOFD)

/*
 * host indicates to process negotiation	
 */
#define BT_HOST_BUSY 		(BT_CTRL_nRESET)

/*
 * host indicates to be idle			
 */
#define BT_HOST_IDLE		(BT_CTRL_nRESET | BT_CTRL_AUTOFD)

/*
 * bitmask to identify if host is idle in BiTronics mode 
 */
#define BT_HOST_IDLE_BIT 	(BT_CTRL_AUTOFD)



#endif /* !BT_IO_H_ */
