/*======================================================================

    A driver for the New Media Bus Toaster SCSI card

    Written by David Hinds, dhinds@allegro.stanford.edu
    
======================================================================*/

#include <linux/config.h>

#ifdef MODULE
#include <linux/module.h>
#include <linux/version.h>
#endif

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/ioport.h>

#include "drivers/block/blk.h"
#include "drivers/scsi/scsi.h"
#include "drivers/scsi/hosts.h"
#include "drivers/scsi/aha152x.h"

#include "version.h"
#include "cs_types.h"
#include "cs.h"
#include "cistpl.h"
#include "ds.h"

#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
static char *version =
    "toaster_cs.c 1.16 1995/04/11 18:26:08 (David Hinds)\n";
#endif

/*====================================================================*/

/* Parameters that can be set with 'insmod' */

/* Bit map of interrupts to choose from */
static u_long irq_mask = 0xdeb8;

/*====================================================================*/

extern void aha152x_setup(char *str, int *ints);

static void toaster_release(u_long arg);
static int toaster_event(event_t event, int priority,
			event_callback_args_t *args);

static dev_link_t *toaster_attach(void);
static void toaster_detach(dev_link_t *);

static Scsi_Host_Template driver_template = AHA152X;

static dev_link_t *dev_list = NULL;

static dev_info_t dev_info = "toaster_cs";

/*====================================================================*/

static void cs_error(int func, int ret)
{
    CardServices(ReportError, dev_info, (void *)func, (void *)ret);
}

/*====================================================================*/

static dev_link_t *toaster_attach(void)
{
    client_reg_t client_reg;
    dev_link_t *link;
    int ret;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("toaster_attach()\n");
#endif

    /* Create new SCSI device */
    link = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);
    memset(link, 0, sizeof(struct dev_link_t));
    link->release.function = &toaster_release;
    link->release.data = (u_long)link;

    link->io.NumPorts1 = 0x20;
    link->io.Attributes1 = IO_DATA_PATH_WIDTH_AUTO;
    link->io.IOAddrLines = 10;
    link->irq.Attributes = IRQ_TYPE_EXCLUSIVE;
    link->irq.IRQInfo1 = IRQ_INFO2_VALID|IRQ_LEVEL_ID;
    link->irq.IRQInfo2 = irq_mask;
    link->conf.Attributes = CONF_ENABLE_IRQ;
    link->conf.Vcc = 50;
    link->conf.IntType = INT_MEMORY_AND_IO;
    link->conf.Present = PRESENT_OPTION;

    /* Register with Card Services */
    link->next = dev_list;
    dev_list = link;
    client_reg.dev_info = &dev_info;
    client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
    client_reg.event_handler = &toaster_event;
    client_reg.EventMask =
	CS_EVENT_RESET_REQUEST | CS_EVENT_CARD_RESET |
	CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
	CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
    client_reg.Version = 0x0210;
    client_reg.event_callback_args.client_data = link;
    ret = CardServices(RegisterClient, &link->handle, &client_reg);
    if (ret != 0) {
	cs_error(RegisterClient, ret);
	toaster_detach(link);
	return NULL;
    }
    
    return link;
} /* toaster_attach */

/*====================================================================*/

static void toaster_detach(dev_link_t *link)
{
    dev_link_t **linkp;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("toaster_detach(0x%p)\n", link);
#endif
    
    /* Locate device structure */
    for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
	if (*linkp == link) break;
    if (*linkp == NULL)
	return;

    if (link->state & DEV_CONFIG) {
	toaster_release((u_long)link);
	if (link->state & DEV_STALE_CONFIG) {
	    link->state |= DEV_STALE_LINK;
	    return;
	}
    }

    if (link->handle)
	CardServices(DeregisterClient, link->handle);
    
    /* Unlink device structure, free bits */
    *linkp = link->next;
    if (link->priv) {
	/* What do we do with the host structure ??? */
    }
    kfree_s(link, sizeof(struct dev_link_t));
    
} /* toaster_detach */

/*====================================================================*/

static void toaster_config(dev_link_t *link)
{
    client_handle_t handle;
    tuple_t tuple;
    cisparse_t parse;
    struct Scsi_Host *host;
    int i, ints[3];
    u_char tuple_data[64];
    
    handle = link->handle;
    host = link->priv;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("toaster_config(0x%p)\n", link);
#endif

    do {
	tuple.DesiredTuple = CISTPL_CONFIG;
	i = CardServices(GetFirstTuple, handle, &tuple);
	if (i != CS_SUCCESS) break;
	tuple.TupleData = tuple_data;
	tuple.TupleDataMax = 64;
	tuple.TupleOffset = 0;
	i = CardServices(GetTupleData, handle, &tuple);
	if (i != CS_SUCCESS) break;
	i = CardServices(ParseTuple, handle, &tuple, &parse);
	if (i != CS_SUCCESS) break;
	link->conf.ConfigBase = (caddr_t)parse.config.base;
    } while (0);
    if (i != CS_SUCCESS) {
	cs_error(ParseTuple, i);
	link->state &= ~DEV_CONFIG_PENDING;
	return;
    }

    /* Configure card */
    driver_template.usage_count = &mod_use_count_;
    link->state |= DEV_CONFIG;

    do {

	tuple.DesiredTuple = CISTPL_CFTABLE_ENTRY;
	i = CardServices(GetFirstTuple, handle, &tuple);
	while (i == CS_SUCCESS) {
	    i = CardServices(GetTupleData, handle, &tuple);
	    if (i != CS_SUCCESS) break;
	    i = CardServices(ParseTuple, handle, &tuple, &parse);
	    if (i != CS_SUCCESS) break;
	    link->conf.ConfigIndex = parse.cftable_entry.index;
	    link->io.BasePort1 = parse.cftable_entry.io.win[0].base;
	    if (link->io.BasePort1 > 0xffff)
		continue;
	    i = CardServices(RequestIO, link->handle, &link->io);
	    if (i == CS_SUCCESS) break;
	    i = CardServices(GetNextTuple, handle, &tuple);
	}
	if (i != CS_SUCCESS) {
	    cs_error(RequestIO, i);
	    break;
	}
	
	i = CardServices(RequestIRQ, link->handle, &link->irq);
	if (i != CS_SUCCESS) {
	    cs_error(RequestIRQ, i);
	    break;
	}
	i = CardServices(RequestConfiguration, link->handle, &link->conf);
	if (i != CS_SUCCESS) {
	    cs_error(RequestConfiguration, i);
	    break;
	}

    } while (0);

    if (i != 0) {
	toaster_release((u_long)link);
	return;
    }

    sprintf(link->dev_name, "toaster");
    /* A bad hack... */
    release_region(link->io.BasePort1, link->io.NumPorts1);
    link->state &= ~DEV_CONFIG_PENDING;

    /* Set configuration options for the aha152x driver */
    ints[0] = 2;
    ints[1] = link->io.BasePort1;
    ints[2] = link->irq.AssignedIRQ;
    aha152x_setup("PCMCIA setup", ints);
    
    scsi_register_module(MODULE_SCSI_HA, &driver_template);
    
} /* toaster_config */

/*====================================================================*/

static void toaster_release(u_long arg)
{
    dev_link_t *link = (dev_link_t *)arg;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("toaster_release(0x%p)\n", link);
#endif

    if (*driver_template.usage_count != 0) {
	printk("toaster_cs: release postponed, '%s' still open\n",
	       link->dev_name);
	link->state |= DEV_STALE_CONFIG;
	return;
    }

    if (link->dev_name[0] != '\0')
	scsi_unregister_module(MODULE_SCSI_HA, &driver_template);
    
    CardServices(ReleaseConfiguration, link->handle);
    CardServices(ReleaseIO, link->handle, &link->io);
    CardServices(ReleaseIRQ, link->handle, &link->irq);
    
    link->state &= ~DEV_CONFIG;
    if (link->state & DEV_STALE_LINK)
	toaster_detach(link);
    
} /* toaster_release */

/*====================================================================*/

static int toaster_event(event_t event, int priority,
			event_callback_args_t *args)
{
    dev_link_t *link = args->client_data;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("toaster_event()\n");
#endif
    
    switch (event) {
#ifdef PCMCIA_DEBUG
    case CS_EVENT_REGISTRATION_COMPLETE:
	if (pc_debug)
	    printk("toaster_cs: registration complete\n");
	break;
#endif
    case CS_EVENT_CARD_REMOVAL:
	link->state &= ~DEV_PRESENT;
	if (link->state & DEV_CONFIG) {
	    link->release.expires = 5;
	    add_timer(&link->release);
	}
	break;
    case CS_EVENT_CARD_INSERTION:
	link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
	toaster_config(link);
	break;
    case CS_EVENT_PM_SUSPEND:
	link->state |= DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_RESET_PHYSICAL:
	if (link->state & DEV_CONFIG)
	    CardServices(ReleaseConfiguration, link->handle);
	break;
    case CS_EVENT_PM_RESUME:
	link->state &= ~DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_CARD_RESET:
	if (link->state & DEV_CONFIG) {
	    CardServices(RequestConfiguration, link->handle, &link->conf);
	    aha152x_reset(NULL);
	}
	break;
    }
    return 0;
} /* toaster_event */

/*====================================================================*/

#ifdef MODULE

char kernel_version[] = UTS_RELEASE;

int init_module(void) {
    servinfo_t serv;
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(version);
#endif
    CardServices(GetCardServicesInfo, &serv);
    if (serv.Revision != CS_RELEASE_CODE) {
	printk("toaster_cs: Card Services release does not match!");
	return -1;
    }
    register_pcmcia_driver(&dev_info, &toaster_attach, &toaster_detach);
    return 0;
}

void cleanup_module(void) {
    printk("toaster_cs: unloading\n");
    unregister_pcmcia_driver(&dev_info);
    while (dev_list != NULL) {
	if (dev_list->state & DEV_CONFIG)
	    panic("toaster_cs: device still configured!");
	else
	    toaster_detach(dev_list);
    }
}

#endif

