/*
 * term.c -- terminal stuff for lsmtool
 *
 * Lars Wirzenius
 * "@(#)lsm:term.c,v 1.2 1994/12/28 13:35:36 wirzeniu Exp"
 */

#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <termios.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <publib.h>
#include <termcap.h>

#include "term.h"


char PC;   /* For tputs.  */
char *BC;  /* For tgoto.  */
char *UP;  /* For something else in termcap */


struct term {
	char buf[10240];
	char buf2[10240];
	char *bufpos;

	char *tc_ti;	/* initialize terminal */
	char *tc_te;	/* termination string */
	char *tc_cm;	/* cursor movement */
	char *tc_cl;	/* clear screen */
	char *tc_ce;	/* clear to end of line */
	char *tc_al;	/* insert line */
	char *tc_dl;	/* delete line */
	char *tc_so;	/* standout */
	char *tc_se;	/* standend */

	char *tc_ks;	/* keypad start */
	char *tc_ke;	/* keypad end */
	char *tc_kl;	/* left arrow */
	char *tc_kr;	/* right arrow */
	char *tc_ku;	/* up arrow */
	char *tc_kd;	/* down arrow */
	char *tc_kN;	/* next page */
	char *tc_kP;	/* previous page */
	char *key[NUM_KEYS];
	int keylen[NUM_KEYS];
};


static struct term term;


/*
 * Original terminal settings for the input stream.
 */
static struct termios inorig;



/*
 * Get a capability string, variations on a theme.
 */
static void xgetcap(struct term *t, const char *capname, char **capstr) {
	*capstr = t->bufpos;
	if (tgetstr(capname, &t->bufpos) == NULL)
		errormsg(1, 0, "No %s capability.  Cannot continue.", capname);
}

static void getcap(struct term *t, const char *capname, char **capstr) {
	*capstr = t->bufpos;
	if (tgetstr(capname, &t->bufpos) == NULL)
		*capstr = "";
}


/*
 * Output function for tputs.
 */
static int char2tty(int c) {
	putchar(c);
	return 0;
}


/*
 * Output strings, variations on a theme.
 */
static void noutstr(const char *cap, int nlines) {
	tputs(cap, nlines, char2tty);
}

static void outstr(const char *cap) {
	tputs(cap, 1, char2tty);
}


/*
 * Return the size of the terminal, or at least a good guess.
 */
static void getwinsize(int *rows, int *cols) {
        struct winsize ws;

        if (ioctl(1, TIOCGWINSZ, &ws) == -1) {
                *rows = 24;
		*cols = 80;
        } else {
                *rows = ws.ws_row;
                *cols = ws.ws_col;
        }
}



/*
 * Initialize the terminal so that everything works nicely.
 */
void terminit(void) {
	struct termios ti;
	char *temp, line[10240];

	/*
	 * Set terminal settings.
	 */
	if (tcgetattr(0, &inorig) == -1)
		errormsg(1, -1, "tcgetattr failed");

	ti = inorig;
	ti.c_lflag &= ~(ISIG | ICANON | ECHO);
	if (tcsetattr(0, TCSADRAIN, &ti) == -1)
		errormsg(1, -1, "tcsetattr failed");

	/*
	 * Initialize termcap.
	 */
	if ((temp = getenv("TERM")) == NULL)
		errormsg(1, 0, "Environment variable TERM not set\n"
				"Terminal type is unknown.  Cannot continue.");

	if (tgetent(term.buf, temp) == -1)
		errormsg(1, -1, "tgetent failed");

	term.bufpos = term.buf2;
	temp = tgetstr("pc", &term.bufpos);
	PC = temp ? *temp : 0;
	BC = tgetstr("le", &term.bufpos);
	UP = tgetstr("up", &term.bufpos);

	getcap(&term, "ti", &term.tc_ti);
	getcap(&term, "te", &term.tc_te);
	xgetcap(&term, "cm", &term.tc_cm);
	xgetcap(&term, "cl", &term.tc_cl);
	xgetcap(&term, "ce", &term.tc_ce);
	xgetcap(&term, "al", &term.tc_al);
	xgetcap(&term, "dl", &term.tc_dl);
	xgetcap(&term, "so", &term.tc_so);
	xgetcap(&term, "se", &term.tc_se);

	getcap(&term, "ks", &term.tc_ks);
	getcap(&term, "ke", &term.tc_ke);
	getcap(&term, "kl", &term.tc_kl);
	getcap(&term, "kr", &term.tc_kr);
	getcap(&term, "ku", &term.tc_ku);
	getcap(&term, "kd", &term.tc_kd);
	getcap(&term, "kN", &term.tc_kN);
	getcap(&term, "kP", &term.tc_kP);

	outstr(term.tc_ti);
	outstr(term.tc_ks);
}



/*
 * Undo any changes to terminal settings.
 */
void termend(void) {
	outstr(term.tc_ke);
	outstr(term.tc_te);
	if (tcsetattr(0, TCSADRAIN, &inorig) == -1)
		errormsg(1, -1, "tcsetattr failed");
}


/*
 * Report the terminal size.
 */
int termwidth(void) {
	int rows, cols;
	getwinsize(&rows, &cols);
	return cols;
}

int termheight(void) {
	int rows, cols;
	getwinsize(&rows, &cols);
	return rows;
}


/*
 * Read one character.  Return -1 for error, 0 for EOF, 1 for timeout, 2
 * for OK.  Return the character via *p.
 */
static int readchar(char *p) {
	fd_set set;
	struct timeval t;
	int n;

	assert(p != NULL);

	FD_ZERO(&set);
	FD_SET(0, &set);
	t.tv_sec = 1;
	t.tv_usec = 0;
	n = select(1, &set, NULL, NULL, &t);

	if (n == -1)
		return -1;	/* error */
	if (n == 0)
		return 1;	/* timeout */

	assert(n == 1);
	assert(FD_ISSET(0, &set));

	n = read(0, p, 1);
	assert(n == -1 || n == 0 || n == 1);
	if (n == 1)
		return 2;
	return n;
}

/*
 * Read a key.  Return -1 for error, 0..UCHAR_MAX if a normal, character
 * key was pressed, or one of the KEY_* codes if a function key
 * was pressed.
 */
int getkey(void) {
	unsigned char ch;

	int key;
	static char keystr[1024];
	static int keylen = 0;

	fflush(stdout);
	for (;;) {
		if (keylen > 0) {
			key = keymatch(keystr, keylen);
			if (key >= 0) {
				keylen = 0;
				return key;
			} else if (key == -1)
				goto return_first_char;
		}

		switch (readchar(&keystr[keylen])) {
		case -1:	/* error */
			return -1;

		case 2:		/* ok */
			++keylen;
			if (keylen < sizeof(keystr))
				break;
			goto return_first_char;

		case 0:		/* eof */
			if (keylen == 0)
				return -1;
			goto return_first_char;

		case 1:		/* timeout */
			if (keylen == 0)
				break;
			goto return_first_char;

		default:
			assert(0);
		}
	}

	assert(0);	/* can't be reached */

return_first_char:
	ch = keystr[0];
	memmove(keystr, keystr+1, keylen-1);
	--keylen;
	return ch;
}



void move(int y, int x) {
	fputs(tgoto(term.tc_cm, x, y), stdout);
}

void clear(void) {
	noutstr(term.tc_cl, 100);
}

void clrtoeol(void) {
	tputs(term.tc_ce, 1, char2tty);
}

void insline(void) {
	tputs(term.tc_al, 100, char2tty);
}

void delline(void) {
	tputs(term.tc_dl, 100, char2tty);
}

void standout(void) {
	tputs(term.tc_so, 1, char2tty);
}

void standend(void) {
	tputs(term.tc_se, 1, char2tty);
}



/*
 * Is `str' (length len, i.e., not zeroterminated) a match for * a key? 
 * Return keycode (>= 0) if it is, -1 if not, -2 if it is a prefix.
 */
int keymatch(const char *str, int len) {
	int i;

	if (term.key[0] == NULL) {
		term.key[KEY_LEFT-MIN_KEY] = term.tc_kl;
		term.key[KEY_RIGHT-MIN_KEY] = term.tc_kr;
		term.key[KEY_UP-MIN_KEY] = term.tc_ku;
		term.key[KEY_DOWN-MIN_KEY] = term.tc_kd;
		term.key[KEY_NEXT-MIN_KEY] = term.tc_kN;
		term.key[KEY_PREV-MIN_KEY] = term.tc_kP;

		for (i = 0; i < NUM_KEYS; ++i)
			term.keylen[i] = strlen(term.key[i]);
	}

	for (i = 0; i < NUM_KEYS; ++i) {
		int n = term.keylen[i];
		if (n == 0 || n < len)
			continue;	/* no key, or can't possibly match */
		if (memcmp(str, term.key[i], len) == 0) {
			if (n == len)
				return MIN_KEY + i; /* keycode */
			return -2;	/* prefix */
		}
	}

	return -1;	/* not a match */
}
