-- ***************************************************************
-- CoLibri
-- ---------------------------------------------------------------
--
-- Map.cs -- 
-- 
--   Copyright (C) sietec Systemtechnik GmbH & Co OHG 1994
--   All rights reserved
-- 
-- Author          : Lutz Hilken
-- 
-- PURPOSE 
--      This module realizes the generic container class Map. It
--      provides the user interface for objects of this collection. 
--
-- REFER
--      see ducoment ITHACA.SNI.CoLibri.91.#4
--
-- ---------------------------------------------------------------
--
-- Created On      : Wed Jan 12 14:57:15 1994
-- Last Modified By: Lutz Hilken
-- Last Modified On: Thu Mar 31 15:16:39 1994
-- Update Count    : 18
-- 
-- ***************************************************************

SPECIFICATION Map

IMPORT IdentityBinaryTree FROM BTree;

TYPE Map [KeyType, EntryType] = OBJECT 

-- -----------------------------------------------------------------
-- General
-- -----------------------------------------------------------------
-- Objects of this class represents a view to a mapping between an
-- object of a key type to an object of an element type. The basic
-- requirement to the key type is, that it comes with to relations,
-- that describe the less and the equal predicate of two key objects.
-- Another requirement is the fact, that it should be possible to use
-- basic cool types for a key. So, this relations have to be written
-- as procedures and not as methods.
-- The elements of this container are in a strong total relation to
-- each other. This means, that there are no double entries in a map.
-- The key value is used to keep track of this relation
-- by using the Less-predicate. To find an element with a specific
-- key value, the Equal-predicate is used.
-- This class also provides a cursor on the elements of this
-- container. With a set of operations to and with this cursor, it
-- should be possible to navigate through the map and operate on the
-- elements, the cursor points to.
-- -----------------------------------------------------------------

PUBLIC
  
-- -----------------------------------------------------------------
-- Creating a map
-- -----------------------------------------------------------------
-- The initialization of a map requires the two predicates of the key
-- value. This predicates are constant throgh the whole life of a
-- container of this class.
-- The cursor is set to an initialized state, the map is set empty.
-- -----------------------------------------------------------------
-- INITIALLY (see obove)

-- -----------------------------------------------------------------
-- Changing the cursor position
-- -----------------------------------------------------------------
-- This operations are used to set the cursors position to a new,
-- absolute or relative position.
-- -----------------------------------------------------------------
  
  METHOD SetCursorToFirst ();
  METHOD SetCursorToLast ();
  METHOD SetCursorToNext ();
  METHOD SetCursorToPrevious ();
  -- -----------------------------------------------------------------
  -- Move the cursor to the first or last, to the next or previous
  -- entry of the map.
  -- -----------------------------------------------------------------

  METHOD SetCursorToElement (IN key : KeyType);
  -- -----------------------------------------------------------------
  -- This method moves the cursor to the element with the key value of
  -- Key. 
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Element access related to the cursor
-- -----------------------------------------------------------------
-- The entry value of the element at the actual cursor position may
-- be accessed by reading and writing. The key value may
-- only accessed by reading to keep the order of the map keys.
-- -----------------------------------------------------------------

  METHOD GetEntryAtCursor () : EntryType;
  METHOD SetEntryAtCursor (IN entry : EntryType);
  -- -----------------------------------------------------------------
  -- The entry value of the element at the actual cursor position is
  -- read or written.
  -- -----------------------------------------------------------------

  METHOD GetKeyAtCursor () : KeyType;
  -- -----------------------------------------------------------------
  -- The key entry of the element at the actual cursor position is 
  -- read. 
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Inserting new elements and replacing entries
-- -----------------------------------------------------------------
-- New elements may be added to the map or the entry may be changed.
-- Both operations work with a given key value. 
-- -----------------------------------------------------------------

  METHOD InsertElement (IN key : KeyType, IN entry : EntryType);
  -- -----------------------------------------------------------------
  -- This method inserts a new element described by Key and
  -- Entry into this map. If there is already an element with the
  -- same key value than Key, an exception is raised. The relation
  -- Less (see INITIALLY) is used to find the position for the new
  -- entry.
  -- -----------------------------------------------------------------
  
  METHOD ReplaceEntry (IN key : KeyType, IN entry : EntryType);
  -- -----------------------------------------------------------------
  -- This Method replaces the entry value of the element described by
  -- Key with Entry in this map. If there is no element with
  -- the Key, an exception is raised. The relation Less (see
  -- INITIALLY) is used to find the position for the new entry.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Delete an element
-- -----------------------------------------------------------------
-- The element with a given key value is removed from the map.
-- -----------------------------------------------------------------

  METHOD RemoveElement (IN key : KeyType);
  -- -----------------------------------------------------------------
  -- This method removes the element with the Key value from this
  -- map. If the cursor was set to the element with Key, it is
  -- undefined after this operation. The relation Equal (see
  -- INITIALLY) is used to find the specific element.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Get an element related to a key
-- -----------------------------------------------------------------

  METHOD GetEntry (IN key : KeyType) : EntryType;
  -- -----------------------------------------------------------------
  -- This method delivers the entry value related to the Key value
  -- from this map. The relation Equal (see INITIALLY) is used to
  -- find the specific element.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Counting elements
-- -----------------------------------------------------------------
-- The number of elements is delivered, which satisfies a given
-- criterion. 
-- -----------------------------------------------------------------

  METHOD GetNumberOfElements () : INT;
  -- -----------------------------------------------------------------
  -- This method delivers the number of elements in this map.
  -- -----------------------------------------------------------------

  METHOD GetNumberOfMatches
    (IN predicate : PROCEDURE (IN EntryType) : BOOL) : INT;
  -- -----------------------------------------------------------------
  -- This method searches in this map for all elements, which satisfy
  -- the given Predicate. The number of matching elements is returned.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Split and merge
-- -----------------------------------------------------------------

  METHOD Split (IN key : KeyType) : Map [KeyType, EntryType];
  -- -----------------------------------------------------------------
  -- This method splits this map at the element with the Key value.
  -- The elements equal to or greater than the element with the Key
  -- value will be removed from this map and inserted in the resulting
  -- map. If this element doesn't exists, only the elements greater
  -- than Key are chosen.
  -- -----------------------------------------------------------------

  METHOD Merge (IN second_map : Map [KeyType, EntryType]);
  -- -----------------------------------------------------------------
  -- This method inserts all elements of the SecondMap into this map. 
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Operation mapping
-- -----------------------------------------------------------------
-- A given operation is applied to all element enties of a map.
-- -----------------------------------------------------------------

  METHOD Apply (IN routine : PROCEDURE (INOUT EntryType));
  -- -----------------------------------------------------------------
  -- This method applies the Routine to all entry values of all
  -- elements in this map. The Routine may modify the entry values.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Predicates
-- -----------------------------------------------------------------
-- Some tests for the entries of a map
-- -----------------------------------------------------------------

  METHOD IsFirst () : BOOL;
  -- ---------------------------------------------------------------
  -- This method tests, whether the cursor is related to the first
  -- element of this map.
  -- ---------------------------------------------------------------

  METHOD IsLast () : BOOL;
  -- ---------------------------------------------------------------
  -- This method tests, whether the cursor is related to the last
  -- element of this map.
  -- ---------------------------------------------------------------

  METHOD IsEmpty () : BOOL;
  -- ---------------------------------------------------------------
  -- This method tests this map for emptyness.
  -- ---------------------------------------------------------------

  METHOD IsEqual 
    (IN second_map : Map [KeyType, EntryType],
     IN equal_entry : PROCEDURE (IN EntryType, IN EntryType) : BOOL)
      : BOOL;
  -- -----------------------------------------------------------------
  -- This method compares this map to SecondMap. Two lists are not
  -- equal, if one map contains an element, the other doesn't contain.
  -- To test the equality of two elements, the key values and the
  -- entry values are compared. The EqualEntry-predicate is used to
  -- compare the entry values of an element.
  -- -----------------------------------------------------------------

  METHOD Contains (IN key : KeyType) : BOOL;
  -- -----------------------------------------------------------------
  -- This method tests the existence of the element with the key value
  -- Key in this map. The procedures Equal and Less (see
  -- INITIALLY) are used to find the element.
  -- -----------------------------------------------------------------

PROTECTED

  METHOD isequalto 
    (IN second_map  : Map [KeyType, EntryType],
     IN equal_entry : PROCEDURE (IN EntryType, IN EntryType) : BOOL,
     IN firsttime   : BOOL) 
      : BOOL;

STATE

  cardinality : INT;
  tree        : IdentityBinaryTree [KeyType, EntryType];
  first       : IdentityBinaryTree [KeyType, EntryType];
  last        : IdentityBinaryTree [KeyType, EntryType];
  cursor      : IdentityBinaryTree [KeyType, EntryType];

END OBJECT;

EXCEPTION CursorUndefined;
EXCEPTION ContainerEmpty;

-- ***************************************************************
CONST spec_vid : STRING = 
   "@(#) $__Header$" ;

END SPECIFICATION (* Map *);
