----------------------------------------------------------------------------
--                  CooL-V2.0 - destination code interface                 --
-----------------------------------------------------------------------------
--                      expression generation routines                     --
--                            Version 1.0, 1993                            --
-----------------------------------------------------------------------------

'module' codeexpr

'export' 
     GenerateEnclosedExpr
     GenerateExpr
     GetArrayDimension
     GetExpressionRealType
     GetExpressionSpecType
     NeededEqualFunction
     ContainsCall
     
'use' ast
      extspecs
      types
      misc
      codetype -- GenerateType
      codedecl -- Current
      codestmt -- Temporaries

-----------------------------------------------------------------------------

-- GenerateEnclosedExpr -------------------------------------------------------


'action' GenerateEnclosedExpr (EXPR)
     
     'rule' GenerateEnclosedExpr (Expr) :
	  Write ("(")
	  GenerateExpr (Expr)
	  Write (")")
	  
'action' GenerateExpr (EXPR)

     'rule' GenerateExpr (nil (_))

     'rule' GenerateExpr (false (_))
	  Write ("C3IFALSE")

     'rule' GenerateExpr (true (_))
	  Write ("C3ITRUE")

     'rule' GenerateExpr (posintliteral (_, Value))
	  WriteUnsignedInt (Value)

     'rule' GenerateExpr (negintliteral (_, Value))
	  Write ("(-")
	  WriteUnsignedInt (Value)
	  Write (")")

     'rule' GenerateExpr (doubleliteral (_, Value))
	  WriteDouble (Value)

     'rule' GenerateExpr (charliteral (_, Value))
	  WriteChar (Value)

     'rule' GenerateExpr (stringliteral (_, Value))
	  Write ("C3IC3IInitWithRefChar( (char (*)[])")
	  WriteString (Value)
	  Write (" )")
 
     'rule' GenerateExpr (expr_nil (_))
	  Write ("0")

     -- string concatenation
	  
     'rule' GenerateExpr (dyop (_, _, plus, Left, Right))
	  IsStringConcatenation (Left, Right -> RtsFunction)
	  Write (RtsFunction)
	  Write (" (")
	  GenerateEnclosedExpr (Left)
	  Write (", ")
	  GenerateEnclosedExpr (Right)
	  Write (")")
	  
     -- address + integer type
     
     'rule' GenerateExpr (dyop (_, _, plus, Left, Right))
	  GetExpressionRealType (Left -> Type)
	  FollowNameChain (Type -> simple (address))
	  Write ("(C3IADDRESS)((char *)") 
	  GenerateEnclosedExpr (Left)
	  Write (" + ")
	  GenerateEnclosedExpr (Right)
	  Write (")")

     -- integer type + address
	  
     'rule' GenerateExpr (dyop (_, _, plus, Left, Right))
	  GetExpressionRealType (Right -> Type)
	  FollowNameChain (Type -> simple (address))
	  Write ("(C3IADDRESS)(" )
	  GenerateEnclosedExpr (Left)
	  Write (" + (char *)")
	  GenerateEnclosedExpr (Right)
	  Write (")")

     -- address - address
	  
     'rule' GenerateExpr (dyop (_, _, minus, Left, Right))
	  GetExpressionRealType (Left -> LeftType)
	  FollowNameChain (LeftType -> simple (address))
	  GetExpressionRealType (Right -> RightType)
	  FollowNameChain (RightType -> simple (address))
	  Write ("(int)((char *)")
	  GenerateEnclosedExpr (Left)
	  Write (" - (char *)")
	  GenerateEnclosedExpr (Right)
	  Write (")")
     
     -- address - integer type
	  
     'rule' GenerateExpr (dyop (_, _, minus, Left, Right))
	  GetExpressionRealType (Left -> Type)
	  FollowNameChain (Type -> simple (address))
	  Write ("(C3IADDRESS)((char *)")
	  GenerateEnclosedExpr (Left)
	  Write (" - ")
	  GenerateEnclosedExpr (Right)
	  Write (")")

     -- dyadic operator / 
	 
     'rule' GenerateExpr (dyop (Pos, _, div, Left, Right))
	  Write( "((" )
	  GenerateDIVArgument (Left)
	  Write (")")
	  GenerateDyadicOperator (div)
	  Write ("(")
	  GenerateDIVArgument (Right)
	  Write( "))" )

     -- dyadic relational operations on strings 
	  
     'rule' GenerateExpr (dyop (_, TypeI, Operator, Left, Right))
	  FollowNameChainIndex (TypeI -> simple (bool))
	  GetExpressionRealType (Left -> LeftType)
	  FollowNameChain (LeftType -> LeftType2)
	  IsStringType (LeftType2)
	  GetRtsStringRelOpFunction (Operator -> RtsFunction)
	  Write (RtsFunction)
	  Write (" (")
	  GenerateEnclosedExpr (Left)
	  Write( ", " )
	  GenerateEnclosedExpr( Right )
	  Write( ")" )
	 
     -- equal on composite types 
     --  use GenerateEqualArgument to code the expression
	  
     'rule' GenerateExpr (dyop (Pos, _, DyOp, Left, Right))
	  (|
	       GetExpressionRealType (Left -> LeftType)
	       FollowNameChain (LeftType -> LeftType2)
	       NeededEqualFunction (LeftType2)
	       let (LeftType -> Type)
	  ||
	       GetExpressionRealType (Right -> RightType)
	       FollowNameChain (RightType -> RightType2)
	       NeededEqualFunction (RightType2)
	       let (RightType -> Type)
	  |)
	  (|
	       where (DyOp -> eq)
	       Write ("C3IC3IEqual( ")
	  ||
	       where (DyOp -> ne)
	       Write ("C3IC3INotEqual( ")
	  |)
	  GenerateEqualArgument (Left) 
	  Write (", ")
	  GenerateEqualArgument (Right)
	  Write (", sizeof( ")
	  GenerateBaseType (Type)
	  GenerateBaseTypeExtension (Type)
	  Write (" ))")
	  
     -- special NIL rule to generate cast 
	  
     'rule' GenerateExpr (dyop (_, _, DyOp, Left:expr_nil (_), 
				      Right)) :
	  GetExpressionRealType (Right -> RightType)
	  Write ("(")
	  GenerateCast (RightType, in)
	  GenerateEnclosedExpr (Left)
	  GenerateDyadicOperator (DyOp)
	  GenerateEnclosedExpr (Right)
	  Write (")")
	  
     -- other dyadic operators
	  
     'rule' GenerateExpr (dyop (_, _, DyOp, Left, Right)) :
	  GetExpressionSpecType (Left -> LeftType)
	  GetExpressionSpecType (Right -> RightType)
	  Write ("(")
	  GenerateEnclosedExpr (Left)
	  GenerateDyadicOperator (DyOp)
	  GenerateCastIfNecessary (RightType, LeftType, in)
	  GenerateEnclosedExpr (Right)
	  Write (")")
	  
    -- monadic opeartor
	  
     'rule' GenerateExpr (monop (_, _, Operator, Operand))
	  GenerateMonadicOperator (Operator)
	  GenerateEnclosedExpr (Operand)
	
    -- In C the address operator is ignored for array types.
	
     'rule' GenerateExpr (adr (_, Designator))
	  GetExpressionRealType (Designator -> Type) 
	  FollowNameChain (Type -> composite (array (_, _)))
	  Write ("(")
	  GenerateBaseType (Type)
	  Write ("(*)")
	  GenerateBaseTypeExtension (Type)
	  Write (")")
	  GenerateEnclosedExpr (Designator)

     'rule' GenerateExpr (adr (_, Designator))
	  GetExpressionRealType (Designator -> Type)
	  FollowNameChain (Type -> composite (openarray (_)))
	  GenerateEnclosedExpr (Designator)

     'rule' GenerateExpr (adr (_, Designator))
	  Write ("&")
	  GenerateEnclosedExpr (Designator)

     -- sizeof
     
     'rule' GenerateExpr (sizeof (_, TypeI))
	  Write ("sizeof (")
	  GenerateTypeIndex (TypeI)
	  Write (")")
	
    -- record selection 
     
     'rule' GenerateExpr (fieldsel (_, _, Record, FieldName))
	  GenerateEnclosedExpr (Record)
	  Write (".")
	  GetExpressionRealType (Record -> RecordType)
	  GenerateFieldName (RecordType, FieldName)
	
    -- union selection
	 
     'rule' GenerateExpr (unionsel (_, _, Union, FieldName))
	  GenerateEnclosedExpr (Union)
	  Write (".")
	  GetExpressionRealType (Union -> UnionType)
	  GenerateFieldName (UnionType, FieldName)

    -- enum selection 	 
	 
     'rule' GenerateExpr (enumsel (_, EnumType, Enumerator, _))
	  EnumType'Type -> composite (typename (EnumTypeId))
	  GenerateQualifiedCooLName (EnumTypeId)
	  GenerateCooLName (Enumerator)

     -- unamed enumeration type is applied
	
     'rule' GenerateExpr (enumsel (_, EnumType, _, Number))
	  EnumType'Type -> composite (enum (_ ))
	  WriteInt (Number)

     -- The expression methodpointer is built by a rts function
     -- called C3IBuildMethodPointer. A direct coding of this
     -- expression is not possible because it is no tempory
     -- avialable which can receive this expression.
     
     'rule' GenerateExpr (methodpointer (_, _, Object:call (_, _, _, _, _), 
					 MethodName))
	  Temporaries -> N
	  Temporaries <- N + 1
	  GenerateTemporary (N)
	  Write (" = ")
	  GenerateEnclosedExpr (Object)
	  Write (", C3IC3IBuildMethodPointer (")
	  GenerateTemporary (N)
	  Write (", (void (*)(void))(")
	  GenerateTemporary (N)
	  Write ("->C3IMTABR->")
	  GenerateCooLName (MethodName)
	  Write ("))")
     
     'rule' GenerateExpr (methodpointer (_, _, Object, MethodName))
	  Write ("C3IC3IBuildMethodPointer (")
	  GenerateEnclosedExpr (Object)
	  Write (", (void (*)(void))(")
	  GenerateEnclosedExpr (Object)
	  Write ("->C3IMTABR->")
	  GenerateCooLName (MethodName)
	  Write ("))")

    -- STRING[Index]
	
     'rule' GenerateExpr (stringsubscr (_, Designator, Index))
	  Write ("C3IC3ICharAtIndex (")
	  GenerateEnclosedExpr (Designator)
	  Write (", ")
	  GenerateEnclosedExpr (Index)
	  Write (")")
	
    -- array subscription	 
	 
     'rule' GenerateExpr (arraysubscr (_, _, Designator, Index))
	  GenerateEnclosedExpr (Designator)
	  Write ("[")
	  GenerateEnclosedExpr (Index)
	  Write (" - 1]")

    -- STRING [Lwb .. Upb]
	 
     'rule' GenerateExpr (substring (_, String, range (_, Lwb, Upb)))
	  Write ("C3IC3ISubString (")
	  GenerateEnclosedExpr (String)
	  Write (", ")
	  GenerateEnclosedExpr (Lwb)
	  Write (", ")
	  GenerateEnclosedExpr (Upb)
	  Write (")")
			      
    -- Expr^	 
	 
     'rule' GenerateExpr (deref (_, _, Reference))
	  Write ("(*")
	  GenerateEnclosedExpr (Reference)
	  Write (")")

    -- Identifier application 
	 
     'rule' GenerateExpr (applied (_, Id))
	  GetIdMeaning (Id -> ActualMeaning)
	  GenerateAppliedIdentifier (ActualMeaning, Id)

    -- CURRENT	 
	 
     'rule' GenerateExpr (current (_))
	  Write ("C3ICURRENT")

    -- calls 
	 
     'rule' GenerateExpr (call (_, _, RealResult, Function, Params))
	  GenerateReceiver (Function, RealResult, Params)
	  IfArrayResultMakeNewFirstParameter (RealResult, Params)
	  GetFParamList (Function -> FParams)
	  GenerateActualParams (Params, FParams)
	  Write (")")
	
-- GenerateFieldName -------------------------------------------------------

'action' GenerateFieldName (TYPE, ID)
	  
     'rule' GenerateFieldName (composite (typename (Id)), FieldId) :
	  GetIdMeaning (Id -> foreigntype (_))
	  GenerateCName (FieldId)
     
     'rule' GenerateFieldName (_, FieldId) :
	  GenerateCooLName (FieldId)
	  
-- GenerateDIVArgument -----------------------------------------------------
-- This predicate generates a /-expression. This rule is necessary, because 
-- in C is INT / INT = INT and in CooLV2.0 is INT / INT = FLOAT. To map this
-- rule correctly to C the operands of / have to be casted to float.
	 
'action' GenerateDIVArgument (EXPR)
     
     'rule' GenerateDIVArgument (Expr) :
	  [|
	       GetExpressionRealType (Expr -> Type)
	       FollowNameChain (Type -> Type2)
	       IsIntegerType (Type2)
	       Write ("(float)")
	  |]
	  GenerateEnclosedExpr (Expr)
	  
-- GenerateEqualArgument ---------------------------------------------------
-- expr_nil has to coded as C3IC3INIL (method types)
-- a temporary is necessary in the case of a call returning a composite type
	 
'action' GenerateEqualArgument (EXPR)
	  
     'rule' GenerateEqualArgument (expr_nil) :
	  Write ("&")
	  GenerateHiddenName ("NIL")
	  
     'rule' GenerateEqualArgument (Expr)
	  GetExpressionRealType (Expr -> RealType)
	  (|
	       FollowNameChain (RealType -> composite (array (_, _)))
	  ||
	       FollowNameChain (RealType -> composite (openarray (_)))
	  |)
	  GenerateEnclosedExpr (Expr)
     
     'rule' GenerateEqualArgument (Expr:call (_, _, _, _, _))
	  Temporaries -> N
	  Temporaries <- N + 1
	  Write ("( ")
	  GenerateTemporary (N)
	  Write (" = ")
	  GenerateEnclosedExpr (Expr)
	  Write (", &")
	  GenerateTemporary (N)
	  Write (")")
	  
     'rule' GenerateEqualArgument (Expr) :
	  Write ("&")
	  GenerateEnclosedExpr (Expr)
	  
-- NeededEqualFunction -----------------------------------------------------
	 
'condition' NeededEqualFunction (TYPE)
	 
     'rule' NeededEqualFunction (composite (record (_))) :
     'rule' NeededEqualFunction (composite (union (_))) :
     'rule' NeededEqualFunction (composite (array (_, _))) :
     'rule' NeededEqualFunction (composite (openarray (_))) :
     'rule' NeededEqualFunction (composite (method (_, _))) :
	  
-- IfArrayResultMakeNewFirstParameter --------------------------------------
	
'action' IfArrayResultMakeNewFirstParameter (TYPE, APARAMLIST)
	
     'rule' IfArrayResultMakeNewFirstParameter (Type, AParams)
	  FollowNameChain (Type -> composite (array( _, _ )))
	  Temporaries -> Number
	  GenerateTemporary (Number)
	  (|
	       where (AParams -> nil)
	  ||
	       Write (", ")
	  |)
	  Temporaries <- Number + 1

     'rule' IfArrayResultMakeNewFirstParameter (_, _)

-- GenerateActualParams ----------------------------------------------------
	
'action' GenerateActualParams (APARAMLIST, FPARAMLIST)

     'rule' GenerateActualParams (aparamlist( Param, nil ), FParamList)
	  GenerateActualParameter (Param, FParamList -> NewFParamList)
				
     'rule' GenerateActualParams (aparamlist( Param, List ), FParamList)
	  GenerateActualParameter (Param, FParamList -> NewFParamList)
	  Write (", ")
	  GenerateActualParams (List, NewFParamList)
     
     'rule' GenerateActualParams (nil, _)

-- GenerateActualParameter --------------------------------------------------

'action' GenerateActualParameter (APARAM, FPARAMLIST -> FPARAMLIST)

     'rule' GenerateActualParameter (in (stringliteral (_, Value)), 
				     FParamList -> NewFParamList)
	  NextFParamList (FParamList -> NewFParamList)
	  (|
	       IsFParamTypeArrayOfChar (FParamList)
	       Write ("(char *)")
	  ||
	       IsFParamTypeRefArrayOfChar (FParamList)
	       Write ("(char (*) [])")
	  ||
	       where (FParamList -> ellipsis (_))
	  |)
	  WriteString (Value)
     
     'rule' GenerateActualParameter (in (Expr:expr_nil), FParamList -> 
				     NewFParamList) :
	  NextFParamList (FParamList -> NewFParamList)
	  (|
	       IsFParamTypeMethod (FParamList)
	       GenerateHiddenName ("NIL")
	  ||
	       GenerateEnclosedExpr (Expr)
	  |)
	  
     'rule' GenerateActualParameter (in (Expr), FParamList -> NewFParamList)
	  GetExpressionSpecType (Expr -> Type)
	  GenParameterCast (FParamList, Type, in -> NewFParamList)
	  GenerateEnclosedExpr (Expr)

     'rule' GenerateActualParameter (AParam, FParamList -> NewFParamList)
	  (|
	       where (AParam -> out (Expr))
	       let (PMODE'out -> Mode)
	  ||
	       where (AParam -> inout (Expr))
	       let (PMODE'inout -> Mode)
	  |)
	  GetExpressionSpecType (Expr -> Type)
	  GenParameterCast (FParamList, Type, Mode -> NewFParamList)
	  FollowNameChain (Type -> Type2)
	  (|
	       where (Type2 -> composite (array (_, _)))
	  ||
	       where (Type2 -> composite (openarray (_)))
	  ||
	       Write ("&")
	  |)
	  GenerateEnclosedExpr (Expr)

-- IsFParamTypeArrayOfChar ---------------------------------------------------

'condition' IsFParamTypeArrayOfChar (FPARAMLIST) 

     'rule' IsFParamTypeArrayOfChar (fparamlist (fparam (_, _, _, TypeI), _))
	  FollowNameChainIndex (TypeI -> composite (ArrayType))
	  (|
	       where (ArrayType -> array (_, BaseTypeI))
	  ||
	       where (ArrayType -> openarray (BaseTypeI))
	  |)
	  FollowNameChainIndex (BaseTypeI -> simple (char))

-- IsFParamTypeRefArrayOfChar ------------------------------------------------

'condition' IsFParamTypeRefArrayOfChar (FPARAMLIST) 

     'rule' IsFParamTypeRefArrayOfChar (fparamlist (fparam (_, _, _, TypeI),
						    _))
	  FollowNameChainIndex (TypeI -> composite (ref (RefTypeI)))
	  FollowNameChainIndex (RefTypeI -> composite (ArrayType))
	  (|
	       where (ArrayType -> array (_, BaseTypeI))
	  ||
	       where (ArrayType -> openarray (BaseTypeI))
	  |)
	  FollowNameChainIndex (BaseTypeI -> simple (char))

-- IsFParamTypeMethod ------------------------------------------------

'condition' IsFParamTypeMethod (FPARAMLIST) 

     'rule' IsFParamTypeMethod (fparamlist (fparam (_, _, _, TypeI),
						    _))
	  FollowNameChainIndex (TypeI -> composite (method (_, _)))

-- GenParameterCast ---------------------------------------------------------
	  
'action' GenParameterCast (FParams : FPARAMLIST, ActualType : TYPE, 
			   Mode : PMODE -> FPARAMLIST)
     
     'rule' GenParameterCast (fparamlist (fparam (_, _, _, TypeI), Tail), 
			      ActualType, Mode -> Tail) :
	  TypeI'Type -> ExpectedType
	  GenerateCastIfNecessary (ActualType, ExpectedType, Mode)
	  
     'rule' GenParameterCast (FP:ellipsis (_), _, _ -> FP) :
	  
	  
-- NextFParamList -----------------------------------------------------------

'action' NextFParamList( FPARAMLIST -> FPARAMLIST )

    'rule' NextFParamList( fparamlist( _, List ) -> List )

    'rule' NextFParamList( List -> List )

-- GetFParamList ------------------------------------------------------------

'action' GetFParamList (RECEIVER -> FPARAMLIST)

     'rule' GetFParamList (proc (ProcId) -> FParams)
	  GetIdMeaning (ProcId -> proc (_, _, FParams, _))

     'rule' GetFParamList (foreignproc (ProcId) -> FParams)
	  GetIdMeaning (ProcId -> foreignproc (_, FParams, _))

     'rule' GetFParamList (method (_, MethodId) -> FParams)
	  GetIdMeaning (MethodId -> method (_, _, FParams, _)) 

     'rule' GetFParamList (localmethod (_, MethodId) -> FParams)
	  GetIdMeaning (MethodId -> method (_, _, FParams, _)) 

     'rule' GetFParamList (super (_, MethodId) -> FParams)
	  GetIdMeaning (MethodId -> method (_, _, FParams, _)) 

     'rule' GetFParamList (new (TypeI) -> FParams)
	  (|
	       FollowNameChainIndex (TypeI -> 
				     composite (classtype (_, _, Interf)))
	  ||
	       FollowNameChainIndex (TypeI -> 
				     generic (genericinst (TypeName, _)))
	       GetIdMeaning (TypeName -> type (TypeI2))
	       FollowNameChainIndex (TypeI2 -> 
				     composite (classtype (_, _, Interf)))
	  |)
	  Interf'Objparams -> FParams

     'rule' GetFParamList (procexpr (Function) -> FParams)
	  GetExpressionRealType (Function -> Type)
	  FollowNameChain (Type -> composite (procedure (FParams, _)))

     'rule' GetFParamList (methodexpr (Method) -> FParams)
	  GetExpressionRealType (Method -> Type)
	  FollowNameChain (Type -> composite (method (FParams, _)))

-- GenerateReceiver ---------------------------------------------------------
	
'action' GenerateReceiver (RECEIVER, TYPE, APARAMLIST)

     'rule' GenerateReceiver (proc (ProcName), _, _)
	  GenerateQualifiedCooLName (ProcName)
	  Write (" (")

     'rule' GenerateReceiver (foreignproc (ProcName), _, _)
	  GenerateCName (ProcName)
	  Write (" (")
     
     'rule' GenerateReceiver (method (Object, MethodId), Result, AParams) :
	  (|
	       where (Object -> call (_, _, _, _, _))
	  ||
	       ContainsCall (AParams)
	  |)
	  Temporaries -> Number
	  Temporaries <- Number + 1
	  GetExpressionRealType (Object -> RealType)
	  GenerateTemporary (Number)
	  Write ("= ")
	  GenerateCast (RealType, in)
	  GenerateEnclosedExpr (Object)
	  Write (", ")
	  GenerateTemporary (Number)
	  Write ("->C3IMTABR->")
	  GenerateCooLName (MethodId)
	  Write (" (") 
	  GenerateCast (RealType, in)
	  GenerateTemporary (Number)
	  IfExistsParamsGenerateCm (AParams, Result)
	  
     'rule' GenerateReceiver (method (Object, MethodId), Result, AParams) :
	  GetExpressionRealType (Object -> RealType)
	  GenerateEnclosedExpr (Object)
	  Write ("->C3IMTABR->")
	  GenerateCooLName (MethodId)
	  Write (" (")
	  GenerateCast (RealType, in)
	  GenerateEnclosedExpr (Object)
	  IfExistsParamsGenerateCm (AParams, Result)
	  
     'rule' GenerateReceiver (localmethod (ClassId, MethodId), Result, Params)
	  GenerateQualifiedCooLName (ClassId)
	  GenerateCooLName (MethodId)
	  Write (" (C3ICURRENT")
	  IfExistsParamsGenerateCm (Params, Result)

     'rule' GenerateReceiver (super (SuperClassId, MethodId), Result, Params)
	  GenerateMTabVarName (SuperClassId)
	  Write (".")
	  GenerateCooLName (MethodId)
	  Write ("((")
	  GenerateClassName (SuperClassId)
	  Write (") C3ICURRENT")
	  IfExistsParamsGenerateCm (Params, Result)

     'rule' GenerateReceiver (new (TypeIndex), _, _)
	  TypeIndex'Type -> composite (typename( ClassId))
	  GenerateQualifiedCooLName (ClassId)
	  Write ("C3INEW (")

     'rule' GenerateReceiver (new (TypeIndex), Result, Params)
	  TypeIndex'Type -> generic (genericinst (ClassId, ActualTypes))
	  GetIdMeaning (ClassId -> type (ClassTypeIndex))
	  GenerateQualifiedCooLName (ClassId)
	  Write ("C3INEW (")
	  GenerateTypeExpressionList (ActualTypes)
	  IfExistsParamsGenerateCm (Params, Result)

     'rule' GenerateReceiver (procexpr (Function), _, _)
	  GenerateEnclosedExpr (Function)
	  Write (" (")

     'rule' GenerateReceiver (methodexpr (MethodPtr), Result, Params)
	  GetExpressionRealType (MethodPtr -> ExprType)
	  FollowNameChain (ExprType -> composite (method (FParams, FResult)))
	  Temporaries -> Number
	  GenerateTemporary (Number)
	  Temporaries <- Number + 1
	  Write ("= ")
	  GenerateEnclosedExpr (MethodPtr)
	  Write (", (")
	  GenerateMethodCast (FParams, FResult)
	  GenerateTemporary (Number)
	  Write (".C3IMETHOD) (")
	  GenerateTemporary (Number)
	  Write (".C3IOBJECT")
	  IfExistsParamsGenerateCm (Params, Result)

-- ContainsCall -------------------------------------------------------------

'condition' ContainsCall (APARAMLIST)
     
     'rule' ContainsCall (aparamlist (Head, Tail)) :
	  (|
	       where (Head -> in (call (_, _, _, _, _)))
	  ||
	       ContainsCall (Tail)
	  |)
	  
-- IfExistsParamsGenerateCm -------------------------------------------------

'action' IfExistsParamsGenerateCm (APARAMLIST, TYPE)

     'rule' IfExistsParamsGenerateCm (nil, Result)
	  FollowNameChain (Result -> composite (array( _, _ )))
	  Write (", ")

     'rule' IfExistsParamsGenerateCm (nil, _)

     'rule' IfExistsParamsGenerateCm (aparamlist (_, _), _)
	  Write (",")

-- GenerateTypeExpressionList -----------------------------------------------

'action' GenerateTypeExpressionList (ActualParams : TYPEINDEXLIST)

     'rule' GenerateTypeExpressionList (typeindexlist (TypeI, nil))
	  FollowNameChainIndex (TypeI -> Type)
	  GenerateTypeExpression (Type)
	  
     'rule' GenerateTypeExpressionList (typeindexlist (TypeI, Tail))
	  FollowNameChainIndex (TypeI -> Type)
	  GenerateTypeExpression (Type)
	  Write (", ")
	  GenerateTypeExpressionList (Tail)

     'rule' GenerateTypeExpressionList (nil)

-- GenerateTypeExpression ---------------------------------------------------
	 
'action' GenerateTypeExpression (ActualType : TYPE)
	 
     'rule' GenerateTypeExpression (composite (classtype (Id, _, _))) :
	  Write ("\"")
	  GenerateClassName (Id)
	  Write ("\"")
	  
     'rule' GenerateTypeExpression (generic (genericinst (Id, _))) :
	  Write ("\"")
	  GenerateClassName (Id)
	  Write ("\"")
	  
     'rule' GenerateTypeExpression (generic (unconstrained (Id))) :
	  Write ("C3ICURRENT->C3IIVR->")
	  GenerateCooLName (Id)
	  Write (".C3IName")
	  
     'rule' GenerateTypeExpression (generic (constrained (Id, _))) :
	  Write ("C3ICURRENT->C3IIVR->")
	  GenerateCooLName (Id)
	  Write (".C3IName")

-- GenerateAppliedIdentifier ------------------------------------------------
	
'action' GenerateAppliedIdentifier( MEANING, ID )

    -- The following rules are used in connection with appling variables
    -- constants.
	
       -- const should not be generated by the context analysis
     'rule' GenerateAppliedIdentifier (const (_, _), Id)
	  GenerateQualifiedCooLName (Id)

     'rule' GenerateAppliedIdentifier (proc (Export, _, _, _), Id)
	  GenerateExportedName (Export, Id)

     'rule' GenerateAppliedIdentifier (globalvar (Export, _), Id)
	  GenerateExportedName (Export, Id)

     'rule' GenerateAppliedIdentifier (localvar (TypeI), Id)
	  GenerateCooLName (Id)
	  [|
	       IsFormalGenericTypeIndex (TypeI -> TypeId)
	       GenerateCooLName (TypeId)
	  |]

     'rule' GenerateAppliedIdentifier (instvar (TypeI), Id)
	  Write ("C3ICURRENT->C3IIVR->")
	  GenerateCooLName (Id)
	  [|
	       IsFormalGenericTypeIndex (TypeI -> TypeId)
	       GenerateCooLName (TypeId)
	  |]

     'rule' GenerateAppliedIdentifier (implicitvar, Id)
	  GetIdMeaning (Id -> implicitvar (TypeIndex))
	  Write( "((" )
	  GenerateTypeIndex (TypeIndex)
	  Write (")")
	  GenerateCooLName (Id)
	  Write (")")

     'rule' GenerateAppliedIdentifier (fparam (_, TypeI), Id)
	  FollowNameChainIndex (TypeI -> composite (array (_, _)))
	  GenerateCooLName (Id)
     
     'rule' GenerateAppliedIdentifier (fparam (_, TypeI), Id)
	  FollowNameChainIndex (TypeI -> composite (openarray (_)))
	  GenerateCooLName (Id)
     
     'rule' GenerateAppliedIdentifier (fparam (in, TypeI), Id)
	  GenerateCooLName (Id)
	  [|
	       IsFormalGenericTypeIndex (TypeI -> TypeId)
	       GenerateCooLName (TypeId)
	  |]
	  
     'rule' GenerateAppliedIdentifier (fparam (OtherMode, TypeI), Id)
	  Write ("(*")
	  GenerateCooLName (Id)
	  [|
	       IsFormalGenericTypeIndex (TypeI -> TypeId)
	       GenerateCooLName (TypeId)
	  |]
	  Write (")")

     'rule' GenerateAppliedIdentifier (foreignvar (_), Id)
	  GenerateCName (Id)

     'rule' GenerateAppliedIdentifier (foreignproc (_, _, _), Id) :
	  GenerateCName (Id)
	  
-- GenerateDyadicOperator ---------------------------------------------------

'action' GenerateDyadicOperator( DOP )

    'rule' GenerateDyadicOperator( or )
        Write( " || " )

    'rule' GenerateDyadicOperator( and )
        Write( " && " )

    'rule' GenerateDyadicOperator( eq )
        Write( " == " )

    'rule' GenerateDyadicOperator( ne )
       Write( " != " )

    'rule' GenerateDyadicOperator( lt )
       Write( " < " )

    'rule' GenerateDyadicOperator( le )
       Write( " <= " )

    'rule' GenerateDyadicOperator( gt )
       Write( " > " )

    'rule' GenerateDyadicOperator( ge )
       Write( " >= " )

    'rule' GenerateDyadicOperator( plus )
       Write( " + " )

    'rule' GenerateDyadicOperator( minus )
       Write( " - " )

    'rule' GenerateDyadicOperator( times )
        Write( " * " )

    'rule' GenerateDyadicOperator( div )
        Write( " / " )

    'rule' GenerateDyadicOperator( idiv )
        Write( " / " )

    'rule' GenerateDyadicOperator( mod )
        Write( " % " )

-- GenerateMonadicOperator --------------------------------------------------

'action' GenerateMonadicOperator( MOP )

    'rule' GenerateMonadicOperator( plus )
        Write( "+" )

    'rule' GenerateMonadicOperator( minus )
        Write( "-" )

    'rule' GenerateMonadicOperator( not )
        Write( "!" )

-- GetRtsStringRelOpFunction -------------------------------------------------

'action' GetRtsStringRelOpFunction (DOP -> STRING)	 
     
     'rule' GetRtsStringRelOpFunction (eq -> "C3IC3IEqualString") :
     'rule' GetRtsStringRelOpFunction (ne -> "C3IC3INotEqualString") :
     'rule' GetRtsStringRelOpFunction (gt -> "C3IC3IGreaterString") :
     'rule' GetRtsStringRelOpFunction (ge -> "C3IC3IGreaterEqualString") :
     'rule' GetRtsStringRelOpFunction (lt -> "C3IC3ILessString") :
     'rule' GetRtsStringRelOpFunction (le -> "C3IC3ILessEqualString") :
	  
-- IsStringConcatenation -----------------------------------------------
	  
'condition' IsStringConcatenation (EXPR, EXPR -> STRING)
     
     'rule' IsStringConcatenation (Left, Right -> "C3IC3IConcatStrings") :
	  GetExpressionRealType (Left -> LeftType)
	  FollowNameChain (LeftType -> LeftType2)
	  IsStringType (LeftType2)
	  GetExpressionRealType (Right -> RightType)
	  FollowNameChain (RightType -> RightType2)
	  IsStringType (RightType2)
	  
     'rule' IsStringConcatenation (Left, Right -> "C3IC3IAppendChar") :
	  GetExpressionRealType (Left -> LeftType)
	  FollowNameChain (LeftType -> LeftType2)
	  IsStringType (LeftType2)
	  GetExpressionRealType (Right -> RightType)
	  FollowNameChain (RightType -> simple (char))
     
     'rule' IsStringConcatenation (Left, Right -> "C3IC3IPrependChar") :
	  GetExpressionRealType (Left -> LeftType)
	  FollowNameChain (LeftType -> simple (char))
	  GetExpressionRealType (Right -> RightType)
	  FollowNameChain (RightType -> RightType2)
	  IsStringType (RightType2)
     
     'rule' IsStringConcatenation (Left, Right -> "C3IC3IConcatChars") :
	  GetExpressionRealType (Left -> LeftType)
	  FollowNameChain (LeftType -> simple (char))
	  GetExpressionRealType (Right -> RightType)
	  FollowNameChain (RightType -> simple (char))
		  
-- GetExpressionRealType ----------------------------------------------------

'action' GetExpressionRealType (EXPR -> TYPE)
     
     'rule' GetExpressionRealType (call (_, _, RealType, _, _) -> RealType) :
	  
     'rule' GetExpressionRealType (Expr -> Type) :
	  GetExpressionType (Expr -> Type)

-- GetExpressionSpecType ----------------------------------------------------

'action' GetExpressionSpecType (EXPR -> TYPE)
     
     'rule' GetExpressionSpecType (call (_, SpecTypeI, _, _, _) -> SpecType) :
	  SpecTypeI'Type -> SpecType
	  
     'rule' GetExpressionSpecType (Expr -> Type) :
	  GetExpressionType (Expr -> Type)

-- GetExpressionType --------------------------------------------------------

-- This predicate returns the top level type of an expression. The returned
-- type must not be used afterwards excepted to test this type.

'action' GetExpressionType( EXPR -> TYPE )

    'rule' GetExpressionType( false( _ ) -> simple( bool ) )
    'rule' GetExpressionType( true( _ ) -> simple( bool ) )
    'rule' GetExpressionType( posintliteral( _, _ ) -> simple( int ) )
    'rule' GetExpressionType( negintliteral( _, _ ) -> simple( int ) )
    'rule' GetExpressionType( doubleliteral( _, _ ) -> simple( double ) )
    'rule' GetExpressionType( charliteral( _, _ ) -> simple( char ) )
    'rule' GetExpressionType( stringliteral( _, _ ) -> simple( string ) )
	      
     'rule' GetExpressionType (expr_nil (_) -> simple (niltype)) :
	  
    'rule' GetExpressionType( dyop( _, TypeIndex, _, _, _ ) -> ActualType )
        TypeIndex'Type -> ActualType

    'rule' GetExpressionType( monop( _, TypeIndex, _, _ ) -> ActualType )
        TypeIndex'Type -> ActualType

     'rule' GetExpressionType (adr (_, _ ) -> simple (address))

    'rule' GetExpressionType( sizeof( _, _ ) -> simple( int ) )

    'rule' GetExpressionType( fieldsel( _, TypeIndex, _, _ ) -> ActualType )
        TypeIndex'Type -> ActualType

    'rule' GetExpressionType( unionsel( _, TypeIndex, _, _ ) -> ActualType )
        TypeIndex'Type -> ActualType

    'rule' GetExpressionType( enumsel( _, _, _, _ ) -> simple( int ) )

    'rule' GetExpressionType( methodpointer( _, _, Object, Id ) -> 
			      composite( method( Params, Result ) ) )
        GetIdMeaning( Id -> method( _, _, Params, Result ) )

    'rule' GetExpressionType( arraysubscr( _, TypeIndex, _, _ ) -> ActualType )
        TypeIndex'Type -> ActualType

    'rule' GetExpressionType( stringsubscr( _, _, _ ) -> simple (char) )

    'rule' GetExpressionType( substring( _, _, _ ) -> simple( string ) )

    'rule' GetExpressionType( deref( _, TypeIndex, _ ) -> ActualType )
        TypeIndex'Type -> ActualType

    'rule' GetExpressionType( applied( _, Id ) -> ActualType )
        GetTypeOfIdentifier( Id -> ActualType )

    'rule' GetExpressionType( current( _ ) -> ActualType )
        Current -> CurrentId
        CurrentId'Meaning -> type( TypeIndex )
        FollowNameChainIndex( TypeIndex -> ActualType )

-- GetTypeOfIdentifier ------------------------------------------------------
	
'action' GetTypeOfIdentifier( ID -> TYPE )

    'rule' GetTypeOfIdentifier( Ident -> ActualType )
        Ident'Meaning -> definingid( Id2 )
        GetTypeOfIdentifier( Id2 -> ActualType )

     'rule' GetTypeOfIdentifier (Ident -> 
				 composite (procedure (FParams, Result)))
	  Ident'Meaning -> proc (_, _, FParams, Result)

    'rule' GetTypeOfIdentifier( Ident -> ActualType )
        Ident'Meaning -> const( Type, _ )
        Type'Type -> ActualType

    'rule' GetTypeOfIdentifier( Ident -> ActualType )
        Ident'Meaning -> globalvar( _, Type )
        Type'Type -> ActualType

    'rule' GetTypeOfIdentifier( Ident -> ActualType )
        Ident'Meaning -> localvar( Type )
        Type'Type -> ActualType

    'rule' GetTypeOfIdentifier( Ident -> ActualType )
        Ident'Meaning -> instvar( Type )
        Type'Type -> ActualType

    'rule' GetTypeOfIdentifier( Ident -> ActualType )
        Ident'Meaning -> implicitvar( Type )
        Type'Type -> ActualType

    'rule' GetTypeOfIdentifier( Ident -> ActualType )
        Ident'Meaning -> fparam( _, Type )
        Type'Type -> ActualType

    'rule' GetTypeOfIdentifier( Ident -> ActualType )
        Ident'Meaning -> foreignvar( Type )
        Type'Type -> ActualType

    'rule' GetTypeOfIdentifier( Ident -> ActualType )
        Ident'Meaning -> foreignproc( _, _, Type )
        Type'Type -> ActualType

-- GetArrayDimension --------------------------------------------------------

-- This predicate returns the C-dimension of an array type.

'action' GetArrayDimension (EXPR -> INT)

     'rule' GetArrayDimension (range (Pos, Lwb, Upb) -> UpbVal - LwbVal + 1)
        GetConstINTValue (Lwb -> LwbVal)
        GetConstINTValue (Upb -> UpbVal)

-- GetConstValue ------------------------------------------------------------

'action' GetConstINTValue (EXPR -> INT)

     'rule' GetConstINTValue (posintliteral (_, Value) -> Value)

     'rule' GetConstINTValue (negintliteral( _, Value) -> - Value)

     'rule' GetConstINTValue (applied (_, Ident) -> Value)
	  GetIdMeaning (Ident -> const (_, Expr))
	  GetConstINTValue (Expr -> Value)

