'module' const

'export'
     IsInvalidConstRec
     IsEqualConstRange CheckArrayRange
     CheckConstantDefinition CheckVariableInitialization 
     EvalConstant
     CheckAnd CheckOr 
     CheckEqCompare CheckNeCompare
     CheckConstEqCompare CheckConstNeCompare 
     CheckLtCompare CheckLeCompare CheckGtCompare CheckGeCompare
     CheckPlusArith CheckMinusArith CheckTimesArith CheckDivArith
     CheckIDivArith CheckModArith
     
'use' ast extspecs misc decls types

-- ========================================================================
--  Check on invalid recursion of constants
--  Example :
--    CONST C : INT = C;
--  The additional parameter of IsInvalidConstRec_h contains a list of
--  names of constants used within the constant definition. If a name of a
--  constant stored in the list is applied inside the constant definition, 
--  a recursive constant definition is detected.
-- ========================================================================

'condition' IsInvalidConstRec (name : ID, value : EXPR)

     'rule' IsInvalidConstRec (Id, Value) :
	  IsInvalidConstRec_h (Value, idlist (Id, nil))
	  
'condition' IsInvalidConstRec_h (value : EXPR, usedNames : IDLIST)

     'rule' IsInvalidConstRec_h (dyop (_, _, _, Left, _), L)
	  IsInvalidConstRec_h (Left, L)
     
     'rule' IsInvalidConstRec_h (dyop (_, _, _, _, Right), L)
	  IsInvalidConstRec_h (Right, L)
     
     'rule' IsInvalidConstRec_h (monop (_, _, _, Operand), L) :
	  IsInvalidConstRec_h (Operand, L)
	  
     'rule' IsInvalidConstRec_h (applied (_, Id), L) :
	  GetIdMeaning (Id -> const (_, _))
	  IsQualifiedIdInList (Id, L)

     'rule' IsInvalidConstRec_h (applied (_, Id), L) :
	  GetIdMeaning (Id -> const (_, Value))
	  IsInvalidConstRec_h (Value, idlist (Id, L))

-- ========================================================================
--  Array ranges
--  CheckArrayRange (E -> Range) checks whether E is a constant range of 
--    type INT with an positive number of elements. Range is the evaluated
--    constant range. The lower bound has to be 1 until the MaX interface
--    concering array indexes is implemented.
--  IsEqualConstRange (E1, E2) checks the equivalence of the ranges E1, E2. 
-- ========================================================================
       
'condition' CheckArrayRange (EXPR -> EXPR) 

     'rule' CheckArrayRange (range (Pos, Lwb,  Upb) -> 
			     range (Pos, LwbVal, UpbVal)) :
	  EvalConstant (Lwb -> LwbType, LwbVal)
	  FollowNameChain (LwbType -> LwbType2)
	  IsIntegerType (LwbType2)
	  [|
               where (LwbVal -> posintliteral (_, N))
               ne (N, 1)
	       Lwb'Pos -> LPos
	       Error ("lower bound of array index unequal 1 : not yet implemented", LPos)
	  |]
	  EvalConstant (Upb -> UpbType, UpbVal)
	  FollowNameChain (UpbType -> UpbType2)
	  IsIntegerType (UpbType2)
	  (|
	       IsErrorType (LwbType2)
	  ||
	       IsErrorType (UpbType2)
	  ||
	       EvalConstLe (Pos, LwbVal, UpbVal -> true (_))
	  |)
	  
     'rule' CheckArrayRange (range (Pos, _, _) -> error (Pos))
	  Error ("positive INT range expected", Pos)

---------------------------------------------------------------------------

'condition' IsEqualConstRange (EXPR, EXPR)

     'rule' IsEqualConstRange (range (Pos1, Lwb1,  Upb1), 
			       range (Pos2, Lwb2,  Upb2)) :
	  UnspecTypeIndex -> TypeI
	  EvalConstant (dyop (Pos1, TypeI, eq, Lwb1, Lwb2) -> Type1, Val1)
	  EvalConstant (dyop (Pos1, TypeI, eq, Upb1, Upb2) -> Type2, Val2)
	  (|
	       IsErrorType (Type1)
	  ||
	       IsErrorType (Type2)
	  ||
	       where (Val1 -> true (_))
	       where (Val2 -> true (_))
	  |)
	  
-- ========================================================================
--  CheckConstantDefinition evaluates the given expression as a constant
--    expression and checks the given type. The type of the expression must
--    be compatible to the given type.
--  CheckVariableInitialization evaluates the given expression as a constant
--    expression.
-- ========================================================================

'action' CheckConstantDefinition (TYPEINDEX, EXPR -> EXPR)
     
     'rule' CheckConstantDefinition (TypeI, Expr -> Expr2) :
	  FollowNameChainIndex (TypeI -> Type)
	  Expr'Pos -> ExprPos
	  TypeI'Pos -> TypePos
	  (|
	       IsConstantType (Type)
	       (|
		    IsValidConstantInit (Type, Expr -> Expr2)
	       ||
		    Error ("incompatible types in constant definition", 
			   ExprPos)
		    let (EXPR'error (ExprPos) -> Expr2)
		    TypeI'Type <- error
	       |)
	  ||
	       Error ("constant type expected", TypePos)
	       let (EXPR'error (ExprPos) -> Expr2)
	       TypeI'Type <- error
	  |)

---------------------------------------------------------------------------
	  
'action' CheckVariableInitialization (TYPEINDEX, EXPR -> EXPR)
     
     'rule' CheckVariableInitialization (TypeI, Init -> Init2) :
	  FollowNameChainIndex (TypeI -> Type)
	  Init'Pos -> Pos
	  (|
	       where (Init -> nil (_))
	       let (Init -> Init2)
	  ||
	       IsValidConstantInit (Type, Init -> Init2)
	  ||
	       Error ("incompatible types in variable initialization", Pos)
	       let (EXPR'error (Pos) -> Init2)
	  |)
	  
---------------------------------------------------------------------------
	  
'condition' IsValidConstantInit (TYPE, EXPR -> EXPR)
     
     'rule' IsValidConstantInit (error, Expr -> error (Pos)) :
	  Expr'Pos -> Pos
	  
     'rule' IsValidConstantInit (Type, Expr -> Value2) :
	  EvalConstant (Expr -> ExprType, Value)
	  FollowNameChain (ExprType -> ExprType2)
	  IsAssignmentCompatible (ExprType2, Type)
	  ConvertConstValue (Type, Value -> Value2)

---------------------------------------------------------------------------
	  
'action' ConvertConstValue (TYPE, EXPR -> EXPR)
     
     'rule' ConvertConstValue (Type, Value -> error (Pos)) :
	  IsErrorType (Type)
	  Value'Pos -> Pos
	  
     'rule' ConvertConstValue (Type, Value -> doubleliteral (Pos, D2)) :
	  IsFloatingPointType (Type)
	  (|
	       where (Value -> posintliteral (Pos, I))
	       IntToDouble (I -> D2)
	  ||
	       where (Value -> negintliteral (Pos, I))
	       IntToDouble (I -> D1)
	       NegateDouble (D1 -> D2)
	  |)
     
     'rule' ConvertConstValue (Type, doubleliteral (Pos, D) -> L) :
	  IsIntegerType (Type)
	  (|
	       IsNegativeDouble (D)
	       NegateDouble (D -> D2)
	       DoubleToInt (D2 -> I)
	       let (negintliteral (Pos, I) -> L)
	  ||
	       DoubleToInt (D -> I)
	       let (posintliteral (Pos, I) -> L)
	  |)
	  
     'rule' ConvertConstValue (_ , Value -> Value) :
	  
-- ========================================================================
--  Evalutation of constant expressions.
-- ========================================================================

'action' EvalConstant (ConstExpr : EXPR -> TYPE, EXPR)
     
     'rule' EvalConstant (C:false (_) -> simple (bool), C) :
     'rule' EvalConstant (C:true (_) -> simple (bool), C) :
     'rule' EvalConstant (C:posintliteral (_, _) -> 
			  simple (intliteraltype), C) :
     'rule' EvalConstant (C:negintliteral (_, _) -> 
			  simple (intliteraltype), C) :
     'rule' EvalConstant (C:doubleliteral (_, _) -> 
			  simple (doubleliteraltype), C) :
     'rule' EvalConstant (C:charliteral (_, _) -> simple (char), C) :
     'rule' EvalConstant (C:stringliteral (_, _) -> simple (string), C) :
     'rule' EvalConstant (C:expr_nil (_) -> simple (niltype), C) :
	  
     'rule' EvalConstant (dyop (Pos, _, DyOp, Left, Right) -> Type, Value) :
	  EvalConstant (Left -> LType, LVal)
	  EvalConstant (Right -> RType, RVal)
	  FollowNameChain (LType -> LType2)
	  FollowNameChain (RType -> RType2)
	  EvalConstDyOp (Pos, DyOp, LType2, LVal, RType2, RVal ->
			 Type, Value)
     
     'rule' EvalConstant (monop (Pos, _, MonOp, Operand) -> Type, Value) :
	  EvalConstant (Operand -> OpType, OpValue)
	  FollowNameChain (OpType -> OpType2)
	  EvalConstMonOp (Pos, MonOp, OpType2, OpValue -> Type, Value)

     'rule' EvalConstant (dot (Pos, EnumDesig:applied (_, _), Id) -> 
			  Type, Value) :
	  EvalConstantEnumSelection (Pos, EnumDesig, Id -> Type, Value)
	  
     'rule' EvalConstant (Expr:enumsel (_, TypeI, _, _) -> Type, Expr)
	  TypeI'Type -> Type
	  
     'rule' EvalConstant (applied (Pos, Id) -> Type, Value)
	  CheckId (Id)
	  GetIdMeaning (Id -> Meaning)
	  EvalConstantId (Id, Meaning -> Type, Value)
	  
     'rule' EvalConstant (Expr:error -> error, Expr) :
	  
     'rule' EvalConstant (Expr -> error, error (Pos)) :
	  Expr'Pos -> Pos
	  Error ("invalid constant expression", Pos)
	  
-- ========================================================================
--  Evalutation of dyadic expressions.
-- ========================================================================

'action' EvalConstDyOp (POS, DOP, LeftType : TYPE, LeftValue : EXPR, 
			RightType : TYPE, RightValue : EXPR ->
			TYPE, EXPR)
     
     'rule' EvalConstDyOp (Pos, _, T, _, _, _ -> error, error (Pos)) :
	  IsErrorType (T)
	  
     'rule' EvalConstDyOp (Pos, _, _, _, T, _ -> error, error (Pos)) :
	  IsErrorType (T)

     'rule' EvalConstDyOp (Pos, or, LType, LVal, RType, RVal -> 
			   Type, Value) :
	  CheckOr (Pos, LType, RType -> Type)
	  (|
	       where (Type -> error)
	       let (EXPR'error (Pos) -> Value)
	  ||
	       EvalConstOr (Pos, LVal, RVal -> Value)
	  |)
	  
     'rule' EvalConstDyOp (Pos, and, LType, LVal, RType, RVal -> 
			   Type, Value) :
	  CheckAnd (Pos, LType, RType -> Type)
	  (|
	       where (Type -> error)
	       let (EXPR'error (Pos) -> Value)
	  ||
	       EvalConstAnd (Pos, LVal, RVal -> Value)
	  |)
	  
     'rule' EvalConstDyOp (Pos, eq, LType, LVal, RType, RVal-> Type, Value) :
	  CheckConstEqCompare (Pos, LType, RType -> Type)
	  (|
	       where (Type -> error)
	       let (EXPR'error (Pos) -> Value)
	  ||
	       EvalConstEq (Pos, LVal, RVal -> Value)
	  |)
	       

     'rule' EvalConstDyOp (Pos, ne, LType, LVal, RType, RVal-> Type, Value) :
	  CheckConstNeCompare (Pos, LType, RType -> Type)
	  (|
	       where (Type -> error)
	       let (EXPR'error (Pos) -> Value)
	  ||
	       EvalConstNe (Pos, LVal, RVal -> Value)
	  |)
	       

     'rule' EvalConstDyOp (Pos, lt, LType, LVal, RType, RVal-> Type, Value) :
	  CheckLtCompare (Pos, LType, RType -> Type)
	  (|
	       where (Type -> error)
	       let (EXPR'error (Pos) -> Value)
	  ||
	       EvalConstLt (Pos, LVal, RVal -> Value)
	  |)
	       

     'rule' EvalConstDyOp (Pos, le, LType, LVal, RType, RVal-> Type, Value) :
	  CheckLeCompare (Pos, LType, RType -> Type)
	  (|
	       where (Type -> error)
	       let (EXPR'error (Pos) -> Value)
	  ||
	       EvalConstLe (Pos, LVal, RVal -> Value)
	  |)
	       

     'rule' EvalConstDyOp (Pos, gt, LType, LVal, RType, RVal-> Type, Value) :
	  CheckGtCompare (Pos, LType, RType -> Type)
	  (|
	       where (Type -> error)
	       let (EXPR'error (Pos) -> Value)
	  ||
	       EvalConstGt (Pos, LVal, RVal -> Value)
	  |)
	       

     'rule' EvalConstDyOp (Pos, plus, LType, LVal, RType, RVal -> 
			   Type, Value) :
	  CheckPlusArith (Pos, LType, RType -> Type)
	  (|
	       where (Type -> error)
	       let (EXPR'error (Pos) -> Value)
	  ||
	       EvalConstPlus (Pos, LVal, RVal -> Value)
	  |)
	  
     'rule' EvalConstDyOp (Pos, minus, LType, LVal, RType, RVal -> 
			   Type, Value) :
	  CheckMinusArith (Pos, LType, RType -> Type)
	  (|
	       where (Type -> error)
	       let (EXPR'error (Pos) -> Value)
	  ||
	       EvalConstMinus (Pos, LVal, RVal -> Value)
	  |)
	  
     
     'rule' EvalConstDyOp (Pos, times, LType, LVal, RType, RVal -> 
			   Type, Value) :
	  CheckTimesArith (Pos, LType, RType -> Type)
	  (|
	       where (Type -> error)
	       let (EXPR'error (Pos) -> Value)
	  ||
	       EvalConstTimes (Pos, LVal, RVal -> Value)
	  |)
	  
     
     'rule' EvalConstDyOp (Pos, div, _, _, _, RVal -> error, error (Pos)) :
	  IsZeroConstValue (RVal)
	  Error ("division by zero", Pos)
	  
      'rule' EvalConstDyOp (Pos, div, LType, LVal, RType, RVal -> 
			   Type, Value) :
	  CheckDivArith (Pos, LType, RType -> Type)
	  (|
	       where (Type -> error)
	       let (EXPR'error (Pos) -> Value)
	  ||
	       EvalConstDiv (Pos, LVal, RVal -> Value)
	  |)
	  
     
     'rule' EvalConstDyOp (Pos, idiv, _, _, _, RVal-> error, error (Pos)) :
	  IsZeroConstValue (RVal)
	  Error ("division by zero", Pos)
	  
     'rule' EvalConstDyOp (Pos, idiv, LType, LVal, RType, RVal -> 
			   Type, Value) :
	  CheckIDivArith (Pos, LType, RType -> Type)
	  (|
	       where (Type -> error)
	       let (EXPR'error (Pos) -> Value)
	  ||
	       EvalConstIDiv (Pos, LVal, RVal -> Value)
	  |)
	  
     
     'rule' EvalConstDyOp (Pos, mod, LType, LVal, RType, RVal -> 
			   Type, Value) :
	  CheckModArith (Pos, LType, RType -> Type)
	  (|
	       where (Type -> error)
	       let (EXPR'error (Pos) -> Value)
	  ||
	       EvalConstMod (Pos, LVal, RVal -> Value)
	  |)
	  
---------------------------------------------------------------------------

'condition' CheckAnd (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckAnd (_ , T:simple (bool), simple (bool) -> T) :
     
     'rule' CheckAnd (Pos, _, _ -> error) :
	  Error ("invalid operands of 'AND'", Pos)

---------------------------------------------------------------------------

'condition' CheckOr (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckOr (_ , T:simple (bool), simple (bool) -> T) :
	  
     'rule' CheckOr (Pos, _, _ -> error) :
	  Error ("invalid operands of 'OR'", Pos)

---------------------------------------------------------------------------

'condition' CheckEqCompare (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckEqCompare (Pos , LType, RType -> simple (bool)) :
	  (|
	       IsAssignmentCompatible (LType, RType)
	  ||
	       IsAssignmentCompatible (RType, LType)
	  |)
	  
     'rule' CheckEqCompare (Pos, _, _ -> error) :
	  Error ("invalid operands of '='", Pos)

---------------------------------------------------------------------------

'condition' CheckNeCompare (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckNeCompare (Pos , LType, RType -> simple (bool)) :
	  (|
	       IsAssignmentCompatible (LType, RType)
	  ||
	       IsAssignmentCompatible (RType, LType)
	  |)
     
     'rule' CheckNeCompare (Pos, _, _ -> error) :
	  Error ("invalid operands of '<>'", Pos)

---------------------------------------------------------------------------

'condition' CheckConstEqCompare (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckConstEqCompare (_, composite (enum (EnumList1)), 
				 composite (enum (EnumList2)) -> 
				 simple (bool)) :
	  IsEquivEnumList (EnumList1, EnumList2)
	  
     'rule' CheckConstEqCompare (_ , simple (LType), simple(RType) -> 
				 simple (bool)) :
	  (|
	       IsRelOpCompare (LType, RType)
	  ||
	       IsBoolCompare (LType, RType)
	  |)
     
     'rule' CheckConstEqCompare (Pos, _, _ -> error) :
	  Error ("invalid operands of '='", Pos)

---------------------------------------------------------------------------

'condition' CheckConstNeCompare (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckConstNeCompare (_, composite (enum (EnumList1)), 
				 composite (enum (EnumList2)) -> 
				 simple (bool)) :
	  IsEquivEnumList (EnumList1, EnumList2)
     
     'rule' CheckConstNeCompare (_ , simple (LType), simple(RType) -> 
				 simple (bool)) :
	  (|
	       IsRelOpCompare (LType, RType)
	  ||
	       IsBoolCompare (LType, RType)
	  |)
     
     'rule' CheckConstNeCompare (Pos, _, _ -> error) :
	  Error ("invalid operands of '<>'", Pos)

---------------------------------------------------------------------------

'condition' CheckLtCompare (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckLtCompare (_, composite (enum (EnumList1)), 
			    composite (enum (EnumList2)) -> 
			    simple (bool)) :
	  IsEquivEnumList (EnumList1, EnumList2)
	  
     'rule' CheckLtCompare (_ , simple (LType), simple(RType) -> 
			    simple (bool)) :
	  IsRelOpCompare (LType, RType)
	  
     'rule' CheckLtCompare (Pos, _, _ -> error) :
	  Error ("invalid operands of '<'", Pos)

---------------------------------------------------------------------------

'condition' CheckLeCompare (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckLeCompare (_, composite (enum (EnumList1)), 
				 composite (enum (EnumList2)) -> 
				 simple (bool)) :
	  IsEquivEnumList (EnumList1, EnumList2)
	  
     'rule' CheckLeCompare (_ , simple (LType), simple(RType) -> 
			    simple (bool)) :
	  IsRelOpCompare (LType, RType)
	  
     'rule' CheckLeCompare (Pos, _, _ -> error) :
	  Error ("invalid operands of '<='", Pos)

---------------------------------------------------------------------------

'condition' CheckGtCompare (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckGtCompare (_, composite (enum (EnumList1)), 
			    composite (enum (EnumList2)) -> 
			    simple (bool)) :
	  IsEquivEnumList (EnumList1, EnumList2)
	  
     'rule' CheckGtCompare (_ , simple (LType), simple(RType) -> 
			    simple (bool)) :
	  IsRelOpCompare (LType, RType)
	  
     'rule' CheckGtCompare (Pos, _, _ -> error) :
	  Error ("invalid operands of '>'", Pos)

---------------------------------------------------------------------------

'condition' CheckGeCompare (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckGeCompare (_, composite (enum (EnumList1)), 
			    composite (enum (EnumList2)) -> 
			    simple (bool)) :
	  IsEquivEnumList (EnumList1, EnumList2)
	  
     'rule' CheckGeCompare (_ , simple (LType), simple(RType) -> 
			    simple (bool)) :
	  IsRelOpCompare (LType, RType)
	  
     'rule' CheckGeCompare (Pos, _, _ -> error) :
	  Error ("invalid operands of '>='", Pos)

---------------------------------------------------------------------------

'condition' CheckPlusArith (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckPlusArith (_ , simple (LType), simple(RType) ->
			    simple(Type)) :
	  (|
	       IsIntArith (LType, RType -> Type)
	  ||
	       IsFloatingPointArith (LType, RType -> Type)
	  ||
	       IsStringArith (LType, RType -> Type)
	  ||
	       IsAddressPlusArith (LType, RType -> Type)
	  |)
     
     'rule' CheckPlusArith (Pos, _, _ -> error) :
	  Error ("invalid operands of '+'", Pos)

---------------------------------------------------------------------------
	  
'condition' CheckMinusArith (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckMinusArith (_ , simple (LType), simple(RType) ->
			     simple(Type)) :
	  (|
	       IsIntArith (LType, RType -> Type)
	  ||
	       IsFloatingPointArith (LType, RType -> Type)
	  ||
	       IsAddressMinusArith (LType, RType -> Type)
	  |)
     
     'rule' CheckMinusArith (Pos, _, _ -> error) :
	  Error ("invalid operands of '-'", Pos)

---------------------------------------------------------------------------
	  
'condition' CheckTimesArith (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckTimesArith (_ , simple (LType), simple(RType) ->
			     simple(Type)) :
	  (|
	       IsIntArith (LType, RType -> Type)
	  ||
	       IsFloatingPointArith (LType, RType -> Type)
	  |)
     
     'rule' CheckTimesArith (Pos, _, _ -> error) :
	  Error ("invalid operands of '*'", Pos)

---------------------------------------------------------------------------
	  
'condition' CheckDivArith (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckDivArith (_ , simple (LType), simple(RType) ->
			   simple(Type)) :
	  (|
	       IsIntArith (LType, RType -> _)
	       let (float -> Type)
	  ||
	       IsFloatingPointArith (LType, RType -> Type)
	  |)
     
     'rule' CheckDivArith (Pos, _, _ -> error) :
	  Error ("invalid operands of '/'", Pos)

---------------------------------------------------------------------------
	  
'condition' CheckIDivArith (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckIDivArith (_ , simple (LType), simple(RType) ->
			    simple(Type)) :
	  IsIntArith (LType, RType -> Type)
     
     'rule' CheckIDivArith (Pos, _, _ -> error) :
	  Error ("invalid operands of 'DIV'", Pos)

---------------------------------------------------------------------------
	  
'condition' CheckModArith (POS, TYPE, TYPE -> TYPE)
     
     'rule' CheckModArith (_ , simple (LType), simple(RType) ->
			   simple(Type)) :
	  IsIntArith (LType, RType -> Type)
     
     'rule' CheckModArith (Pos, _, _ -> error) :
	  Error ("invalid operands of 'MOD'", Pos)

---------------------------------------------------------------------------

'condition' IsZeroConstValue (EXPR)
     
     'rule' IsZeroConstValue (posintliteral (_, Val)) :
	  eq (Val, 0)
     
     'rule' IsZeroConstValue (negintliteral (_, Val)) :
	  eq (Val, 0)
	  
     'rule' IsZeroConstValue (doubleliteral (_, Val) ) :
	  IsZeroDouble (Val)
	  
-- ========================================================================
--  Dyadic relational operations
-- ========================================================================

'action' EvalConstEq (POS, EXPR, EXPR -> EXPR)

     'rule' EvalConstEq (Pos, posintliteral (_, I1), 
			 posintliteral (_, I2) -> true (Pos)) :
	  eq (I1, I2)
     
     'rule' EvalConstEq (Pos, negintliteral (_, I1), 
			 negintliteral (_, I2) -> true (Pos)) :
	  eq (I1, I2)
	  
     'rule' EvalConstEq (Pos, doubleliteral (_, D1), 
			  doubleliteral (_, D2) -> true (Pos)) :
	  IsEqualDouble (D1, D2)
			      
     'rule' EvalConstEq (Pos, doubleliteral (_, D1), L2 -> true (Pos)) :
	  IntLiteralToDoubleLiteral (L2 -> doubleliteral (_, D2))
	  IsEqualDouble (D1, D2)
	  
     'rule' EvalConstEq (Pos, L1, doubleliteral (_, D2) -> true (Pos)) :
	  IntLiteralToDoubleLiteral (L1 -> doubleliteral (_, D1))
	  IsEqualDouble (D1, D2)
	  
     'rule' EvalConstEq (Pos, true (_), true (_) -> true (Pos)) :
     
     'rule' EvalConstEq (Pos, false (_), false (_) -> true (Pos)) :
     
     'rule' EvalConstEq (Pos, charliteral (_, C1), 
			charliteral (_, C2) -> true (Pos)) :
	  IsEqualChar (C1, C2)
	  
     'rule' EvalConstEq (Pos, stringliteral (_, S1), 
			 stringliteral (_, S2) -> true (Pos)) :
	  IsEqualString (S1, S2)
     
     'rule' EvalConstEq (Pos, enumsel (_, _, _, N1), 
			 enumsel (_, _, _, N2) -> true (Pos)) :
	  eq (N1, N2)
	  
     'rule' EvalConstEq (Pos, _, _ -> false (Pos)) :

---------------------------------------------------------------------------
	  
'action' EvalConstNe (POS, EXPR, EXPR -> EXPR)
     
     'rule' EvalConstNe (Pos, L1, L2 -> NotVal) :
 	  EvalConstEq (Pos, L1, L2 -> Val)
	  EvalConstNot (Val -> NotVal)
	  
---------------------------------------------------------------------------

'action' EvalConstLt (POS, EXPR, EXPR -> EXPR)
     
     'rule' EvalConstLt (Pos, posintliteral (_, I1), 
			 posintliteral (_, I2) -> true (Pos)) :
	  IsLessInt (I1, I2)
     
     'rule' EvalConstLt (Pos, negintliteral (_, I1), 
			 negintliteral (_, I2) -> true (Pos)) :
	  IsLessInt (I2, I1)
	  
     'rule' EvalConstLt (Pos, negintliteral (_, _), 
			 posintliteral (_, _) -> true (Pos)) :
	  
     'rule' EvalConstLt (Pos, doubleliteral (_, D1), 
			 doubleliteral (_, D2) -> true (Pos)) :
	  IsLessDouble (D1, D2)
	  
     'rule' EvalConstLt (Pos, doubleliteral (_, D1), L2 -> true (Pos)) :
	  IntLiteralToDoubleLiteral (L2 -> doubleliteral (_, D2))
	  IsLessDouble (D1, D2)
	  
     'rule' EvalConstLt (Pos, L1, doubleliteral (_, D2) -> true (Pos)) :
	  IntLiteralToDoubleLiteral (L1 -> doubleliteral (_, D1))
	  IsLessDouble (D1, D2)
	  
     'rule' EvalConstLt (Pos, charliteral (_, C1), 
			 charliteral (_, C2) -> true (Pos)) :
	  IsLessChar (C1, C2)
	  
     'rule' EvalConstLt (Pos, stringliteral (_, C1), 
			 stringliteral (_, C2) -> true (Pos)) :
	  IsLessString (C1, C2)
	  
     'rule' EvalConstLt (Pos, enumsel (_, _, _, N1), 
			 enumsel (_, _, _, N2) -> true (Pos)) :
	  IsLessInt (N1, N2)
	  
     'rule' EvalConstLt (Pos, _, _ -> false (Pos)) :

---------------------------------------------------------------------------
	  
'action' EvalConstLe (POS, EXPR, EXPR -> EXPR)
     
     'rule' EvalConstLe (Pos, L1, L2 -> Val) :
	  (|
	       EvalConstEq (Pos, L1, L2 -> true (_))
	       let (true (Pos) -> Val)
	  ||
	       EvalConstLt (Pos, L1, L2 -> true (_))
	       let (true (Pos) -> Val)
	  ||
	       let (false (Pos) -> Val)
	  |)
	  
---------------------------------------------------------------------------

'action' EvalConstGt (POS, EXPR, EXPR -> EXPR)
     
     'rule' EvalConstGt (Pos, L1, L2 -> NotVal) :
	  EvalConstLe (Pos, L1, L2 -> Val)
	  EvalConstNot (Val -> NotVal)
	  
---------------------------------------------------------------------------

'action' EvalConstGe (POS, EXPR, EXPR -> EXPR)
     
     'rule' EvalConstGe (Pos, L1, L2 -> NotVal) :
	  EvalConstLt (Pos, L1, L2 -> Val)
	  EvalConstNot (Val -> NotVal)
	  

-- ========================================================================
--  Dyadic arithmetic operations 
-- ========================================================================

'action' EvalConstPlus (POS, EXPR, EXPR -> EXPR)
     
     'rule' EvalConstPlus (Pos, posintliteral (_, I1), 
			   posintliteral (_, I2) -> 
			   posintliteral (Pos, I)) :
	  let (I1 + I2 -> I)
     
     'rule' EvalConstPlus (Pos, posintliteral (_, I1), 
			   negintliteral (_, I2) -> Val) :
	  (|
	       IsLessInt (I1, I2)
	       let (I2 - I1 -> I)
	       let (negintliteral (Pos, I) -> Val)
	  ||
	       let (I1 - I2 -> I)
	       let (posintliteral (Pos, I) -> Val)
	  |)
	  
     'rule' EvalConstPlus (Pos, L1:negintliteral (_, I1), 
			   L2:posintliteral (_, I2) -> L) :
	  EvalConstPlus (Pos, L2, L1 -> L) 
     
     'rule' EvalConstPlus (Pos, negintliteral (_, I1), 
			   negintliteral (_, I2) -> 
			   negintliteral (Pos, I)) :
	  let (I1 + I2 -> I)
	  
     'rule' EvalConstPlus (Pos, doubleliteral (_, D1), 
			   doubleliteral (_, D2) -> 
			   doubleliteral (Pos, D)) :
	  PlusDouble (D1, D2 -> D)
	  
     'rule' EvalConstPlus (Pos, doubleliteral (_, D1), 
			   posintliteral (_, I2) -> 
			   doubleliteral (Pos, D)) :
	  IntToDouble (I2 -> D2)
	  PlusDouble (D1, D2 -> D)
     
     'rule' EvalConstPlus (Pos, doubleliteral (_, D1), 
			   negintliteral (_, I2) -> 
			   doubleliteral (Pos, D)) :
	  IntToDouble (I2 -> D2)
	  MinusDouble (D1, D2 -> D)
	  
     'rule' EvalConstPlus (Pos, L1, L2:doubleliteral (_, _) -> L) :
	  EvalConstPlus (Pos, L2, L1 -> L)
     
     'rule' EvalConstPlus (Pos, stringliteral (_, S1),
			   stringliteral (_, S2) -> 
			   stringliteral (Pos, S)) :
	  PlusStringString (S1, S2 -> S)
     
     'rule' EvalConstPlus (Pos, stringliteral (_, S1), 
			   charliteral (_, C2) -> 
			   stringliteral (Pos, S)) :
	  PlusStringChar (S1, C2 -> S)

     'rule' EvalConstPlus (Pos, charliteral (_, C1), 
			   stringliteral (_, S2) -> 
			   stringliteral (Pos, S)) :
	  PlusCharString (C1, S2 -> S)

     'rule' EvalConstPlus (Pos, charliteral (_, C1), 
			   charliteral (_, C2) -> 
			   stringliteral (Pos, S)) :
	  PlusCharChar (C1, C2 -> S)

---------------------------------------------------------------------------

'action' EvalConstMinus (POS, EXPR, EXPR -> EXPR)
     
     'rule' EvalConstMinus (Pos, L1, L2 -> L)
	  EvalConstNegate (L2 -> NegL2)
	  EvalConstPlus (Pos, L1, NegL2 -> L)
	  
---------------------------------------------------------------------------

'action' EvalConstTimes (POS, EXPR, EXPR -> EXPR)
     
     'rule' EvalConstTimes (Pos, posintliteral (_, I1), 
			    posintliteral (_, I2) -> 
			    posintliteral (Pos, I)) :
	  let (I1 * I2 -> I)
	  
     'rule' EvalConstTimes (Pos, posintliteral (_, I1), 
			    negintliteral (_, I2) -> 
			    negintliteral (Pos, I)) :
	  let (I1 * I2 -> I)
     
     'rule' EvalConstTimes (Pos, negintliteral (_, I1), 
			    posintliteral (_, I2) -> 
			    negintliteral (Pos, I)) :
	  let (I1 * I2 -> I)
     
     'rule' EvalConstTimes (Pos, negintliteral (_, I1), 
			    negintliteral (_, I2) -> 
			    posintliteral (Pos, I)) :
	  let (I1 * I2 -> I)

     'rule' EvalConstTimes (Pos, doubleliteral (_, D1), 
			    doubleliteral (_, D2) -> 
			    doubleliteral (Pos, D)) :
	  TimesDouble (D1, D2 -> D)
	  
     'rule' EvalConstTimes (Pos, doubleliteral (_, D1), 
			    posintliteral (_, I2) -> 
			    doubleliteral (Pos, D)) :
	  IntToDouble (I2 -> D2)
	  TimesDouble (D1, D2 -> D)
     
     'rule' EvalConstTimes (Pos, doubleliteral (_, D1), 
			    negintliteral (_, I2) -> 
			    doubleliteral (Pos, D4)) :
	  IntToDouble (I2 -> D2)
	  TimesDouble (D1, D2 -> D3)
	  NegateDouble (D3 -> D4)
	  
     'rule' EvalConstTimes (Pos, L1, L2:doubleliteral (_, _) -> L) :
	  EvalConstTimes (Pos, L2, L1 -> L)

---------------------------------------------------------------------------

'action' EvalConstDiv (POS, EXPR, EXPR -> EXPR)
     
     'rule' EvalConstDiv (Pos, I1, I2 -> D3) :
	  IntLiteralToDoubleLiteral (I1 -> D1)
	  IntLiteralToDoubleLiteral (I2 -> D2)
	  EvalConstFloatDiv_h (Pos, D1, D2 -> D3)

'action' EvalConstFloatDiv_h (POS, EXPR, EXPR -> EXPR)
     
     'rule' EvalConstFloatDiv_h (Pos, doubleliteral (_, D1), 
				 doubleliteral (_, D2) -> 
				 doubleliteral (Pos, D)) :
	  DivDouble (D1, D2 -> D)
	  
     'rule' EvalConstFloatDiv_h (Pos, doubleliteral (_, D1), 
				 posintliteral (_, I2) -> 
				 doubleliteral (Pos, D)) :
	  IntToDouble (I2 -> D2)
	  DivDouble (D1, D2 -> D)
     
     'rule' EvalConstFloatDiv_h (Pos, doubleliteral (_, D1), 
				 negintliteral (_, I2) -> 
				 doubleliteral (Pos, D4)) :
	  IntToDouble (I2 -> D2)
	  DivDouble (D1, D2 -> D3)
	  NegateDouble (D3 -> D4)
     
     'rule' EvalConstFloatDiv_h (Pos, posintliteral (_, I1),
				 doubleliteral (_, D2) ->
				 doubleliteral (Pos, D)) :
	  IntToDouble (I1 -> D1)
	  DivDouble (D1, D2 -> D)
     
     'rule' EvalConstFloatDiv_h (Pos, negintliteral (_, I1),
				 doubleliteral (_, D2) -> 
				 doubleliteral (Pos, D4)) :
	  IntToDouble (I1 -> D1)
	  DivDouble (D1, D2 -> D3)
	  NegateDouble (D3 -> D4)

---------------------------------------------------------------------------

'action' EvalConstIDiv (POS, EXPR, EXPR -> EXPR)
     
     'rule' EvalConstIDiv (Pos, posintliteral (_, I1), 
			   posintliteral (_, I2) -> 
			   posintliteral (Pos, I)) :
	  let (I1 / I2 -> I)
	  
     'rule' EvalConstIDiv (Pos, posintliteral (_, I1), 
			   negintliteral (_, I2) -> 
			   negintliteral (Pos, I)) :
	  let (I1 / I2 -> I)
     
     'rule' EvalConstIDiv (Pos, negintliteral (_, I1), 
			   posintliteral (_, I2) -> 
			   negintliteral (Pos, I)) :
	  let (I1 / I2 -> I)
     
     'rule' EvalConstIDiv (Pos, negintliteral (_, I1), 
			   negintliteral (_, I2) -> 
			   posintliteral (Pos, I)) :
	  let (I1 / I2 -> I)

---------------------------------------------------------------------------

'action' EvalConstMod (POS, EXPR, EXPR -> EXPR)
     
     'rule' EvalConstMod (Pos, posintliteral (_, I1), 
			  posintliteral (_, I2) -> 
			  posintliteral (Pos, I)) :
	  ModInt (I1, I2 -> I)
     
     'rule' EvalConstMod (Pos, posintliteral (_, I1), 
			  negintliteral (_, I2) -> 
			  negintliteral (Pos, I)) :
	  ModInt (I1, I2 -> I)
	  
     'rule' EvalConstMod (Pos, negintliteral (_, I1), 
			  posintliteral (_, I2) -> 
			  negintliteral (Pos, I)) :
	  ModInt (I1, I2 -> I)
	  
     'rule' EvalConstMod (Pos, negintliteral (_, I1), 
			  negintliteral (_, I2) -> 
			  posintliteral (Pos, I)) :
	  ModInt (I1, I2 -> I)
	  
---------------------------------------------------------------------------

'action' IntLiteralToDoubleLiteral (EXPR -> EXPR)
     
     'rule' IntLiteralToDoubleLiteral (posintliteral (Pos, I) -> 
				       doubleliteral (Pos, D)) :
	  IntToDouble (I -> D)
     
     'rule' IntLiteralToDoubleLiteral (negintliteral (Pos, I) -> 
				       doubleliteral (Pos, D2)) :
	  IntToDouble (I -> D1)
	  NegateDouble (D1 -> D2)
	  
-- ========================================================================
--  Dyadic boolean operations
-- ========================================================================

'action' EvalConstOr (POS, EXPR, EXPR -> EXPR)
	  
     'rule' EvalConstOr (Pos, false (_), false (_) -> true (Pos)) :
     
     'rule' EvalConstOr (Pos, _, _ -> false (Pos)) :
	  
---------------------------------------------------------------------------

'action' EvalConstAnd (POS, EXPR, EXPR -> EXPR)
	  
     'rule' EvalConstAnd (Pos, true (_), true (_) -> false (Pos)) :
     
     'rule' EvalConstAnd (Pos, _, _ -> true (Pos)) :
	  
-- ========================================================================
--  Monadic operations
-- ========================================================================

'action' EvalConstMonOp (POS, MOP, OperandType : TYPE, 
			 OperandValue : EXPR -> TYPE, EXPR)
     
     'rule' EvalConstMonOp (Pos, _, Type, _ -> error, error (Pos)) :
	  IsErrorType (Type)
     
     'rule' EvalConstMonOp (Pos, minus, Type, Val -> Type2, NegVal) :
	  CheckNumericType (Pos, Type -> Type2)
	  (|
	       where (Type2 -> error)
	       let (EXPR'error (Pos) -> NegVal)
	  ||
	       EvalConstNegate (Val -> NegVal)
	  |)
     
     'rule' EvalConstMonOp (Pos, plus, Type, Val -> Type2, PlusVal) :
	  CheckNumericType (Pos, Type -> Type2)
	  (|
	       where (Type2 -> error)
	       let (EXPR'error (Pos) -> PlusVal)
	  ||
	       let (Val -> PlusVal)
	  |)
     
     'rule' EvalConstMonOp (Pos, not, Type, Val -> Type2, NotVal) :
	  CheckBoolType (Pos, Type -> Type2)
	  (|
	       where (Type2 -> error)
	       let (EXPR'error (Pos) -> NotVal)
	  ||
	       EvalConstNot (Val -> NotVal)
	  |)
	  
---------------------------------------------------------------------------
	  
'action' EvalConstNegate (EXPR -> EXPR)
     
     'rule' EvalConstNegate (posintliteral (P, I) -> negintliteral (P, I)) :
	  
     'rule' EvalConstNegate (negintliteral (P, I) -> posintliteral (P, I)) :
	  
     'rule' EvalConstNegate (doubleliteral (P, D1) -> 
			     doubleliteral (P, D2)) :
	  NegateDouble (D1 -> D2)
	  
     'rule' EvalConstNegate (C:error (_) -> C) :
	  
---------------------------------------------------------------------------
	  
'action' EvalConstNot (EXPR -> EXPR)
     
     'rule' EvalConstNot (true (Pos) -> false (Pos)) :
     'rule' EvalConstNot (false (Pos) -> true (Pos)) :
     'rule' EvalConstNot (C:error (Pos) -> C) :
	  
-- ========================================================================
--  Evalutation of enumeration selection.
-- ========================================================================

'action' EvalConstantEnumSelection (POS, EnumType : EXPR, ID ->
				    TYPE, EXPR)
     
     'rule' EvalConstantEnumSelection (Pos, applied (TPos, TId), Id -> 
				       T, Value) :
	  GetTypeIdDef (TId -> T:composite (enum (Enumerators)))
	  (|
	       LookupEnumerator (Id, Enumerators -> N)
	       NewTypeIndex (TPos, composite (typename (TId)) -> TypeI)
	       let (enumsel (Pos, TypeI, Id, N) -> Value)
	  ||
	       Id'Pos -> IPos
	       Id'Ident -> I
	       ErrorI ("'", I, "' is not declared as enumerator", Pos)
	       let (EXPR'error (Pos) -> Value)
	  |)
     
     'rule' EvalConstantEnumSelection (Pos, applied (TPos, _), Id -> 
				       error, error (Pos)) :
	  Error ("enumeration type expected", TPos)
    
-- ========================================================================
--  Evalutation of an identifier application 
--  EvalConstantId returns the type and the value of an identifier defined
--   as constant.
--  ReplacePosInConstantExpr replaces the Pos in the defining expression
--   with the position of the identifier application.
-- ========================================================================

'action' EvalConstantId (ID, MEANING -> TYPE, EXPR)
     
     'rule' EvalConstantId (Id , const (TypeI, Expr:error (_)) -> 
			    error, Expr) :
     
     'rule' EvalConstantId (Id, const (TypeI, Expr) -> Type, Expr2) :
	  Id'Pos -> Pos
	  ReplacePosInConstantExpr (Expr, Pos -> Expr2)
	  TypeI'Type -> Type
			    
     'rule' EvalConstantId (Id, error -> error, error (Pos)) :
	  Id'Pos -> Pos
     
     'rule' EvalConstantId (Id, M -> error, error (Pos)) :
	  Id'Pos -> Pos
	  Id'Ident -> I
	  ErrorI ("'", I, "' is not declared as constant", Pos)
	  
---------------------------------------------------------------------------

'action' ReplacePosInConstantExpr (EXPR, POS -> EXPR)
     
     'rule' ReplacePosInConstantExpr (false (_), P -> 
				      false (P)) :
     'rule' ReplacePosInConstantExpr (true (_), P -> 
				      true (P)) :
     'rule' ReplacePosInConstantExpr (posintliteral (_, V), P -> 
				      posintliteral (P, V)) :
     'rule' ReplacePosInConstantExpr (negintliteral (_, V), P -> 
				      negintliteral (P, V)) :
     'rule' ReplacePosInConstantExpr (doubleliteral (_, V), P -> 
				      doubleliteral (P, V)) :
     'rule' ReplacePosInConstantExpr (charliteral (_, V), P -> 
				      charliteral (P, V)) :
     'rule' ReplacePosInConstantExpr (stringliteral (_, V), P -> 
				      stringliteral (P, V)) :
	  
---------------------------------------------------------------------------

'end'
