#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include "files.h"
#include "idents.h"
#include "errmsg.h"
#include "pos.h"

/*--------------------------------------------------------------------*/
/* IMPORTS */

extern FILE *yyin;

/*--------------------------------------------------------------------*/
/* IMPLEMENTS */

void InitFiles (void);

void InsertDir (char *);

void OpenFile (char *);
FILE * NextFile (void);
static FileId *GetFileId (char *);
void InsertImportFile (Ident);
static void InsertFileIdEntry (Ident);
static bool IsFileEntryPresent (Ident);

void InsertLinePragmat (char *, long);
void InsertLinePragmatNoFilename (long);
static FileId * IsLinePragmatEntryPresent (FileId *);

void CheckSpecModuleName (long, Ident);
void CheckImplModuleName (long, Ident);
void CheckModuleName (long, Ident);
static bool EqualFilename (char *filename, char *pattern);

void fprintFileId (FILE *, FileId *);
void fprintFileIdExtended (FILE * dev, FileId * file, char * Ext);
void PrintDependencies (void);

FILE* OpenOutputFile (char *);

void DumpFileTab (void);

/*--------------------------------------------------------------------*/

/* single chained list of  dirnames */
typedef struct DirIdRec DirId;    
struct DirIdRec
{
  char * dirname;
  DirId * next;
};

/* Pointer to the first and last dir entry */
DirId * firstDirIdEntry;          
DirId * lastDirIdEntry;           

#ifdef s_win_nt
#define DIRECTORY_SEP '\\'
#ifndef PATH_MAX
#define PATH_MAX _MAX_PATH
#endif /* not defined PATH_MAX */
#else
#define DIRECTORY_SEP '/'
#endif /* defined s_win_nt */

/*--------------------------------------------------------------------*/

#define SPEC_SUFFIX ".cs"
#define IMPL_SUFFIX ".ci"
#define MOD_SUFFIX  ".cm"

#define MAX_SUFFS   2
#define FIRST_SUFF_ENTRY 0
char * (suffs [MAX_SUFFS]) = {SPEC_SUFFIX, MOD_SUFFIX};
                             /* array of suffixes for specification files */
/*--------------------------------------------------------------------*/

/* single chained list of splitted filenames */
struct FileIdRec
{
   Ident  basename;
   char * suffix;
   char * path;
   FileId * next;
};

/* points to the first (source), last and currently scanned entry  */
FileId * sourceFileIdEntry = 0;   
FileId * lastFileIdEntry = 0;     
FileId * currFileIdEntry = 0;      

/* special list of line pragmat file entries */
FileId * firstLinePragmatEntry = 0;
FileId * lastLinePragmatEntry = 0;

/*--------------------------------------------------------------------*/

/* filename of current scanned file set by OpenFile and NextFile */
char currFilename [PATH_MAX + 1];

/*--------------------------------------------------------------------*/

/* string used to print error messages */

char errMsg [PATH_MAX + 50];

/*--------------------------------------------------------------------*/
/* Initialization */
/*--------------------------------------------------------------------*/

void InitFiles (void)
{
  /* Init Dir List with the name of current directory */
  firstDirIdEntry = malloc (sizeof (DirId));
  firstDirIdEntry->dirname = "";
  firstDirIdEntry->next = 0;
  lastDirIdEntry = firstDirIdEntry;
}

/*--------------------------------------------------------------------*/
/* directory stuff (-d option) */
/*--------------------------------------------------------------------*/

void InsertDir (char *dirname)
{
  int len;

  len = strlen (dirname);
  lastDirIdEntry->next = malloc (sizeof (DirId));
  lastDirIdEntry = lastDirIdEntry->next;
  
  lastDirIdEntry->dirname = malloc (len + 2);
  strcpy (lastDirIdEntry->dirname, dirname);
  
  if (lastDirIdEntry->dirname [len - 1] != DIRECTORY_SEP) {
    lastDirIdEntry->dirname [len] = DIRECTORY_SEP;
    lastDirIdEntry->dirname [len + 1] = '\0';
  }
}

/*--------------------------------------------------------------------*/
/* File handling */
/*--------------------------------------------------------------------*/

void OpenFile (char * filename)
{
  char *suffptr;

  yyin = fopen (filename, "r");
  if (yyin == NULL) {
    sprintf(errMsg, "cannot open source file: '%s'", filename);
    ErrorUsage (errMsg);
  }
  
  strcpy (currFilename, filename);
  sourceFileIdEntry = GetFileId (filename);

  suffptr = sourceFileIdEntry->suffix;
  /* check on correct file suffix */
  if (strlen (suffptr) == 0) {
    sprintf (errMsg, "missing suffix: '%s'", filename);
    ErrorUsage (errMsg);
  }
  else if (EqualFilename (suffptr, SPEC_SUFFIX) &&
	   EqualFilename (suffptr, IMPL_SUFFIX) && 
	   EqualFilename (suffptr, MOD_SUFFIX)) {
    sprintf (errMsg, "invalid suffix: '%s'", suffptr);
    ErrorUsage (errMsg);
  }
  
  lastFileIdEntry = sourceFileIdEntry;
  currFileIdEntry = sourceFileIdEntry;
  yyPosToFirstFile (currFileIdEntry);
}

/*--------------------------------------------------------------------*/

FILE * NextFile (void)
{
  DirId * i;
  int j;
  char *basename;
  FILE *next;

  /* close current file */
  fclose (yyin);  /* Ergebnis pruefen ??? */

  currFileIdEntry = currFileIdEntry->next;
  if (currFileIdEntry == 0) /* no more files */ {
    return ((FILE *) 0);
  }
  
  id_to_string (currFileIdEntry->basename, &basename);
   
  /* search and open next file, look for all suffixes in each directory */
  i = firstDirIdEntry;
  while  (i != 0) {
    for (j = FIRST_SUFF_ENTRY; j < MAX_SUFFS; j++) {
      sprintf (currFilename, "%s%s%s", i->dirname, basename, suffs [j]);
      next = fopen (currFilename, "r");

      if (next != (FILE *) 0) {
        /* fill up File entry */
	currFileIdEntry->suffix = suffs [j];
	currFileIdEntry->path = i->dirname;
  
        /* reset pos */ 
        yyPosToNextFile(currFileIdEntry, 1);
	return (next);
      }
    }
    i = i->next;
  }

  sprintf (errMsg, "cannot find file with Module: '%s'", basename);
  ErrorUsage (errMsg);
  return ((FILE *) 0);
}

/*--------------------------------------------------------------------*/

static FileId * GetFileId (char *filename)
{
  FileId * newId;
  int pathlen;
  int baselen;
  int sufflen;
  char basename [PATH_MAX + 1];
  char * baseptr; 
  char * suffptr;
  Ident baseId;

  /* eval first pos of basename */
  baseptr = strrchr (filename, (int) DIRECTORY_SEP);
  if (baseptr == (char *) 0) {
    baseptr = filename;
  }
  else {
    baseptr++;
  }
  
  /* eval first pos of suffix */
  suffptr = strrchr (filename, (int) '.');
  if (suffptr == (char *) 0) {
    suffptr = "";              
  }
  pathlen = strlen (filename) - strlen (baseptr);
  baselen = strlen (baseptr) - strlen (suffptr);
  sufflen = strlen (suffptr);

  /* fill File entry completly */
  newId = malloc (sizeof (FileId));
  newId->path = malloc (pathlen + 1);
  newId->suffix = malloc (sufflen + 1);
  
  strncpy (newId->path, filename, pathlen);
  newId->path [pathlen] = '\0';
  
  strncpy (basename, baseptr, baselen);
  basename [baselen] = '\0';
  string_to_id (basename, &baseId); 
  newId->basename = baseId;
  
  strcpy (newId->suffix, suffptr);

  newId->next = 0;

  /* and return the new File entry */
  return newId;
}
  
/*--------------------------------------------------------------------*/

void InsertImportFile (Ident unit)
{
  if (IsFileEntryPresent (unit) == FALSE) {
    InsertFileIdEntry (unit);
  }
}

/*--------------------------------------------------------------------*/

static void InsertFileIdEntry (Ident unit)
{
  int len;
  char *basename;
  
  id_to_string (unit, &basename);

  len = strlen(basename);

  lastFileIdEntry->next = malloc (sizeof (FileId));
  lastFileIdEntry = lastFileIdEntry->next;
  
  /* fill the basename and set Rest to Nil Value, Rest will be filled  */
  /* in Next File                                                      */
  lastFileIdEntry->basename = unit;
  lastFileIdEntry->suffix = "";
  lastFileIdEntry->path = "";
  lastFileIdEntry->next = 0;
}

/*--------------------------------------------------------------------*/

static bool IsFileEntryPresent (Ident unit)
{
  FileId * i;
  char *Temp1, *Temp2;

  i = sourceFileIdEntry;
  while (i != 0) {
    if (unit == i->basename)
      return TRUE;
    i = i->next;
  }
  return FALSE;
}

/*--------------------------------------------------------------------*/
/* line pragmat stuff */
/*--------------------------------------------------------------------*/

void InsertLinePragmat (char * filename, long linenumber)
{
  FileId *newFileId, *oldFileId;

  newFileId = GetFileId (filename);
  if (firstLinePragmatEntry == (FileId *) 0) { /* first line pragmat? */
    firstLinePragmatEntry = newFileId;
    lastLinePragmatEntry  = newFileId;
  }
  else { 
    /* check on identical file entry of former line pramat */
    oldFileId = IsLinePragmatEntryPresent (newFileId); 
    if (oldFileId != (FileId *) 0) {
      free (newFileId->path);
      free (newFileId->suffix);
      free (newFileId);
      newFileId = oldFileId;
    }
    else { /* new line pragmat file entry */
      lastLinePragmatEntry->next = newFileId;
      lastLinePragmatEntry = newFileId;
    }
  }
  yyPosToNextFile (newFileId, linenumber);
}

/*--------------------------------------------------------------------*/

void InsertLinePragmatNoFilename (long linenumber)
{
  if (lastLinePragmatEntry == 0) {
    yyPosToNextFile (lastFileIdEntry, linenumber);
  }
  else {
    yyPosToNextFile (lastLinePragmatEntry, linenumber);
  }
} 

/*--------------------------------------------------------------------*/

static FileId * IsLinePragmatEntryPresent (FileId *newId)
{
  FileId * i;

  i = firstLinePragmatEntry;
  while (i != 0) {
    if ((strcmp (newId->path, i->path) == 0) &&
	(newId->basename == i->basename) &&
	(strcmp (newId->suffix, i->suffix) == 0))  {
      return i;
    }
    i = i->next;
  }
  return (FileId *) 0;
}

/*--------------------------------------------------------------------*/
/* Check on module names */
/*--------------------------------------------------------------------*/

void CheckSpecModuleName (long pos, Ident modname)
{
  CheckModuleName (pos, modname);
  if (EqualFilename (currFileIdEntry->suffix, IMPL_SUFFIX)) {
    sprintf (errMsg, "invalid specification suffix '%s'", IMPL_SUFFIX);
    Error (errMsg, pos);
    CheckErrorMessages ();
  }
}

/*--------------------------------------------------------------------*/

void CheckImplModuleName (long pos, Ident modname)
{
  CheckModuleName (pos, modname);
  if (EqualFilename (currFileIdEntry->suffix, SPEC_SUFFIX)) {
    sprintf (errMsg, "invalid implementation suffix '%s'", SPEC_SUFFIX);
    Error (errMsg, pos);
    CheckErrorMessages ();
  }
  if (EqualFilename (currFileIdEntry->suffix, IMPL_SUFFIX)) {
    InsertFileIdEntry (modname);
  }
}

/*--------------------------------------------------------------------*/

void CheckModuleName (long pos, Ident modname)
{
  char *modnamestr;
  char *basenamestr;

  id_to_string (modname, &modnamestr);
  id_to_string (currFileIdEntry->basename, &basenamestr);
  if (!EqualFilename (basenamestr, modnamestr)) {
    sprintf (errMsg, 
	     "Modulename '%s' does not correspond to filename '%s'\n", 
	     modnamestr, basenamestr);
    Error (errMsg, pos);
    CheckErrorMessages ();
  }
}

/*--------------------------------------------------------------------*/

static char * UpperCase (char * string, char * upper)
{
  int i;

  for (i=0; string[i] != '\0'; i++) {
    upper [i] = toupper (string[i]);
  }
  upper [i] = '\0';

  return upper;
}

/*--------------------------------------------------------------------*/

static bool EqualFilename (char *filename, char *pattern)
{
  char upperPattern [PATH_MAX + 1];
  if (strcmp (filename, pattern) == 0) {
    return TRUE;
  }
#ifdef s_win_nt
  else if (strcmp (filename, UpperCase (pattern, upperPattern)) == 0) {
    return TRUE;
  }
#endif /* defined s_win_nt */
  return FALSE;
}

/*--------------------------------------------------------------------*/
/* misc */
/*--------------------------------------------------------------------*/

void fprintFileId (FILE * dev, FileId * file)
{
  char * repr;
  
  id_to_string (file->basename, &repr);
  fprintf (dev, "%s%s%s", file->path, repr, file->suffix);
}

/*--------------------------------------------------------------------*/

void fprintFileIdExtended (FILE * dev, FileId * file, char * Ext)            
{
  char * repr;

  id_to_string (file->basename, &repr);
  fprintf (dev, "%s%s%s", file->path, repr, Ext);                      
}

/*--------------------------------------------------------------------*/

void PrintDependencies (void)
{
  FileId * i;
  
  if (EqualFilename (sourceFileIdEntry->suffix, SPEC_SUFFIX)) {
    fprintFileIdExtended (stdout, sourceFileIdEntry, HEADER_SUFFIX);
  }
  else if (EqualFilename (sourceFileIdEntry->suffix, IMPL_SUFFIX)) {
    fprintFileIdExtended (stdout, sourceFileIdEntry, DEST_SUFFIX);
  }
  else if (EqualFilename (sourceFileIdEntry->suffix, MOD_SUFFIX)) {
    fprintFileIdExtended (stdout, sourceFileIdEntry, HEADER_SUFFIX);
    fprintf (stdout, " ");
    fprintFileIdExtended (stdout, sourceFileIdEntry, DEST_SUFFIX);
  }

  fprintf (stdout, ": ");
  fprintFileId (stdout, sourceFileIdEntry);
  i = sourceFileIdEntry-> next;
  while (i != 0) {
    fprintf (stdout, " \\\n\t");
    fprintFileId (stdout, i);
    i = i->next;
  }
  fprintf (stdout, "\n");
  exit (0);
}

/*--------------------------------------------------------------------*/
/* aux function used by output */
/*--------------------------------------------------------------------*/

FILE * OpenOutputFile (char * suffix)
{
  char filename [PATH_MAX + 1];
  FILE * fileptr;
  char * basenamestr;
  
  id_to_string (sourceFileIdEntry->basename, &basenamestr);
  sprintf (filename, "%s%s", basenamestr, suffix);
  fileptr = fopen (filename, "w");
  if (fileptr == NULL) {
    sprintf(errMsg, "cannot open output file: '%s'", filename);
    ErrorUsage (errMsg);
  }
  return (fileptr);
}

/*--------------------------------------------------------------------*/
/* Debug routines */
/*--------------------------------------------------------------------*/

void DumpFileTab (void)
{
  FileId * curr;
  char * filename;
  
  printf ("\n-------------------\n");
  printf ("file table dump\n");
  printf ("-------------------\n");
  curr = sourceFileIdEntry;
  while (curr != 0) {
    printf ("  ");
    fprintFileId (stdout, curr);
    printf ("\n");
    curr = curr->next;
  }
  printf ("\n-------------------\n");
  printf ("line pragmat table dump\n");
  printf ("-------------------\n");
  curr = firstLinePragmatEntry;
  while (curr != 0) {
    printf ("  ");
    fprintFileId (stdout, curr);
    printf ("\n");
    curr = curr->next;
  }
}


