
'module' scopes

'export' 
-- Initialization
     InitScopes 
-- Scope handling
     EnterScope LeaveScope ScopeLevel 
-- Identifier meaning
     Define Apply DefineModule ApplyModule DefineRoot DefineForeignExport
     
'use' ast extspecs misc

-- ========================================================================
--  Types and global variables
--  NEST contains all old definitions of an identifier which are defined with
--    a new meaning in the actual scope
--  ENV is a list of nestings, one for each scope level.
--  CurrentEnv is the actual identifier environment.
--  CurrentModuleEnv is the actual environment of module meaning.
-- ========================================================================

'type' ENV
   env (NEST, ENV)
   emptyenv 

'type' NEST
   nest (Ident, Hidden : MEANING, NEST)
   emptynest

-------------------------------------------------------------------------------

'var' CurrentEnv : ENV
'var' CurrentModuleEnv : ENV
'var' CurrentLevel : INT
'var' RootDecl : IDLIST
'var' ExportForeignDeclList : IDLIST

-- ========================================================================
--  Initialization
-- ========================================================================
     
'action' InitScopes

     'rule' InitScopes :
	  CurrentEnv <- emptyenv
	  CurrentModuleEnv <- emptyenv
	  CurrentLevel <- 0
	  RootDecl <- nil
	  ExportForeignDeclList <- nil

-- ========================================================================
--  Scope handling
--  EnterScope opens a new scope.
--  LeaveScope installs all identifier meanings valid in scope level N-1
--    (predicate ForgetNest) and decrements the scope level.
--  IsUndefined checks if there is no definition of the identifier in the
--    actual scope.
--  Hides is used to define the old meaning in the case of an hidden
--    meaning.
-- ========================================================================
     
'action' EnterScope

     'rule' EnterScope :
	  CurrentEnv -> Env
	  CurrentEnv <- env (emptynest, Env)
	  CurrentModuleEnv -> ModuleEnv
	  CurrentModuleEnv <- env (emptynest, ModuleEnv)
	  CurrentLevel -> N
	  CurrentLevel <- N+1

-------------------------------------------------------------------------------

'action' LeaveScope

     'rule' LeaveScope :
	  CurrentEnv -> env (Nest, Env)
	  ForgetNest (Nest)
	  CurrentEnv <- Env
	  CurrentModuleEnv -> env (ModuleNest, ModuleEnv)
	  ForgetModuleNest (ModuleNest)
	  CurrentModuleEnv <- ModuleEnv
	  CurrentLevel -> N
	  CurrentLevel <- N-1

-------------------------------------------------------------------------------

'action' ScopeLevel (-> INT)

     'rule' ScopeLevel (-> N) : 
	  CurrentLevel -> N

-------------------------------------------------------------------------------

'action' ForgetNest (NEST)

     'rule' ForgetNest (nest (Id, noobj, Nest)) :
	  UndefMeaning (Id)
	  ForgetNest (Nest)

     'rule' ForgetNest (nest (Id, Hidden, Nest)) :
	  DefMeaning (Id, Hidden)
	  ForgetNest (Nest)
     
     'rule' ForgetNest (emptynest) :

-------------------------------------------------------------------------------

'action' ForgetModuleNest (NEST)

     'rule' ForgetModuleNest (nest (Id, noobj, Nest)) :
	  UndefModuleMeaning (Id)
	  ForgetModuleNest (Nest)

     'rule' ForgetModuleNest (nest (Id, module (Hidden), Nest)) :
	  DefModuleMeaning (Id, Hidden)
	  ForgetModuleNest (Nest)
     
     'rule' ForgetModuleNest (emptynest) :

-------------------------------------------------------------------------------

'condition' IsUndefined (Ident, NEST)

     'rule' IsUndefined (Id, nest (Id2, Hidden, Nest)) :
	  ne (Id, Id2)
	  IsUndefined (Id, Nest)

     'rule' IsUndefined (Id, emptynest)

-------------------------------------------------------------------------------

'condition' IsUndefinedModule (Ident, NEST)

     'rule' IsUndefinedModule (Id, nest (Id2, Hidden, Nest)) :
	  ne (Id, Id2)
	  IsUndefinedModule (Id, Nest)

     'rule' IsUndefinedModule (Id, emptynest)

-------------------------------------------------------------------------------

'action' Hides (Ident -> MEANING)

     'rule' Hides (Id -> Obj) : 
	  HasMeaning (Id -> Obj)
	  
     'rule' Hides (Id -> noobj)

-------------------------------------------------------------------------------

'action' HidesModule (Ident -> MEANING)

     'rule' HidesModule (Id -> module (Obj)) : 
	  HasModuleMeaning (Id -> Obj)
	  
     'rule' HidesModule (Id -> noobj)

-- ========================================================================
--  Identifier meaning
--  Define defines the identifier in actual scope, if it was not defined
--    before.  
--  Apply get the actual meaning of the identifier, if present.
-- ========================================================================

'action' Define (POS, Ident, MEANING)

     'rule' Define (Pos, Ident, Meaning) :
	  CurrentEnv -> env (Nest, Env)
	  IsUndefined (Ident, Nest)
	  Hides (Ident -> Hidden)
	  CurrentEnv <- env (nest (Ident, Hidden, Nest), Env)
	  DefMeaning (Ident, Meaning)
	  
     'rule' Define (Pos, Ident, _) :
	  ErrorI ("multiple declaration of '", Ident, "'", Pos)

-------------------------------------------------------------------------------

'action' Apply (POS, Ident -> MEANING)

     'rule' Apply (Pos, Ident -> Meaning) 
	  HasMeaning (Ident -> Meaning)
	  
     'rule' Apply (Pos, Ident -> error) 
	  ErrorI ("'", Ident, "' is not declared", Pos)

-- ========================================================================
--  Module meaning
--  This predicates are nescenssary, bacause module names have their own
--    namespace. 
-- ========================================================================
     
'action' DefineModule (POS, Ident, DECL)

     'rule' DefineModule (Pos, Ident, Decls) :
	  CurrentModuleEnv -> env (Nest, Env)
	  IsUndefinedModule (Ident, Nest)
	  HidesModule (Ident -> Hidden)
	  CurrentModuleEnv <- env (nest (Ident, Hidden, Nest), Env)
	  DefModuleMeaning (Ident, Decls)
	  
     'rule' DefineModule (Pos, Ident, _) :
	  ErrorI ("multiple declaration of module '", Ident, "'", Pos)

-------------------------------------------------------------------------------

'action' ApplyModule (POS, Ident -> DECL)

     'rule' ApplyModule (Pos, Ident -> Decl) 
	  HasModuleMeaning (Ident -> Decl)
	  
     'rule' ApplyModule (Pos, Ident -> error (Pos)) 
	  ErrorI ("'", Ident, "' is not declared as module", Pos)
     
-- ========================================================================
--  check on multiple root clauses
-- ========================================================================

'action' DefineRoot (POS, ID)
     
     'rule' DefineRoot (Pos, Id) :
	  RootDecl -> nil
	  RootDecl <- idlist (Id, nil)
	  
     'rule' DefineRoot (Pos, _) :
	  RootDecl -> idlist (Id, _)
	  (|
	       IsDefinedInActualModule (Id)
	       Error ("multiple root clause", Pos)
	  ||
	       Id'Module -> module (MId)
	       MId'Ident -> Ident
	       ErrorI ("multiple root clause, first defined in module '", 
		       Ident, "'", Pos)
	  |)
	  
-- ========================================================================
--  check, if an identifier is multiple declared as export foreign in
--  different modules 
-- ========================================================================

'action' DefineForeignExport (POS, ID)
	  
     'rule' DefineForeignExport (Pos, Id) :
	  ExportForeignDeclList -> List
	  IsIdInList (Id, List -> Id2)
	  (|
	       IsDefinedInActualModule (Id2)
	  ||
	       Id2'Module -> module (MId)
	       MId'Ident -> Ident
	       ErrorI ("multiple export foreign or root declaration, first defined in module '", Ident, "'", Pos)
	  |)
		    
     'rule' DefineForeignExport (Pos, Id) :
	  ExportForeignDeclList -> Tail
	  ExportForeignDeclList <- idlist (Id, Tail)
	  
-------------------------------------------------------------------------------

'end'
