/*--------------------------------------------------------------------
 *
 *              C - Interface for CooL - Doi
 *
 *  This module contains the basic functionality of Doi
 *
 *------------------------------------------------------------------
 *------- external used functions ----------------------------------
 *------------------------------------------------------------------
 *  Display* DoiGetDisplay         ();
 *    int    DoiCreateLockWindow   (Widget,Window*,int,char*,char*);
 *    void   DoiDestroyLockWindow  (Window);
 *    void   DoiSetLockWindow      (Window);
 *    void   DoiResetLockWindow    (Window);
 *    void   DoiSetAppClass        (char*);
 *    void   DoiLoop               ();
 *    void   DoiDoLocalLoop        (int);
 *    void   DoiRemoveInputEvents  ();
 *    void   DoiUpdateDisplay  ();
 *    void   DoiInit               (char*,Widget*,int*,char*[]);
 *    void   DoiSetIconName        (Widget);
 *    void   DoiBell               (Widget);
 *    void   DoiSetFontCursor      (Widget,int,char*,char*);
 *    void   DoiResetFontCursor    (Widget);
 *    void   DoiManageChild        (Widget,Widget);
 *    void   DoiAddCallback        (Widget, int(*)(char*),char*,void*);
 *    void   DoiAddWMCloseCallback (Widget, Widget, int(*)(char*),void*);
 *    void   DoiRemoveCallback     (Widget, int(*)(char*),char*,void*);
 *    void   DoiShow               (Widget,Widget,int);
 *    void   DoiHide               (Widget);
 *    void   DoiRealizeParentWidget(Widget);
 *    int    DoiGetWidgetName      (Widget,char*,int);
 *    void   DoiDestroyWidget      (Widget);
 *    void   DoiLock               (Widget);
 *    void   DoiUnlock             (Widget);
 *    int    DoiGetDialogElement   (char *,int *);
 *    Widget DoiGetChild           (Widget, int);
 *    void   DoiMalloc             ();
 *    void   DoiFree               ();
 *------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
#include <stdio.h>
#include <malloc.h>
#include <Xm/Xm.h>
#include <Xm/ArrowB.h>
#include <Xm/ArrowBG.h>
#include <Xm/BulletinB.h>
#include <Xm/CascadeB.h>
#include <Xm/CascadeBG.h>
#include <Xm/Command.h>
#include <Xm/DialogS.h>
#include <Xm/DrawnB.h>
#include <Xm/MenuShell.h>
#include <Xm/FileSB.h>
#include <Xm/MessageB.h>
#include <Xm/Protocols.h>
#include <Xm/PushB.h>
#include <Xm/PushBG.h>
#include <Xm/Scale.h>
#include <Xm/ScrollBar.h>
#include <Xm/SelectioB.h>
#include <X11/Shell.h>
#include <Xm/Text.h>
#include <Xm/TextF.h>
#include <Xm/ToggleB.h>
#include <Xm/ToggleBG.h>
#ifdef FORMAT
#include <Xc/FormatAlph.h>
#include <Xc/FormatNum.h>
#include <Xc/Format.h>
#include <Xc/FormatDate.h>
#include <Xc/Table.h>
#endif

/*------------------------------------------------------------------*/
#define StringEnd '\0'


/*------------------------------------------------------------------*/
extern int   DoiDebugFlag;
static XtAppContext doiContext = NULL;
static Display     *doiDisplay = NULL;
static int          doiLoop;


/*------------------------------------------------------------------*/
Display *DoiGetDisplay ()
{
  return doiDisplay;
}/* DoiGetDisplay */

/*------------------------------------------------------------------*/
char *DoiMalloc (int size)
{
  char *p;
  p = malloc (size);
  if (DoiDebugFlag)
    fprintf (stderr,"DoiMalloc(%d) returns %x\n",size,p);
  return p;
}/* DoiMalloc */

/*------------------------------------------------------------------*/
char *DoiFree (char *p)
{
  free (p);
  if (DoiDebugFlag)
    fprintf (stderr,"DoiFree(%x)\n",p);
}/* DoiFree */

/*------------------------------------------------------------------*/
void DoiLoop()
{
  if (DoiDebugFlag) printf("DoiLoop\n");
  XtAppMainLoop (doiContext);
}

/*------------------------------------------------------------------*/
void DoiDoLocalLoop(int blocking)
{
  XEvent event;
  if (!blocking && !XtAppPending (doiContext)) return;

  if (XtAppPeekEvent (doiContext,&event)){
    /* an X event has to be processed */
    XtAppNextEvent (doiContext,&event);
    XtDispatchEvent (&event);
  } else /* process an alternate event */
    XtAppProcessEvent (doiContext, XtIMTimer | XtIMAlternateInput);
}

/*------------------------------------------------------------------*/
void DoiUpdateDisplay(Widget PRwidget)
{
  XmUpdateDisplay (PRwidget);
}

/*------------------------------------------------------------------*/
void DoiRemoveInputEvents()
{
  XEvent event;
  while (XtAppPending (doiContext)){
    /* there is an event in the queue */
    if (DoiDebugFlag) printf ("DoiRemoveInputEvents: event is pending\n");
    if (XtAppPeekEvent (doiContext,&event)){
      /* an X event has to be processed */
      XtAppNextEvent (doiContext,&event);
      /* check the event type */
      /* and discard all keyboard and mouse events */
      if (DoiDebugFlag) printf ("  an X event of type = %d\n",event.type);
      switch (event.type){
      case KeyPress:
      case KeyRelease:
      case ButtonPress:
      case ButtonRelease:
        continue;
      default:
        XtDispatchEvent (&event);
      }
    } else /* process an alternate event */
      XtAppProcessEvent (doiContext, XtIMTimer | XtIMAlternateInput);
  }
}

/*------------------------------------------------------------------*/
static char doiAppShell[] = "Doi-Appshell";
static char doiClient[]   = "Doi-Client";
static char defaultAppClass[] = "Doi-Class";
static char *doiClass = NULL;

/*------------------------------------------------------------------*/
void DoiSetAppClass (char *appClass)
  {/* sets the doiClass */
    doiClass = appClass;
  }

/*------------------------------------------------------------------*/
void DoiSetIconName (Widget widget)
{/* sets the icon name, if parent is an application shell */
  XmString xmString;
  char *iconName; 
  Widget parent;
  WidgetClass wclass;

  if (widget == NULL) return;
  if ((parent = XtParent(widget)) == NULL) return;
  wclass = XtClass (parent);
  if (wclass == applicationShellWidgetClass){
    iconName = NULL;
    if (XtClass(widget) == xmBulletinBoardWidgetClass){
      XtVaGetValues (widget,XmNdialogTitle,&xmString,NULL);
      if (XmStringGetLtoR(xmString,XmSTRING_DEFAULT_CHARSET,&iconName) == True)
      	XtVaSetValues (parent,XmNiconName,iconName,NULL);
    
    }else{
      XtVaGetValues (parent,XmNtitle,&iconName,NULL);
      if (iconName != NULL && strlen(iconName) > (size_t)0)
        XtVaSetValues (parent,XmNiconName,iconName,NULL);
    }
  }
}

/*------------------------------------------------------------------*/
void DoiInit ( char *title, Widget *PRshell, int *PRargc, char *PRargv[] )
  /* creates an application shell */
{
  Arg args[2];
  Widget shell = NULL;

  if (DoiDebugFlag) printf("DoiInit\n");

  XtSetArg (args[0], XmNtitle, title);
  XtSetArg (args[1], XmNmappedWhenManaged, True);
  /*XtSetArg (args[1], XmNmappedWhenManaged, False);*/
  if (*PRshell != NULL) return;


  /* no shell widget is specified, create one */
  if (PRargc != NULL && *PRargc > 0){
    /* args are specified */
    if (doiContext == NULL){
      if (doiClass == NULL)
        doiClass = defaultAppClass; /* sets the default application class */
      shell = XtAppInitialize(&doiContext,doiClass,NULL,0,(Cardinal*)PRargc,PRargv,NULL,args,2);
      doiDisplay = XtDisplay (shell);
    } else { /* Application context exists already, only create Appshell*/
      shell = XtAppCreateShell (PRargv[0],doiAppShell,applicationShellWidgetClass,doiDisplay,args,2);
    }
  } else {/* no arguments are specified, use default */
    if (doiContext == NULL){
      /* application context not yet defined */
      int argc;
      char *argv[2], *p;
      argc = 1;
      argv[0] = p = DoiMalloc (strlen(title)+1);
      strcpy (argv[0],title);
      argv[1] = NULL;
      if (doiClass == NULL)
        doiClass = &defaultAppClass[0]; /* sets the default application class */
      shell = XtAppInitialize(&doiContext,doiClass,NULL,0,(Cardinal*)&argc,argv,NULL,args,2);
      doiDisplay = XtDisplay (shell);
      DoiFree (p);
    } else /* Application context exists already, only create Appshell*/
      shell = XtAppCreateShell (title,doiAppShell,applicationShellWidgetClass,doiDisplay,args,2);
  }
  if (DoiDebugFlag) printf("DoiInit Application shell=%4x\n", shell);
  *PRshell = shell;
}

/*------------------------------------------------------------------*/
void DoiBell(
  /* rings the X bell */
  Widget PRwidget
)
{
  if (DoiDebugFlag) printf("DoiBell\n");
  if (PRwidget == NULL) return;
  XBell (XtDisplay(PRwidget),100);
}

/*------------------------------------------------------------------*/
void DoiSetFontCursor(
  /* sets a font cursor */
  Widget       PRwidget,
  unsigned int PRcursorShape,
  char        *PRforegroundColor,
  char        *PRbackgroundColor
)
{
  Cursor cursor;
  XColor foreground, background, defCol;

  if (DoiDebugFlag) printf("DoiSetFontCursor(%d)\n",PRcursorShape);
  if (PRwidget == NULL) return;
  if (doiDisplay == NULL) return;
  cursor = XCreateFontCursor (doiDisplay,PRcursorShape);
  if (PRforegroundColor)
    XLookupColor (doiDisplay,DefaultColormap(doiDisplay,DefaultScreen(doiDisplay)),PRforegroundColor,&defCol,&foreground);
  else
    XLookupColor (doiDisplay,DefaultColormap(doiDisplay,DefaultScreen(doiDisplay)),"black",&defCol,&foreground);
  if (PRbackgroundColor)
    XLookupColor (doiDisplay,DefaultColormap(doiDisplay,DefaultScreen(doiDisplay)),PRbackgroundColor,&defCol,&background);
  else
    XLookupColor (doiDisplay,DefaultColormap(doiDisplay,DefaultScreen(doiDisplay)),"white",&defCol,&background);

  XRecolorCursor (doiDisplay,cursor,&foreground,&background);
  XDefineCursor (doiDisplay,XtWindow(PRwidget),cursor);
  XFreeCursor (doiDisplay,cursor);
}

/*------------------------------------------------------------------*/
void DoiResetFontCursor(
  /* sets a font cursor */
  Widget       PRwidget
)
{
  if (DoiDebugFlag) printf("DoiResetFontCursor\n");
  if (PRwidget == NULL) return;
  if (doiDisplay == NULL) return;
  XUndefineCursor (doiDisplay,XtWindow(PRwidget));
}

/*------------------------------------------------------------------*/
typedef struct {
   Window window;
   Window lockWindow;
   XWindowAttributes attr;
   Cursor cursor;
   char *foreground;
   char *background;
} LockWindow;
int DoiCreateLockWindow (
  /* creates a lock window, that's a window which lies upon
   * the specified widget
   */
  Widget  PRwidget,
  Window *PRlockWindow,
  int     PRcursorShape,
  char   *PRforegroundColor,
  char   *PRbackgroundColor
)
{
  Cursor cursor;
  XColor foreground, background, defCol;
  int    ret = 1;  /* something goes wrong */
  LockWindow *lock;
  XSetWindowAttributes attr;
  Window window;

  if (DoiDebugFlag) printf("DoiCreateLockWindow\n");

  if (PRwidget == NULL) return ret;
  if (doiDisplay == NULL) return ret;

  if ((cursor = XCreateFontCursor (doiDisplay,PRcursorShape))){
    if (PRforegroundColor && PRbackgroundColor){
      if (XLookupColor (doiDisplay,DefaultColormap(doiDisplay,DefaultScreen(doiDisplay)),PRforegroundColor,&defCol,&foreground) &&
          XLookupColor (doiDisplay,DefaultColormap(doiDisplay,DefaultScreen(doiDisplay)),PRbackgroundColor,&defCol,&background))
        XRecolorCursor (doiDisplay,cursor,&foreground,&background);
    }
     if ((lock = (LockWindow*)malloc(sizeof(LockWindow))) == NULL)
        return ret;;

     lock->window = XtWindow(XtParent(PRwidget));
     lock->cursor = cursor;
     lock->foreground = XtNewString(PRforegroundColor);
     lock->background = XtNewString(PRbackgroundColor);
     if (!XGetWindowAttributes (doiDisplay,lock->window,
                        &lock->attr)) return ret;
     *PRlockWindow = (Window)lock;
     attr.cursor = cursor;
     attr.event_mask = 0;
     attr.do_not_propagate_mask = KeyPressMask | ButtonPressMask;

     if ((lock->lockWindow = XCreateWindow (doiDisplay,lock->window,
                               0,0,
                               DisplayWidth (doiDisplay,0),
                               DisplayHeight (doiDisplay,0),
                               0,
                               CopyFromParent,
                               InputOnly,
                               CopyFromParent,
                               CWCursor|CWEventMask|CWDontPropagate,
                               &attr))){
         XDefineCursor (doiDisplay,lock->lockWindow,lock->cursor);

         ret = 0; /* okay */
      }
  }

  return ret;
}

/*------------------------------------------------------------------*/
void DoiSetLockWindow (
  LockWindow* PRlock
)
{
  XSetWindowAttributes attr;

  if (DoiDebugFlag) printf("DoiSetLockWindow\n");

  if (PRlock == NULL) return;

  XMapWindow (doiDisplay,PRlock->lockWindow);

  attr.event_mask = StructureNotifyMask
                  | SubstructureNotifyMask
                  | SubstructureRedirectMask
                  | ExposureMask
                  ;
  attr.do_not_propagate_mask = KeyPressMask
                             | KeyReleaseMask
                             | ButtonReleaseMask
                             | ButtonPressMask;
  XChangeWindowAttributes (doiDisplay,PRlock->window,
                           CWEventMask|CWDontPropagate,
                           &attr);
}

/*------------------------------------------------------------------*/
void DoiResetLockWindow (
  LockWindow* PRlock
)
{
  XSetWindowAttributes attr;

  if (DoiDebugFlag) printf("DoiResetLockWindow\n");

  if (PRlock == NULL) return;

  XUnmapWindow (doiDisplay,PRlock->lockWindow);

  attr.event_mask = PRlock->attr.your_event_mask;
  attr.do_not_propagate_mask = PRlock->attr.do_not_propagate_mask;
  XChangeWindowAttributes (doiDisplay,PRlock->window,
                           CWEventMask|CWDontPropagate,
                           &attr);
}

/*------------------------------------------------------------------*/
void DoiDestroyLockWindow (
  /* destroys the lock window */
   LockWindow* PRlock
)
{
  if (DoiDebugFlag) printf("DoiDestroyLockWindow\n");

  if (PRlock == NULL) return;
  
  DoiResetLockWindow (PRlock);

  XDestroyWindow (doiDisplay,PRlock->lockWindow);

  XtFree (PRlock->foreground);
  XtFree (PRlock->background);
  XFreeCursor (doiDisplay,PRlock->cursor);
  free (PRlock);

}

/*------------------------------------------------------------------*/
void DoiManageChild(
  /* does a proper manage */
  Widget PRchild,
  Widget PRshell
)
{
  Widget parent;
  WidgetClass wclass;

  if (DoiDebugFlag) printf("DoiManageChild\n");

  if (PRchild == NULL) return;

  parent = XtParent(PRchild);
  if (parent == NULL)  return;
  wclass = XtClass(parent);
  if (wclass == xmMenuShellWidgetClass ||
      wclass == xmDialogShellWidgetClass){
    return;
  }
  XtVaSetValues (PRchild,XmNmappedWhenManaged,False,NULL);
  XtVaSetValues (PRshell,XmNmappedWhenManaged,False,NULL);
  XtManageChild(PRchild);
  XtRealizeWidget(PRshell);
}

/*------------------------------------------------------------------*/
void DoiRealizeParentWidget(
  Widget PRwidget
)
{
  Widget parent;
  WidgetClass wclass;
  if (DoiDebugFlag) printf("DoiRealizeParentWidget\n");

  if (PRwidget  == NULL) return;

  parent = XtParent(PRwidget);
  if (parent == NULL)  return;
  wclass = XtClass(parent);
  if (wclass != xmMenuShellWidgetClass &&
      wclass != xmDialogShellWidgetClass){
    /*XtVaSetValues (PRwidget,XmNmappedWhenManaged,False,NULL);*/
    XtManageChild(PRwidget);
  }
}

/*------------------------------------------------------------------*/
static int getDefaultCallback(
  /* read the default callback reason of a widget and returns
   * 0 : if there is a default defined
   * or
   * -1 : if no default callback reason is defined or
   *      the widget class is unknown
   */
  Widget PRwidget,
  char *PRcallbackReason
  )
{
  WidgetClass wclass;
  if (DoiDebugFlag) printf("getDefaultCallback\n");
  if (PRwidget == NULL) return -1;
  wclass = XtClass(PRwidget);
  if (      wclass == xmArrowButtonWidgetClass ||
            wclass == xmArrowButtonGadgetClass){
                     strcpy(PRcallbackReason,XmNactivateCallback);
  }else if (wclass == xmCascadeButtonWidgetClass ||
            wclass == xmCascadeButtonGadgetClass){
                     strcpy(PRcallbackReason,XmNactivateCallback);
  }else if (wclass == xmCommandWidgetClass){
                     strcpy(PRcallbackReason,XmNokCallback);
  }else if (wclass == xmDrawnButtonWidgetClass){
                     strcpy(PRcallbackReason,XmNactivateCallback);
  }else if (wclass == xmFileSelectionBoxWidgetClass){
                     strcpy(PRcallbackReason,XmNokCallback);
  }else if (wclass == xmMessageBoxWidgetClass){
                     strcpy(PRcallbackReason,XmNokCallback);
  }else if (wclass == xmPushButtonWidgetClass ||
            wclass == xmPushButtonGadgetClass){
                     strcpy(PRcallbackReason,XmNactivateCallback);
  }else if (wclass == xmScaleWidgetClass){
                     strcpy(PRcallbackReason,XmNvalueChangedCallback);
  }else if (wclass == xmScrollBarWidgetClass){
                     strcpy(PRcallbackReason,XmNvalueChangedCallback);
  }else if (wclass == xmSelectionBoxWidgetClass){
                     strcpy(PRcallbackReason,XmNokCallback);
  }else if (wclass == xmTextWidgetClass){
                     strcpy(PRcallbackReason,XmNactivateCallback);
  }else if (wclass == xmTextFieldWidgetClass){
                     strcpy(PRcallbackReason,XmNactivateCallback);
  }else if (wclass == xmToggleButtonWidgetClass ||
            wclass == xmToggleButtonGadgetClass){
                     strcpy(PRcallbackReason,XmNvalueChangedCallback);
#ifdef FORMAT
  }else if (wclass == xcFormatAlphanumWidgetClass){
                     strcpy(PRcallbackReason,XmNactivateCallback);
  }else if (wclass == xcFormatNumericWidgetClass){
                     strcpy(PRcallbackReason,XmNactivateCallback);
  }else if (wclass == xcFormatDateWidgetClass){
                     strcpy(PRcallbackReason,XmNactivateCallback);
#endif
  }else return -1; /* all other widgets */
  if (DoiDebugFlag) printf("getDefaultCallback: attr = %s\n",PRcallbackReason);
  return 0;
}/* getDefaultCallback */

/*------------------------------------------------------------------*/
void DoiAddCallback(
  Widget PRwidget,
  int (*PRfct)(char *PRElement), 
  char *PRcallbackReason,
  void *PRclientData)
{
  char callbackReason[100];
  char *callbReason;
  if (DoiDebugFlag) printf("DoiAddCallback\n");

  if (PRwidget  == NULL) return;

  if (*PRcallbackReason == StringEnd ){
    if (getDefaultCallback(PRwidget,&callbackReason[0])) return;
  } else
      strcpy (callbackReason,PRcallbackReason);

  callbReason = callbackReason;
  if (strlen(callbackReason)>(size_t)3)
    if(callbackReason[0]=='X')
    if(callbackReason[1]=='m')
    if(callbackReason[2]=='N')
      callbReason = &callbackReason[2];

  if (DoiDebugFlag) printf("DoiAddCallback reason('%s') to widget '%x' clientD='%x'\n",callbReason,PRwidget,PRclientData);
 
  XtAddCallback(PRwidget,callbReason,(XtCallbackProc)PRfct,(caddr_t)PRclientData);
}

/*------------------------------------------------------------------*/
void DoiAddWMCloseCallback(
  Widget PRshell,
  Widget PRwidget,
  int (*PRfct)(char *PRElement), 
  void *PRclientData)
{
  static Atom delWindowAtom;
  Widget parent, shell;


  if ((parent = XtParent(PRwidget)) && XmIsDialogShell (parent))
    shell = parent;
  else if (PRshell  == NULL || !XmIsVendorShell(PRshell)) return;
  else shell = PRshell;
  if (DoiDebugFlag)
     printf("DoiAddWMCloseCallback(shell=%x,widgetParent=%x)\n",PRshell,parent);
  if (DoiDebugFlag)
     printf("   WMCloseCb wird registriert (shell=%x)\n",shell);
  delWindowAtom = XmInternAtom(doiDisplay,"WM_DELETE_WINDOW",TRUE);
  XmRemoveWMProtocols(shell,&delWindowAtom,1);
  XmAddWMProtocolCallback(shell,delWindowAtom,(XtCallbackProc)PRfct,(caddr_t)PRclientData);
}

/*------------------------------------------------------------------*/
void DoiRemoveCallback(
  Widget PRwidget,
  int (*PRfct)(char *PRElement), 
  char *PRcallbackReason,
  void *PRclientData)
{
  char callbackReason[100];
  char *callbReason;
  if (DoiDebugFlag) printf("DoiRemoveCallback\n");

  if (PRwidget  == NULL) return;

  if (*PRcallbackReason == StringEnd){
    if (getDefaultCallback(PRwidget,&callbackReason[0])) return;
  } else
      strcpy (callbackReason,PRcallbackReason);

  callbReason = callbackReason;
  if (strlen(callbackReason)>(size_t)3)
    if(callbackReason[0]=='X')
    if(callbackReason[1]=='m')
    if(callbackReason[2]=='N')
      callbReason = &callbackReason[2];

  if (DoiDebugFlag) printf("DoiRemoveCallback reason('%s') to widget '%x' clientD='%s'\n",callbReason,PRwidget,PRclientData);
 
  XtRemoveCallback(PRwidget,callbReason,(XtCallbackProc)PRfct,(caddr_t)PRclientData);
}

/*------------------------------------------------------------------*/
void DoiShow(
  /*  shows the specified widget */
  Widget PRwidget,
  Widget PRshell,
  int    managed
)
{
  WidgetClass wclass;
  Widget      parent;
  if (DoiDebugFlag) printf("DoiShow\n");

  if (PRwidget == NULL) return;

  parent = XtParent(PRwidget);
  wclass = XtClass(parent);
  if (wclass == xmMenuShellWidgetClass ||
      wclass == xmDialogShellWidgetClass){
    XtVaSetValues (PRwidget,XmNmappedWhenManaged,True,NULL);
    XtManageChild(PRwidget);
    if (managed){
      XtMapWidget (parent);
      if (XmIsGadget(PRwidget))
        XtManageChild (PRwidget);
      else
        XtMapWidget (PRwidget);
    }
  }else{
    if (wclass == applicationShellWidgetClass){
      if (managed){
        XtMapWidget (parent);
      }else{
        XtVaSetValues (PRwidget,XmNmappedWhenManaged,True,NULL);
        XtVaSetValues (parent,XmNmappedWhenManaged,True,NULL);
        XtUnmanageChild(PRwidget);
        XtManageChild(PRwidget);
        XtUnrealizeWidget(PRshell);
        XtRealizeWidget(PRshell);
      }
    }else{/* widget has no shell parent */
      if (XmIsGadget(PRwidget))
        XtManageChild (PRwidget);
      else
        XtMapWidget (PRwidget);
    }
  }
  if (!XmIsGadget(PRwidget) &&
       XtIsSubclass (parent,shellWidgetClass))
    XRaiseWindow (doiDisplay,XtWindow(PRwidget));
}

/*------------------------------------------------------------------*/
void DoiHide(
  /* hides the specified widget */
  Widget PRwidget
)
{
  WidgetClass wclass;
  Widget      parent;
  if (DoiDebugFlag) printf("DoiHide\n");

  if (PRwidget == NULL) return;

  parent = XtParent(PRwidget);
  wclass = XtClass(parent);
  if (wclass == applicationShellWidgetClass){
    XtUnmapWidget(parent);
  }else if (wclass == xmMenuShellWidgetClass ||
            wclass == xmDialogShellWidgetClass){
    /*if (XmIsGadget(PRwidget))
     */ XtUnmanageChild (PRwidget);
    /*else
      XtUnmapWidget (PRwidget);
    XtUnmapWidget(parent);
    */
  }else
    if (XmIsGadget(PRwidget))
      XtUnmanageChild (PRwidget);
    else
      XtUnmapWidget (PRwidget);
}

/*------------------------------------------------------------------*/
int DoiGetWidgetName(
  /* fetches the name of a widget */
  Widget PRwidget,
  char*  PRpopupName,
  int    PRmaxLength
)
{
  char *name;
  if (DoiDebugFlag) printf("DoiGetWidgetName\n");

  if (PRwidget  == NULL) return -1;

  name = XtName(PRwidget);
  if (name && strlen(name) < (size_t)PRmaxLength){
    strcpy (PRpopupName,name);
    return 0;
  }else
    return -1;
}

/*------------------------------------------------------------------*/
void DoiDestroyWidget(
  /* destroys a widget  and if necessary its underlying shell */
  Widget PRwidget
)
{
  Widget parent;

  if (DoiDebugFlag) printf("DoiDestroyWidget\n");

  if (PRwidget  == NULL) return;

  parent = XtParent(PRwidget);
  if (parent  == NULL)
    return;

  if (XtClass (parent) == xmDialogShellWidgetClass){
    XtDestroyWidget(PRwidget);
    return;
  }
  if (XtClass (parent) == applicationShellWidgetClass){
    /* if the parent of the widget is an applicationShell
      destroy the parent too */
    XtDestroyWidget(PRwidget);
    XtDestroyWidget(parent);
  } else{
    XtDestroyWidget(PRwidget);
  }
}

/*------------------------------------------------------------------*/
void DoiLock(
  /* makes a widget insensitive */
  Widget PRwidget
)
{
  if (DoiDebugFlag) printf("DoiLock\n");

  if (PRwidget == NULL) return;

  /*XtVaSetValues (PRwidget,XmNsensitive,False,NULL);*/
  XtSetSensitive (PRwidget,False);
}

/*------------------------------------------------------------------*/
void DoiUnlock(
  /* makes a widget sensitive */
  Widget PRwidget
)
{
  if (DoiDebugFlag) printf("DoiUnlock\n");

  if (PRwidget == NULL) return;

  /*XtVaSetValues (PRwidget,XmNsensitive,True,NULL);*/
  XtSetSensitive (PRwidget,True);
}

/*------------------------------------------------------------------*/
/* The following array is used by 'DoiGetDialogElement'             */
/*------------------------------------------------------------------*/
typedef struct {char *name; int id;} dialogId;
static dialogId STdialogElements[] = {
          "APPLY_BUTTON"   , XmDIALOG_APPLY_BUTTON,
          "CANCEL_BUTTON"  , XmDIALOG_CANCEL_BUTTON,
          "COMMAND_TEXT"   , XmDIALOG_COMMAND_TEXT,
          "DEFAULT_BUTTON" , XmDIALOG_DEFAULT_BUTTON,
          "DIR_LIST"       , XmDIALOG_DIR_LIST,
          "DIR_LIST_LABEL" , XmDIALOG_DIR_LIST_LABEL,
          "FILTER_LABEL"   , XmDIALOG_FILTER_LABEL,
          "FILTER_TEXT"    , XmDIALOG_FILTER_TEXT,
          "HELP_BUTTON"    , XmDIALOG_HELP_BUTTON,
          "HISTORY_LIST"   , XmDIALOG_HISTORY_LIST,
          "LIST"           , XmDIALOG_LIST,
          "LIST_LABEL"     , XmDIALOG_LIST_LABEL,
          "MESSAGE_LABEL"  , XmDIALOG_MESSAGE_LABEL,
          "OK_BUTTON"      , XmDIALOG_OK_BUTTON,
          "PROMPT_LABEL"   , XmDIALOG_PROMPT_LABEL,
          "SELECTION_LABEL", XmDIALOG_SELECTION_LABEL,
          "SEPARATOR"      , XmDIALOG_SEPARATOR,
          "SYMBOL_LABEL"   , XmDIALOG_SYMBOL_LABEL,
          "TEXT"           , XmDIALOG_TEXT,
          "WORK_AREA"      , XmDIALOG_WORK_AREA,
          };

/*------------------------------------------------------------------*/
int DoiGetDialogElement(
  /* returns the child id of an Dialog element*/
  char* PRdialog,
  int*  PRchild
  )
{
  int i;
  if (DoiDebugFlag) printf("DoiGetDialogElement\n");

  for (i=0; i < sizeof(STdialogElements)/sizeof(dialogId); i++)
    if (!strcmp(PRdialog,STdialogElements[i].name)){
      /* dialog element found */
      *PRchild = STdialogElements[i].id;
      return 0;
    }
  return -1;
}/* DoiGetDialogElement */

/*------------------------------------------------------------------*/
Widget DoiGetChild(
  /* returns the widget id of the specified child */
  Widget PRwidget,
  int    PRchild
  )
{
  WidgetClass wclass;
  if (DoiDebugFlag) printf("DoiGetChild\n");

  if (PRwidget == NULL) return NULL;

  wclass = XtClass(PRwidget);
  if (wclass == xmCommandWidgetClass){          /* Command */
    return XmCommandGetChild (PRwidget,PRchild);
  }else if (wclass == xmFileSelectionBoxWidgetClass){/* FileSelectionBox */
    return XmFileSelectionBoxGetChild (PRwidget,PRchild);
  }else if (wclass == xmMessageBoxWidgetClass){   /* MessageBox */
    return XmMessageBoxGetChild (PRwidget,PRchild);
  }else if (wclass == xmSelectionBoxWidgetClass){   /* SelectionBox */
    return XmSelectionBoxGetChild (PRwidget,PRchild);
  }
  return NULL;
}/* DoiGetChild */
