/***************************************************************************/
/*                     SQL_DB.C - database interface                       */
/*-------------------------------------------------------------------------*/
/*      Copyright (C) Siemens Nixdorf Informationssysteme AG 1992          */
/*      All rights reserved                                                */
/***************************************************************************/

# include "sql_defs.h"

static char sccsid[] = "@(#) sql_db.c 1.0 1992-11-29";

# ifdef DATABASE_PRESENT

# ifdef INFORMIX

# include <sqlca.h>
# include <sqlhdr.h>

static _SQCURSOR prep_cursor;

# endif /* INFORMIX */

/*---- CONNECT_DATABASE ---------------------------------------------------*/

void connect_Database()
{

# ifdef INFORMIX
          
          /* 
        Here it is cheched whether the given database connection
        is correct or not. If the database connection is not correct
        then sqlgen reports an error and the execution terminates.
        The folling part depends on the used INFORMIX system. This
        file must be compiled with 'esql' when the open database
        operation is denoted with the $DATABASE statement. Here the
        runtime function for $DATABASE is used, therefore, this file
        can be compiled with 'cc'. To link sqlgen the runtime library
        $(INFORMIXDIR)/lib/libsql4.a must be added.
          */
          /*
           *     $database Database;
           */

          _iqdatabase(Database, 0, 0, (char *) 0);
          if ( sqlca.sqlcode != 0 )
        {
          char msg[200];

          sprintf( msg, "database %s not found or permission denied",
                        Database );
          sql_fatal( msg );
        }
# endif /* INFORMIX */

}

# endif /* DATABASE_PRESENT */

/*---- CHECK_SQL ---------------------------------------------------------*/

# ifdef DATABASE_PRESENT

static void check_sql( sql_text, exists_Parameter )
char *sql_text;
int exists_Parameter;
{
    char *Pointer_to_SQL_Text;
    char *Copy_Pointer;

    Pointer_to_SQL_Text = sql_text;
    Copy_Pointer = sql_text;
    if ( exists_Parameter )
    {
    /* In this part the given SQL statement is prepared to check it.
       If a host variable is found this variable is replaced by a
       joker symbol as it is required by the used database system. The
       host variable identificator is fully deleted. To do this the
       'Copy_Pointer' is used. */
    while ( *Pointer_to_SQL_Text )
        {
        if ( *Pointer_to_SQL_Text & 0x80 )
            {
            /* host variable found */
            Pointer_to_SQL_Text++;
            *Copy_Pointer++ = '?'; /* host variable joker */
            while ( CHARACTER_OF_IDENTIFIER(*Pointer_to_SQL_Text) )
                Pointer_to_SQL_Text++;
            }
        else
            {
            *Copy_Pointer++ = *Pointer_to_SQL_Text++;
            }
        }
    *Copy_Pointer = '\0';
    }
# ifdef INFORMIX
    /* check statement */
    _iqprepare( &prep_cursor, sql_text );
    if ( sqlca.sqlcode != 0 )
      {
	char msg[401];
	char msg2[201];
	int length;
	rgetmsg( sqlca.sqlcode, msg, 200 );
	sprintf( msg2, "(DB errno: %d) ", sqlca.sqlcode );
	sprintf( msg2+strlen(msg2), msg, sqlca.sqlerrm );
	length = strlen( msg2 );
	if ( *(msg2 + length - 1) == '\n' )
	  *(msg2 + length -1 ) = '\0';
	sql_error( msg2 );
      }
# endif
}

# endif


/**************************************************************************/
/*               procedures for generating hostvariables                  */
/**************************************************************************/

/*---- GET_ESQL_TYPE ------------------------------------------------------*/

static char *get_ESQL_type( SQL_type )
SQL_TYPE *SQL_type;
{
    switch( SQL_type->typecode )
        {
        case CHAR:
        case VARCHAR:
            return "char";
        case INTEGER:
        case SERIAL:
            return "long";
        case SMALLINT:
            return "short int";
        case SMALLFLOAT:
            return "float";
        case FLOAT:
            return "double";
        case DATE:
            return "long";
        case DECIMAL:
        case MONEY:
            return "dec_t";
        case DATETIME:
            return "dtime_t";
        case INTERVAL:
            return "intrvl_t";
        }
}

/*---- GENERATE_HOSTVARIABLES ---------------------------------------------*/

void generate_Hostvariables( View_Name, Variables )
YY_IDENT *View_Name;
PAR_LIST *Variables;
{
    PAR_LIST *Current_Variable;

    fprintf( ESQLfile, "\n" );
    for ( Current_Variable = Variables; Current_Variable != PAR_LIST_NULL; 
            Current_Variable = Current_Variable->next )
        {
        fprintf( ESQLfile, "$static %s H_%s_%s", 
                get_ESQL_type( &(Current_Variable->type) ), View_Name,
                            get_symbol( Current_Variable->ident.Hash ) );
        if ( Current_Variable->type.typecode == CHAR ) 
                fprintf( ESQLfile, "[%d+1];\n",
                            Current_Variable->type.Char.length );
        else
            if ( Current_Variable->type.typecode == VARCHAR )
                fprintf( ESQLfile, "[%d+1];\n",
                            Current_Variable->type.VarChar.maxlength );
            else
                fprintf( ESQLfile, ";\n" );
        fprintf( ESQLfile, "$static short I_%s_%s;\n", View_Name, 
                            get_symbol( Current_Variable->ident.Hash ) );
        }
}

/**************************************************************************/
/*                 procedures for marking hostvariables                   */
/**************************************************************************/

/*---- GET_NEXT_SQL_IDENTIFIER ---------------------------------------------*/

static int get_next_sql_identifier( sql_text, sql_text_length, ident, length )
char *sql_text;
int sql_text_length;
char **ident;
int *length;
{
  char *Pointer_to_sql_text;
  int current_index;
  char *new_identifier;
  int new_length;

  Pointer_to_sql_text = sql_text;
  for ( current_index = 0; current_index < sql_text_length; current_index++ )
    {
      if ( SQL_WHITESPACE( Pointer_to_sql_text[current_index] ) )
      continue;
      if (FIRST_CHARACTER_OF_IDENTIFIER(Pointer_to_sql_text[current_index]))
    {
      new_identifier = &Pointer_to_sql_text[current_index];
      new_length = 1;
      for ( current_index++; current_index < sql_text_length; 
                                                          current_index++ )
        {
          if ( CHARACTER_OF_IDENTIFIER(Pointer_to_sql_text[current_index]) )
        {
          new_length++;
          continue;
        }
              break;
             }
      *ident = new_identifier;
      *length = new_length;
      return TRUE;
    }
      if ( Pointer_to_sql_text[current_index] == '\"' ||
      Pointer_to_sql_text[current_index] == '\'' )
    {
      /* string literal has to be excluded from identifier analysing */
      char delimiter;
      
      delimiter = Pointer_to_sql_text[current_index];
      for ( current_index++ ; current_index < sql_text_length;
           current_index++ )
        {
          if ( Pointer_to_sql_text[current_index] == delimiter )
        break;
        }
    }
    }
  return FALSE;
}

/*---- MARK_HOSTVARIABLES -------------------------------------------------*/

void mark_hostvariables( parameters, sql_clause )
PAR_LIST *parameters;
SQL_TEXT *sql_clause;
{
    char *sql_text;
    int sql_text_length;
    char *Symbol;
    int Length;
    PAR_LIST *Actual_Parameter;
    char *actual_Parameter_symbol;
    int actual_Parameter_length;

    sql_text_length = sql_clause->Length;
    sql_text = sql_clause->Begin;

    while( get_next_sql_identifier( sql_text, sql_text_length, 
                         &Symbol, &Length ) )
      {
    for ( Actual_Parameter = parameters; 
                    Actual_Parameter != PAR_LIST_NULL;
                            Actual_Parameter = Actual_Parameter->next )
      {
        actual_Parameter_symbol = 
                              get_symbol( Actual_Parameter->ident.Hash );
        actual_Parameter_length = strlen( actual_Parameter_symbol );
        if ( actual_Parameter_length != Length )
          continue;
        if ( strncmp( actual_Parameter_symbol, Symbol, Length ) == 0 )
          {
        *Symbol = *Symbol | 0x80;
        break;
          }
      }
    sql_text_length = sql_text_length - (Symbol - sql_text) - Length;
    sql_text = Symbol + Length;
      }
}

/***************************************************************************/
/*               procedures for generating esql view record                */
/***************************************************************************/

/*---- GENERATE_ESQL_RECORD -----------------------------------------------*/

void generate_ESQL_record( ident, fields )
YY_IDENT *ident;
PAR_LIST *fields;
{
    PAR_LIST *Current_Field;

    fprintf( ESQLfile, "\ntypedef struct\n{\n" );
    for ( Current_Field = fields; Current_Field != PAR_LIST_NULL; 
            Current_Field = Current_Field->next )
        {
        fprintf( ESQLfile, "  %s %s;\n",
                                get_CLanguage_type( &(Current_Field->type) ),
                                    get_symbol( Current_Field->ident.Hash ) );
        }
    fprintf( ESQLfile, "} %s_R;\n", get_symbol( ident->Hash ) );
}

/***************************************************************************/
/*               procedures for generating esql statements                 */
/***************************************************************************/

/*---- PRINT_STATEMENT_PARAMETERS -----------------------------------------*/

static void print_statement_parameters( Parameters )
PAR_LIST *Parameters;
{
    PAR_LIST *Current_Parameter;

    for ( Current_Parameter = Parameters;
        Current_Parameter != PAR_LIST_NULL;
          Current_Parameter = Current_Parameter->next )
        {
        fprintf( ESQLfile, "%s", get_symbol( Current_Parameter->ident.Hash ) );
        if ( Current_Parameter->next != PAR_LIST_NULL )
            fprintf( ESQLfile, ", " );
        }
    fprintf( ESQLfile, " )\n" );
    for ( Current_Parameter = Parameters;
        Current_Parameter != PAR_LIST_NULL;
              Current_Parameter = Current_Parameter->next )
        {
      switch ( Current_Parameter->mode )
        {
        case M_OUT:
        case M_INOUT:
          fprintf( ESQLfile, "%s *%s;\n",
              get_CLanguage_type( &(Current_Parameter->type) ),
              get_symbol( Current_Parameter->ident.Hash ) );
          break;
        case M_IN:
          fprintf( ESQLfile, "%s %s;\n",
              get_CLanguage_type( &(Current_Parameter->type) ),
              get_symbol( Current_Parameter->ident.Hash ) );
          break;
        }
        }
}

/*---- ASSIGN_PARAMETERS_TO_HOSTS -----------------------------------------*/

static void assign_parameters_to_hosts( Name, Parameters)
char *Name;
PAR_LIST *Parameters;
{
    PAR_LIST *Current_Parameter;

    for ( Current_Parameter = Parameters;
        Current_Parameter != PAR_LIST_NULL;
          Current_Parameter = Current_Parameter->next )
        {
      char *actual_parameter_name;
      
      actual_parameter_name = get_symbol( Current_Parameter->ident.Hash );

      /* Only IN and INOUT parameter have to be assigned to host and
         indicator variables */

      switch ( Current_Parameter->mode )
        {
        case M_IN:
          /* In this mode parameters are passed by value */
          if ( Current_Parameter->type.typecode == CHAR ||
                 Current_Parameter->type.typecode == VARCHAR )
        {
          fprintf( ESQLfile, "  strcpy( H_%s_%s, %s.value );\n", Name,
                                 actual_parameter_name, actual_parameter_name );
        }
          else
        fprintf( ESQLfile, "  H_%s_%s = %s.value;\n", Name,
                         actual_parameter_name, actual_parameter_name );
          fprintf( ESQLfile, "  I_%s_%s = %s.indicator;\n", Name,
                                actual_parameter_name, actual_parameter_name );
          break;
        case M_INOUT:
          /* In this mode parameters are passed by reference */
          if ( Current_Parameter->type.typecode == CHAR ||
                 Current_Parameter->type.typecode == VARCHAR )
        {
          fprintf( ESQLfile, "  strcpy( H_%s_%s, %s->value );\n", Name,
                                actual_parameter_name, actual_parameter_name );
        }
          else
        fprintf( ESQLfile, "  H_%s_%s = %s->value;\n", Name,
                         actual_parameter_name, actual_parameter_name );
          fprintf( ESQLfile, "  I_%s_%s = %s->indicator;\n", Name,
                                actual_parameter_name, actual_parameter_name );
          break;
        }
    }
}

/*---- ASSIGN_HOSTS_TO_PARAMETERS  -----------------------------------------*/

static void assign_hosts_to_parameters( Name, Parameters)
char *Name;
PAR_LIST *Parameters;
{
  PAR_LIST *Current_Parameter;
  
  for ( Current_Parameter = Parameters;
    Current_Parameter != PAR_LIST_NULL;
       Current_Parameter = Current_Parameter->next )
    {
      char *actual_parameter_name;
      
      actual_parameter_name = get_symbol( Current_Parameter->ident.Hash );
      
      /* Only OUT and INOUT parameter have to be assigned to the parameters */
      
      switch( Current_Parameter->mode )
          {
          case M_OUT:
          case M_INOUT:
          if ( Current_Parameter->type.typecode == CHAR ||
                 Current_Parameter->type.typecode == VARCHAR )
              {
              fprintf( ESQLfile, "  strcpy( %s->value, H_%s_%s );\n",
                        actual_parameter_name, Name, actual_parameter_name );
              }
          else
              fprintf( ESQLfile, "  %s->value = H_%s_%s;\n",
                    actual_parameter_name, Name, actual_parameter_name );
          fprintf( ESQLfile, "  %s->indicator = I_%s_%s;\n",
          actual_parameter_name, Name, actual_parameter_name );
          break;
          }
    }
}

/*---- SQL_STRNCMP -------------------------------------------------------*/

static int sql_strncmp( s1, s2, n )
char *s1;
char *s2;
int n;
{
    while ( n-- )
        {
        /* In difference to INFORMIX the characters of the identifier
           are converted temporary in capital letters, furthermore the
           host variable bit is cleared */
        if ( (*s1++&0x5f) == (*s2++&0x5f) )
            continue;
        else
            return -1;
        }
    if ( CHARACTER_OF_IDENTIFIER( *s2 & 0x5f ) )
        return -1; /* identifier continues here */
    return 0; /* identifier checked fully */
}

/*---- PRINT_PARAMETERIZED_SQL_TEXT --------------------------------------*/

static int Select_active = FALSE;
static int generate_indicator = FALSE;

static int print_parameterized_sql_text( Statement_Name, 
                                            SQL_Text, Text_Length )
char *Statement_Name;
char *SQL_Text;
int Text_Length;
{
    char *Pointer_to_SQL_Text;

    for ( Pointer_to_SQL_Text = SQL_Text; *Pointer_to_SQL_Text != ';'; )
    {
        if ( sql_strncmp( "SELECT", Pointer_to_SQL_Text, 6 ) == 0 )
            {
            Select_active = TRUE;
            generate_indicator = TRUE;
            /* SELECT has to be printed before calling recursively
               'print_parameterized_sql_text' */
            while ( CHARACTER_OF_IDENTIFIER( *Pointer_to_SQL_Text ) )
                {
                fputc( *Pointer_to_SQL_Text++, ESQLfile );
                Text_Length--;
                }
            print_parameterized_sql_text( Statement_Name,
                                        Pointer_to_SQL_Text, Text_Length );
            break;
            }
        if ( Select_active )
            {
            if ( sql_strncmp( "FROM", Pointer_to_SQL_Text, 4 ) == 0 )
                {
                generate_indicator = FALSE;
                }
            }
        else
            {
            if ( sql_strncmp( "FETCH", Pointer_to_SQL_Text, 5 ) == 0 ||
                  sql_strncmp( "INSERT", Pointer_to_SQL_Text, 6 ) == 0 ||
                   sql_strncmp( "UPDATE", Pointer_to_SQL_Text, 6 ) == 0 )
                {
                generate_indicator = TRUE;
                }
            }
        if ( *Pointer_to_SQL_Text & 0x80 )
            {
            char *actual_host;

            /* Here host and indicator (if necessary) are generated */

            actual_host = Pointer_to_SQL_Text;
            fprintf( ESQLfile, "$H_%s_%c", Statement_Name, 
                                            *Pointer_to_SQL_Text & 0x7f );
            Pointer_to_SQL_Text++;
            Text_Length--;
            while ( CHARACTER_OF_IDENTIFIER( *Pointer_to_SQL_Text ) )
                {
                fputc( *Pointer_to_SQL_Text++, ESQLfile );
                Text_Length--;
                }
            if ( generate_indicator )
                {
                fprintf( ESQLfile, ":I_%s_%c", Statement_Name, 
                                                *actual_host & 0x7f );
                actual_host++;
                while ( CHARACTER_OF_IDENTIFIER( *actual_host ) )
                    {
                    fputc( *actual_host++, ESQLfile );
                    }
                }
            }
        else
            {
            fputc( *Pointer_to_SQL_Text++, ESQLfile );
            Text_Length--;
            }
        }
}

/*---- GENERATE_SINGLE_SQL_STATEMENT -------------------------------------*/

static void generate_single_ESQL_statement( Procedure_Name,
                        SQL_Text, Text_Length, Parameters )
char *Procedure_Name;
char *SQL_Text;
int Text_Length;
PAR_LIST *Parameters;
{
    while( SQL_WHITESPACE( *SQL_Text ) )
    {
    SQL_Text++;
    Text_Length--;
    }

# ifdef DATABASE_PRESENT
    if ( check_Flag )
    {
    char *check_buffer;

    check_buffer = Get_String_Space( Text_Length + 2 ); /* ; + '\0' */
    strncpy( check_buffer, SQL_Text, Text_Length + 1 );
    *(check_buffer + Text_Length + 1 ) = '\0';
    if ( Parameters == PAR_LIST_NULL )
            check_sql( check_buffer, FALSE );
        else
            check_sql( check_buffer, TRUE );
        Free_String_Space( check_buffer );
        }
# endif
    fprintf( ESQLfile, "  $" );
    if ( Parameters == PAR_LIST_NULL )
        {
        while( *SQL_Text != ';' )
            fputc( *SQL_Text++, ESQLfile );
        }
    else
        {
        print_parameterized_sql_text( Procedure_Name, SQL_Text, Text_Length );
        Select_active = FALSE;
        generate_indicator = FALSE;
        }
        fprintf( ESQLfile, ";\n" );
}

/*---- GENERATE_ESQL_STATEMENT ---------------------------------------------*/

void generate_ESQL_statement( Statement_Name, Parameters, SQL_Text )
char *Statement_Name;
PAR_LIST *Parameters;
STATEMENT_LIST *SQL_Text;
{
  STATEMENT_LIST *actual_statement;

  if ( Parameters == PAR_LIST_NULL )
    {
      fprintf( ESQLfile, "\nint SQL_%s()\n{\n", Statement_Name );
    }
  else
    {
      fprintf( ESQLfile, "\nint SQL_%s( ", Statement_Name );
      print_statement_parameters( Parameters );
      fprintf( ESQLfile, "{\n" );
      assign_parameters_to_hosts( Statement_Name, Parameters );
    }
  for ( actual_statement = SQL_Text; actual_statement != STATEMENT_LIST_NULL;
                                    actual_statement = actual_statement->next )
    {
      generate_single_ESQL_statement( Statement_Name, 
                       actual_statement->statement.Begin,
                       actual_statement->statement.Length,
                                       Parameters );
      if ( actual_statement->next != STATEMENT_LIST_NULL )
    {
      fprintf( ESQLfile, "  if ( sqlca.sqlcode != 0 )\n" );
      fprintf( ESQLfile, "    return sqlca.sqlcode;\n" );
    }
    }
  if ( Parameters != PAR_LIST_NULL )
    assign_hosts_to_parameters( Statement_Name, Parameters );
  fprintf( ESQLfile, "  return sqlca.sqlcode;\n}\n" );
}

/***************************************************************************/
/*               procedures for generating cursor functions                */
/***************************************************************************/

/*---- GENERATE_INSERT_CURSOR_DECLARE -------------------------------------*/

static void generate_insert_cursor_declare( View_Name, Table_Name, Columns )
char *View_Name;
char *Table_Name;
LIST *Columns;
{
    LIST *Current_Column;
    PAR_LIST *Current_Component;

    fprintf( ESQLfile, "  INSERT INTO %s ", Table_Name );
    Current_Column = Columns;                
    if ( Current_Column != LIST_NULL )
        {
        fprintf( ESQLfile, "( " );
        while ( Current_Column != LIST_NULL )
            {
            fprintf( ESQLfile, "%s",
                               get_symbol( Current_Column->ident.Hash ) );
            if ( Current_Column->next != LIST_NULL )
                fprintf( ESQLfile, ",\n" );
            Current_Column = Current_Column->next;
            }
        fprintf( ESQLfile, " ) " );
        }
    fprintf( ESQLfile, "VALUES (" );
    for ( Current_Component = Actual_View; 
            Current_Component != PAR_LIST_NULL;
                Current_Component = Current_Component->next )
        {
        fprintf( ESQLfile, " $H_%s_%s", View_Name, 
                            get_symbol( Current_Component->ident.Hash ) );

        fprintf( ESQLfile, ":I_%s_%s", View_Name,
                            get_symbol( Current_Component->ident.Hash ) );

        if ( Current_Component->next != PAR_LIST_NULL )
            fprintf( ESQLfile, ",\n" );
        }
    fprintf( ESQLfile, " )" );
}

/*---- GENERATE_DECLARE_FUNCTION ------------------------------------------*/

static void generate_declare_function( View_Name, Cursor_Name, 
                                                Cursor_Parameter, Cursor )
char *View_Name;
char *Cursor_Name;
PAR_LIST *Cursor_Parameter;
SQL_CURSOR *Cursor;
{
    fprintf( ESQLfile, "\nint SQL_%s_Declare()\n{\n", View_Name );
    switch ( Cursor->Select.type )
        {
        case C_INSERT:
            fprintf( ESQLfile, "  $DECLARE %s CURSOR FOR\n  ", Cursor_Name );
            generate_insert_cursor_declare( View_Name, 
                                    get_symbol( Cursor->Insert.table.Hash),
                                    Cursor->Insert.columns );
            break;
        case C_SELECT:
        case C_UPDATE:
            {
            char *SQL_Text;
            int Text_Length;

            SQL_Text = Cursor->Select.text.Begin;
            Text_Length = Cursor->Select.text.Length;
            if ( Cursor->Select.type == C_SELECT )
                fprintf( ESQLfile, "  $DECLARE %s SCROLL CURSOR FOR\n  ",
                                Cursor_Name );
            else
                fprintf( ESQLfile, "  $DECLARE %s CURSOR FOR\n  ",
                                Cursor_Name );
            /* Here after the select statement a semicolon is inserted,
               space for this character is always available */
            *(SQL_Text + Text_Length) = ';';
            Text_Length++;
            if ( Cursor_Parameter == PAR_LIST_NULL )
                {
                while ( *SQL_Text != ';' )
                    fputc( *SQL_Text++, ESQLfile );
                }
            else
                {
        char newView_Name[401];

        sprintf( newView_Name, "%s%s", View_Name, View_Name );
                print_parameterized_sql_text( newView_Name, SQL_Text, 
                                                Text_Length );
                Select_active = FALSE;
                generate_indicator = FALSE;
                }
            if ( Cursor->Select.type == C_UPDATE )
                fprintf( ESQLfile, "FOR UPDATE" );
            }
        }
    fprintf( ESQLfile, ";\n  return sqlca.sqlcode;\n}\n" );
}

/*---- GENERATE_OPEN_FUNCTION ---------------------------------------------*/

static void generate_open_function( View_Name, Cursor_Name, Cursor_Parameter )
char *View_Name;
char *Cursor_Name;
PAR_LIST *Cursor_Parameter;
{
    PAR_LIST *Current_Parameter;

    if ( Cursor_Parameter == PAR_LIST_NULL )
    fprintf( ESQLfile, "\nint SQL_%s_Open()\n{\n", View_Name );
    else
    {
    fprintf( ESQLfile, "\nint SQL_%s_Open( ", View_Name );
    for ( Current_Parameter = Cursor_Parameter;
        Current_Parameter != PAR_LIST_NULL;
          Current_Parameter = Current_Parameter->next )
        {
        fprintf( ESQLfile, "%s",
                get_symbol( Current_Parameter->ident.Hash ) );
        if ( Current_Parameter->next != PAR_LIST_NULL )
        fprintf( ESQLfile, ", " );
        }
    fprintf( ESQLfile, " )\n" );
    for ( Current_Parameter = Cursor_Parameter;
        Current_Parameter != PAR_LIST_NULL;
          Current_Parameter = Current_Parameter->next )
        {
        fprintf( ESQLfile, "%s %s;\n",
                            get_CLanguage_type( &(Current_Parameter->type) ),
                                get_symbol( Current_Parameter->ident.Hash ) );
            }
    fprintf( ESQLfile, "{\n" );
    {
    char temp_View_Name[401];

    sprintf( temp_View_Name, "%s%s", View_Name, View_Name );
    assign_parameters_to_hosts( temp_View_Name, Cursor_Parameter );
    }
    }
    fprintf( ESQLfile, "  $OPEN %s;\n", Cursor_Name );
    fprintf( ESQLfile, "  return sqlca.sqlcode;\n}\n" );
}

/*---- ASSIGN_VIEW_PARAMETERS_TO_HOSTS ------------------------------------*/

static void assign_view_parameters_to_hosts( Name, Parameters)
char *Name;
PAR_LIST *Parameters;
{
    PAR_LIST *Current_Parameter;

    for ( Current_Parameter = Parameters;
        Current_Parameter != PAR_LIST_NULL;
          Current_Parameter = Current_Parameter->next )
        {
        char *actual_parameter_name;

    actual_parameter_name = get_symbol( Current_Parameter->ident.Hash );
    if ( Current_Parameter->type.typecode == CHAR ||
           Current_Parameter->type.typecode == VARCHAR )
        {
        fprintf( ESQLfile, "  strcpy( H_%s_%s, row.%s.value );\n", Name,
                    actual_parameter_name, actual_parameter_name );
        }
    else
        fprintf( ESQLfile, "  H_%s_%s = row.%s.value;\n", Name,
                actual_parameter_name, actual_parameter_name );
    fprintf( ESQLfile, "  I_%s_%s = row.%s.indicator;\n", Name,
            actual_parameter_name, actual_parameter_name );
    }
}

/*---- ASSIGN_HOSTS_TO_PARAMETERS  -----------------------------------------*/

void assign_hosts_to_view( View_Name, Parameters)
char *View_Name;
PAR_LIST *Parameters;
{
    PAR_LIST *Current_Parameter;

    for ( Current_Parameter = Parameters;
    Current_Parameter != PAR_LIST_NULL;
      Current_Parameter = Current_Parameter->next )
    {
    char *actual_parameter_name;

    actual_parameter_name = get_symbol( Current_Parameter->ident.Hash );
	if ( Current_Parameter->type.typecode == CHAR ||
          Current_Parameter->type.typecode == VARCHAR )
    {
    fprintf( ESQLfile, "  strcpy( record->%s.value, H_%s_%s );\n",
        actual_parameter_name, View_Name, actual_parameter_name );
    }
    else
    fprintf( ESQLfile, "  record->%s.value = H_%s_%s;\n",
        actual_parameter_name, View_Name, actual_parameter_name );
    fprintf( ESQLfile, "  record->%s.indicator = I_%s_%s;\n",
        actual_parameter_name, View_Name, actual_parameter_name );
    }
}

/*---- GENERATE_FETCH_HOST_ASSIGNMENT -------------------------------------*/

static void generate_fetch_host_assignment( View_Name )
char *View_Name;
{
    fprintf( ESQLfile, "\nstatic int %s_Hosts_to_View( record )\n", View_Name );
    fprintf( ESQLfile, "%s_R *record;\n{\n", View_Name );
    assign_hosts_to_view( View_Name, Actual_View );
    fprintf( ESQLfile, "}\n" );
}

/*---- GENERATE_FETCH_OPERATIONS ------------------------------------------*/

static void generate_fetch_operations( View_Name, Cursor_Name, Operation_Name )
char *View_Name;
char *Cursor_Name;
char *Operation_Name;
{
    PAR_LIST *Current_Component;

    fprintf( ESQLfile, "\nint SQL_%s_Fetch_%s( record )\n", View_Name,
                                Operation_Name );
    fprintf( ESQLfile, "%s_R *record;\n{\n", View_Name );
    fprintf( ESQLfile, "  $FETCH " );
    while ( *Operation_Name )
    fputc( (*Operation_Name++)&0x5f, ESQLfile );
    fprintf( ESQLfile, " %s INTO", Cursor_Name );
    for ( Current_Component = Actual_View; Current_Component != PAR_LIST_NULL;
        Current_Component = Current_Component->next )
    {
    fprintf( ESQLfile, " $H_%s_%s", View_Name,
                get_symbol( Current_Component->ident.Hash ) );

    fprintf( ESQLfile, ":I_%s_%s", View_Name,
                get_symbol( Current_Component->ident.Hash ) );

        if ( Current_Component->next != PAR_LIST_NULL )
            fprintf( ESQLfile, ",\n" );
        }
    fprintf( ESQLfile, ";\n" );
    fprintf( ESQLfile, "  %s_Hosts_to_View( record );\n ", View_Name );
    fprintf( ESQLfile, "  return sqlca.sqlcode;\n}\n" );
}

/*---- GENERATE_SINGLE_CURSOR_FUNCTION ------------------------------------*/

static void generate_single_cursor_function( View_Name, Cursor_Name,
                         Operation, Parameter )
char *View_Name;
char *Cursor_Name;
char *Operation;
char *Parameter;
{
    fprintf( ESQLfile, "\nint SQL_%s_%s(%s)\n", View_Name, Operation,
                        Parameter );
    if ( *Parameter != '\0')
        {
        fprintf( ESQLfile, "%s_R row;\n{\n", View_Name ); 
    assign_view_parameters_to_hosts( View_Name, Actual_View );
    fprintf( ESQLfile, "  $" );
        }
    else
        fprintf( ESQLfile, "{\n  $" );
    while ( *Operation )
    fputc( (*Operation++)&0x5f, ESQLfile );
    fprintf( ESQLfile, " %s;\n", Cursor_Name );
       fprintf( ESQLfile, "  return sqlca.sqlcode;\n}\n" );
}

/*---- EXTRACT_COLUMNS ----------------------------------------------------*/

char *extract_columns( SQL_Text )
char *SQL_Text;
{
    char *first_column;
    char *end_column;
    char *new_columns;
    int column_length;

    first_column = SQL_Text + 6; /* The first 6 characters represent always
                    the keyword 'select' */
    while ( SQL_WHITESPACE( *first_column ) )
    first_column++;
    if ( sql_strncmp( first_column, "all", 3 ) == 0 )
      first_column += 3;
    if ( sql_strncmp( first_column, "distinct", 8 ) == 0 )
      first_column += 8;
    if ( sql_strncmp( first_column, "unique", 6 ) == 0 )
      first_column += 6;
    while ( SQL_WHITESPACE( *first_column ) )
    first_column++;

    end_column = first_column;
    while ( (sql_strncmp( end_column, "into", 4 ) != 0) &&
         (sql_strncmp( end_column, "from", 4 ) ) )
      end_column++;
    end_column--;
    while ( SQL_WHITESPACE( *end_column ) )
      end_column--;

    column_length = end_column - first_column + 1;
    new_columns = Get_String_Space( column_length + 1 );
    strncpy( new_columns, first_column, column_length );
    *(new_columns + column_length ) = '\0';
    return new_columns;
}

/*---- EXTRACT_TABLES ------------------------------------------------------*/

char *extract_tables( SQL_Text, Text_Length )
char *SQL_Text;
int Text_Length;
{
    char *first_table;
    char *end_table;
    char *new_tables;
    int table_length;

    first_table = SQL_Text + 6; /* The first 6 characters represent always
                    the keyword 'select' */
    Text_Length -= 6;
    while ( SQL_WHITESPACE( *first_table ) )
      {
    first_table++;
    Text_Length--;
      }

    while ( sql_strncmp( first_table, "from", 4 ) != 0 )
      {
    first_table++;
    Text_Length--;
      }

    first_table += 4;
    Text_Length -=4;
    while ( SQL_WHITESPACE( *first_table ) )
      {
    first_table++;
    Text_Length--;
      }

    end_table = first_table;

    while ( Text_Length )
      {
    switch ( (*end_table&0x5f) )
      {
      case 'W':
        if ( sql_strncmp( end_table, "WHERE", 5 ) == 0 )
          break;
        end_table++;
        Text_Length--;
        continue;
      case 'G':
        if ( sql_strncmp( end_table, "GROUP", 5 ) == 0 )
          break;
        end_table++;
        Text_Length--;
        continue;
      case 'H':
        if ( sql_strncmp( end_table, "HAVING", 6 ) == 0 )
          break;
        end_table++;
        Text_Length--;
        continue;
      case 'O':
        if ( sql_strncmp( end_table, "ORDER", 5 ) == 0 )
          break;
        end_table++;
        Text_Length--;
        continue;
      case 'I':
        if ( sql_strncmp( end_table, "INTO", 4 ) == 0 )
          break;
        end_table++;
        Text_Length--;
        continue;
      case 'U':
        if ( sql_strncmp( end_table, "UNION", 5 ) == 0 )
          break;
        end_table++;
        Text_Length--;
        continue;
      case ';':
        break;
      default:
        end_table++;
        Text_Length--;
        continue;
      }
    break;
      }

    end_table--;
    while ( SQL_WHITESPACE( *end_table ) )
      {
    end_table--;
      }

    table_length = end_table - first_table + 1;
    new_tables = Get_String_Space( table_length + 1 );
    strncpy( new_tables, first_table, table_length );
    *(new_tables + table_length ) = '\0';
    return new_tables;
}

static void generate_delete_function( View_Name, Cursor_Name, Cursor )
char *View_Name;
char *Cursor_Name;
SQL_CURSOR *Cursor;
{
  fprintf( ESQLfile, "\nint SQL_%s_Delete()\n{\n", View_Name );
  fprintf( ESQLfile, "  $DELETE FROM %s ", 
                             extract_tables( Cursor->Select.text.Begin,
                                        Cursor->Select.text.Length ) );
  fprintf( ESQLfile, "WHERE CURRENT OF %s;\n", Cursor_Name );
  fprintf( ESQLfile, "  return sqlca.sqlcode;\n}\n" );
} 

static void generate_update_function( View_Name, Cursor_Name, Cursor )
char *View_Name;
char *Cursor_Name;
SQL_CURSOR *Cursor;
{
  char *column_list;
  PAR_LIST *Current_Component;

  fprintf( ESQLfile, "\nint SQL_%s_Update(row)\n", View_Name );
  fprintf( ESQLfile, "%s_R row;\n{\n", View_Name );
  assign_view_parameters_to_hosts( View_Name, Actual_View );
  fprintf( ESQLfile, "  $UPDATE %s SET ", 
                           extract_tables( Cursor->Select.text.Begin,
                                      Cursor->Select.text.Length ) );
  column_list = extract_columns(Cursor->Select.text.Begin);
  if ( *column_list == '*' )
    fprintf( ESQLfile, "* = ( " );
  else
    {
      char *temp_list;
      int comma_present;
      
      comma_present = FALSE;
      for ( temp_list = column_list; *temp_list; temp_list++ )
    {
      if ( *temp_list == ',' )
        {
          comma_present = TRUE;
          break;
        }
    }
      if ( comma_present )
    {
      fprintf( ESQLfile, "( " );
      while ( *column_list )
        fputc( (*column_list++)&0x7f, ESQLfile );
      fprintf( ESQLfile, ") = ( " );
    }
      else
    {
      while ( *column_list )
        fputc( (*column_list++)&0x7f, ESQLfile );
      fprintf( ESQLfile, " = ( " );
    }
    }

  for ( Current_Component = Actual_View; 
            Current_Component != PAR_LIST_NULL;
                Current_Component = Current_Component->next )
    {
      fprintf( ESQLfile, " $H_%s_%s", View_Name, 
                                get_symbol( Current_Component->ident.Hash ) );

      fprintf( ESQLfile, ":I_%s_%s", View_Name,
                                get_symbol( Current_Component->ident.Hash ) );

      if ( Current_Component->next != PAR_LIST_NULL )
    fprintf( ESQLfile, ",\n" );
    }

  fprintf( ESQLfile, " )  WHERE CURRENT OF %s;\n", Cursor_Name );
  fprintf( ESQLfile, "  return sqlca.sqlcode;\n}\n" );
} 

/*---- GENERATE_CURSOR_FUNCTIONS ------------------------------------------*/

void generate_cursor_functions( View_Name, Cursor_Parameter, Cursor )
char *View_Name;
PAR_LIST *Cursor_Parameter;
SQL_CURSOR *Cursor;
{
    int Cursor_Name_Length;
    char Cursor_Name[INFORMIX_NAME_LENGTH + 1];

    Cursor_Name_Length = strlen( View_Name );
    if ( Cursor_Name_Length > INFORMIX_NAME_LENGTH )
        {
        char msg[200];

        sprintf( msg, "cursor name %s truncated", View_Name );
        sql_information( msg );
        strncpy( Cursor_Name, View_Name, INFORMIX_NAME_LENGTH );
        Cursor_Name[INFORMIX_NAME_LENGTH] = '\0';
        }
    else
        strcpy( Cursor_Name, View_Name );

# ifdef DATABASE_PRESENT
    if ( Cursor->Select.type == C_SELECT || Cursor->Select.type == C_UPDATE )
        {
        if ( check_Flag )
        {
            char *check_buffer;
            char *SQL_Text;
            int Text_Length;

            SQL_Text = Cursor->Select.text.Begin;
            Text_Length = Cursor->Select.text.Length;
            check_buffer = Get_String_Space( Text_Length + 2 ); /* ; + '\0' */
            strncpy( check_buffer, SQL_Text, Text_Length );
            /* The following statement ends the given select statement
               whether a semicolon is given or not (FOR UPDATE) */
            *(check_buffer + Text_Length ) = ';';
            *(check_buffer + Text_Length + 1 ) = '\0';
            if ( Cursor_Parameter == PAR_LIST_NULL )
                check_sql( check_buffer, FALSE );
            else
                check_sql( check_buffer, TRUE );
            Free_String_Space( check_buffer );
            }
        }
# endif

    /* cursor function generating routines */

    if ( Cursor->Select.type == C_INSERT && 
                                        Cursor_Parameter != PAR_LIST_NULL )
        {
        sql_warning( "parameter for insert cursor ignored" );
        Cursor_Parameter = PAR_LIST_NULL;
        }

    if ( Cursor_Parameter != PAR_LIST_NULL )
    {
    char temp_View_Name[401];

    sprintf( temp_View_Name, "%s%s", View_Name, View_Name );
        generate_Hostvariables( temp_View_Name, Cursor_Parameter );
    }

    if ( current_Language == L_COOL )
        fprintf( ESQLfile, "\nint %s_count = 0;\n", View_Name );

    generate_declare_function( View_Name, Cursor_Name, 
                                            Cursor_Parameter, Cursor );
    generate_open_function( View_Name, Cursor_Name, Cursor_Parameter );

    generate_single_cursor_function( View_Name, Cursor_Name, 
                            "Free", "" );
    generate_single_cursor_function( View_Name, Cursor_Name,
                            "Close", "" );

    switch( Cursor->Select.type )
    {
    case C_INSERT:
        generate_single_cursor_function( View_Name, Cursor_Name,
                            "Put", "row" );
        generate_single_cursor_function( View_Name, Cursor_Name,
                            "Flush", "" );
        break;
    case C_SELECT:
        generate_fetch_host_assignment( View_Name );
        generate_fetch_operations( View_Name, Cursor_Name, "next");
        generate_fetch_operations( View_Name, Cursor_Name, "previous" );
        generate_fetch_operations( View_Name, Cursor_Name, "last" );
        generate_fetch_operations( View_Name, Cursor_Name, "first" );
        break;
    case C_UPDATE:
        generate_fetch_host_assignment( View_Name );
        generate_fetch_operations( View_Name, Cursor_Name, "next");
        generate_delete_function( View_Name, Cursor_Name, Cursor );
        generate_update_function( View_Name, Cursor_Name, Cursor );
    }
}

/*---- GENERATE_INFORMIX_PROLOG -------------------------------------------*/

void generate_informix_prolog()
{
    fprintf( ESQLfile, "/* SQL-CooL-Interface for INFORMIX 4.x/5.x */\n\n");
    fprintf( ESQLfile, "$include sqlca;\n$include sqltypes;\n" );
    fprintf( ESQLfile, "$include datetime;\n" );
    fprintf( ESQLfile, "#include <string.h>\n" );

    fprintf( ESQLfile, "\ntypedef struct\n" );
    fprintf( ESQLfile, "{\nchar *value;\nint indicator;\nint length;");
    fprintf( ESQLfile, " } SQL_CHAR;\n" );

    fprintf( ESQLfile, "\ntypedef struct\n" );
    fprintf( ESQLfile, "{\nlong value;\nint indicator;");
    fprintf( ESQLfile, " } SQL_INTEGER;\n" );

    fprintf( ESQLfile, "\ntypedef struct\n" );
    fprintf( ESQLfile, "{\nlong value;\nint indicator;");
    fprintf( ESQLfile, " } SQL_SERIAL;\n" );

    fprintf( ESQLfile, "\ntypedef struct\n" );
    fprintf( ESQLfile, "{\nshort int value;\nint indicator;");
    fprintf( ESQLfile, " } SQL_SMALLINT;\n" );

    fprintf( ESQLfile, "\ntypedef struct\n" );
    fprintf( ESQLfile, "{\nfloat value;\nint indicator;");
    fprintf( ESQLfile, " } SQL_SMALLFLOAT;\n" );

    fprintf( ESQLfile, "\ntypedef struct\n" );
    fprintf( ESQLfile, "{\ndouble value;\nint indicator;");
    fprintf( ESQLfile, " } SQL_FLOAT;\n" );

    fprintf( ESQLfile, "\ntypedef struct\n" );
    fprintf( ESQLfile, "{\ndec_t value;\nint indicator;");
    fprintf( ESQLfile, " } SQL_DECIMAL;\n" );

    fprintf( ESQLfile, "\ntypedef struct\n" );
    fprintf( ESQLfile, "{\ndec_t value;\nint indicator;");
    fprintf( ESQLfile, " } SQL_MONEY;\n" );

    fprintf( ESQLfile, "\ntypedef struct\n" );
    fprintf( ESQLfile, "{\nlong value;\nint indicator;");
    fprintf( ESQLfile, " } SQL_DATE;\n" );

    fprintf( ESQLfile, "\ntypedef struct\n" );
    fprintf( ESQLfile, "{\ndtime_t value;\nint indicator;");
    fprintf( ESQLfile, "\nint from;\nint to;" );
    fprintf( ESQLfile, " } SQL_DATETIME;\n" );

    fprintf( ESQLfile, "\ntypedef struct\n" );
    fprintf( ESQLfile, "{\nintrvl_t value;\nint indicator;");
    fprintf( ESQLfile, "\nint length;\nint from;\nint to;" );
    fprintf( ESQLfile, " } SQL_INTERVAL;\n\n" );
}

