#include <stdio.h>
#include <malloc.h>
#include <string.h>

#include "page.h"
#include "vector.h"
#include "target.h"
#include "tree.h"
#include "hotword.h"
#include "topic.h"
#include "lex.h"
#include "code.h"

static vector page_list;
static vector target_page_list;
static vector global_scope;
static vector local_scope;
static vector curr_scope;
static tree topic_list;

extern FILE *hlpfile,*hdffile,*hrffile;
extern char *hlpname,*hdfname,*hrfname;
extern int warnings;
extern long line_cnt;

static void error(s)
	char *s;
	{
	fprintf(stderr,"\nSyntax error \"%s\" in file %s near line %ld\n",
		s,hlpname,line_cnt);
	exit(-1);
	}

static int detect(h1,h2)
	hotw h1,h2;
	{
	if(strcmp(hotw_name(h1),hotw_name(h2))==0)
		return(0);
	if(strstr(hotw_name(h1),hotw_name(h2))!=NULL)
		return(1);
	if(strstr(hotw_name(h2),hotw_name(h1))!=NULL)
		return(1);
	return(0);
	}

void proc_hot(to_prn)
	int to_prn;
	{
	char *name,*buff;
	hotw h;
	target newt;
	int action,pos;

	get_token();
	if(curr_tok!=STRING) error("Missing hotword");
	name=(char*)malloc(strlen(tok_value)+1);
	strcpy(name,tok_value);
	get_token();
	switch(curr_tok)
		{
		case GOTO:
			get_token();
			if(curr_tok!=STRING)
				error("Missing GOTO target");
			action=ACTGOTO;
			newt=new_target(tok_value,line_cnt);
			vector_insert(target_page_list,newt);
			break;
		case DEF:
			get_token();
			if(curr_tok!=STRING)
				error("Missing definition");
			action=ACTDEF;
			break;
		case DO:
			get_token();
			if(curr_tok!=NOTHING)
				error("Do what??");
			action=ACTNULL;
			break;
		case LINK:
			get_token();
			if(curr_tok!=STRING)
				error("Can't understand what to link");
			action=ACTLINK;
			break;
		default:
			error("Wrong or missing action");
		}
	h=new_hotw(name,action,tok_value);
	if(to_prn==1)
		{
		buff=hotw_strput(h);
		fprintf(hdffile,"%c",COMTEXT);
		fprintf(hdffile,NUMFORMAT,(long)strlen(buff));
		fprintf(hdffile,"%s",buff);
		free(buff);
		hotw_free(h);
		}
	else
		{
		pos=vector_loopover(curr_scope,detect,h);
		if(pos>=0)
			{
			warnings++;
			fprintf(stderr,
	"Warning: hotword collision between \"%s\" and \"%s\"\n",
				hotw_name(h),
				hotw_name(vector_read(curr_scope,pos)));
			}
		if(action==ACTNULL)
			vector_delete(curr_scope,h,hotw_cmp,hotw_free);
		else
			vector_replace(curr_scope,h,hotw_cmp,hotw_free);
		}
	free(name);
	get_token();
	}

void proc_print()
	{
	get_token();
	proc_hot(1);
	}

void proc_text()
	{
	get_token();
	if(curr_tok!=STRING)
		error("Wrong or missing text");
	vector_traverse(curr_scope,hotw_subst,tok_value);
	fprintf(hdffile,"%c",COMTEXT);
	fprintf(hdffile,NUMFORMAT,(long)strlen(tok_value));
	fprintf(hdffile,"%s",tok_value);
	get_token();
	}


void proc_sub()
	{
	get_token();
	if(curr_tok!=STRING)
		error("No subtitle");
	if(curr_tok!=STRING)
		error("Wrong or missing text");
	fprintf(hdffile,"%c",COMSUB);
	fprintf(hdffile,NUMFORMAT,(long)strlen(tok_value));
	fprintf(hdffile,"%s",tok_value);
	get_token();
	}

void proc_title()
	{
	get_token();
	if(curr_tok!=STRING)
		error("No title");
	if(curr_tok!=STRING)
		error("Wrong or missing text");
	fprintf(hdffile,"%c",COMTITLE);
	fprintf(hdffile,NUMFORMAT,(long)strlen(tok_value));
	fprintf(hdffile,"%s",tok_value);
	get_token();
	}


void proc_source()
	{
	get_token();
	if(curr_tok!=STRING)
		error("No source code to display");
	if(curr_tok!=STRING)
		error("Wrong or missing text");
	fprintf(hdffile,"%c",COMSOURCE);
	fprintf(hdffile,NUMFORMAT,(long)strlen(tok_value));
	fprintf(hdffile,"%s",tok_value);
	get_token();
	}

void proc_tlist(p)
	char *p;
	{
	do
		{
		get_token();
		if(curr_tok!=STRING) error("Topic expected");
		tree_insert(topic_list,new_topic(tok_value,p),topic_cmp);
		get_token();
		}
	while(curr_tok==SEP);
	}

void proc_page()
	{
	long pos,endpos;
	char *name;
	char *buffer;
	page newp;

	get_token();
	if(curr_tok!=STRING) error("Page name expected");
	pos=ftell(hdffile);
	fprintf(hdffile,NUMFORMAT,0L);
	name=(char*)malloc(strlen(tok_value)+1);
	if(name==NULL) error("Allocation error");
	strcpy(name,tok_value);

	newp=new_page(name,pos);
	if(vector_search(page_list,newp,page_cmp)!=-1)
		{
		buffer=(char*)malloc(strlen("Duplicate page ")+strlen(page_name(newp))+1);
		if(buffer==NULL) error("Allocation error");
		sprintf(buffer,"Duplicate page %s",page_name(newp));
		error(buffer);
		}
	vector_insert(page_list,newp);

	local_scope=vector_copy(global_scope,hotw_copy);
	curr_scope=local_scope;

	get_token();
	if(curr_tok==ABOUT)
		proc_tlist(name);
	free(name);
	if(curr_tok!=END)
		error("Wrong page definition");
	get_token();
	while(curr_tok!=ENDPAGE)
		{
		switch(curr_tok)
			{
			case PRINT:
				proc_print();
				break;
			case HOT:
				proc_hot(0);
				break;
			case TEXT:
				proc_text();
				break;
			case SUBTITLE:
				proc_sub();
				break;
			case SOURCE:
				proc_source();
				break;
			case TITLE:
				proc_title();
				break;
			case END:
				break;
			default:
				error("Wrong or missing internal command");
				break;
			}
		if(curr_tok!=END) error("Missing ;");
		get_token();
		}
	curr_scope=global_scope;

	vector_free(local_scope,hotw_free);

	fprintf(hdffile,"%c",COMENDPAGE);
	endpos=ftell(hdffile);
	fseek(hdffile,pos,0);
	fprintf(hdffile,NUMFORMAT,endpos-pos-NUMLEN);
	fseek(hdffile,endpos,0);
	get_token();
	}

void proc_inp()
	{
	int i;
 	page p;
	target targ;

	target_page_list=new_vector(1024);
	global_scope=new_vector(1024);
	page_list=new_vector(1024);
	topic_list=new_tree();
	curr_scope=global_scope;

	get_token();
	while(curr_tok!=END_FILE)
		{
		switch(curr_tok)
			{
			case END:
				break;
			case PAGE:
				proc_page();
				break;
			case HOT:
				proc_hot();
				break;
			default:
				error("Hot or page expected");
			}
		if(curr_tok!=END) error("Missing ;");
		get_token();
		}
	fprintf(stderr,"\nChecking targets...\n");
	for(i=0;i<vector_num(target_page_list);i++)
		{
		targ=(target)vector_read(target_page_list,i);
		p=new_page(target_name(targ),0);
		if(vector_search(page_list,p,page_cmp)<0)
			{
			fprintf(stderr,
				"Warning: missing GOTO target in line %ld\n",
				target_line(targ));
			warnings++;
			}
		page_free(p);
		}
	vector_put(page_list,hrffile,page_put);
	tree_put(topic_list,hrffile,topic_put);
	
	vector_free(global_scope,hotw_free);
	vector_free(page_list,page_free);
	tree_free(topic_list,topic_free);
	vector_free(target_page_list,target_free);
	}
