// File lat_man.c :  lattice_manager - class

#include "lattice.h"

// to display infos in the lattice_window with Button3 in a moving window
class display_cursor : public window {
public:
  char val[50];
  Window ll; // ein Window fuer eine simple vertikale Linie
  display_cursor(window &parent) : window(parent,60,20,0,0,1) {
    set_save_under();
    ll = XCreateSimpleWindow(display,parent.Win,0,0,2,10,0,0,1);
    XSelectInput(display, Win, ExposureMask);
  }
  void Move(XPoint p) { 
    width = 7*strlen(val); 
    XMoveResizeWindow(display,Win,p.x-width/2,p.y-30,width,20);     
    XMoveWindow(display,ll,p.x,p.y-8); 
    XMapWindow(display,ll); 
    redraw(); XMapWindow(display,Win);
  }
  virtual void redraw() { clear(); PlaceText(val);}
  void hide() { Unmap(); XUnmapWindow(display,ll); }
};

// *************** lattice_manager *************
// class lattice_manager : public lattice_window 
// generates a lattice_window and the buttons to manage it

void lattice_manager::set_defaults() { 
  set_angles(40/grd2rad, 55/grd2rad, 1, 0); 
  set_toggles(1,1,1,0);
  init_region();
}

void lattice_manager::draw_interior() 
{ char headline[200]; 
  WatchCursor(); // show watch cursor here
  if (body)   
    make_body(nx, ny, qptr, alpha, beta, gamma, opaque, dist,a_light,b_light);
  else 
    make_lattice(nx, ny, qptr, alpha, beta, gamma, opaque, dist);
  ResetCursor();
  sprintf(headline,"alpha = %d beta = %d gamma = %g d =%5.2f",
	  (int) (alpha*grd2rad), (int) (beta*grd2rad), gamma,dist); 
  PlaceText(headline,0,height - 10); 
  // unterer Rand,
}

// zeichnet mitbewegtes display-window im Originalbild,[ x,y,z ]
static display_cursor *dpcur = NULL;

static int ixdp, iydp;
void lattice_manager::show_infos(int px, int py) {
  // Ruecktransformation fuer einen Punkt der x,y-Ebene
  // Maus-Koordinaten (px,py) -> index-Koordinaten (ix,iy)
  float xs = x_org(px), ys = y_org(py)/cb,
        x = ca*xs + sa*ys + xp, y = ca*ys - sa*xs + yp;
  int ix = (int) (x+0.5) + ixstart, iy = (int) (y+0.5) + iystart;  
  if (dpcur == NULL) dpcur = new display_cursor(*this);
  if (ix < ixstart || ix >= ixend || iy < iystart || iy >= iyend) return; 

  if (ix != ixdp || iy != iydp) {
    ixdp = ix; iydp = iy;
    // das Feld der Gitterwerte
    XPoint (*scp)[ny] = (XPoint (*)[ny]) scptr;
    float (*ff)[ny] = (float (*) [ny]) qptr;
    sprintf(dpcur->val,"x: %d y: %d z: %f",ix,iy,ff[ix][iy]); 
    dpcur->Move(scp[ix][iy]);    
  } 
}

/* Drehung der Darstellung mit Maus :
   es kann ein beliebiger Punkt (gedacht in der Koordinaten-Ebene z=0)
   mit dem linken Button angeklickt werden (i.a. ist das der Ursprung)
   er laesst sich dann mit gedruecktem Button ziehen, wobei das neue KS
   als Frame mitwandert 
*/
static int xctr,yctr; 
static float phi, yzs, r2; // Parameter fuer Best. neuer alpha, beta
static XPoint scoord[6];
static Bool clear_old;

void lattice_manager::BPress_1_CB(XButtonEvent ev) {  
  xctr = width/2; yctr = height/2; // zentrierte Koordinaten, Anfangswerte
  int xs = ev.x - xctr, ys = ev.y - yctr; 
  float xsf = xs/float(width); // Normierung wegen Aspekt-verzerrung
  yzs = ys/cb/height; r2 = SQR(yzs) + SQR(xsf);
  phi = alpha + atan2(yzs,xsf); 
  clear_old = False;
}

void lattice_manager::BPress_3_CB(XButtonEvent ev) {
  ixdp = -1; // make sure it is new point
  show_infos(ev.x,ev.y);
}

void lattice_manager::BRelease_CB(XButtonEvent ev) 
{ if (ev.state & Button1Mask) redraw(); else
    if (ev.state & Button3Mask) dpcur->hide();
}

// zeichnet das gezogene KS 
void temp_coord(Window Win, int body, XPoint scoord[]) {
  int i;
  for (i=0; i<3; i++) XDrawLines(display,Win,gc_rubber,
				 scoord + 2*i, 2, CoordModeOrigin);
}

void lattice_manager::Motion_CB(XMotionEvent ev) { 
 if (ev.state & Button1Mask) { // Button 1 gedrueckt
   int xss = ev.x - xctr, yss = ev.y - yctr;
   if (SQR(xss) + SQR(yss) < 10) return; // zu indifferent
   float xsf = xss/float(width), ysf = yss/float(height);
   float nom = r2 - SQR(ysf) - SQR(xsf);
   if (nom < 0) return; // oder : beta = 0;
   beta = atan2(sqrt(nom), fabs(ysf)); 
   // falls ysf anstelle fabs(ysf)-> Unterseite wird anst. Rueckseite gezeigt
   alpha = phi  - atan2(ysf/cos(beta),xsf); 
   
   if (clear_old) temp_coord(Win,body,scoord); // altes KS loeschen
   clear_old = True;
   ca = cos(alpha); sa = sin(alpha); cb = cos(beta); sb = sin(beta);
     
   scoord[0] = scoord[2] = scoord[4] = screen_project(0,0,0);
   scoord[1] = screen_project(0,0,zmax); 
   scoord[3] = screen_project(xspan,0,0);
   scoord[5] = screen_project(0,yspan,0);
   temp_coord(Win,body,scoord);
 } else if (ev.state & Button3Mask) show_infos(ev.x,ev.y); 
}

// the arrow-keys turn the image for 5 grd
void lattice_manager::KeyPress_CB(XKeyEvent ev) 
{ KeySym keysym = XLookupKeysym(&ev,0);
  /*char * sym = XKeysymToString(keysym);
    printf("0x%x sym = '%s'\n", keysym,sym);
    */
  int change = TRUE;
  switch (keysym) {
  case XK_Left:  alpha -= 0.1; break;
  case XK_Right: alpha += 0.1; break;  
  case XK_Up:    beta += 0.1; break;
  case XK_Down:  beta -= 0.1; break;
  default: change = FALSE;
  }
  if (change) redraw();
}

// erzeugt button mit spezieller callback-Funktion zum Aendern eines T-Wertes
// mit dem Pointer "T *ref" un dem uebergebenen Wert "T val"
template <class T>
class modifier_button : public button { 
lattice_manager * lat; 
void (lattice_manager::*callback)(T*,T);
T *v1, v2;
public:
  modifier_button(menu_bar &parent,char * Name, 
		  void (lattice_manager::*cb)(T*,T), 
		  lattice_manager *lm, T *ref, T val) :
    button(parent,Name) { callback = cb; lat = lm; v1 = ref; v2 = val;}
  virtual void BRelease_1_action() { (lat->*callback)(v1,v2); }
};

/* nicht mehr benutzte Variante :
typedef void (lattice_manager::*LMFF)(float*,float);

class float_modifier_button : public button { 
lattice_manager * lat; 
LMFF callback;
float *v1, v2;
public:
  float_modifier_button(menu_bar &parent,char * Name, LMFF cb, 
			lattice_manager *lm, float *ref, float val) :
    button(parent,Name) { callback = cb; lat = lm; v1 = ref; v2 = val;}
  void BRelease_CB(XButtonEvent) { (lat->*callback)(v1,v2); }
};  */


static char *lattice_help[] = { 
 "Die Buttons dieser beiden Reihen dienen der Anpassung der Darstellungsform.",
 "       Einfach alles austesten, es kann nichts passieren !","",
 "a+/- : Drehung der x,y-Ebene (horizontal) um 5 Grad",
 "b+/- : Drehung der y,z-Ebene (vertikal 'kippen') um 5 Grad","", 
 "Ausserdem koennen die 4 Pfeiltasten zur Drehung benutzt werden ",
 "Und durch Anklicken eines beliebigen Punktes in der xy-Ebene mit",
 "der linken Maustaste und ziehen erscheint eingestricheltes KS,",
 "das beim Loslassen des Knopfes als neues KS dargestellt wird",
 "z+/- : Vergoesserung/Verkleinerung der Hoehe um 25%",
 "d+/- : perspektivische Verzerrung um +/- 1%","",
 "                          Toggle Buttons ",
 "rand   : Darstellung der Randlinien (Gitter)",
 "opaque : 'undurchsichtige' Darstellung (Gitter)","",
 "body : Toggle-button zwischen Gitter- und Flaechen-Darstellung",
 "flat : toggle zwischen flacher und 3d-Perspektive",
 "dump : Erzeugt X-Window dump-file 'lattice.dump' des aktuellen Bildes",
 "hardcopy: analoger dump direkt auf den Drucker","",
 "region : ein Popup zu Einstellung eines Auschnitts (zooming)",
 "palette : Popup zur Einstellung der Farbpalette (fuer Flaechendarst.)",
 "light : Popup zur Einstellung der Einfallswinkel (Flaechendarst.)",
 "clone : erzeugt eine Kopie des Bildes, das ggf. mitintegriert wird",
 0 };

lattice_manager::lattice_manager(window & parent, int w, int h, int x, int y,
				 int nxp, int nyp, float * q) : 
lattice_window(parent,w,h-40,x,y), nx(nxp), ny(nyp), qptr(q)
{ 
  selection_mask |= ButtonPressMask | ButtonReleaseMask | PointerMotionMask |
                    KeyPressMask;
  set_defaults();
  int bh = 20;
  int bwd = w/11-1; // die mittlere button-Breite auf den menu_bars
  // die menu-bars werden direkt auf dem lattice-window plaziert (unten)
  //  Ueberschneidungen & Verdeckung ist unproblematisch
  mb1 = new menu_bar(parent, w, bh, x, y+h-40, 30, 100, 0);
  mb2 = new menu_bar(parent, w, bh, x, y+h-20, 30, 100, 0);
  // es gibt nur eine globale Palette auch bei mehreren lattice_managern
  if (pal_win == NULL) 
    pal_win = new palette_popup(ncolors); // popup for palette
  light_win = new light_popup(this,240);
  region_win = new main_window("region manager",200,222);
  region_int = new region_manager(*region_win,this);
  clone_lm = NULL;
  selection_mask |= KeyPressMask; // enable Keypress events
  button * fblist[] = 
  { new modifier_button <float> (*mb1,"a+", inc_float,this,&alpha,0.1), 
    new modifier_button <float> (*mb2,"a-", inc_float,this,&alpha,-0.1),

    new modifier_button <float> (*mb1,"b+", inc_float,this,&beta,0.1),
    new modifier_button <float> (*mb2,"b-", inc_float,this,&beta,-0.1), 

    new modifier_button <float> (*mb1,"z+", mult_float, this, &gamma, 1.25), 
    new modifier_button <float> (*mb2,"z-", mult_float, this, &gamma, 0.80), 

    new modifier_button <float> (*mb1,"d+", inc_float, this, &dist, 0.01),   
    new modifier_button <float> (*mb2,"d-", inc_float, this, &dist, -0.01),  

    new toggle_redraw_button(*mb1," rand ",&rand,this),
    new toggle_redraw_button(*mb2,"opaque",&opaque,this),
    
    new toggle_redraw_button(*mb1," body ",&body,this), 
    new toggle_redraw_button(*mb2," flat ",&flat_mode,this),

    // hardcopy des inneren windows !
    new xwd_button(*mb1,"  dump  ","-out lattice.dump",this),
    new xwd_button(*mb2,"hardcopy"," | xpr | lpr",this),
 
    new popup_button (*mb1,  pal_win,"palette"),
    new popup_button (*mb2,light_win," light "),
    new instance_button <lattice_manager> (*mb1,"default",
					  &lattice_manager::default_draw,this),
    new instance_button <lattice_manager> (*mb2," clone ",
					   &lattice_manager::make_clone, this),
    new popup_button (*mb1,region_win,"region"),
    new help_button (*mb2,            " help ",lattice_help)
  };
}

lattice_manager* lattice_manager::make_popup(char * Name,float *q) {
  main_window *pp = new main_window(Name,width,height+20);
  lattice_manager* lm = new lattice_manager(*pp,width,height,0,20,nx,ny,q);
  lm->set_toggles(body, rand, opaque, flat_mode);
  lm->set_angles(alpha, beta, gamma, dist);  
  return lm;
}

void lattice_manager::delete_clone() {
  clone_lm->mainw->Unmap();
  delete clone_lm->mainw;
  clone_lm = NULL;
}  

void lattice_manager::make_clone() { 
  clone_lm = make_popup("clone",qptr);  
  main_window *pp = clone_lm->mainw;  // der Zugriff auf das popup-Window

  // der button zum Loeschen des clones ist auf dem popup des clones 
  button *del = new instance_button <lattice_manager> (*pp, "delete",
		   &lattice_manager::delete_clone, this, 100, 20,0,0);
  pp->do_popup(0,0);
}

