/*
 * File:	wx_text.cc
 * Purpose:	wxTextWindow implementation
 * Author:	Julian Smart
 * Created:	1993
 * Updated:	August 1994
 * RCS_ID:      $Id: wx_text.cc,v 1.1 1994/08/14 21:59:17 edz Exp $
 * Copyright:	(c) 1993, AIAI, University of Edinburgh
 */

/*
 * Modified: July 5th 1994 (happy birthday to me) by Andrew Davison
 * (andrewd@sfe.com.au) to attempt working edit window functionality.
 * Note: Normally all edit controls share the one 64K local heap, so
 * availability is limited and competitive. Also they are limited by
 * default to a max of 32K each. However by use of undocumented Windows
 * features we can arrange for each edit control to get it's own
 * guaranteed (almost) 64K slice of pie from global memory.
 */


static const char sccsid[] = "%W% %G%";

#include "wx.h"
#pragma hdrstop

#include "wx_privt.h"
#include "wx_clipb.h"
#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#if CTL3D
#include "ctl3d.h"
#endif

#include <sys\stat.h>
#ifdef __BORLANDC__
#include <alloc.h>
#else
#include <malloc.h>
#define farmalloc malloc
#define farfree free
#endif
#include <windowsx.h>

#if (!EDITABLE_TEXT_WINDOW)
class wxTextWnd : public wxSubWnd
{
public:
  wxStringList lines;
  int no_lines;
  int current_line;

  int cxChar;
  int cyChar;

  int cxClient;
  int cyClient;

  wxTextWnd(wxWnd *parent, wxWindow *wx_win,
             int x, int y, int width, int height, DWORD style);
  ~wxTextWnd();

  void CalcNewSize(int x, int y);

  void OnCreate(void);
  BOOL OnPaint(void);
  void OnSize(int x, int y, UINT);

  void WriteText(char *text);

};
#endif // EDITABLE_TEXT_WINDOW

wxTextWindow::wxTextWindow(void)
{
  file_name = NULL;
  globalHandle = 0;
  wxWinType = wxTYPE_HWND;
  handle = NULL;
}

wxTextWindow::wxTextWindow(wxFrame *frame, int x, int y, int width, int height,
                           long style, char *name):
  wxbTextWindow(frame, x, y, width, height, style, name)

{
  Create(frame, x, y, width, height, style, name);
}

Bool wxTextWindow::Create(wxFrame *frame, int x, int y, int width, int height,
                           long style, char *name)
{
  window_parent = frame;
  file_name = NULL;
  insertionPoint = 0;
  windowStyle = style;
  globalHandle = 0;
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
    wxWinType = wxTYPE_HWND;
  else
    wxWinType = wxTYPE_XWND;

  wxWnd *cparent = NULL;
  if (frame)
    cparent = (wxWnd *)frame->handle;

#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    // Use global memory (1K initially but can grow). Normally the instance
    // handle is passed on window creation, the sole purpose being that
    // at offset 6 is a near pointer to the local heap. If this heap
    // pointer is zero then CreateWindow() thinks it needs to create
    // a new one and does so. By allocating some (zeroed) global
    // memory and passing to CreateWindow() it gets tricked into creating
    // a new heap for each edit control, up to nearly 64K...

    globalHandle = GlobalAlloc(GHND, 1024);

    windows_id = (int)NewId();

  long msStyle = ES_MULTILINE | ES_LEFT | ES_WANTRETURN |
               WS_BORDER | WS_VISIBLE | WS_CHILD | WS_TABSTOP |
               WS_VSCROLL;

  if (windowStyle & wxREADONLY)
    msStyle |= ES_READONLY;

  if (windowStyle & wxHSCROLL)
    msStyle |= (WS_HSCROLL | ES_AUTOHSCROLL) ;

    HWND edit = CreateWindowEx(0, "EDIT", 0,
//                 ES_MULTILINE | ES_AUTOHSCROLL | ES_LEFT | ES_WANTRETURN |
//                 WS_BORDER | WS_VISIBLE | WS_CHILD | WS_TABSTOP |
//                 WS_HSCROLL | WS_VSCROLL,
                 msStyle,
                 0, 0, 0, 0, cparent->handle, (HMENU)windows_id,
                 globalHandle, NULL);

    ms_handle = edit;

    // Bypass the default 32K limit (now 64K)
    SendMessage(edit, EM_LIMITTEXT, 0, 0);

    // Set tab stops every 4 (default is 8).. units in 1/4 characters!
    WORD wTabSpacing = 4 * 4;
    SendMessage((HWND)edit, EM_SETTABSTOPS, 1, (LPARAM)&wTabSpacing);

    // Initialize
//    SendMessage(edit, WM_SETTEXT, 0, (LPARAM)"");
    SetWindowText((HWND)edit, "");

#if CTL3D
    Ctl3dSubclassCtl(edit);
#endif
    SetSize(x, y, width, height);
  }
#if !EDITABLE_TEXT_WINDOW
  else
  {
    DWORD ms_flags = WS_HSCROLL | WS_VSCROLL | WS_CHILD | WS_VISIBLE;
    if (style & wxBORDER)
      ms_flags |= WS_BORDER;
    handle = (char *)new wxTextWnd(cparent, this, x, y, width, height, ms_flags);
  }
#endif

  if (frame) frame->AddChild(this);

  return TRUE;
}

wxTextWindow::~wxTextWindow(void)
{
  if (file_name)
    delete[] file_name;
}

void wxTextWindow::SetFont(wxFont *theFont)
{
  font = theFont;
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    HWND hWnd = GetHWND();
    HDC the_dc = GetWindowDC(hWnd);
    if (font && font->GetInternalFont(the_dc))
      SendMessage(hWnd,WM_SETFONT, (WPARAM)font->GetInternalFont(the_dc),0L);
    ReleaseDC(hWnd,the_dc) ;
  }
}

Bool wxTextWindow::LoadFile(char *file)
{
  if (!file || !FileExists(file))
    return FALSE;

  if (file_name)
    delete[] file_name;

  file_name = copystring(file);

  Clear();

  ifstream input(file, ios::nocreate | ios::in);

  if (!input.bad())
  {
#if !EDITABLE_TEXT_WINDOW
    if (windowStyle & wxNATIVE_IMPL)
#else
    if (TRUE)
#endif
    {
      // Previously a SETSEL/REPLACESEL call-pair were done to insert
      // line by line into the control. Apart from being very slow this
      // was limited to 32K of text by the external interface presenting
      // positions as signed shorts. Now load in one chunk...
      // Note use of 'farmalloc' as in Borland 3.1 'size_t' is 16-bits...

      struct stat stat_buf;
      if (stat(file, &stat_buf) < 0)
        return FALSE;
//      char *tmp_buffer = (char*)farmalloc(stat_buf.st_size+1);
      // This may need to be a bigger buffer than the file size suggests,
      // if it's a UNIX file. Give it an extra 1000 just in case.
      char *tmp_buffer = (char*)farmalloc(stat_buf.st_size+1+1000);
      long no_lines = 0;
      long pos = 0;
      while (!input.eof() && input.peek() != EOF)
      {
        input.getline(wxBuffer, 500);
	int len = strlen(wxBuffer);
	wxBuffer[len] = 13;
	wxBuffer[len+1] = 10;
	wxBuffer[len+2] = 0;
	strcpy(tmp_buffer+pos, wxBuffer);
	pos += strlen(wxBuffer);
	no_lines++;
      }

//      SendMessage((HWND)ms_handle, WM_SETTEXT, 0, (LPARAM)tmp_buffer);
      SetWindowText((HWND)ms_handle, tmp_buffer);
      farfree(tmp_buffer);
    }
#if !EDITABLE_TEXT_WINDOW
    else
    {
      wxTextWnd *text_win = (wxTextWnd *)handle;
      while (!input.eof() && input.peek() != EOF)
      {
        input.getline(wxBuffer, 500);
        text_win->lines.Add(wxBuffer);
        text_win->no_lines ++;
      }
      if (text_win->no_lines > 0)
        text_win->current_line = text_win->no_lines - 1;
      else text_win->current_line = 0;

      RECT rect;
      GetClientRect(text_win->handle, &rect);
      text_win->OnSize(rect.right, rect.bottom, 0);
      InvalidateRgn(text_win->handle, NULL, TRUE);
      UpdateWindow(text_win->handle);
    }
#endif
    return TRUE;
  }
  return FALSE;
}

// If file is null, try saved file name first
// Returns TRUE if succeeds.
Bool wxTextWindow::SaveFile(char *file)
{
  if (!file)
    file = file_name;
  if (!file)
    return FALSE;
  if (file_name) delete[] file_name;
  file_name = copystring(file);

  ofstream output(file);

#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    // This will only save 64K max
    unsigned long nbytes = SendMessage(ms_handle, WM_GETTEXTLENGTH, 0, 0);
    char *tmp_buffer = (char*)farmalloc(nbytes+1);
    SendMessage(ms_handle, WM_GETTEXT, (WPARAM)(nbytes+1), (LPARAM)tmp_buffer);
    char *pstr = tmp_buffer;

    if (!output.bad())
    {
	// Convert \r\n to just \n
	while (*pstr)
	{
		if (*pstr != '\r')
			output << *pstr;
		pstr++;
	}
    }

    farfree(tmp_buffer);

    return TRUE;
  }
#if !EDITABLE_TEXT_WINDOW
  else
  {
    if (!output.bad())
    {
     wxTextWnd *text_win = (wxTextWnd *)handle;
     wxNode *node = text_win->lines.First();
     while (node)
     {
       char *s = (char *)node->Data();
       if (s)
         output << s << "\n";
       node = node->Next();
     }
     return TRUE;
    }
  }
#endif
  return FALSE;
}

void wxTextWindow::WriteText(char *text)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    // Covert \n to \r\n
    char *newtext = new char[strlen(text)*2];
    char *dst = newtext;
    while (*text)
    {
      if (*text == '\n')
        *dst++ = '\r';
      *dst++ = *text++;
    }
    *dst++ = 0;
    SendMessage((HWND)ms_handle, EM_REPLACESEL, 0, (LPARAM)newtext);
    delete [] newtext;
  }
#if !EDITABLE_TEXT_WINDOW
  else
  {
    char *the_text = copystring(text);  // Necessary in case text points to
                                        // wxBuffer

    wxTextWnd *text_win = (wxTextWnd *)handle;
    int len = strlen(the_text);

    Bool text_end = FALSE;

    int i = 0;
    while (!text_end)
    {
      int j = 0;
      Bool eol = FALSE;
      wxNode *current_node = text_win->lines.Nth(text_win->current_line);
      char *s = (char *)current_node->Data();
      int old_line_length = strlen(s);
      strcpy(wxBuffer, s);

      while (!eol && !text_end)
      {
        if (i == len)
        {
          wxBuffer[j+old_line_length] = 0;
          text_end = TRUE;
        }
        else
        {
          char ch = the_text[i];

          if (ch == '\n' || (j+old_line_length) > 490)
          {
            eol = TRUE;
            wxBuffer[j+old_line_length] = 0;
            if ((j + old_line_length) > 490)
            {
              i --; j --;
            }
          }
          else
          {
            wxBuffer[j+old_line_length] = ch;
          }
          i ++;
          j ++;
        }
      }
      delete[] s;
      current_node->SetData((wxObject *)copystring(wxBuffer));

      HDC dc = GetDC(text_win->handle);
      SetTextColor(dc, GetSysColor(COLOR_WINDOWTEXT));
      SetBkColor(dc, GetSysColor(COLOR_WINDOW));
      HFONT oldFont = ::SelectObject(dc, font->GetInternalFont(dc));

      int x = (text_win->cxChar) * (1 - text_win->xscroll_position);
      int y = (text_win->cyChar) * (text_win->current_line - text_win->yscroll_position);
      TextOut(dc, x, y, wxBuffer, strlen(wxBuffer));
      ::SelectObject(dc, oldFont);
      ReleaseDC(text_win->handle, dc);

      if (eol)
      {
        text_win->current_line ++;
        text_win->no_lines ++;
        text_win->lines.Add("");

        RECT rect;
        GetClientRect(text_win->handle, &rect);
        text_win->CalcNewSize(rect.right, rect.bottom);

        if (y >= (rect.bottom - text_win->cyChar))
          text_win->OnVScroll(SB_BOTTOM, 0, NULL);

//        (void)wxYield();
      }
    }
    delete[] the_text;
  }
#endif
}

void wxTextWindow::SetSize(int x, int y, int w, int h)
{
  int currentX, currentY;
  GetPosition(&currentX, &currentY);
  if (x == -1)
    x = currentX;
  if (y == -1)
    y = currentY;

  int currentW,currentH;
  GetSize(&currentW, &currentH);
  if (w == -1)
    w = currentW ;
  if (h == -1)
    h = currentH ;

#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    MoveWindow(ms_handle, x, y, w, h, TRUE);
  }
#if !EDITABLE_TEXT_WINDOW
  else
  {
    wxWnd *wnd = (wxWnd *)handle;
    if (wnd)
      MoveWindow(wnd->handle, x, y, w, h, TRUE);
  }
#endif
  OnSize(w, h);
}

void wxTextWindow::Clear(void)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
//    SendMessage((HWND)ms_handle, WM_SETTEXT, 0, (LPARAM)"");
    SetWindowText((HWND)ms_handle, "");
  }
#if !EDITABLE_TEXT_WINDOW
  else
  {
    wxTextWnd *text_win = (wxTextWnd *)handle;
    wxNode *node = text_win->lines.First();
    while (node)
    {
      char *s = (char *)node->Data();
      delete[] s;
      delete node;
      node = text_win->lines.First();
    }
    text_win->lines.Add("");
    text_win->no_lines = 1;
    text_win->current_line = 0;

    RECT rect;
    GetClientRect(text_win->handle, &rect);
    text_win->OnSize(rect.right, rect.bottom, 0);
    InvalidateRgn(text_win->handle, NULL, TRUE);
  }
#endif
}

Bool wxTextWindow::Modified(void)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
    return (Bool)SendMessage((HWND)ms_handle, EM_GETMODIFY, 0, 0);
#if !EDITABLE_TEXT_WINDOW
  else
    return FALSE;
#endif
}

// Not clear whether Clear is required as well as DiscardEdits
void wxTextWindow::DiscardEdits(void)
{
  Clear();
}

char *wxTextWindow::GetContents(void)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    return 0;
  }
#if !EDITABLE_TEXT_WINDOW
  else
  {
    wxTextWnd *text_win = (wxTextWnd *)handle;
    // Count size of buffer required
    int i = 0;
    wxNode *node = text_win->lines.First();
    while (node)
    {
      char *s = (char *)node->Data();
      i += strlen(s) + 1; // Add one for a newline
      node = node->Next();
    }
    char *buf = new char[i+1];
    i = 0;
    node = text_win->lines.First();
    while (node)
    {
      char *s = (char *)node->Data();
      int len = strlen(s);
      for (int j = 0; j < len; j++)
      {
        buf[i] = s[j];
        i ++;
      }
      buf[i] = '\n';
      i ++;
    
      node = node->Next();
    }
    buf[i] = 0;
  
    return buf;
  }
#endif
}

/*
 * Some of the following functions are yet to be implemented
 *
 */
 
int wxTextWindow::GetNumberOfLines(void)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
    return (int)SendMessage((HWND)ms_handle, EM_GETLINECOUNT, (WPARAM)0, (LPARAM)0);
#if !EDITABLE_TEXT_WINDOW
  else
  {
    wxTextWnd *text_win = (wxTextWnd *)handle;
    return text_win->lines.Number();
  }
#endif
}

void wxTextWindow::SetInsertionPoint(long pos)
{
  insertionPoint = pos;
}

void wxTextWindow::SetInsertionPointEnd(void)
{
}

long wxTextWindow::GetInsertionPoint(void)
{
  return insertionPoint;
}

long wxTextWindow::GetLastPosition(void)
{
  return 0;
}

long wxTextWindow::XYToPosition(long x, long y)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    HWND hWnd = GetHWND();
    
    // This gets the char index for the _beginning_ of this line
    int charIndex = (int)SendMessage(hWnd, EM_LINEINDEX, (WPARAM)y, (LPARAM)0);
    return (long)(x + charIndex);
  }
  return 0;
}

void wxTextWindow::PositionToXY(long pos, long *x, long *y)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    HWND hWnd = GetHWND();

    // This gets the line number containing the character
    int lineNo = (int)SendMessage(hWnd, EM_LINEFROMCHAR, (WPARAM)pos, (LPARAM)0);
    // This gets the char index for the _beginning_ of this line
    int charIndex = (int)SendMessage(hWnd, EM_LINEINDEX, (WPARAM)lineNo, (LPARAM)0);
    // The X position must therefore be the different between pos and charIndex
    *x = (long)(pos - charIndex);
    *y = (long)lineNo;
  }
}

void wxTextWindow::ShowPosition(long pos)
{
}

int wxTextWindow::GetLineLength(long lineNo)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    long charIndex = XYToPosition(0, lineNo);
    HWND hWnd = GetHWND();
    int len = (int)SendMessage(hWnd, EM_LINELENGTH, (WPARAM)charIndex, (LPARAM)0);
    return len;
  }
#if !EDITABLE_TEXT_WINDOW
  else
  {
    wxTextWnd *text_win = (wxTextWnd *)handle;
    wxNode *node = text_win->lines.Nth((int)lineNo);
    if (node)
    {
      char *s = (char *)node->Data();
      return strlen(s);
    }
  }
#endif
  return -1;
}

int wxTextWindow::GetLineText(long lineNo, char *buf)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    HWND hWnd = GetHWND();
    *(WORD *)buf = 128;
    int noChars = (int)SendMessage(hWnd, EM_GETLINE, (WPARAM)lineNo, (LPARAM)buf);
    buf[noChars] = 0;
    return noChars;
  }
#if !EDITABLE_TEXT_WINDOW
  else
  {
    wxTextWnd *text_win = (wxTextWnd *)handle;
    wxNode *node = text_win->lines.Nth((int)lineNo);
    if (node)
    {
      char *s = (char *)node->Data();
      strcpy(buf, s);
      return strlen(s);
    }
  }
#endif
  buf[0] = 0;
  return -1;
}

void wxTextWindow::Replace(long from, long to, char *value)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
#if USE_CLIPBOARD
    HWND hWnd = GetHWND();
    long fromChar = from;
    long toChar = to;
    
    // Set selection and remove it
    SendMessage(hWnd, EM_SETSEL, (WPARAM)0, (LPARAM)MAKELONG(fromChar, toChar));
    SendMessage(hWnd, WM_CUT, (WPARAM)0, (LPARAM)0);

    if (value)
    {
      // Now replace with 'value', by pasting.
      wxSetClipboardData(wxCF_TEXT, (wxObject *)value, 0, 0);

      // Paste into edit control
      SendMessage(hWnd, WM_PASTE, (WPARAM)0, (LPARAM)0L);
    }
#endif
  }
}

void wxTextWindow::Remove(long from, long to)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    HWND hWnd = GetHWND();
    long fromChar = -1;
    long toChar = -1;
    
    // Cut all selected text
    SendMessage(hWnd, EM_SETSEL, (WPARAM)0, (LPARAM)MAKELONG(fromChar, toChar));
  }
}

void wxTextWindow::SetSelection(long from, long to)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    HWND hWnd = GetHWND();
    long fromChar = from;
    long toChar = to;
    // if from and to are both -1, it means
    // (in wxWindows) that all text should be selected.
    // This translates into Windows convention
    if ((from == -1) && (to == -1))
    {
      fromChar = 0;
      toChar = -1;
    }
    
    // WPARAM is 0: selection is scrolled into view
    SendMessage(hWnd, EM_SETSEL, (WPARAM)0, (LPARAM)MAKELONG(fromChar, toChar));
  }
}

void wxTextWindow::SetEditable(Bool editable)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    HWND hWnd = GetHWND();
    SendMessage(hWnd, EM_SETREADONLY, (WPARAM)!editable, (LPARAM)0L);
  }
}

// Copy selection to clipboard
void wxTextWindow::Copy(void)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    HWND hWnd = GetHWND();
    SendMessage(hWnd, WM_COPY, 0, 0L);
  }
}

// Paste clipboard into text window
void wxTextWindow::Paste(void)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    HWND hWnd = GetHWND();
    SendMessage(hWnd, WM_PASTE, 0, 0L);
  }
}

// Copy selection to clipboard, then remove selection.
void wxTextWindow::Cut(void)
{
#if !EDITABLE_TEXT_WINDOW
  if (windowStyle & wxNATIVE_IMPL)
#else
  if (TRUE)
#endif
  {
    HWND hWnd = GetHWND();
    SendMessage(hWnd, WM_CUT, 0, 0L);
  }
}

#if (!EDITABLE_TEXT_WINDOW)
wxTextWnd::wxTextWnd(wxWnd *parent, wxWindow *wx_win,
                       int x, int y, int width, int height, DWORD style):
  wxSubWnd(parent, "wxCanvasClass", wx_win, x, y, width, height, style)
{
  no_lines = 1;
  current_line = 0;
  lines.Add("");

  OnCreate();
  ShowScrollBar(handle, SB_BOTH, TRUE);
}

wxTextWnd::~wxTextWnd()
{
}

BOOL wxTextWnd::OnPaint()
{
  RECT rect;
  if (GetUpdateRect(handle, &rect, FALSE))
  {
    PAINTSTRUCT ps;
    // Hold a pointer to the dc so long as the OnPaint() message
    // is being processed
    HDC dc = BeginPaint(handle, &ps);

    COLORREF bkgnd_color = GetSysColor(COLOR_WINDOW);
    ::SetTextColor(dc, GetSysColor(COLOR_WINDOWTEXT));
    ::SetBkColor(dc, bkgnd_color);
    HFONT oldFont = ::SelectObject(dc, wx_window->font->GetInternalFont(dc));

    HBRUSH brush = CreateSolidBrush(bkgnd_color);
    ::FillRect(dc, &rect, brush);
    DeleteObject(brush);

    int nStart = yscroll_position;
    int nEnd = min(no_lines, yscroll_position + (rect.bottom/cyChar) + 1);

    int i;
    int x,y;
    wxNode *node = lines.Nth(nStart);
    i = nStart;
    while (node && (i < nEnd))
    {
      char *s = (char *)node->Data();
      x = cxChar * (1 - xscroll_position);
      y = cyChar * (i - yscroll_position);
      TextOut(dc, x, y, s, strlen(s));
      i ++;
      node = node->Next();
    }
    ::SelectObject(dc, oldFont);
    EndPaint(handle, &ps);
    return 0;
  }
  return 1;
}

void wxTextWnd::OnCreate(void)
{
  TEXTMETRIC tm;
  HDC dc = GetDC(handle);
  GetTextMetrics(dc, &tm);
  ReleaseDC(handle, dc);
  cxChar = tm.tmAveCharWidth;
  cyChar = tm.tmHeight + tm.tmExternalLeading;
  yscroll_pixels_per_line = cyChar;
  xscroll_pixels_per_line = cxChar;
  xscroll_lines = 300;
  yscroll_lines = 0;
  ReleaseDC(handle, dc);
}

void wxTextWnd::OnSize(int x, int y, UINT)
{
  CalcNewSize(x, y);
  InvalidateRgn(handle, NULL, TRUE);
}

void wxTextWnd::CalcNewSize(int x, int y)
{
  cxClient = x;
  cyClient = y;

  int nMaxWidth = xscroll_lines*xscroll_pixels_per_line;

  int nVscrollMax = max(0, (int)(no_lines + 2 - cyClient/cyChar));
  yscroll_position = min(yscroll_position, nVscrollMax);

  SetScrollRange(handle, SB_VERT, 0, nVscrollMax, FALSE);
  SetScrollPos(handle, SB_VERT, yscroll_position, TRUE);

  int nHscrollMax = max(0, (int)(2 + nMaxWidth - cxClient/cxChar));
  xscroll_position = min(xscroll_position, nHscrollMax);

  SetScrollRange(handle, SB_HORZ, 0, nHscrollMax, FALSE);
  SetScrollPos(handle, SB_HORZ, xscroll_position, TRUE);

  yscroll_lines = no_lines;

  yscroll_lines_per_page = max(1, cyClient/cyChar);
  xscroll_lines_per_page = 10;
}

#endif // EDITABLE_TEXT_WINDOW
