/*
 * File:      wx_utils.cc
 * Purpose:     Various utilities (X version)
 * Author:      Julian Smart
 * Created:     1993
 * Updated:	August 1994
 * RCS_ID:      $Id: wx_utils.cc,v 1.7 1994/08/15 21:53:06 edz Exp edz $
 * Copyright:   (c) 1993, AIAI, University of Edinburgh
 */

static const char sccsid[] = "%W% %G%";

#ifdef __GNUG__
#pragma implementation
#endif

#include <iostream.h>
#include <fstream.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "common.h"

#include "wx_setup.h"
#include "wx_utils.h"
#include "wx_main.h"
#include "wx_dialg.h"

// Sun CC compatibility (interference with xview/pkg.h, apparently...)
#if defined(SUN_CC) && defined(wx_xview)
#undef va_start
#undef va_end
#undef va_arg
#undef va_list
#endif

#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef wx_motif
#if !(defined(linux) || defined(__sgi) || defined(__hpux) || defined(__ultrix)) || defined(__bsdi__)
#include <tiuser.h>
#endif
#endif

#include <X11/Xlib.h>
#include <X11/Xresource.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include <sys/types.h>
#include <sys/time.h>
#include <sys/syscall.h>
#include <sys/file.h>
#include <sys/wait.h>
#include <signal.h>
#include <netdb.h>
#include <dirent.h>
#include <unistd.h>
#include <pwd.h>

#ifdef wx_xview
#include <xview/canvas.h>
#endif

#if USE_RESOURCES
#ifdef wx_xview
#define use_xview_code
#include <xview/defaults.h>
#endif
#endif

#ifdef sun
# ifndef __GNUG__
#  ifndef SUN_CC
#   define SUN_CC 1
#  endif
# endif
#endif

// Yuck this is really BOTH site and platform dependent
// so we should use some other strategy!
#ifdef sun
# define DEFAULT_XRESOURCE_DIR "/usr/openwin/lib/app-defaults"
#else
# define DEFAULT_XRESOURCE_DIR "/usr/lib/X11/app-defaults"
#endif

#if !(defined(SUN_CC) || defined(__CLCC__) || defined(__hpux) || defined(__osf__) || defined(SVR4))
extern "C"
{
  int select (int, fd_set *, fd_set *, fd_set *, struct timeval *);
  int sigsetmask (int);
  int sigblock (int);
#ifndef __GNUG__
  int gethostname (char *host, unsigned const namelen);
#endif
}
#endif

#ifdef __osf__  
extern "C" int gethostname(char *host , int len);
extern "C" struct passwd *getpwuid(uid_t uid);
extern "C" uid_t getuid(void);
#endif

#ifdef SVR4
#include <sys/systeminfo.h>
#endif

#if (defined(SUN_CC) || defined(__CLCC__))
#include <sysent.h>
#endif

#ifdef wx_xview
#include <xview/screen.h>
#include <xview/server.h>
#include <xview/notify.h>
extern Xv_Server xview_server;
#endif

#ifndef use_xview_code
static XrmDatabase wxResourceDatabase = 0;
#endif

#if USE_RESOURCES
void wxXMergeDatabases (wxApp * theApp, Display * display);
#endif

static char *GetIniFile (char *dest, const char *filename);

// Get fully qualified hostname e.g. foo.bar.edu
Bool 
wxGetHostName (char *buf, int maxSize)
{
#ifdef SVR4
  return (sysinfo (SI_HOSTNAME, buf, maxSize) != -1);
#else /* BSD Sockets */
  char name[255];
  struct hostent *h;

  // Get hostname
  if (gethostname (name, sizeof (name) / sizeof (char) - 1) == -1)
      return FALSE;
  // Get official full name of host
  strncpy (buf
	   ,(h = gethostbyname (name)) != NULL ? h->h_name : name
	   ,maxSize - 1);
  return TRUE;
#endif
}

// Get user ID e.g. jacs
Bool 
wxGetUserId (char *buf, int maxSize)
{
  struct passwd *who;

  if ((who = getpwuid (getuid ())) != NULL)
    {
      strncpy (buf, who->pw_name, maxSize - 1);
      return TRUE;
    }
  return FALSE;
}

// Get user name e.g. Julian Smart
Bool 
wxGetUserName (char *buf, int maxSize)
{
  struct passwd *who;

  if ((who = getpwuid (getuid ())) != NULL)
    {
      strncpy (buf, who->pw_gecos, maxSize - 1);
      return TRUE;
    }
  return FALSE;
}

// Execute a command (e.g. another program) in a
// system-independent manner.

// Planned:
// wxExecute(const char *command, wxFunction *func)
//

Bool 
wxExecute (char **argv, Bool Async)
{
  if (*argv == NULL)
    return FALSE;	// Nothing???
  // Run a program the recomended way under X (XView) 

  /* fork the process */
#if defined(sun) || defined(__ultrix) || defined(__bsdi__)
  pid_t pid = vfork ();
#else
  pid_t pid = fork ();
#endif
  if (pid == -1)
    {
      perror ("fork failed");
      return FALSE;
    }
  else if (pid == 0)
    {
      /* child */
      execvp (*argv, argv);
      if (errno == ENOENT)
	printf ("%s: command not found\n", *argv);
      else
	perror (*argv);
      printf ("wxWindows: could not execute '%s'\n", *argv);
      _exit (-1);
    }

#ifdef wx_xview
  static Notify_client sys_client = 42;
  notify_set_wait3_func (sys_client, (Notify_func) notify_default_wait3, pid);
#endif
  // Code below is NOT really acceptable!
  // One should NEVER use wait under X
  // Ideas? A Sleep idle callback?

  // WARNING: WARNING: WARNING: WARNING:
  // The CODE BELOW IS BAD BAD BAD BAD!
  if (Async)
    {
      int status;
      wxSleep (2);		// Give a little time
#if !defined(SVR4) && !defined(sun) && !defined(__hpux)
      while (wait ((union wait *) &status) != pid)
#else
      while (wait (&status) != pid)
#endif
	wxSleep(3);	// 3 sec?
    }

  return TRUE;
}


Bool 
wxExecute (const char *command, Bool Async)
{
  if (command == NULL || *command == '\0')
    return FALSE; // Nothing to do

  // Run a program the recomended way under X (XView) 
  int argc = 0;
  char *argv[127];
  char tmp[1024];
  const char *IFS = " \t\n";

  // Build argument vector 
  strncpy (tmp, command, sizeof (tmp) / sizeof (char) - 1);
  tmp[sizeof (tmp) / sizeof (char) - 1] = '\0';
  argv[argc++] = strtok (tmp, IFS);
  while ((argv[argc++] = strtok (NULL, IFS)) != NULL)
    /* loop */ ;

  return wxExecute(argv, Async);
}

//
// Execute a program in an Interactive Shell
//
Bool
wxShell(const char *command)
{
#if defined(sun) || defined(__ultrix) || defined(__bsdi__)
  pid_t pid = vfork ();
#else
  pid_t pid = fork ();
#endif
  switch( pid ) {
    case -1:			/* error */
	return(FALSE);
    case 0:			/* child */
#ifdef wx_xview
	execlp("shelltool", "-c", (char *) command, NULL);
#else
	// Generic X windows terminal window
	if (command && *command)
	  execlp("xterm", "-e", (char *) command, NULL);
	else
	  execlp("xterm", NULL);
#endif
	_exit(127);
  }
#ifdef wx_xview
  static Notify_client sys_client = 42;
  notify_set_wait3_func (sys_client, (Notify_func) notify_default_wait3, pid);
#endif
  return TRUE;
}

// Get a temporary filename, opening and closing the file.
char *wxGetTempFileName (const char *prefix, char *dest)
{
  static short last_temp = 0;	// cache last to speed things a bit
  // At most 1000 temp files to a process! We use a ring count.
  char buf[64];

  for (short suffix = last_temp + 1; suffix != last_temp; ++suffix %= 1000)
    {
      sprintf (buf, "/tmp/%s%d.%03x", prefix, (int) getpid (), (int) suffix);
      if (!FileExists (buf))
	{
	  // Touch the file to create it (reserve name)
	  FILE *fd = fopen (buf, "w");
	  if (fd)
	    fclose (fd);
	  last_temp = suffix;
          if (dest)
	    strcpy(dest, buf);
	  else
	    dest = copystring(buf);
	  return dest;
	}
    }
  cerr << "wxWindows: error finding temporary file name.\n";
  if (dest) dest[0] = 0;
  return NULL;
}

Bool 
wxRemoveFile (const char *file)
{
  return ((unlink (file) == 0) ? TRUE : FALSE);
}

Bool 
wxMkdir (const char *dir)
{
  // give default perms of owner read and write, group read and
  // others read. The interface to this func should be changed
  // to pass the perms info in.
  // Since directory it must also be searchable @@@
  // Added S_IXUSR | S_IXGRP | S_IXOTH
  return (mkdir (dir, S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH) == 0);
}

Bool 
wxRmdir (const char *dir)
{
  return (rmdir (dir) == 0);
}

Bool 
DirExists (const char *dir)
{
  struct stat sbuf;
  return (stat(dir, &sbuf) != -1) && S_ISDIR(sbuf.st_mode) ? TRUE : FALSE;
}


// Get first file name matching given wild card.
// Flags are reserved for future use.

static DIR *wxDirStream = NULL;
static char *wxFileSpec = NULL;

char *wxFindFirstFile(const char *spec, int flags)
{
  if (wxFileSpec)
    delete[] wxFileSpec;
  wxFileSpec = copystring(spec);

  static char buf[400];

  // Find path only so we can concatenate
  // found file onto path
  char *p = PathOnly(wxFileSpec);
  char *n = FileNameFromPath(wxFileSpec);

  if ((wxDirStream=opendir(p))==NULL)
    return NULL;

  // Do the reading
  struct dirent *nextDir;
  for (nextDir = readdir(wxDirStream); nextDir != NULL; nextDir = readdir(wxDirStream))
  {
    if ((strcmp(nextDir->d_name, ".") == 0) ||
        (strcmp(nextDir->d_name, "..") == 0))
    {
      // Skip
    }
    else if (!wxIsWild(n) || wxMatchWild(n, nextDir->d_name))
    {
      buf[0] = 0;
      if (p && (strlen(p) > 0))
      {
        strcpy(buf, p);
        strcat(buf, "/");
      }
      strcat(buf, nextDir->d_name);
      return buf;
    }
  }
  closedir(wxDirStream);
  return NULL;
}

char *wxFindNextFile(void)
{
  static char buf[400];

  // Find path only so we can concatenate
  // found file onto path
  char *p = PathOnly(wxFileSpec);
  char *n = FileNameFromPath(wxFileSpec);

  // Do the reading
  struct dirent *nextDir;
  for (nextDir = readdir(wxDirStream); nextDir != NULL; nextDir = readdir(wxDirStream))
  {
    if ((strcmp(nextDir->d_name, ".") == 0) ||
        (strcmp(nextDir->d_name, "..") == 0))
    {
      // Skip
    }
    else if (!wxIsWild(n) || wxMatchWild(n, nextDir->d_name))
    {
      buf[0] = 0;
      if (p && (strlen(p) > 0))
      {
        strcpy(buf, p);
        strcat(buf, "/");
      }
      strcat(buf, nextDir->d_name);
      return buf;
    }
  }
  closedir(wxDirStream);
  return NULL;
}

// Get current working directory.
// If buf is NULL, allocates space using new, else
// copies into buf.
char *wxGetWorkingDirectory(char *buf, int sz)
{
  if (!buf)
    buf = new char[1000];
  (void)getcwd(buf, sz);
  return buf;
}

// Get free memory in bytes, or -1 if cannot determine amount (e.g. on UNIX)
long 
wxGetFreeMemory (void)
{
  return -1;
}

// Sleep for nSecs seconds under UNIX, do nothing under Windows.
// XView implementation according to the Heller manual
void 
wxSleep (int nSecs)
{
#ifdef __sgi
  sleep (nSecs);
#else
#if defined(SVR4)
  struct sigset_t oldmask, mask;
  struct timeval tv;

  tv.tv_sec = nSecs;
  tv.tv_usec = 0;

  sigemptyset (&mask);
  sigaddset (&mask, SIGIO);
  sigaddset (&mask, SIGALRM);
  sigprocmask (SIG_BLOCK, &mask, &oldmask);
  if ((select (0, 0, 0, 0, &tv)) == -1)
    {
      perror ("select in wxSleep");
    }
//  sigprocmask(SIG_BLOCK, &oldmask, (sigset_t *) NULL); // Bug according to Kari
  sigprocmask (SIG_SETMASK, &oldmask, (sigset_t *) NULL);
#else
  int oldmask, mask;
  struct timeval tv;

  tv.tv_sec = nSecs;
  tv.tv_usec = 0;

  mask = sigmask (SIGIO);
  mask |= sigmask (SIGALRM);
  oldmask = sigblock (mask);
  if ((select (0, 0, 0, 0, &tv)) == -1)
    {
      perror ("select in wxSleep");
    }
  sigsetmask (oldmask);
#endif
#endif // __sgi
}

// Consume all events until no more left
#ifdef wx_xview
extern "C" int xv_input_pending (Display *, int);
extern "C" int ndis_dispatch (void);
#endif

void 
wxFlushEvents (void)
{
#ifdef wx_motif
  XSync (XtDisplay (wxTheApp->topLevel), FALSE);
  XEvent event;
  while (XtAppPending (wxTheApp->appContext))
    {
      XFlush (XtDisplay (wxTheApp->topLevel));
      XtAppNextEvent (wxTheApp->appContext, &event);
      XtDispatchEvent (&event);
    }
#endif
#ifdef wx_xview

  Xv_Screen screen = xv_get (xview_server, SERVER_NTH_SCREEN, 0);
  Xv_opaque root_window = xv_get (screen, XV_ROOT);
  Display *display = (Display *) xv_get (root_window, XV_DISPLAY);

  XSync (display, FALSE);
  XFlush (display);
/* Causes nasty internal problems. Pity, I thought I'd cracked it...
   while(XPending(display))
   {
   XEvent event;
   XPeekEvent(display, &event);
   xv_input_pending(display, 0);
   ndis_dispatch();
   }
 */
#endif
}

// Output a debug mess., in a system dependent fashion.

void 
wxDebugMsg (const char *fmt...)
{
#ifndef __sgi
  va_list ap;
  char buffer[BUFSIZ];

  if (!wxTheApp->wantDebugOutput)
    return ;

  va_start (ap, fmt);

  vsprintf (buffer, fmt, ap);
  cerr << buffer;

  va_end (ap);
#else
  cerr << "Error: cannot use variable-argument functions on SGI!\n";
#endif
}

// Non-fatal error: write error and continue
void 
wxError (const char *msg, const char *title)
{
  cerr << title << ": " << msg << "\n";
}

// Fatal error: pop up message box and abort
void 
wxFatalError (const char *msg, const char *title)
{
  cerr << title << ": " << msg << "\n";
  exit (1);
}

// Emit a beeeep...

void 
wxBell ()
{
#ifdef wx_motif
  Display *display = XtDisplay (wxTheApp->topLevel);
#endif
#ifdef wx_xview
  Xv_Screen screen = xv_get (xview_server, SERVER_NTH_SCREEN, 0);
  Xv_opaque root_window = xv_get (screen, XV_ROOT);
  Display *display = (Display *) xv_get (root_window, XV_DISPLAY);
#endif

  // Use current setting for the bell
  XBell (display, 0);

}

int 
wxGetOsVersion (int *majorVsn, int *minorVsn)
{
#ifdef wx_motif
  Display *display = XtDisplay (wxTheApp->topLevel);
#endif
#ifdef wx_xview
  Xv_Screen screen = xv_get (xview_server, SERVER_NTH_SCREEN, 0);
  Xv_opaque root_window = xv_get (screen, XV_ROOT);
  Display *display = (Display *) xv_get (root_window, XV_DISPLAY);
#endif
  if (majorVsn)
    *majorVsn = ProtocolVersion (display);
  if (minorVsn)
    *minorVsn = ProtocolRevision (display);
#ifdef wx_xview
  return wxXVIEW_X;
#endif
#ifdef wx_motif
  return wxMOTIF_X;
#endif
}

// Reading and writing resources (eg WIN.INI, .Xdefaults)

#if USE_RESOURCES

/*
 * We have a cache for writing different resource files,
 * which will only get flushed when we call wxFlushResources().
 * Build up a list of resource databases waiting to be written.
 *
 */

wxList wxResourceCache (wxKEY_STRING);

void 
wxFlushResources (void)
{
  char nameBuffer[512];

  wxNode *node = wxResourceCache.First ();
  while (node)
    {
      char *file = node->key.string;
      // If file doesn't exit, create it first.
      if (!FileExists (file))
	{
	  if (file[0] == '/')
	    strcpy (nameBuffer, file);
	  else
	    {
	      // Put in standard place for resource files if not absolute
#ifdef wx_xview
	      // OpenWindows home dir (don't know about relative Motif)
              char *openwin = getenv("OPENWINHOME");
	      if (openwin) {
		strcpy(nameBuffer, openwin);
		strcat(nameBuffer, "/lib/app-defaults");
	      } else
#endif
	      strcpy (nameBuffer, DEFAULT_XRESOURCE_DIR);
	      strcat (nameBuffer, "/");
	      strcat (nameBuffer, FileNameFromPath (file));
	    }
	  if (!FileExists (nameBuffer))
	    {
	      // Touch the file to create it
	      FILE *fd = fopen (nameBuffer, "w");
	      if (fd)
		fclose (fd);
	    }
	}
      else
	strcpy (nameBuffer, file);

      XrmDatabase database = (XrmDatabase) node->Data ();
      XrmPutFileDatabase (database, nameBuffer);
      XrmDestroyDatabase (database);
      wxNode *next = node->Next ();
      delete node;
      node = next;
    }
}

Bool 
wxWriteResource (const char *section, const char *entry, char *value, const char *file)
{
  char buffer[500];

  (void) GetIniFile (buffer, file);

  XrmDatabase database;
  wxNode *node = wxResourceCache.Find (buffer);
  if (node)
    database = (XrmDatabase) node->Data ();
  else
    {
      database = XrmGetFileDatabase (buffer);
      wxResourceCache.Append (buffer, (wxObject *) database);
    }

  char resName[300];
  strcpy (resName, section);
  strcat (resName, ".");
  strcat (resName, entry);

  XrmPutStringResource (&database, resName, value);
  return TRUE;
}

Bool 
wxWriteResource (const char *section, const char *entry, float value, const char *file)
{
  char buf[50];
  sprintf (buf, "%.4f", value);
  return wxWriteResource (section, entry, buf, file);
}

Bool 
wxWriteResource (const char *section, const char *entry, long value, const char *file)
{
  char buf[50];
  sprintf (buf, "%ld", value);
  return wxWriteResource (section, entry, buf, file);
}

Bool 
wxWriteResource (const char *section, const char *entry, int value, const char *file)
{
  char buf[50];
  sprintf (buf, "%d", value);
  return wxWriteResource (section, entry, buf, file);
}

Bool 
wxGetResource (const char *section, const char *entry, char **value, const char *file)
{
#ifdef use_xview_code		/* @@@ */
  // New code using Xview
  char buf[1024];
  char *result;

  defaults_load_db (GetIniFile (buf, file));

  strcpy (buf, section);
  strcat (buf, ".");
  strcat (buf, entry);
  result = (char *) defaults_get_string (buf, buf, "$$default");
  if (strcmp (result, "$$default") == 0)
    return FALSE;
  *value = copystring (result);
  return TRUE;
#else // Old code for XView and code for Motif
  if (!wxResourceDatabase)
    {
#ifdef wx_motif
      Display *display = XtDisplay (wxTheApp->topLevel);
#endif
#ifdef wx_xview
      Xv_Screen screen = xv_get (xview_server, SERVER_NTH_SCREEN, 0);
      Xv_opaque root_window = xv_get (screen, XV_ROOT);
      Display *display = (Display *) xv_get (root_window, XV_DISPLAY);
#endif
      wxXMergeDatabases (wxTheApp, display);
    }

  XrmDatabase database;

  if (file)
    {
      wxNode *node = wxResourceCache.Find (file);
      if (node)
	database = (XrmDatabase) node->Data ();
      else
	{
	  database = XrmGetFileDatabase (file);
	  wxResourceCache.Append (file, (wxObject *) database);
	}
    }
  else
    database = wxResourceDatabase;

  XrmValue xvalue;
  char *str_type[20];
  char buf[150];
  strcpy (buf, section);
  strcat (buf, ".");
  strcat (buf, entry);

  Bool success = XrmGetResource (database, buf, "*", str_type,
				 &xvalue);
  // Try different combinations of upper/lower case, just in case...
  if (!success)
    {
      buf[0] = (isupper (buf[0]) ? tolower (buf[0]) : toupper (buf[0]));
      success = XrmGetResource (database, buf, "*", str_type,
				&xvalue);
    }
  if (success)
    {
      if (*value)
        delete[] *value;

      *value = new char[xvalue.size + 1];
      strncpy (*value, xvalue.addr, (int) xvalue.size);
      return TRUE;
    }
  return FALSE;
#endif // use_xview_code
}


Bool 
wxGetResource (const char *section, const char *entry, float *value, const char *file)
{
  char *s = NULL;
  Bool succ = wxGetResource (section, entry, &s, file);
  if (succ)
    {
      *value = (float) strtod (s, NULL);
      delete[]s;
      return TRUE;
    }
  else
    return FALSE;
}

Bool 
wxGetResource (const char *section, const char *entry, long *value, const char *file)
{
  char *s = NULL;
  Bool succ = wxGetResource (section, entry, &s, file);
  if (succ)
    {
      *value = strtol (s, NULL, 10);
      delete[]s;
      return TRUE;
    }
  else
    return FALSE;
}

Bool 
wxGetResource (const char *section, const char *entry, int *value, const char *file)
{
  char *s = NULL;
  Bool succ = wxGetResource (section, entry, &s, file);
  if (succ)
    {
      // Handle True, False here 
      // True, Yes, Enables, Set or  Activated 
      if (*s == 'T' || *s == 'Y' || *s == 'E' || *s == 'S' || *s == 'A')
	*value = TRUE;
      // False, No, Disabled, Reset, Cleared, Deactivated
      else if (*s == 'F' || *s == 'N' || *s == 'D' || *s == 'R' || *s == 'C')
	*value = FALSE;
      // Handle as Integer
      else
	*value = (int) strtol (s, NULL, 10);

      delete[]s;
      return TRUE;
    }
  else
    return FALSE;
}

#ifdef XXXX
#ifdef wx_motif
/*
 * Not yet used but may be useful.
 *
 */
void 
wxSetDefaultResources (const Widget w, const char **resourceSpec, const char *name)
{
  int i;
  Display *dpy = XtDisplay (w);	// Retrieve the display pointer

  XrmDatabase rdb = NULL;	// A resource data base

  // Create an empty resource database
  rdb = XrmGetStringDatabase ("");

  // Add the Component resources, prepending the name of the component

  i = 0;
  while (resourceSpec[i] != NULL)
    {
      char buf[1000];

      sprintf (buf, "*%s%s", name, resourceSpec[i++]);
      XrmPutLineResource (&rdb, buf);
    }

  // Merge them into the Xt database, with lowest precendence

  if (rdb)
    {
#if (XlibSpecificationRelease>=5)
      XrmDatabase db = XtDatabase (dpy);
      XrmCombineDatabase (rdb, &db, FALSE);
#else
      XrmMergeDatabases (dpy->db, &rdb);
      dpy->db = rdb;
#endif
    }
}
#endif
#endif // 0


#ifndef use_xview_code
/*
 * Merging defaults databases. We need to find resource information
 * from various sources and merge them before we query resources.
 *
 */

void 
wxXMergeDatabases (wxApp * theApp, Display * display)
{
  XrmDatabase homeDB, serverDB, applicationDB;
  char filenamebuf[1024];

  char *filename = &filenamebuf[0];
  char *environment;
  char *classname = theApp->wx_class;
  char name[256];
  (void) strcpy (name, "/usr/lib/X11/app-defaults/");
  (void) strcat (name, classname);

  /* Get application defaults file, if any */
  applicationDB = XrmGetFileDatabase (name);
  (void) XrmMergeDatabases (applicationDB, &wxResourceDatabase);

  /* Merge server defaults, created by xrdb, loaded as a property of the root
   * window when the server initializes and loaded into the display
   * structure on XOpenDisplay;
   * if not defined, use .Xdefaults
   */

  if (XResourceManagerString (display) != NULL)
    {
      serverDB = XrmGetStringDatabase (XResourceManagerString (display));
    }
  else
    {
      (void) GetIniFile (filename, NULL);
      serverDB = XrmGetFileDatabase (filename);
    }
  XrmMergeDatabases (serverDB, &wxResourceDatabase);

  /* Open XENVIRONMENT file, or if not defined, the .Xdefaults,
   * and merge into existing database
   */

  if ((environment = getenv ("XENVIRONMENT")) == NULL)
    {
      size_t len;
      environment = GetIniFile (filename, NULL);
      len = strlen (environment);
#ifndef SVR4
      (void) gethostname (environment + len, 1024 - len);
#else
      (void) sysinfo (SI_HOSTNAME, environment + len, 1024 - len);
#endif
    }
  homeDB = XrmGetFileDatabase (environment);
  XrmMergeDatabases (homeDB, &wxResourceDatabase);
}

#endif // !use_xview_code

#endif /* USE_RESOURCES */

// Read $HOME for what it says is home, if not
// read $USER or $LOGNAME for user name else determine
// the Real User, then determine the Real home dir.
static char *
GetIniFile (char *dest, const char *filename)
{
  if (filename && *filename == '/')
  {
//    *dest = '\0';		// have a full path
    strcpy(dest, filename);
  }
  else
  {
    (void)wxGetHomeDir(dest);

    strcat (dest, "/");
    if (filename == NULL)
      {
        if ((filename = getenv ("XENVIRONMENT")) == NULL)
          filename = ".Xdefaults";
      }
    else if (*filename != '.')
      strcat (dest, ".");
    strcat (dest, filename);
  }
  return dest;
}

char *wxGetHomeDir(char *dest)
{
  struct passwd *pw;
  register char *ptr;

  if ((ptr = getenv ("HOME")) != NULL)
    strcpy (dest, ptr);
  else
    {
      if ((ptr = getenv ("USER")) != NULL ||
	  (ptr = getenv ("LOGNAME")) != NULL)
	{
	  pw = getpwnam (ptr);
	}
      else
	pw = getpwuid (getuid ());
      if (pw)
	strcpy (dest, pw->pw_dir);
      else
	*dest = '\0';
    }

  return dest;
}

// Helper function for XView
#ifdef wx_xview
static void SetXCursor(wxWindow *win, wxCursor *cursor)
{
  Display *dpy = win->GetXDisplay();
  Window xwin = win->GetXWindow();
  if (cursor)
  {
    if (cursor->x_cursor)
    {
      if (cursor->use_raw_x_cursor)
      {
        XDefineCursor(dpy, xwin, cursor->x_cursor);
      }
      else
      {
        Xv_opaque x_win = (Xv_opaque)win->handle;
        if (wxSubType(win->__type, wxTYPE_CANVAS))
        {
          Xv_Window win2 = xv_get(x_win, CANVAS_NTH_PAINT_WINDOW, 0);

          xv_set(win2, WIN_CURSOR, cursor->x_cursor, NULL);
        }
        else
          xv_set(x_win, WIN_CURSOR, cursor->x_cursor, NULL);
      }
    }
  }
  else
  {
    XSetWindowAttributes attrs;
    attrs.cursor = None;
    XChangeWindowAttributes (dpy, xwin, CWCursor, &attrs);
  }
}
#endif

// Old cursor
static int wxBusyCursorCount = FALSE;

// Helper function
static void 
wxXSetBusyCursor (wxWindow * win, wxCursor * cursor)
{
  Display *display = win->GetXDisplay();

#ifdef wx_motif
  Window xwin = win->GetXWindow();
  XSetWindowAttributes attrs;

  if (cursor)
    {
      attrs.cursor = cursor->x_cursor;
    }
  else
    {
      // Restore old cursor
      if (win->wx_cursor)
	attrs.cursor = win->wx_cursor->x_cursor;
      else
	attrs.cursor = None;
    }
  if (xwin)
    XChangeWindowAttributes (display, xwin, CWCursor, &attrs);
#endif
#ifdef wx_xview
  if (cursor)
    SetXCursor(win, cursor);
  else
    SetXCursor(win, win->wx_cursor);
#endif

  XFlush (display);

  // Forget old cursor if we're resetting
  //  if (!cursor)
  //    win->currentWindowCursor = 0;

  for(wxNode *node = win->children->First (); node; node = node->Next())
    {
      wxWindow *child = (wxWindow *) node->Data ();
      if (wxSubType (child->__type, wxTYPE_FRAME) ||
	  wxSubType (child->__type, wxTYPE_CANVAS) ||
	  wxSubType (child->__type, wxTYPE_PANEL) ||
	  wxSubType (child->__type, wxTYPE_TEXT_WINDOW))
	wxXSetBusyCursor (child, cursor);
    }
}

// Set the cursor to the busy cursor for all windows
void 
wxBeginBusyCursor (wxCursor * cursor)
{
  wxBusyCursorCount++;
  if (wxBusyCursorCount == 1)
    {
      for(wxNode *node = wxTopLevelWindows.First (); node; node = node->Next())
	{
	  wxWindow *win = (wxWindow *) node->Data ();
	  wxXSetBusyCursor (win, cursor);
	}
    }
}

// Restore cursor to normal
void 
wxEndBusyCursor (void)
{
  if (wxBusyCursorCount == 0)
    return;

  wxBusyCursorCount--;
  if (wxBusyCursorCount == 0)
    {
      for(wxNode *node = wxTopLevelWindows.First (); node; node = node->Next())
	{
	  wxWindow *win = (wxWindow *) node->Data ();
	  wxXSetBusyCursor (win, NULL);
	}
    }
}

// TRUE if we're between the above two calls
Bool 
wxIsBusy (void)
{
  return (wxBusyCursorCount > 0);
}
