;;;
;;; stream toys from Touretsky and Gabriel modeled after Sussman
;;;

;;;
;;; delayed evaluation tools
;;;
(defmacro delay (form)
  `(let ((computation-completed nil) (result nil))
     #'(lambda (op)
	 (ecase op
		(:DELAYEDP (not computation-completed))
		(:FORCE (unless computation-completed
				(setq result ,form)
				(setq computation-completed t))
			result)))))

(defun delayedp (x) (funcall x :DELAYEDP))

(defun force (x) (funcall x :FORCE))

;;;
;;; scheme-like stream
;;;
(defstruct (stream-cell (:print-function print-stream-cell))
  car cdr)

(defun print-stream-cell (x io-stream depth)
  (declare (ignore depth))
  (format io-stream "#<[")
  (loop
   (format io-stream "~S " (scar x))
   (when (delayedp (stream-cell-cdr x))
	 (format io-stream "...]>")
	 (return-from print-stream-cell nil))
   (setq x (scdr x))))

(defmacro scons (x yform)
  `(make-stream-cell :car ,x :cdr (delay ,yform)))

(defun scar (cell) (stream-cell-car cell))

(defun scdr (cell) (force (stream-cell-cdr cell)))

(defun snthcdr (n s) (if (zerop n) s (snthcdr (1- n) (scdr s))))

;;;
;;; samples
;;;
(defun add (x y) (print "Adding") (+ x y))

;(setq a (delay (add 2 3)))

(defun integers-from (n) (scons n (integers-from (1+ n))))

(defun fibonacci ()
  (labels ((fib1 (i j k)
		 (scons i (fib1 j k (+ j k)))))
	  (fib1 1 1 2)))

(defun merge-streams (s1 s2)
  (scons (scar s1) (merge-streams s2 (scdr s1))))

(defun constant-stream (x)
  (scons x (constant-stream x)))

(defun filter (pred x)
  (if (funcall pred (scar x))
      (scons (scar x) (filter pred (scdr x)))
      (filter pred (scdr x))))

;(setq b (filter #'evenp (fibonacci)))
;(setq c (snthcdr 5 b))

(defun scale-stream (k s)
  (scons (* k (scar s)) (scale-stream k (scdr s))))

(defun add-streams (s1 s2)
  (scons (+ (scar s1) (scar s2))
	 (add-streams (scdr s1) (scdr s2))))

(defun fib ()
  (scons 1
	 (scons 1
		(add-streams (fib) (scdr (fib))))))

(defun ones ()
  (constant-stream 1))

(defun integers ()
  (add-streams (ones) (scons 0 (integers))))

;;;
;;; sieve
;;;
(defun not-divisible-by (n)
  #'(lambda (x) (plusp (mod x n))))

(defun sieve (s)
  (scons (scar s)
	 (sieve (filter (not-divisible-by (scar s))
			(scdr s)))))

(defun primes ()
  (sieve (integers-from 2)))

;(setq d (primes))
;(snthcdr 10 d)
