#include <stdio.h>
/*
 * 			Copyright 1993, 1994 by AT&T
 * 
 * 			 All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of AT&T not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * AT&T DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * AT&T BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * 
 * AT&T's dontation of this software does not imply a licence granted for
 * patents nor transfer of ownership of any patents which may inadvertently
 * be implemented in this code.
 * 
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Intrinsic.h>
#include "tiffio.h"
	
#include "C_P_args.h"

C_PROTOS_BEGIN_EXTERN

extern void
main C_P_ARGS((int argc, char **argv));

static XImage *
CreateImage C_P_ARGS((Display *display, int out_width, int out_height));

static void
ImageToTiffFile C_P_ARGS((Display *display, XImage *image,
			  Colormap colormap, char *filename));

C_PROTOS_END_EXTERN

#ifndef TRUE
#define	TRUE	1
#define	FALSE	0
#endif

#define MAX_IMAGE		(64 * 1024)

void
main (argc, argv)
int argc;
char **argv;
{
    Display *display;
    int screen;
    XSizeHints sizehints;
    XRectangle rect, subrect;
    XImage *image_out, *image_in;
    char *data_out, *data_in;
    int current_line, lines_per_sample, i;
    char buffer[BUFSIZ + 64];

    if ((display = XOpenDisplay(NULL)) == NULL)
    {
	fprintf (stderr, "%s: Can't open display.\n");
	exit (1);
    }
    screen = DefaultScreen (display);

    if (argc != 2 || *argv[1] == '-')
    {
	fprintf (stderr, "%s usage:\n\t%s <output filename>\n",
		 argv[0], argv[0]);
	exit (1);
    }

    if ((int) strlen (argv[1]) > BUFSIZ)
    {
	fprintf (stderr, "%s usage:\n\t%s <output filename>\n",
		 argv[0], argv[0]);
	fprintf (stderr, "%s: filename is more than %d\n",
		 argv[0], BUFSIZ);
	exit (1);
    }

    sprintf (buffer, "%s: x=%%d y=%%d width=%%d height=%%d",
	     argv[1]);
    sizehints.flags = PSize | PMinSize;
    sizehints.width = sizehints.min_width = 8;
    sizehints.height = sizehints.min_height = 8;
    XGetRect (display, screen, RootWindow (display, screen), &sizehints,
	      buffer, "fixed",
	      FALSE, TRUE, &rect);

    /*
     * create an XImage large enough to store the entire image
     */
    image_out = CreateImage (display, rect.width, rect.height);
    if (image_out == NULL)
	exit (1);

    /*
     * Loop getting parts of the image
     */
    lines_per_sample = MAX_IMAGE / image_out -> bytes_per_line;
    current_line = 0;
    data_out = image_out -> data;
    for (current_line = 0; current_line < image_out -> height;
	 current_line += lines_per_sample)
    {
	if (current_line + lines_per_sample > image_out -> height)
	    lines_per_sample = image_out -> height - current_line;
		
	image_in = XGetImage (display,
			      RootWindow (display, screen),
			      rect.x,
			      rect.y + current_line,
			      rect.width,
			      lines_per_sample,
			      AllPlanes,
			      image_out -> format);
		
	data_in = image_in -> data;
	for (i = 0; i < image_in -> height; i++)
	{
	    memcpy (data_out, data_in, image_in -> bytes_per_line);
	    data_in += image_in -> bytes_per_line;
	    data_out += image_out -> bytes_per_line;
	}
		
	(*image_in -> f.destroy_image) (image_in);
    }

    XPutImage (display, RootWindow (display, screen),
	       DefaultGC (display, screen),
	       image_out, 0, 0, 0, 0,
	       image_out -> width, image_out -> height);

    ImageToTiffFile (display, image_out, DefaultColormap (display, screen),
		     argv[1]);
}

/*
 * create an image that matches the display
 */
static XImage *
CreateImage (display, out_width, out_height)
Display *display;
int out_width;
int out_height;
{
    XImage *out;
    int screen = DefaultScreen (display);
    int depth = DefaultDepth (display, screen);

    out = XCreateImage (display, DefaultVisual (display, screen),
			depth,
			(depth > 1) ? ZPixmap : XYPixmap,
			0,	/* offset */
			(char *) NULL,
			out_width, out_height,
			8,	/* bitmap quantum */
			0);

    if (out == (XImage *) NULL)
    {
	fprintf (stderr, "Cannot allocate image space.\n");
	exit (1);
    }

    out -> data = (char *) malloc (out -> bytes_per_line *
				   out -> height);
    if (out -> data == (char *) NULL)
    {
	fprintf (stderr, "Cannot allocate image space.\n");
	exit (1);
    }

    return out;
}

#define CMAP_MAX	256

static void
ImageToTiffFile (display, image, colormap, filename)
Display *display;
XImage *image;
Colormap colormap;
char *filename;
{
    TIFF *out;
    unsigned char *data;
    unsigned char map[CMAP_MAX][3];
    unsigned short rmap[CMAP_MAX], gmap[CMAP_MAX], bmap[CMAP_MAX];
    unsigned short *rptr, *gptr, *bptr;
    int marks[CMAP_MAX];
    XColor colors[CMAP_MAX];
    XColor min_colors[CMAP_MAX];
    XColor *cptr;
    int colors_used;
    int i, j;
    unsigned long pixel;
    int cmap_count;
    unsigned long depth_mask;
    Visual *visual;
    unsigned char *buf;

    visual = DefaultVisual (display, DefaultScreen (display));

    if (visual -> class == DirectColor)
    {
	fprintf (stderr,
"Silly us, we do not know how to write DirectColor images.\n");
	return;
    }

    if (visual -> class == TrueColor &&
	(image -> bits_per_pixel > 24 ||
	 image -> red_mask > 8 ||
	 image -> green_mask > 8 ||
	 image -> blue_mask > 8))
    {
	fprintf (stderr,
"Silly us, we do not know how to write this TrueColor image - too many bits.\n");
	return;
    }

    if ((visual -> class == StaticGray  ||
	 visual -> class == PseudoColor ||
	 visual -> class == StaticColor ||
	 visual -> class == PseudoColor) &&
	image -> depth > 1)
    {
	if (image -> depth > 8)
	{
	    fprintf (stderr,
"Silly us, we do not know how to write colormapped images that are more than 8 bits deep,\n");
		return;
	}

	cmap_count = 1 << image -> depth;
	depth_mask = cmap_count - 1;

	for (i = 0; i < cmap_count; i++)
	    marks[i] = FALSE;

	for (colors_used = 0, i = 0; i < image -> height; i++)
	{
	    for (j = 0; j < image -> width; j++)
	    {
		pixel = XGetPixel (image, j, i) & depth_mask;
		if (marks[pixel] == FALSE)
		{
		    marks[pixel] = TRUE;
		    colors_used++;
		}
	    }
	}
		
	fprintf (stderr,
		 "The file \"%s\" uses %d color table entries.\n",
		 filename, colors_used);
		
	for (i = 0; i < cmap_count; i++)
	    colors[i].pixel = i;
	XQueryColors (display, colormap, colors, cmap_count);
	for (j = 0, i = 0; i < cmap_count; i++)
	{
	    if (marks[i])
	    {
		colors[i].pixel = j;
		min_colors[j] = colors[i];
		j++;
	    }
	}
	for (i = 0; i < image -> height; i++)
	{
	    for (j = 0; j < image -> width; j++)
		XPutPixel (image, j, i,
			   colors[XGetPixel (image, j, i)].pixel);
	}
		
	for (rptr = rmap, gptr = gmap, bptr = bmap, cptr = min_colors, i = 0;
	     i < colors_used; i++, cptr++)
	{
	    *rptr++ = (cptr -> red >> 8);
	    *gptr++ = (cptr -> green >> 8);
	    *bptr++ = (cptr -> blue >> 8);
	}
	for (; i < CMAP_MAX; i++)
	{
	    *rptr++ = 0;
	    *gptr++ = 0;
	    *bptr++ = 0;
	}
    }

    out = TIFFOpen (filename, "w");
    if (out == NULL)
    {
	fprintf (stderr, "Cannot open output file: %s.\n", filename);
	return;
    }

    TIFFSetField (out, TIFFTAG_IMAGEWIDTH, (long) image -> width);
    TIFFSetField (out, TIFFTAG_IMAGELENGTH, (long) image -> height);
    TIFFSetField (out, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
    TIFFSetField (out, TIFFTAG_COMPRESSION, COMPRESSION_NONE);
    TIFFSetField (out, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
    TIFFSetField (out, TIFFTAG_RESOLUTIONUNIT, RESUNIT_NONE);

    if (WhitePixel (display, DefaultScreen (display)) == 0)
	TIFFSetField (out, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISWHITE);
    else
	TIFFSetField (out, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISBLACK);

    buf = (unsigned char *) NULL;
    if (visual -> class == TrueColor)
    {
	if (image -> format == ZPixmap && image -> depth > 8)
	{
	    TIFFSetField (out, TIFFTAG_SAMPLESPERPIXEL, 3);
	    TIFFSetField (out, TIFFTAG_BITSPERSAMPLE, 8);
	    i = TIFFScanlineSize (out);
	    if (i < image -> bytes_per_line)
	    {
		buf = (unsigned char *) malloc (i);
		if (buf == (unsigned char *) NULL)
		{
		    fprintf (stderr, "Cannot allocate memory.\n");
		    exit (1);
		}
	    }
	}
	else
	{
	    TIFFSetField (out, TIFFTAG_SAMPLESPERPIXEL, 1);
	    TIFFSetField (out, TIFFTAG_BITSPERSAMPLE, image -> depth);
	}
	TIFFSetField (out, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);
    }
    else
    {
	TIFFSetField (out, TIFFTAG_SAMPLESPERPIXEL, 1);
	TIFFSetField (out, TIFFTAG_BITSPERSAMPLE, image -> depth);
	if (image -> depth > 1)
	{
	    TIFFSetField (out, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_PALETTE);
	    TIFFSetField (out, TIFFTAG_COLORMAP, rmap, gmap, bmap);
	}
    }

    if (buf)
    {
	int k;

	/*
	 * this assumes that the masks are contiguous bits - no gaps!
	 */
	for (i = 0; i < image -> height; i++)
	{
	    k = 0;
	    for (j = 0; j < image -> width; j++)
	    {
		pixel = XGetPixel (image, j, i);
		buf[k++] = (pixel & image -> red_mask) /
		    (1);

		buf[k++] = (pixel & image -> green_mask) /
		    (image -> red_mask + 1);

		buf[k++] = (pixel & image -> blue_mask) /
		    (image -> red_mask + image -> green_mask + 1);
	    }
	    TIFFWriteScanline (out, buf, i, 0);
	}
    }
    else
    {
	data = (unsigned char *) image -> data;
	for (i = 0; i < image -> height; i++)
	{
	    TIFFWriteScanline (out, data, i, 0);
	    data += image -> bytes_per_line;
	}
    }

    TIFFClose (out);
}
