static char TM_xtent_tif_c[] = "<%W%	%D% %T%>";
/*
 * 			Copyright 1993, 1994 by AT&T
 * 
 * 			 All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of AT&T not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * AT&T DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * AT&T BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * 
 * AT&T's dontation of this software does not imply a licence granted for
 * patents nor transfer of ownership of any patents which may inadvertently
 * be implemented in this code.
 * 
 */

#include <stdio.h>
#include <X11/Xos.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <sys/types.h>
#include "tiffio.h"

#include "C_P_args.h"

C_PROTOS_BEGIN_EXTERN

extern XImage *
LoadTifImage C_P_ARGS((Display *display, int screen, char *fname, int new_map,
		       Colormap *colormap_return));

extern Pixmap
LoadTifPixmap C_P_ARGS((Display *display, int screen, char *fname,
			int new_map, Colormap *colormap_return,
			int *rwidth, int *rheight));

static Colormap
AllocateColormap C_P_ARGS((Display *display, int screen, int new_map,
			   XColor *xcolors, int xcolor_count));

static void
PseudoMapToGreyMap C_P_ARGS((XColor *xcolors, unsigned char *greymap,
			     int size));

static void
DitherImage C_P_ARGS((XColor *xcolors, XImage *in, XImage *out));

static XImage *
ConvertOneToMany C_P_ARGS((Display *display, int screen, int depth,
			   XImage *image));

extern void
ImageToTifFile C_P_ARGS((Display *display, XImage *image,
			 Colormap colormap, char *filename));

C_PROTOS_END_EXTERN

#ifndef TRUE
#define	TRUE	1
#endif
#ifndef FALSE
#define	FALSE	0
#endif

#define	MAX_CMAP_SIZE	256

XImage *
LoadTifImage (display, screen, fname, new_map, colormap_return)
Display *display;
int screen;
char *fname;
int new_map;
Colormap *colormap_return;			/* RETURNS */
{
    unsigned char *wptr;
    unsigned char *data;
    int i, j;
    XImage *image;
    XImage *out;
    unsigned long width, height;
    unsigned short bitspersample, samplesperpixel;
    unsigned short *rm, *gm, *bm;
    int cmap_size;
    XColor xcolors[MAX_CMAP_SIZE];
    TIFF *in;
    int depth = DefaultDepth (display, screen);
    int eight_bit;
    unsigned short config;
    unsigned short resolve;
    unsigned short photo;
    int photo_reverse;
    Visual *visual;
    int bits;

    /*
     * open the tif file and check the arguments
     */
    if ((in = TIFFOpen (fname, "r")) == NULL)
    {
	fprintf (stderr, "%s: Could not open tif image file: %s.\n",
		 XtentProgramName (), fname);
	return (XImage *) NULL;
    }

    TIFFGetField (in, TIFFTAG_PLANARCONFIG, &config);
    if (TIFFIsTiled (in) || config != PLANARCONFIG_CONTIG)
    {
	fprintf (stderr,
"%s: Silly us, we can only read planar, contiguous, non-tiled images: %s\n",
		 XtentProgramName (), fname);
	return (XImage *) NULL;
    }

    TIFFGetFieldDefaulted (in, TIFFTAG_BITSPERSAMPLE, &bitspersample);
    TIFFGetFieldDefaulted (in, TIFFTAG_SAMPLESPERPIXEL, &samplesperpixel);

    visual = DefaultVisual (display, DefaultScreen (display));
    if (visual -> class == TrueColor || visual -> class == DirectColor)
    {
	if (bitspersample > 8 || samplesperpixel > 3)
	{
	    fprintf (stderr,
"%s: Silly us, we do not know how to read this TrueColor or DirectColor image - too many bits: %s.\n", XtentProgramName (), fname);
	    return (XImage *) NULL;
	}
    }
    else
    {
	if (bitspersample > 8 || samplesperpixel > 1)
	{
	    fprintf (stderr,
"%s: Silly us, we do not know how to read this PseudoColor image - too many bits: %s.\n",
		     XtentProgramName (), fname);
	    TIFFClose (in);
	    return (XImage *) NULL;
	}
    }
    bits = bitspersample * samplesperpixel;

    TIFFGetField (in, TIFFTAG_IMAGEWIDTH, &width);
    TIFFGetField (in, TIFFTAG_IMAGELENGTH, &height);

    TIFFGetField (in, TIFFTAG_PHOTOMETRIC, &photo);
    if ((photo == PHOTOMETRIC_MINISWHITE &&
	 BlackPixel (display, DefaultScreen (display)) == 0) ||
	(photo == PHOTOMETRIC_MINISBLACK &&
	 WhitePixel (display, DefaultScreen (display)) == 0))
	photo_reverse = True;
    else
	photo_reverse = False;

    /*
     * create an image
     */
    image = XCreateImage (display, DefaultVisual (display, screen),
			  bits,
			  (bits > 1) ? ZPixmap : XYPixmap,
			  0,
			  (char *) NULL,
			  (int) width, (int) height,
			  8, 0);
    if (image == NULL)
    {
	fprintf (stderr,
	"%s: Could not allocate the X image for the file: %s\n",
		 XtentProgramName (), fname);
	TIFFClose (in);
	return (XImage *) NULL;
    }

    data = (unsigned char *) XtMalloc (image -> bytes_per_line * height);
    image -> data = (char *) data;
	
    /*
     * read in the bits a line at a time
     */
    wptr = data;
    for (i = 0; i < height; i++)
    {
	if (TIFFReadScanline(in, wptr, i, 0) <= 0)
	    break;

	if (bitspersample == 1 && photo_reverse)
	{
	    for (j = 0; j < image -> bytes_per_line; j++)
		wptr[j] = ~(wptr[j]);
	}
	wptr += image -> bytes_per_line;
    }

    /*
     * one bit deep images work in all cases (cough)
     */
    if (bitspersample == 1)
    {
	TIFFClose (in);
	if (depth == 1)
	    return image;

	return ConvertOneToMany (display, screen, depth, image);
    }

    /*
     * get the colormap entries - no colormap means grey scale
     */
    if (photo != PHOTOMETRIC_RGB)
    {
	cmap_size = 1 << bitspersample;
	if (TIFFGetField (in, TIFFTAG_COLORMAP, &rm, &gm, &bm))
	{
	    for (i = 0; i < cmap_size; i++)
		if (rm[i] > 255 || gm[i] > 255 || bm[i] > 255)
		    break;

	    if (i < cmap_size)
		for (i = 0; i < cmap_size; i++)
		{
		    xcolors[i].red   = rm[i];
		    xcolors[i].green = gm[i];
		    xcolors[i].blue  = bm[i];
		}
	    else
		for (i = 0; i < cmap_size; i++)
		{
		    xcolors[i].red   = rm[i] | (rm[i] << 8);
		    xcolors[i].green = gm[i] | (gm[i] << 8);
		    xcolors[i].blue  = bm[i] | (bm[i] << 8);
		}
	}
	else
	{
	    int inc = (MAX_CMAP_SIZE + cmap_size - 1) / cmap_size;

	    if (photo_reverse)
	    {
		for (j = MAX_CMAP_SIZE, i = 0; i < cmap_size; j -= inc, i++)
		{
		    xcolors[i].green = xcolors[i].blue =
			xcolors[i].red = (j << 8) | j;
		}
	    }
	    else for (j = 0, i = 0; i < cmap_size; j += inc, i++)
	    {
		xcolors[i].green = xcolors[i].blue =
		    xcolors[i].red = (j << 8) | j;
	    }
	}

	*colormap_return = AllocateColormap (display, screen, new_map,
					     xcolors, cmap_size);
    }

    /*
     * close up the tiff file - we have all the info that we need.
     */
    TIFFClose (in);

    /*
     * convert the pixel values to the ones allocated on this display
     */
    if (visual -> class == TrueColor || visual -> class == DirectColor)
	return image;

    if (depth == 8 && bitspersample == 8) /* 8 bit to 8 bit */
    {
	for (j = 0; j < image -> height; j++)
	{
	    wptr = (unsigned char *) image -> data +
		image -> bytes_per_line * j;

	    for (i = 0; i < image -> width; i++)
	    {
#ifdef LINT
		unsigned char order_of_eval;

		order_of_eval = xcolors[*wptr & 255].pixel;
		*wptr++ = order_of_eval;
#else
		*wptr++ = xcolors[*wptr & 255].pixel;
#endif
	    }
	}
    }
    else if (depth != bitspersample) /* 8 bit to 1 or odd bits */
    {
	out = XCreateImage (display, DefaultVisual (display, screen),
			    depth,
			    (depth == 1) ? XYPixmap : ZPixmap,
			    0,	/* offset */
			    (char *) NULL,
			    (int) image -> width,
			    (int) image -> height,
			    8,	/* bitmap quantum */
			    0);
	if (out == NULL)
	{
	    fprintf (stderr, "%s: Could not allocate the X image\n",
		     XtentProgramName ());
	    XDestroyImage (image);
	    return (XImage *) NULL;
	}
		
	data = (unsigned char *) XtMalloc (out -> bytes_per_line *
					   out -> height);
	out -> data = (char *) data;
	if (depth == 1)
	{
	    DitherImage (xcolors, image, out);
	    XDestroyImage (image);
	    return out;
	}

	for (j = 0; j < image -> height; j++)
	    for (i = 0; i < image -> width; i++)
		XPutPixel (out, i, j,
			   xcolors[XGetPixel (image, i, j)].pixel &
			   (cmap_size - 1));
	XDestroyImage (image);
	return out;
    }
    else				/* arbitrary in and out depths */
    {
	for (j = 0; j < image -> height; j++)
	    for (i = 0; i < image -> width; i++)
		XPutPixel (image, i, j,
			   xcolors[XGetPixel (image, i, j)].pixel &
			   (cmap_size - 1));
    }

    return image;
}

Pixmap
LoadTifPixmap (display, screen, fname, new_map, colormap_return, rwidth, rheight)
Display *display;
int screen;
char *fname;
int new_map;
Colormap *colormap_return;			/* RETURNS */
int *rwidth;
int *rheight;
{
    XImage *image;
    Pixmap image_pixmap;
    int depth = DefaultDepth (display, screen);

    image = LoadTifImage (display, screen, fname, new_map, colormap_return);
    if (image == (XImage *) NULL)
	return (Pixmap) NULL;

    *rwidth = image -> width;
    *rheight = image -> height;

    image_pixmap = XCreatePixmap (display, RootWindow (display, screen),
				  image -> width,
				  image -> height,
				  depth);
	
    XPutImage (display, image_pixmap,
	       DefaultGC (display, screen),
	       image,
	       0, 0, 0, 0,
	       image -> width, image -> height);
	
    XDestroyImage (image);
	
    return image_pixmap;
}

static Colormap
AllocateColormap (display, screen, new_map, xcolors, xcolor_count)
Display *display;
int screen;
int new_map;
XColor *xcolors;
int xcolor_count;
{
    int i, j;
    Colormap colormap;
    char *reject_array = (char *) NULL;
    int reject_count;
    Visual *visual;
    int cmap_size;
    XColor *colors;
    int diff, smallest_diff, grey, gdiff, smallest_gdiff;
    int index;

    if (new_map)
	colormap = XCreateColormap (display,
				    RootWindow (display, screen),
				    DefaultVisual (display, screen),
				    AllocNone);
    else
	colormap = DefaultColormap (display, screen);
	
    for (i = 0; i < xcolor_count; i++)
    {
	if (!XAllocColor (display, colormap, &xcolors[i]))
	{
	    if (reject_array == (char *) NULL)
	    {
		reject_array = (char *) XtMalloc (xcolor_count);
		memset (reject_array, '\0', xcolor_count);
		reject_count = 0;
	    }
	    reject_array[i] = 1;
	    reject_count++;
	}
    }

    if (reject_array == (char *) NULL)
	return colormap;

    visual = DefaultVisual (display, DefaultScreen (display));
    cmap_size = visual -> map_entries;
    colors = (XColor *) XtMalloc (sizeof (XColor) * cmap_size);
    for (i = 0; i < cmap_size; i++)
	colors[i].pixel = i;
    XQueryColors (display, colormap, colors, cmap_size);

    for (i = 0; i < xcolor_count; i++)
    {
	if (reject_array[i] == 0)
	    continue;

	grey = (long) ((long) xcolors[i].red * 77 +
		       (long) xcolors[i].green * 151 +
		       (long) xcolors[i].blue * 28) >> 8;

	smallest_diff = 256 * 256 * 256;
	smallest_gdiff = 256 * 256 * 256;
	index = 0;
	for (j = 0; j < cmap_size; j++)
	{
	    diff = abs (xcolors[i].red - colors[j].red) +
		   abs (xcolors[i].green - colors[j].green) +
		   abs (xcolors[i].blue - colors[j].blue);

	    gdiff = abs (((long) ((long) colors[j].red * 77 +
				  (long) colors[j].green * 151 +
				  (long) colors[j].blue * 28) >> 8) - grey);
	    if (gdiff == 0)
		gdiff = 1;

	    if (diff <= smallest_diff  &&
		16384 + 65536 / gdiff >= 65536 / smallest_gdiff)
	    { /* 1/4 the range is aprrox. 1 sd */
		smallest_diff = diff;
		smallest_gdiff = gdiff;
		index = j;
	    }
	}
	if (!XAllocColor (display, colormap, &colors[index]))
	{
	    fprintf (stderr,
	     "%s: Could not RE-allocate the color(%d): [%d][%d][%d].\n",
		     XtentProgramName (),
		     i,
		     colors[index].red & 255,
		     colors[index].green & 255,
		     colors[index].blue & 255);
	}
	xcolors[i].pixel = colors[index].pixel;
    }
    XFree ((char *) reject_array);
    XFree ((char *) colors);
    fprintf (stderr, "%s: Requested %d colors - got %d\n",
	     XtentProgramName (), xcolor_count, xcolor_count - reject_count);

    return colormap;
}

static void
PseudoMapToGreyMap (xcolors, greymap, size)
XColor *xcolors;
unsigned char *greymap;
int size;
{
    XColor *cptr = xcolors;
    unsigned char *gptr = greymap;
    unsigned long grey;
    int i;

    for (i = 0; i < size; i++)
    {
	grey = ((cptr -> red >> 8) & 255) * 77 +
	    ((cptr -> green >> 8) & 255) * 151 +
		((cptr -> blue >> 8) & 255) * 28;
	*gptr++ = grey >> 8;
	cptr++;
    }
}

/*
 * a cheap and fast dither
 */
static void
DitherImage (xcolors, in, out)
XColor *xcolors;
XImage *in;
XImage *out;
{
    unsigned char greymap[MAX_CMAP_SIZE];
    register int i, j;
    register unsigned char pixel;
    unsigned char *in_ptr, *out_ptr;
    static int dither_matrix[8][8] =
    {
	2, 194,  50, 242,  14, 206,  62, 254,
	130,  66, 178, 114, 142,  78, 190, 126,
	34, 226,  18, 210,  46, 238,  30, 222,
	162,  98, 146,  82, 174, 110, 158,  94,
	10, 202,  58, 250,   6, 198,  54, 246,
	138,  74, 186, 122, 134,  70, 182, 118,
	42, 234,  26, 218,  38, 230,  22, 214,
	170, 106, 154,  90, 166, 102, 150,  86
	};

    PseudoMapToGreyMap (xcolors, greymap, 1 << in -> depth);

    for (j = 0; j < out -> height; j++)
    {
	in_ptr = (unsigned char *) in -> data + j *
	    in -> bytes_per_line;
	out_ptr = (unsigned char *) out -> data + j *
	    out -> bytes_per_line;
	pixel = 0;
	for (i = 0; i < out -> width; i++)
	{
	    pixel <<= 1;
	    if ((int) greymap [*in_ptr++] < dither_matrix[i & 7][j & 7])
		pixel |= 1;
	    if ((i & 7) == 7)
	    {
		*out_ptr++ = pixel;
		pixel = 0;
	    }
	}
	if (i & 7)
	    *out_ptr = pixel << (8 - (i & 7));
    }
}		 		   

static XImage *
ConvertOneToMany (display, screen, depth, image)
Display *display;
int screen;
int depth;
register XImage *image;
{
    register XImage *out;
    register int i, j;
    long black, white;
    unsigned char *data;

    out = XCreateImage (display, DefaultVisual (display, screen),
			depth,	/* bitspersample */
			ZPixmap,
			0,	/* offset */
			(char *) NULL,
			(int) image -> width, (int) image -> height,
			8,	/* bitmap quantum */
			0);
    if (out == NULL)
    {
	fprintf (stderr, "%s: Could not allocate the X image.\n",
		 XtentProgramName ());
	XDestroyImage (image);
	return (XImage *) NULL;
    }

    data = (unsigned char *) XtMalloc (out -> bytes_per_line *
				       out -> height);
    out -> data = (char *) data;

    black = BlackPixel (display, screen);
    white = WhitePixel (display, screen);

    if (white == 0)
    {
	memset (data, (char) 0, out -> bytes_per_line * out -> height);

	for (j = 0; j < image -> height; j++)
	{
	    for (i = 0; i < image -> width; i++)
	    {
		if (XGetPixel (image, i, j))
		    XPutPixel (out, i, j, black);
	    }
	}
    }
    else
    {
	for (j = 0; j < image -> height; j++)
	{
	    for (i = 0; i < image -> width; i++)
	    {
		if (XGetPixel (image, i, j))
		    XPutPixel (out, i, j, black);
		else
		    XPutPixel (out, i, j, white);
	    }
	}
    }

    XDestroyImage (image);
    return out;
}

void
ImageToTifFile (display, image, colormap, filename)
Display *display;
XImage *image;
Colormap colormap;
char *filename;
{
    TIFF *out;
    unsigned char *data;
    unsigned char map[MAX_CMAP_SIZE][3];
    unsigned short rmap[MAX_CMAP_SIZE], gmap[MAX_CMAP_SIZE], bmap[MAX_CMAP_SIZE];
    unsigned short *rptr, *gptr, *bptr;
    int marks[MAX_CMAP_SIZE];
    XColor colors[MAX_CMAP_SIZE];
    XColor min_colors[MAX_CMAP_SIZE];
    XColor *cptr;
    int colors_used;
    int i, j;
    unsigned long pixel;
    int cmap_count;
    unsigned long depth_mask;
    Visual *visual;
    unsigned char *buf;

    visual = DefaultVisual (display, DefaultScreen (display));

    if (visual -> class == DirectColor)
    {
	fprintf (stderr,
"%s: Silly us, we do not know how to write DirectColor images.\n",
		 XtentProgramName ());
	return;
    }

    if (visual -> class == TrueColor &&
	(image -> bits_per_pixel > 24 ||
	 image -> red_mask > 8 ||
	 image -> green_mask > 8 ||
	 image -> blue_mask > 8))
    {
	fprintf (stderr,
"%s: Silly us, we do not know how to write this TrueColor image - too many bits.\n",
		 XtentProgramName ());
	return;
    }

    if ((visual -> class == StaticGray  ||
	 visual -> class == PseudoColor ||
	 visual -> class == StaticColor ||
	 visual -> class == PseudoColor) &&
	image -> depth > 1)
    {
	if (image -> depth > 8)
	{
	    fprintf (stderr,
"%s: Silly us, we do not know how to write colormapped images that are more than 8 bits deep,\n",
		     XtentProgramName ());
		return;
	}

	cmap_count = 1 << image -> depth;
	depth_mask = cmap_count - 1;

	for (i = 0; i < cmap_count; i++)
	    marks[i] = FALSE;

	for (colors_used = 0, i = 0; i < image -> height; i++)
	{
	    for (j = 0; j < image -> width; j++)
	    {
		pixel = XGetPixel (image, j, i) & depth_mask;
		if (marks[pixel] == FALSE)
		{
		    marks[pixel] = TRUE;
		    colors_used++;
		}
	    }
	}
		
	fprintf (stderr,
		 "%s: The file \"%s\" uses %d color table entries.\n",
		 XtentProgramName (), filename, colors_used);
		
	for (i = 0; i < cmap_count; i++)
	    colors[i].pixel = i;
	XQueryColors (display, colormap, colors, cmap_count);
	for (j = 0, i = 0; i < cmap_count; i++)
	{
	    if (marks[i])
	    {
		colors[i].pixel = j;
		min_colors[j] = colors[i];
		j++;
	    }
	}
	for (i = 0; i < image -> height; i++)
	{
	    for (j = 0; j < image -> width; j++)
		XPutPixel (image, j, i,
			   colors[XGetPixel (image, j, i)].pixel);
	}
		
	for (rptr = rmap, gptr = gmap, bptr = bmap, cptr = min_colors, i = 0;
	     i < colors_used; i++, cptr++)
	{
	    *rptr++ = (cptr -> red >> 8);
	    *gptr++ = (cptr -> green >> 8);
	    *bptr++ = (cptr -> blue >> 8);
	}
	for (; i < MAX_CMAP_SIZE; i++)
	{
	    *rptr++ = 0;
	    *gptr++ = 0;
	    *bptr++ = 0;
	}
    }

    out = TIFFOpen (filename, "w");
    if (out == NULL)
    {
	fprintf (stderr, "%s: Cannot open output file: %s.\n",
		 XtentProgramName (), filename);
	return;
    }

    TIFFSetField (out, TIFFTAG_IMAGEWIDTH, (long) image -> width);
    TIFFSetField (out, TIFFTAG_IMAGELENGTH, (long) image -> height);
    TIFFSetField (out, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
    TIFFSetField (out, TIFFTAG_COMPRESSION, COMPRESSION_NONE);
    TIFFSetField (out, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
    TIFFSetField (out, TIFFTAG_RESOLUTIONUNIT, RESUNIT_NONE);

    if (WhitePixel (display, DefaultScreen (display)) == 0)
	TIFFSetField (out, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISWHITE);
    else
	TIFFSetField (out, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISBLACK);

    buf = (unsigned char *) NULL;
    if (visual -> class == TrueColor)
    {
	if (image -> format == ZPixmap && image -> depth > 8)
	{
	    TIFFSetField (out, TIFFTAG_SAMPLESPERPIXEL, 3);
	    TIFFSetField (out, TIFFTAG_BITSPERSAMPLE, 8);
	    i = TIFFScanlineSize (out);
	    if (i < image -> bytes_per_line)
	    {
		buf = (unsigned char *) malloc (i);
		if (buf == (unsigned char *) NULL)
		{
		    fprintf (stderr, "%s: Cannot allocate memory.\n",
			     XtentProgramName ());
		    exit (1);
		}
	    }
	}
	else
	{
	    TIFFSetField (out, TIFFTAG_SAMPLESPERPIXEL, 1);
	    TIFFSetField (out, TIFFTAG_BITSPERSAMPLE, image -> depth);
	}
	TIFFSetField (out, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);
    }
    else
    {
	TIFFSetField (out, TIFFTAG_SAMPLESPERPIXEL, 1);
	TIFFSetField (out, TIFFTAG_BITSPERSAMPLE, image -> depth);
	if (image -> depth > 1)
	{
	    TIFFSetField (out, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_PALETTE);
	    TIFFSetField (out, TIFFTAG_COLORMAP, rmap, gmap, bmap);
	}
    }

    if (buf)
    {
	int k;

	/*
	 * this assumes that the masks are contiguous bits - no gaps!
	 */
	for (i = 0; i < image -> height; i++)
	{
	    k = 0;
	    for (j = 0; j < image -> width; j++)
	    {
		pixel = XGetPixel (image, j, i);
		buf[k++] = (pixel & image -> red_mask) /
		    (1);

		buf[k++] = (pixel & image -> green_mask) /
		    (image -> red_mask + 1);

		buf[k++] = (pixel & image -> blue_mask) /
		    (image -> red_mask + image -> green_mask + 1);
	    }
	    TIFFWriteScanline (out, buf, i, 0);
	}
    }
    else
    {
	data = (unsigned char *) image -> data;
	for (i = 0; i < image -> height; i++)
	{
	    TIFFWriteScanline (out, data, i, 0);
	    data += image -> bytes_per_line;
	}
    }

    TIFFClose (out);
}
