/*
 * Copyright 1992 the Board of Trustees of the Leland Stanford Junior
 * University. Official permission to use this software is included in
 * the documentation. It authorizes you to use this file for any
 * non-commercial purpose, provided that this copyright notice is not
 * removed and that any modifications made to this file are commented
 * and dated in the style of the example below.
 */

/*
 *  expand a string expression.
 *  --------------------------
 *
 *  char* expand_string( char *name, char* input_string )
 *
 *  Variable names $???? or $(????) are expanded to
 *  their values, and backquoted expressions are evaluated.
 *
 *
 * Steve Cole, Dave Nichols (SEP), January 16 1993
 *      Inserted this sample edit history entry.
 *      Please log any further modifications made to this file:
 *
 * Steve Losen (scl@virginia.edu) rewrote expand_string() Mar 3 1993
 * to use a single pass that does not rescan the value of any $????
 *
 * Dave Nichols (SEP), May 4 1993
 *      Changed double backquote generation so that it is independent
 *      of evaluation order of different compilers.
 */

#include <stdio.h>
#include <string.h>

#include "tree.h"

#include "object.h"
#include "eval_command.h"
#include "string_buf.h"
#include "expand_string.h"

/*
 * get the arguments from the object list and format the
 * output string in the buffer
 */

char *
expand_string(name, input_string)
     char* name;
     char* input_string;
{
    char *p, *ret, *start, savechar;
    string_buf *buffer, *outbuf, *evalbuf;
    int leftparen;

    buffer = outbuf = buf_start();

    p = start = input_string;
    while (*p) {
	switch(*p) {

	default:
	    p++;
	    break;

	case '`':
	    if (start < p)
		buf_cat(buffer, start, p - start);
	    p++;
	    if (*p == '`') {  /* double backquote `` */
		buf_cat (buffer, "`", 1);p++;
	    }
	    else if (buffer == outbuf) {         /* left backquote */
		buffer = evalbuf = buf_start();  /* switch buffers */
	    }
	    else {  /* right backquote, evaluate string in evalbuf */
		ret = buf_fetch(evalbuf);
		buf_free(evalbuf);
		start = eval_command(ret);
		free(ret);
		buf_cat(outbuf, start, strlen(start));
		free(start);
		buffer = outbuf;
	    }
	    start = p;
	    break;

	case '$':
	    if (start < p)
		buf_cat(buffer, start, p - start);
	    p++;
	    leftparen = 0;
	    switch (*p) {
	    case '$':
		buf_cat (buffer, "$", 1); p++; /* $$ */
		start = p;
		continue;   /* next while iteration */
	    case '(':
		leftparen = 1;
		start = ++p;
		while (*p && *p != ')')
		    p++;
		if (! *p) {
		    if (name != 0) {
			fprintf(stderr, "In action for object %s\n", name);
		    }
		    fprintf(stderr, "expand_string(), couldn't find matching \")\"\nstarting here->$(%s\n", start);
		    exit(-1);
		}
		break;
	    default:
		start = p;            /*   {      (for editor) */
		while (*p && !strchr("\"'`$}\t\n ", *p))
		    p++;
		break;
	    }
	    savechar = *p;
	    *p = '\0';
	    if (!strcmp("val", start)) {
		if (name == 0) {
		    fprintf(stderr,"You can only use \"$val\" in an action\n");
		    exit(-1);
		}
		ret = get_string(name);
	    }
	    else { 
		ret = get_string(start);
	    }
	    *p = savechar;
	    buf_cat (buffer, ret, strlen(ret));
	    free(ret);
	    if (leftparen)  /* skip right paren */
		p++;
	    start = p;
	    break;
	}  /* switch */
    }      /* while */
    if (buffer == evalbuf) {  /* still inside backquotes */
	if (name != 0) {
	    fprintf(stderr, "In action for object %s\n", name);
	}
	fprintf(stderr, "expand_string(), couldn't find matching backquote\nstarting here->`%s\n", start);
    }

    /* tack on remainder of string */

    if (start < p)
	buf_cat(buffer, start, p - start);

    ret = buf_fetch(buffer);
    buf_free(buffer);
    return ret;
}
