/* edit.h -- Data structures for the editor (buffers, marks, etc...)
   Copyright (C) 1993, 1994 John Harper <jsh@ukc.ac.uk>

   This file is part of Jade.

   Jade is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   Jade is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Jade; see the file COPYING.	If not, write to
   the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifndef _EDIT_H
#define _EDIT_H

/*
 * Line structure -- an array of these is in the TX->tx_Lines
 */
typedef struct LINE {
    u_char	   *ln_Line;
    long	    ln_Strlen;	/* includes '\0' */
} LINE;

/*
 * Structure to uniquely identify a character position
 */
typedef struct POS {
    long	    pos_Col, pos_Line;
} POS;

#define POS_EQUAL_P(s,e) \
    (((s)->pos_Line == (e)->pos_Line) && ((s)->pos_Col == (e)->pos_Col))
#define POS_GREATER_P(s,e) \
    (((s)->pos_Line > (e)->pos_Line) \
     || (((s)->pos_Line == (e)->pos_Line) && ((s)->pos_Col > (e)->pos_Col)))
#define POS_GREATER_EQUAL_P(s,e) \
    (((s)->pos_Line > (e)->pos_Line) \
     || (((s)->pos_Line == (e)->pos_Line) && ((s)->pos_Col >= (e)->pos_Col)))
#define POS_LESS_P(s,e) \
    (((s)->pos_Line < (e)->pos_Line) \
     || (((s)->pos_Line == (e)->pos_Line) && ((s)->pos_Col < (e)->pos_Col)))
#define POS_LESS_EQUAL_P(s,e) \
    (((s)->pos_Line < (e)->pos_Line) \
     || (((s)->pos_Line == (e)->pos_Line) && ((s)->pos_Col <= (e)->pos_Col)))

/*
 * Each bookmark has one of these in the tx_Marks list
 */
typedef struct _Mark {
    u_char	    mk_Type;
    bool	    mk_Resident;

    /* When the file is resident this node is linked into its tx_Marks list,
       otherwise it's in a list of all non-resident marks.  */
    struct _Mark   *mk_Next;

    /* next allocated MARK  */
    struct _Mark   *mk_NextAlloc;

    VALUE	    mk_Pos;

    /* This union tells me where to look for the file this mark is in.
       if (mk_Resident == 0) then the file (mk_File.name) has to be loaded and
       used. Otherwise (mk_File.tx) is used.  */
    union {
	VALUE		name;
	/* this TX should not be marked for gc */
	struct _TX     *tx;
    }		    mk_File;
} Mark;

/*
 * A buffer, strangely called `TX'
 */
typedef struct _TX {
    u_char	    tx_Type;
    u_char	    tx_Flags;
    u_char	    tx_Pad1, tx_Pad2;

    struct _TX	   *tx_Next;
    Mark	   *tx_MarkChain;
    LINE	   *tx_Lines;
    long	    tx_NumLines;
    VALUE	    tx_FileName;
    VALUE	    tx_BufferName;
    VALUE	    tx_ModeName;
    VALUE	    tx_MinorModeNameList;
    VALUE	    tx_MinorModeNameString;
    long	    tx_Changes;
    long	    tx_AutoSaveInterval; /* seconds between saves */
    long	    tx_LastSaveTime;	 /* time at last save (auto or user) */
    long	    tx_LastSaveChanges;	 /* changes at last save (any type) */
    long	    tx_ProperSaveChanges; /* changes at last `proper' save */

    long	    tx_TabSize;

    /* Section of buffer which may have changed since last refresh.  */
    POS		    tx_ModStart, tx_ModEnd;
    /* How many more lines in the above area than at the last refresh.	*/
    long	    tx_ModDelta;
    /* `tx_Changes' at last refresh.  */
    long	    tx_LastChanges;

    VALUE	    tx_LocalVariables; /* alist of (SYMBOL . VALUE) */
    VALUE	    tx_GlyphTable;

    /* Undo information */
    VALUE	    tx_UndoList;
    VALUE	    tx_ToUndoList;
    VALUE	    tx_UndoneList;

    /* Saved state for buffers which are not being displayed.  */
    POS		    tx_SavedCPos;
    POS		    tx_SavedWPos;
    POS		    tx_SavedBlockPos[2];
    char	    tx_SavedBlockStatus;
} TX;

/* For tx_Flags */
#define TXFF_RDONLY   1	    /* No modifications to file */
#define TXFF_SPECIAL  2	    /* No mod flag, buffer never killed. */
#define TXFF_REFRESH_ALL 4  /* *All* buffer has changed. */
#define TXFF_NO_UNDO  8	    /* No recording of undo information */

/*
 * Each window is like this
 */
typedef struct _VW
{
    u_char	    vw_Type;
    u_char	    vw_Flags;
    u_char	    vw_Pad1, vw_Pad2;

    struct _VW	   *vw_Next;
    TX		   *vw_Tx;

    /* Data that the window-system needs.  */
    VW_WindowSys    vw_WindowSys;

    /* Cursor positioning data.  */
    POS		    vw_CursorPos;
    u_long	    vw_LastCursorOffset; /* number of glyphs from col 0 */
    POS		    vw_LastCursorPos;
    u_long	    vw_LastCursorChanges;
    TX		   *vw_LastCursorTx;

    POS		    vw_DisplayOrigin;
#define vw_StartLine vw_DisplayOrigin.pos_Line
#define vw_StartCol vw_DisplayOrigin.pos_Col

    POS		    vw_BlockS, vw_BlockE;
    /* 0=block marked, 1=start marked, 2=end marked, -1=none marked */
    char	    vw_BlockStatus;

    u_char	   *vw_Message;
    u_long	    vw_MessageLen;
    u_long	    vw_LastClickMics;

    int		    vw_MaxX, vw_MaxY;
    int		    vw_XStartPix, vw_YStartPix;
    int		    vw_XEndPix, vw_YEndPix;
    int		    vw_XWidthPix, vw_YHeightPix;
    short	    vw_XStep, vw_YStep;
    short	    vw_XStepRatio, vw_YStepRatio;

    /* Position at which to draw the separator line and the base-line
       for the message text.  */
    int		    vw_MessageLineY;
    int		    vw_MessageFontY;

    VALUE	    vw_FontName;
    short	    vw_FontStart;
    short	    vw_FontX, vw_FontY;

    TX		   *vw_LastRefTx;
    POS		    vw_LastDisplayOrigin;
    short	    vw_DeferRefresh;
    u_short	    vw_MaxScroll;

#ifndef NOSCRLBAR
    ScrollBar	    vw_SBar;
#endif

    /* List of buffers accessible in this window.  This is not used by the
       C code at all; access is via the `buffer-list' variable.  */
    VALUE	    vw_BufferList;
} VW;

/* For vw_Flags	 */
#define VWFF_RECTBLOCKS     1	/* mark rectangular blocks */
#define VWFF_FORCE_REFRESH  2	/* full redraw next time */
#define VWFF_REFRESH_BLOCK  4	/* redraw the block */
#define VWFF_REFRESH_STATUS 8	/* redraw the status line */
#define VWFF_SLEEPING	    16	/* window is iconified */
#define VWFF_MESSAGE	    32	/* a message is currently displayed */
#define VWFF_STATUS_CURS    64	/* cursor is draw at end of status msg */

#define CURS_ON	 1
#define CURS_OFF 0

#endif /* _EDIT_H */
