/*
 * grid.c: functions for grid type
 */

#include <malloc.h>
#include <stdio.h>

#include "grid.h"

#define DELTA 	3

extern int GridColor;

static void warning(char* s)
	{
	fprintf(stderr,"Grid error: %s\n",s);
	}

/*
 * Allocation function for a grid
 */

grid new_grid(int width,int height,int xgran,int ygran)
	{
	grid g;
	int Size,i,x,y;
	int stepx,stepy;

	if((g=(grid)malloc(sizeof(struct GRID)))==NULL)
		return(NULL);
	g->GWidth=width;
	g->GHeight=height;
	g->XGran=xgran;
	g->YGran=ygran;
	
	/* Let's allocate the grid as a matrix of areas */

	if((g->theGrid=(AreaRow*)malloc(ygran*sizeof(AreaRow*)))==NULL)
		{
		warning("Unable to allocate memory required.");
		return(NULL);
		}
	Size=xgran*sizeof(struct AREA);/* Size of a row of areas */
	for(i=0;i<ygran;i++)
		if((g->theGrid[i]=(AreaRow)malloc(Size))==NULL)
			{
			warning("Unable to allocate memory required.");
			return(NULL);
			}	
	g->CPCnt=(xgran-1)*(ygran-1);

	/* Let's allocate the crosspoint vector */

	Size=(g->CPCnt)*sizeof(struct POINT);
	if((g->CrossPoint=(Point*)malloc(Size))==NULL)
		{
		warning("Unable to allocate memory required.");
		return(NULL);
		}
	
	/* Let's build a regular grid */

	stepx=width/xgran;
	stepy=height/ygran;
	for(y=0;y<ygran;y++)
		for(x=0;x<xgran;x++)
			{
			g->theGrid[y][x].v[0].x=stepx*x;
			g->theGrid[y][x].v[0].y=y*stepy;
			g->theGrid[y][x].v[1].x=
				(x==xgran-1)?width-1:stepx*(x+1);
			g->theGrid[y][x].v[1].y=y*stepy;
			g->theGrid[y][x].v[2].x=
				(x==xgran-1)?width-1:stepx*(x+1);
			g->theGrid[y][x].v[2].y=
				(y==ygran-1)?height-1:stepy*(y+1);		
			g->theGrid[y][x].v[3].x=stepx*x;
			g->theGrid[y][x].v[3].y=
				(y==ygran-1)?height-1:stepy*(y+1);
			}

	/* Let's init the crosspoint vector */

	i=0;		
	for(y=0;y<ygran-1;y++)
		for(x=0;x<xgran-1;x++)
			{
			g->CrossPoint[i].x=g->theGrid[y][x].v[2].x;
			g->CrossPoint[i].y=g->theGrid[y][x].v[2].y;
			i++;
			}			
	return(g);
	}


/*
 * Function that draws an area. moveto() and lineto() are implementation
 * dependent and are external (graph.c)
 */


void gpoly(Area a)
	{
	moveto(a.v[0].x,a.v[0].y);
	lineto(a.v[1].x,a.v[1].y);
        lineto(a.v[2].x,a.v[2].y);
        lineto(a.v[3].x,a.v[3].y);
	lineto(a.v[0].x,a.v[0].y);
	}

/*
 * Show a grid
 */

void show_grid(grid g)
	{
	int x,y;
	int xm,ym;
	xm=g->XGran;
	ym=g->YGran;

	ChangeFGColor(GridColor);

	/* First area ...(0,0) */

	gpoly(g->theGrid[0][0]);

	/* First row of areas... */

	for(x=1;x<xm;x++)
		{
		moveto(g->theGrid[0][x].v[0].x,
			g->theGrid[0][x].v[0].y);
		lineto(g->theGrid[0][x].v[1].x,
                        g->theGrid[0][x].v[1].y);
                lineto(g->theGrid[0][x].v[2].x,
                        g->theGrid[0][x].v[2].y);
                lineto(g->theGrid[0][x].v[3].x,
                        g->theGrid[0][x].v[3].y);
		}

	/* Other rows of areas ... */

	for(y=1;y<ym;y++)
		{

		/* First area ... */

		moveto(g->theGrid[y][0].v[1].x,
			g->theGrid[y][0].v[1].y);
		lineto(g->theGrid[y][0].v[2].x,
                        g->theGrid[y][0].v[2].y);	
                lineto(g->theGrid[y][0].v[3].x,
                        g->theGrid[y][0].v[3].y);
                lineto(g->theGrid[y][0].v[0].x,
                        g->theGrid[y][0].v[0].y);

		/* Other areas... */

		for(x=1;x<xm;x++)
			{
			moveto(g->theGrid[y][x].v[1].x,
				g->theGrid[y][x].v[1].y);
			lineto(g->theGrid[y][x].v[2].x,
                                g->theGrid[y][x].v[2].y);
			lineto(g->theGrid[y][x].v[3].x,
                                g->theGrid[y][x].v[3].y);
			}
		}

	/* Hightlight the crosspoints... */

	for(x=0;x<g->CPCnt;x++)
		{
		ChangeFGColor(!GridColor);
		putpixel(g->CrossPoint[x].x,g->CrossPoint[x].y);
		}
	}

/*
 * Returns >=0 if point p lies inside a square of size DELTA cenetered
 * in a crosspoint. In this case the value returned is the index of the 
 * crosspoint in the crosspoint vector
 */

int hit_point(grid g,Point p)
	{
	int i;

	for(i=0;i<g->CPCnt;i++)
		{
		if(p.x>=(g->CrossPoint[i].x-DELTA) &&
		      p.x<=(g->CrossPoint[i].x+DELTA))
			if(p.y>=(g->CrossPoint[i].y-DELTA) &&
				p.y<=(g->CrossPoint[i].y+DELTA))
				return(i);

				/* Returns the index */

		}
	return(-1);
	}

/*
 * Deallocation function for a grid
 */

void free_grid(grid g)
	{
	if(g!=NULL)	
		{
		free(g->theGrid);
		free(g->CrossPoint);
		free(g);
		}
	}

/*
 * Moves a crosspoint 
 */


void move_hit_point(grid g,Point NewPnt,int index)
	{
	Area ULArea,URArea,LLArea,LRArea;
	int ULXIndex,ULYIndex;
	Point NPnt,EPnt,SPnt,WPnt,OldPnt;

	/* Let's save the old crosspoint... */

	OldPnt.x=g->CrossPoint[index].x;
	OldPnt.y=g->CrossPoint[index].y;
	
	/* From the index of the crosspoint derive the four areas 
  	   to modify */

	ULXIndex=index%(g->XGran-1);
	ULYIndex=index/(g->XGran-1);
	ULArea=g->theGrid[ULYIndex][ULXIndex];
	URArea=g->theGrid[ULYIndex][ULXIndex+1];
	LLArea=g->theGrid[ULYIndex+1][ULXIndex];
	LRArea=g->theGrid[ULYIndex+1][ULXIndex+1];

	/* We have to redraw 4 segments; one vertex is the crosspoint,
	   the others derive from ULArea and LRArea... */

	NPnt.x=ULArea.v[1].x;
	NPnt.y=ULArea.v[1].y;
	EPnt.x=LRArea.v[1].x;
	EPnt.y=LRArea.v[1].y;
	SPnt.x=LRArea.v[3].x;
	SPnt.y=LRArea.v[3].y;
	WPnt.x=ULArea.v[3].x;
	WPnt.y=ULArea.v[3].y;

	/* Storing new crosspoint vector */

	g->CrossPoint[index].x=NewPnt.x;
	g->CrossPoint[index].y=NewPnt.y;

	/* Let's update the new vertex in the four areas */

	g->theGrid[ULYIndex][ULXIndex].v[2].x=NewPnt.x; /* ULArea */
	g->theGrid[ULYIndex][ULXIndex].v[2].y=NewPnt.y;
	g->theGrid[ULYIndex][ULXIndex+1].v[3].x=NewPnt.x; /* URArea */
	g->theGrid[ULYIndex][ULXIndex+1].v[3].y=NewPnt.y;
	g->theGrid[ULYIndex+1][ULXIndex].v[1].x=NewPnt.x; /* LLArea */
	g->theGrid[ULYIndex+1][ULXIndex].v[1].y=NewPnt.y;
	g->theGrid[ULYIndex+1][ULXIndex+1].v[0].x=NewPnt.x; /* LRArea */
	g->theGrid[ULYIndex+1][ULXIndex+1].v[0].y=NewPnt.y;

	}	

/*
 * Save a grid onto a file
 */

int save_grid(grid g,char* filename)
	{
	FILE* f;		
	int x,y;
	
	if(g==NULL) return(0);

	/* Open the file */

	if((f=fopen(filename,"w"))==NULL)
		return(0);

	/* Write GWidth,GHeight,XGran,YGran */

	fprintf(f,"%d,%d,%d,%d\n",g->GWidth,g->GHeight,g->XGran,g->YGran);
	
	/* then write every area as v[0].x,v[0].y,v[1].x ... */

	for(y=0;y<g->YGran;y++)		
		for(x=0;x<g->XGran;x++)
			fprintf(f,"%d,%d,%d,%d,%d,%d,%d,%d\n",
				g->theGrid[y][x].v[0].x,
				g->theGrid[y][x].v[0].y,
				g->theGrid[y][x].v[1].x,
				g->theGrid[y][x].v[1].y,
				g->theGrid[y][x].v[2].x,
				g->theGrid[y][x].v[2].y,
				g->theGrid[y][x].v[3].x,
				g->theGrid[y][x].v[3].y);
	fclose(f);
	return(1);

	/* returns 1 if OK */

	}

/*
 * Read a grid from a file 
 */

grid load_grid(char* filename)
	{
	FILE *f;
	grid g;
	int x,y,i;
	int gw,gh,xg,yg;
	int v0x,v0y,v1x,v1y,v2x,v2y,v3x,v3y;

	/* Open the file */

	if((f=fopen(filename,"r"))==NULL)
		return(NULL);

	/* Read dimensions and granularity */

	fscanf(f,"%d,%d,%d,%d",&gw,&gh,&xg,&yg);
	if(gw<=0||gh<=0||xg<=1||yg<=1)
		return(NULL);

	/* Allocate a new grid */

	if((g=new_grid(gw,gh,xg,yg))==NULL)
		return(NULL);

	/* Read each area */

        for(y=0;y<g->YGran;y++)
                for(x=0;x<g->XGran;x++)
			{
			fscanf(f,"%d,%d,%d,%d,%d,%d,%d,%d",
				&v0x,&v0y,&v1x,&v1y,&v2x,&v2y,&v3x,&v3y);
			g->theGrid[y][x].v[0].x=v0x;
			g->theGrid[y][x].v[0].y=v0y;
			g->theGrid[y][x].v[1].x=v1x;
			g->theGrid[y][x].v[1].y=v1y;
			g->theGrid[y][x].v[2].x=v2x;
			g->theGrid[y][x].v[2].y=v2y;
			g->theGrid[y][x].v[3].x=v3x;
			g->theGrid[y][x].v[3].y=v3y;
			}

	/* Update the crosspoint vector */

        i=0;            
        for(y=0;y<(yg-1);y++)
                for(x=0;x<(xg-1);x++)
                        {
                        g->CrossPoint[i].x=g->theGrid[y][x].v[2].x;
                        g->CrossPoint[i].y=g->theGrid[y][x].v[2].y;
                        i++;
                        }                       
	fclose(f);
	return(g);
	}

/*
 * Copy a grid into another
 */


grid copy_grid(grid s)
	{
	grid g;
	int x,y,i;

	/* Allocate a new grid */

	if((g=new_grid(s->GWidth,s->GHeight,
			s->XGran,s->YGran))==NULL)
		return(NULL);

	/* Copy the matrix of areas */

        for(y=0;y<g->YGran;y++)
                for(x=0;x<g->XGran;x++)
			{
			g->theGrid[y][x]=s->theGrid[y][x];
			}

	/* Update the crosspoint vector */

        i=0;            
        for(y=0;y<(g->YGran-1);y++)
                for(x=0;x<(g->XGran-1);x++)
                        {
                        g->CrossPoint[i].x=g->theGrid[y][x].v[2].x;
                        g->CrossPoint[i].y=g->theGrid[y][x].v[2].y;
                        i++;
                        }                       
	return(g);
	}

