static char rcsid[] = "@(#)$Id: builtin.c,v 2.6.2.1 1995/01/26 04:48:03 peter Exp $";
/*
 * builtin.c - make builtin objects 

 * Copyright 1994 and 1995, 26th January.
 * By Peter Chang
 * peterc@a3.ph.man.ac.uk

 * See notice in misc.h for details of permissions and warranty of this
 * software.

 *
 * $Log: builtin.c,v $
 * Revision 2.6.2.1  1995/01/26  04:48:03  peter
 * Guess #segments in various builtins (for new findedge()).
 *
 * Revision 2.6  1994/11/18  04:03:03  peter
 * xpgs-2.5-patch 01: Changes in header files for clean compile
 *
 * Revision 2.5.1.1  1994/11/17  03:34:26  peter
 * Import of actual public release of xpgs-2.5: lots of cosmetic changes to docs
 *
 * Revision 2.5  1994/11/16  09:19:19  peter
 * Putting xpgs-2.5 in trunk.
 *
 * Revision 2.0.1.1  1994/11/16  09:10:30  peter
 * Import of xpgs-2.5: archive of new release to alt.sources (11/94)
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "polyh.h"

#if __GNUC__ == 2
USE(rcsid);
#endif

static const char *ppnames[] = { /* preset names */
   "triangle",
   "square",
   "box",
   "cylinder",
   "cone",
   "torus",
   "sphere",
   "spring",
   NULL
};

static const char *penames[] = { /* equivalent specifications */
   "polygon",
   "polygon(4)",
   "prism(4)",
   "prism(36)",
   "pyramid(36)",
   "spiral(36)",
   "spiral(36,-9,10,1,0.0)",
   "spiral(18,0,18,1,2.0,3.0,18,54,1)",
   NULL
};

static const char *pbnames[] = { /* builtin names */
   "polygon",
   "pyramid",
   "prism",
   "spiral",
   NULL
};

#ifdef BI_TEST
Object *shapes = NULL;
usint nobj = 0;
float theta = 0.0, phi = 0.0;

int main(int argc, char *argv[])
{
   Object net;
   char name[NLEN];
   int i = 0;
   int (*pfun)(Object *);

   int (*getbuiltin(const char *, int ))(Object *);

   strcpy(name, argv[++i]);

   pfun = (*getbuiltin)(argv[++i], 1);

   if (pfun != NULL) {
      i = (*pfun)(&net);

      sprintf(net.fname, "%s.polyh", name);
      savepolyh(&net, 0);
      printf("saved %s which has %d pts, %d polys with %d indices\n",
	     net.fname, net.ipts-1, i, net.nindex);
      free(net.ri);
      free(net.polygon);
      exit(EXIT_SUCCESS);
   } else exit(EXIT_FAILURE);
}
#endif


/* most complicated function definition I've done (so far) */
/*
 * should be a function (of two args - char *, int) returning
 * a pointer to a function (of one arg - Object *) returning int
 * 
 * From: cdecl
 * cdecl> declare getbuiltin as function (cstring as pointer to char, verbose
 *        as int) returning pointer to function (pobj as pointer to float)
 *        returning int
 * int (*getbuiltin(char *cstring, int verbose))(float *pobj) { }
 */
static int ppolygon(Object *);
static int ppyramid(Object *);
static int pprism(Object *);
static int pspiral(Object *);

int (*getbuiltin(const char *cstring, int verbose))(Object *)
{
   char fname[NLEN];
   const char **sptr;
   const char *ccstring;
   int w, pbno;

   /* is it a preset */
   w = strlen(cstring);
   for (sptr = ppnames, pbno = 0; *sptr != NULL; sptr++, pbno++) {
      if (!strncmp(cstring, *sptr, w)) break;
   }
   ccstring = (*sptr != NULL) ? penames[pbno] : cstring;

   pparse(ccstring, fname);

   w = strlen(fname);
   for (sptr = pbnames, pbno = 0; *sptr != NULL; sptr++, pbno++) {
      if (!strncmp(fname, *sptr, w)) break;
   }
#ifdef BI_TEST
   printf("function name is %s\n   it has %d parameters", fname, parno);
   if (parno > 0) {
      printf(" which are:");
      for (w=0; w<parno; w++) printf("\n   %d: %f %d", w, parameter[w],
				     (int) parameter[w]);
   }
   printf("\n");
#endif
   switch(pbno) {
    case 0:
      return ppolygon;
    case 1:
      return ppyramid;
    case 2:
      return pprism;
    case 3:
      return pspiral;
    default:
      if (verbose) printf("%s is not built-in\n", fname);
      return NULL;
   }
}


/*
 * regular polygon centred at origin
 *  pdiv pbeg pend chord
 */
static int ppolygon(Object *pobj)
{
   int pdiv, pbeg, pend;
   int i, chord;
   float lphi, pinc;
   float r;

   chord = 0;
   if (parno < 1) {
      pbeg = 0;
      pend = 3;
      pdiv = 3;
   } else {
      pdiv = (int) parameter[0];
      pbeg = 0;
      pend = pdiv;
      i = (parno > 4) ? 4 : parno;
      switch (i) { /* deliberate fall through */
	 /*@ -casebreak */
       case 4:
	 chord = (int) parameter[3];
       case 3:
	 pbeg = (int) parameter[2];
       case 2:
	 pend = (int) parameter[1];
       default:
	 break;
	 /*@ =casebreak */
      }
   }

   if (pbeg > pend) {
      i = pbeg; pbeg = pend; pend = i;
   }
   if (pend == pbeg) return 0;
   
   if (pdiv < 3) return 0;

   pinc = 2*PI/pdiv;
   lphi = pbeg*pinc;

   pobj->ipts = pend - pbeg;
   if (pobj->ipts > pdiv) {
      pobj->ipts = pdiv;
      pend = pbeg + pdiv;
   }
   if (pobj->ipts == pdiv) chord = 1;

   if (chord == 0) pobj->ipts += 2;
   else pobj->ipts++;
   pobj->nindex = pobj->ipts;

   DIEIF((pobj->ri = (Point *) calloc(pobj->ipts, sizeof(Point))) == NULL,
	 "Too many points, unable to allocate memory\n");
   DIEIF((pobj->polygon = (usint *) calloc(pobj->nindex, sizeof(int))) == NULL,
	 "Unable to allocate memory for polygons\n");

   r = 1.0;
   if (pdiv == 4 && pbeg == 0 && pend == 4) {
      lphi += PI/4;
      r = sqrt(0.5);
   }
   i = 1;
   if (chord == 0) {
      pobj->ri[i].x = 0.0;
      pobj->ri[i].y = 0.0;
      pobj->ri[i].z = 0.0;
      i++;
   }
   for (; i<pobj->ipts; i++) {
      pobj->ri[i].x = r*cos(lphi);
      pobj->ri[i].y = r*sin(lphi);
      pobj->ri[i].z = 0.0;
      lphi += pinc;
   }


   pobj->polygon[0] = pobj->ipts-1;
   for (i=1; i<pobj->nindex; i++) pobj->polygon[i] = i;
   pobj->nlines = pobj->nindex-1;

   return 1;
}


/*
 * polygon-based pyramid with peak at (0,0,1)
 */
static int ppyramid(Object *pobj)
{
   Object tobj;
   int i, k;
   int f;

   if ((f = ppolygon(&tobj)) == 0) return 0;

   pobj->ipts = tobj.ipts + 1;
   pobj->nindex = tobj.nindex + 4*tobj.ipts - 4;
   pobj->nlines = 2*tobj.nlines;

   DIEIF((pobj->ri = (Point *) calloc(pobj->ipts, sizeof(Point))) == NULL,
	 "Too many points, unable to allocate memory\n");
   DIEIF((pobj->polygon = (usint *) calloc(pobj->nindex, sizeof(int))) == NULL,
	 "Unable to allocate memory for polygons\n");

   for (i=1; i<tobj.ipts; i++) {
      pobj->ri[i].x = tobj.ri[i].x;
      pobj->ri[i].y = tobj.ri[i].y;
      pobj->ri[i].z = 0.0;
   }
   pobj->ri[i].x = 0.0;
   pobj->ri[i].y = 0.0;
   pobj->ri[i].z = 1.0;
   free(tobj.ri);
   free(tobj.polygon);
   pobj->polygon[0] = tobj.ipts - 1;
   for (i=1; i<tobj.nindex; i++) pobj->polygon[i] = i;
   for (k=1; k<tobj.ipts; k++) {
      pobj->polygon[i++] = 3;
      f++;
      pobj->polygon[i++] = tobj.ipts;
      pobj->polygon[i++] = k;
      pobj->polygon[i++] = k+1;
   }
   pobj->polygon[i-1] = 1;

   return f;
}


/*
 * wraps a rectangular net round an array of points
 */
static int wrappolygon(int num, int pts, usint *polygon, int lid)
{
   int f;
   int i, j, k;
   int l, m, ol, om;
   usint *poly;

   poly = polygon;
   /* base */
   *poly++ = pts-1;
   for (i=1; i<pts; i++) *poly++ = i;
   f = 1;
   l = 1;
   m = pts;
   for (j=0; j<num; j++) {
      ol = l;
      om = m;
      for (k=2; k<pts; k++) {
	 *poly++ = 4;
	 f++;
	 *poly++ = l;
	 *poly++ = l+1;
	 *poly++ = m+1;
	 *poly++ = m;
	 l++;
	 m++;
      }
      *poly++ = 4;
      f++;
      *poly++ = l++;
      *poly++ = ol;
      *poly++ = om;
      *poly++ = m++;
   }
   if (lid == 1) { /* lid */
      *poly++ = pts-1;
      f++;
      for (k=1; k<pts; k++) *poly++ = l++;
   }

   return f;
}


/*
 * polygon-based prism
 *  zdiv zno lid
 */
static int pprism(Object *pobj)
{
   Object tobj;
   float z, zinc;
   int zno, zdiv;
   int i, j, k;
   int f;
   int lid;

   if ((f = ppolygon(&tobj)) == 0) return 0;

   lid = 1;
   if (parno < 5) {
      zdiv = 1;
      zno = zdiv;
   } else {
      zdiv = (int) parameter[4];
      zno = (parno>5) ? (int) parameter[5] : zdiv;
      if (parno>6) lid = (int) parameter[6];
   }

   if (zdiv == 0) return 0;
   zinc = 1.0/zdiv;
   z = 0.0;
   
   pobj->ipts = (zno+1)*(tobj.ipts-1) + 1;
   if (zdiv == zno) lid = 1;
   if (lid == 0) pobj->nindex = tobj.nindex + 5*zno*(tobj.ipts-1);
   else pobj->nindex = 2*tobj.nindex + 5*zno*(tobj.ipts-1);

   DIEIF((pobj->ri = (Point *) calloc(pobj->ipts, sizeof(Point))) == NULL,
	 "Too many points, unable to allocate memory\n");
   DIEIF((pobj->polygon = (usint *) calloc(pobj->nindex, sizeof(int))) == NULL,
	 "Unable to allocate memory for polygons\n");

   k = 1;
   for (j=0; j<=zno; j++) {
      for (i=1; i<tobj.ipts; i++) {
	 pobj->ri[k].x = tobj.ri[i].x;
	 pobj->ri[k].y = tobj.ri[i].y;
	 pobj->ri[k].z = z;
	 k++;
      }
      z += zinc;
   }
   free(tobj.ri);
   free(tobj.polygon);
   f = wrappolygon(zno, tobj.ipts, pobj->polygon, lid);
   pobj->nlines = (2*zno+1)*(tobj.ipts-1);
   return f;
}


/*
 * spiral or torus with polygonal cross-section
 *  xs yinc adiv ano lid
 */
static int pspiral(Object *pobj)
{
   Object tobj;
   float x, alpha, ainc;
   float xs, ys, yinc;
   int ano, adiv;
   int i, j, k;
   int f;
   int lid;

   if ((f = ppolygon(&tobj)) == 0) return 0;

   xs = 2.0;
   yinc = 0.0;
   adiv = 36;
   lid = 0;
   if (parno < 5) {
      ano = adiv;
   } else {
      xs = parameter[4];
      if (parno>5) yinc = parameter[5];
      if (parno>6) adiv = (int) parameter[6];
      ano = (parno>7) ? (int) parameter[7] : adiv;
      if (parno>8) lid = (int) parameter[8];
   }

   if (adiv == 0) return 0;
   ainc = 2.0*PI/adiv;
   alpha = 0.0;
   yinc = yinc/adiv;
   ys = 0.0;

   pobj->ipts = (ano+1)*(tobj.ipts-1) + 1;
   if (adiv == ano) lid = 1;
   if (lid == 0) pobj->nindex = tobj.nindex + 5*ano*(tobj.ipts-1);
   else pobj->nindex = 2*tobj.nindex + 5*ano*(tobj.ipts-1);
   DIEIF((pobj->ri = (Point *) calloc(pobj->ipts, sizeof(Point))) == NULL,
	 "Too many points, unable to allocate memory\n");
   DIEIF((pobj->polygon = (usint *) calloc(pobj->nindex, sizeof(int))) == NULL,
	 "Unable to allocate memory for polygons\n");

   k = 1;
   for (j=0; j<=ano; j++) {
      for (i=1; i<tobj.ipts; i++) {
	 x = tobj.ri[i].x + xs;
	 pobj->ri[k].x = -x*sin(alpha);
	 pobj->ri[k].y = tobj.ri[i].y + ys;
	 pobj->ri[k].z = x*cos(alpha);
	 k++;
      }
      alpha += ainc;
      ys += yinc;
   }
   free(tobj.ri);
   free(tobj.polygon);
   f = wrappolygon(ano, tobj.ipts, pobj->polygon, lid);
   pobj->nlines = (2*ano+1)*(tobj.ipts-1);
   return f;
}
