static char rcsid[] = "@(#)$Id: cvtopolyh.c,v 2.6 1994/11/18 04:03:03 peter Exp $";
/*
 *  cvtopolyh - a program to convert various 3D data formats to the AVS
 *            polyh form.

 * Copyright 1994, 16th November.
 * Copyright 1994, 14th February.
 * By Peter Chang
 * peterc@v2.ph.man.ac.uk

 * See notice in misc.h for details of permissions and warranty of this
 * software.

 *
 * $Log: cvtopolyh.c,v $
 * Revision 2.6  1994/11/18  04:03:03  peter
 * xpgs-2.5-patch 01: Changes in header files for clean compile
 *
 * Revision 2.5.1.1  1994/11/17  03:34:24  peter
 * Import of actual public release of xpgs-2.5: lots of cosmetic changes to docs
 *
 * Revision 2.5  1994/11/16  09:19:19  peter
 * Putting xpgs-2.5 in trunk.
 *
 * Revision 2.0.1.1  1994/11/16  09:10:31  peter
 * Import of xpgs-2.5: archive of new release to alt.sources (11/94)
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "polyh.h"

#if __GNUC__ == 2
USE(rcsid);
#endif

#define  FERROR (1.0e-5)   /* absolute error in compare */ 

/* change this values if program gets too big */
#define SIDES    20

#define MLEN 360

Object *shapes = NULL;
usint  nobj = 0;
float theta, phi;

int main(int argc, char *argv[])
{
   char  name[NLEN], oname[NLEN];
   int   i, j, k;
   float max, rad;
   float xmin, xmax, ymin, ymax, zmin, zmax;
   static const char usage1[] = 
          "usage: cvtopolyh <options> <object file> <polyh file>";
   static const char usage2[] = " try cvtopolyh -help for list of options";

   int   npts, s;
   usint *same = NULL;
   Point *rn = NULL;
   Point *rt;
 
   int load3d(Object *, char );
   int loadobj(Object *);
   int loadwobj(Object *);

   if (argc == 1 || (argc < 4 && argv[1][1] != 'h')) {
      printf("no options or not enough parameters\n");
      printf("%s\n%s\n", usage1, usage2);
      exit(EXIT_FAILURE);
   }
   if (argc == 4) {
      strcpy(name, argv[2]);
      strcpy(oname, argv[3]);
   }
   switch (argv[1][1]) {
    case 'p': case '3': case 'b': case 'o': case 'w':
      break;
    case 'h':
      printf("%s\n", usage1);
      printf("Options:\n");
      printf(" -p[c]         load AVS polyh file\n");
      printf(" -3[c]    load a certain 3d format\n");
      printf(" -b[c]       ....with bounding box\n");
      printf(" -o[c]        load X3d object file\n");
      printf(" -w[c]  load Wavefront object file\n");
      printf("\n");
      printf("Putting c after centres object.\n");
      printf("\n");
      exit(EXIT_SUCCESS);
    default:
      printf("wrong option\n");
      printf("%s\n%s\n", usage1, usage2);
      exit(EXIT_FAILURE);
   }

   nobj = 1;
   DIEIF((shapes = (Object *) calloc(1, sizeof(Object))) == NULL,
         "Unable to allocate memory for shapes array\n");

   i = 0;
   strcpy(shapes->fname, name);
   switch (argv[1][1]) {
    case 'p':
      i = loadpolyh(shapes);
      break;
    case '3':
      i = load3d(shapes, 0);
      break;
    case 'b':
      i = load3d(shapes, 1);
      break;
    case 'o':
      i = loadobj(shapes);
      break;
    case 'w':
      i = loadwobj(shapes);
      break;
   }

   printf("\nLoaded %s:\n\t %d points %d ies %d polygons\n",
     name, shapes->ipts-1, shapes->nindex, i);

   /* check for redundant/replicated points */
   /* have a counter of repeated points and an i of "same" array */
   /* the array indicates which point is the same as another */

   printf("Checking points\n");
   DIEIF((same = (usint *) calloc(shapes->ipts,sizeof(int))) == NULL,
         "Unable to allocate memory for same array\n");
   s = 0;

   for (i=1; i<shapes->ipts; i++) same[i] = i;

   rt = shapes->ri + 1;
   for (i=1; i<shapes->ipts; i++, rt++) {
     if (same[i] == i)  {     /* the original */
         xmin = rt->x;
         ymin = rt->y;
         zmin = rt->z;
         for (j=i+1; j<shapes->ipts; j++) {
            if (same[j] == j &&
		fabs(xmin - shapes->ri[j].x) < FERROR && 
                fabs(ymin - shapes->ri[j].y) < FERROR &&
                fabs(zmin - shapes->ri[j].z) < FERROR) {
               s++;
               same[j] = i;        /* the copies */
            }
         }
      }
   }

   printf("Found %d points replicated\n", s);

   /* now we replace compress the points list and replace the i's */
   if (s != 0) {
      DIEIF((rn = (Point *) calloc(shapes->ipts-s,sizeof(Point))) == NULL,
         "Unable to allocate memory for points\n");
      npts = 1;
      rt = shapes->ri + 1;
      for (i=1; i<shapes->ipts; i++, rt++) {
         if (same[i] == i) {              /* original */
            rn[npts].x = rt->x;
            rn[npts].y = rt->y;
            rn[npts].z = rt->z;
            if (npts != i) {              /* change i */
               for (j=i; j<shapes->ipts; j++) {
                  if (same[j] == i) same[j] = npts;
               }
            }
            npts++;
         }
      }
      for (j=0; j<shapes->nindex;) {      /* search and replace */
         s = shapes->polygon[j++];
         for (k=0; k<s; k++) {
            i = shapes->polygon[j++];
            if (same[i] != i) shapes->polygon[j-1] = same[i];
         }
      }
      free(shapes->ri);
      shapes->ri = rn;
      printf("Reduced points to %d (%d less)\n", npts-1, shapes->ipts-npts);
      shapes->ipts = npts;
   }
   free(same);

   /* centre points */
   if (argv[1][2] == 'c') {
      xmin = 0.0;
      xmax = 0.0;
      ymin = 0.0;
      ymax = 0.0;
      zmin = 0.0;
      zmax = 0.0;

      printf("Centring points\n");
      rt = shapes->ri + 1;
      for (i=1; i<shapes->ipts; i++, rt++) {
         MYMIN(xmin,rt->x);
         MYMAX(xmax,rt->x);
         MYMIN(ymin,rt->y);
         MYMAX(ymax,rt->y);
         MYMIN(zmin,rt->z);
         MYMAX(zmax,rt->z);
      }
      printf("Bounding box is (%7.5f, %7.5f, %7.5f)\n", xmin, ymin, zmin);
      printf("\t to (%7.5f, %7.5f, %7.5f)\n", xmax, ymax, zmax);
      xmax = (xmax+xmin)*0.5;
      ymax = (ymax+ymin)*0.5;
      zmax = (zmax+zmin)*0.5;

      printf("Centre of object is (%7.5f, %7.5f, %7.5f)\n", xmax, ymax, zmax);

      rt = shapes->ri + 1;
      for (i=1; i<shapes->ipts; i++, rt++) {
         rt->x -= xmax;
         rt->y -= ymax;
         rt->z -= zmax;
      }
   }

   /* Scale the Points */

   printf("Scaling points\n");
   max = 0.0;

   rt = shapes->ri + 1;
   for (i=1; i<shapes->ipts; i++, rt++) {
      rad = rt->x*rt->x + rt->y*rt->y + rt->z*rt->z;
      MYMAX(max,rad);
   }

   /* Normalise to 1 */

   rad = sqrt(max);
   printf("Max radius %7.5f\n", rad);
   rad = 1.0/rad;

   rt = shapes->ri + 1;
   for (i=1; i<shapes->ipts; i++, rt++) {
      rt->x *= rad;
      rt->y *= rad;
      rt->z *= rad;
   }

   strcpy(shapes->fname, oname);
   savepolyh(shapes, 1);

   killallshapes();
   return EXIT_SUCCESS;
}


/* load .wobj file */

int loadwobj(Object *pobj)
{
   FILE  *fi;
   int   nver, ver, tmp, j, k;
   int npoly = 0, ipts;
   char  line[MLEN], s[MLEN], s2[MLEN], ch[3];

   DIEIF3((fi = fopen(pobj->fname, "r")) == NULL,
          "Unable to open file: %s\n", pobj->fname);
   ipts=1;
   free(pobj->ri);
   free(pobj->polygon);
   DIEIF((pobj->ri = (Point *) calloc(100,sizeof(Point))) == NULL,
         "Unable to allocate memory for points\n");

   DIEIF((pobj->polygon = (usint *) calloc(100,sizeof(int))) == NULL,
         "Unable to allocate memory for polygons\n");

   j = 0;
   while(fgets(line, MLEN-1, fi) != NULL) {
      sscanf(line, "%2s%[^#]", ch, s);
      if ((tmp = strncmp("v", ch,2)) == 0) {
         DIEIF(sscanf(s, "%f %f %f", &pobj->ri[ipts].x,
		      &pobj->ri[ipts].y, &pobj->ri[ipts].z) !=3,
               "Expected more points\n");
         ipts++;
         if ((ipts % 100)==0) {
	    pobj->ri = (Point *) realloc(pobj->ri, (ipts+100)*sizeof(Point));
	    DIEIF(pobj->ri == NULL,
		  "Unable to reallocate memory for points\n");
         }
      } else if ((tmp = strncmp("f", ch,2))==0 ||
		 (tmp = strncmp("fo", ch,2))==0) {
         k = j++;
         if ((j % 100)==0) {
	    pobj->polygon=(usint *)realloc(pobj->polygon,(j+100)*sizeof(int));
	    DIEIF(pobj->polygon== NULL,
		  "Unable to reallocate memory for polygons\n");
         }
         nver=0;
         while((tmp = sscanf(s, "%s%[^#]", s2, s)) == 2) {
            sscanf(s2, "%d%s", &ver, s2);
            nver++;
            pobj->polygon[j++]=ver;
            if ((j % 100)==0) {
	       pobj->polygon=(usint *)realloc(pobj->polygon,
					      (j+100)*sizeof(int));
	       DIEIF(pobj->polygon== NULL,
		  "Unable to reallocate memory for polygons\n");
            }
         }
         if (tmp > 0) {
            if (sscanf(s2, "%d%s", &ver, s2) > 0) {
               pobj->polygon[j++]=ver;
               if ((j % 100)==0) {
		  pobj->polygon=(usint *)realloc(pobj->polygon,
						 (j+100)*sizeof(int));
		  DIEIF(pobj->polygon== NULL,
			"Unable to reallocate memory for polygons\n");
               }
            }
         }
         pobj->polygon[k]=nver;
         npoly++;
      }
   }
   fclose(fi);

   pobj->ipts = ipts;
   pobj->nindex = j;
   return npoly;
}


int load3d(Object *pobj, char bb)
{
   FILE  *fi;
   int   nver, i, j, k;
   int   npoly = 0, ipts;
   float t;

   DIEIF3((fi = fopen(pobj->fname, "r")) == NULL,
          "Unable to open file: %s\n", pobj->fname);

   fscanf(fi, "%d", &ipts);
   fscanf(fi, "%d", &npoly);
   ipts++;

   if (bb) {      /* bounding box */
      fscanf(fi, "%f", &t);
      fscanf(fi, "%f", &t);
      fscanf(fi, "%f", &t);
      fscanf(fi, "%f", &t);
      fscanf(fi, "%f", &t);
      fscanf(fi, "%f", &t);
   }

   free(pobj->ri);
   free(pobj->polygon);
   DIEIF((pobj->ri = (Point *) calloc(ipts, sizeof(Point))) == NULL,
         "Too many points, unable to allocate memory\n");

   DIEIF((pobj->polygon = (usint *) calloc(100,sizeof(int))) == NULL,
               "Unable to allocate memory for polygons\n");

   for (j=1; j<ipts; j++)
      fscanf(fi, "%f %f %f", &pobj->ri[j].x, &pobj->ri[j].y, &pobj->ri[j].z);

   i = 0; j = 0;
   while (fscanf(fi, "%d", &nver)==1) {
      pobj->polygon[j++] = nver;
      if ((j % 100)==0) {
	 pobj->polygon = (usint *) realloc(pobj->polygon,(j+100)*sizeof(int));
	 DIEIF(pobj->polygon == NULL,
               "Unable to reallocate memory for polygons\n");
      }
      for (k=0; k<nver; k++) {       
         fscanf(fi, "%d", &pobj->polygon[j++]);
         if ((j % 100)==0) {
	    pobj->polygon = (usint *) realloc(pobj->polygon,
					      (j+100)*sizeof(int));
	    DIEIF(pobj->polygon == NULL,
		  "Unable to reallocate memory for polygons\n");
         }
      }
      i++;
   }
   DIEIF(i < npoly, "Polygons missing\n");
   fclose(fi);

   pobj->ipts = ipts;
   pobj->nindex = j;
   return npoly;
}


/*
 * load X3d .obj file
 * Read the point list and segment list from a file.  We allocate space
 * for both lists.
 */

int loadobj(Object *pobj)
{
   int     numColors, numSegs, i, j, k, l;
   FILE    *fd;
   char    line[MLEN], s[MLEN];
   int     lineNo = 0, p, q, num, color;
   Segment *segs, segt[SIDES];
   int     nver, segno, pair, tmp;
   int     npoly = 0, ipts;

   int readLine(FILE *, char *, int *);

   DIEIF3((fd = fopen(pobj->fname, "r")) == NULL,
          "Unable to open file: %s\n", pobj->fname);

   DIEIF(readLine(fd, line, &lineNo) != 0,
	 "Number of colors not specified\n");
   sscanf(line, "%d", &numColors);

   for(i=0; i<numColors; i++) readLine(fd, line, &lineNo);

   /* Read points */

   DIEIF(readLine(fd, line, &lineNo) != 0,
	 "Number of points not specified\n");
   sscanf(line, "%d", &ipts);

   ipts++;
   free(pobj->ri);
   DIEIF((pobj->ri = (Point *) calloc(ipts, sizeof(Point))) == NULL,
         "Too many points, unable to allocate memory\n");

   for (j = 1; j<ipts; j++) {
      DIEIF3(readLine(fd, line, &lineNo) != 0,
	     "Expected point on line %d\n", lineNo);

      DIEIF3(sscanf(line, "%d %f %f %f", &num,
		    &pobj->ri[j].x, &pobj->ri[j].y, &pobj->ri[j].z) != 4,
             "Error reading point on line %d\n", lineNo);
   }

   /* Read segments */

   DIEIF(readLine(fd, line, &lineNo) != 0,
	 "Number of segments not specified\n");
   sscanf(line, "%d", &numSegs);

   DIEIF((segs = (Segment *) calloc(numSegs, sizeof(Segment))) == NULL,
         "Too many points, unable to allocate memory\n");

   for (i=0; i<numSegs; i++) {
      readLine(fd, line, &lineNo);
      DIEIF3(sscanf(line, "%d %d %d %d", &num, &color, &p, &q) != 4,
             "Error reading segment on line %d\n", lineNo);

      DIEIF3((p < 0) || (q < 0) || (p > ipts) || (q > ipts),
             "Segment value out of range on line %d\n", lineNo);
      segs[i].f = p;
      segs[i].t = q;
   }

   /* Read polygons */

   DIEIF(readLine(fd, line, &lineNo) != 0,
	 "Number of polygons not specified\n");
   sscanf(line, "%d", &npoly);

   free(pobj->polygon);
   DIEIF((pobj->polygon = (usint *) calloc(100, sizeof(int))) == NULL,
         "Too many polygons, unable to allocate memory\n");
   j = 0;
   for (i=0; i< npoly; i++) {
      DIEIF3(readLine(fd, line, &lineNo) != 0,
	"Expected polygon on line %d\n", lineNo);

      DIEIF3(sscanf(line, "%d %d %d %[^#]", &num, &color, &nver, s) != 4,
             "Error reading polygon on line %d\n", lineNo);

      pobj->polygon[j++] = nver;
      if ((j % 100)==0) {
	 pobj->polygon=(usint *) realloc(pobj->polygon,(j+100)*sizeof(int));
	 DIEIF(pobj->polygon == NULL,
	       "Unable to reallocate memory for polygons\n");
      }

      /* store relevant segments in temporary array */
      for (l=1; l<nver; l++) {
         sscanf(s, "%d %[^#]",&segno, s);
         segt[l].f = segs[segno].f+1;
         segt[l].t = segs[segno].t+1;
      }
      sscanf(s, "%d",&segno);
      segt[l].f = segs[segno].f+1;
      segt[l].t = segs[segno].t+1;

      /*
       * sort segments as vectors that go around the sides of the polygon
       * by matching endpoints of the sides
       */

      nver++;
      for (k=2; k<nver; k++) {
         pair = segt[k-1].t;
         for (l=k; l<nver && (segt[l].f!=pair); l++) ;
         if (l == nver) {
            for (l=k; l<nver && (segt[l].t!=pair); l++) ;
            if (l == nver) {
               fprintf(stderr, "Mismatch in polygon %d on line %d\n",
               i, lineNo);
               exit(EXIT_FAILURE);
            } else {
               tmp = segt[l].f;
               segt[l].f = segt[l].t;
               segt[l].t = tmp;
            }
         }
         if (k != l) {
            tmp = segt[l].f;
            segt[l].f = segt[k].f;
            segt[k].f = tmp;
            tmp = segt[l].t;
            segt[l].t = segt[k].t;
            segt[k].t = tmp;
         }
      }

      DIEIF4(segt[1].f != segt[nver-1].t,
             "Polygon %d is not closed on line %d\n", i, lineNo);

      for (k=1;k<nver; k++) {
         pobj->polygon[j++] = segt[k].f;
         if ((j % 100)==0) {
	    pobj->polygon=(usint *) realloc(pobj->polygon,
					    (j+100)*sizeof(int));
	    DIEIF(pobj->polygon == NULL,
		  "Unable to reallocate memory for polygons\n");
         }
      }
   }
   fclose(fd);
   free(segs);

   pobj->ipts = ipts;
   pobj->nindex = j;
   return npoly;
}

/*
 * Read a significant line from a file.
 */
int readLine(FILE *fd, char *line, int *lineNo)
{
   float value;

   do {
      if (fgets(line, MLEN-1, fd) == NULL) {
         if (feof(fd)) {
            return (-1);
         } else {
            fprintf(stderr, "fgets() failed on line %d\n", *lineNo);
            exit(EXIT_FAILURE);
         }
      } else {
         (*lineNo)++;
         DIEIF3(strlen(line) > MLEN-2,
                "Line too long on line %d, line may be truncated.\n", *lineNo);
      }
   } while(sscanf(line, "%f", &value) <= 0);

   return 0;
}
