static char rcsid[] = "@(#)$Id: meshfn.c,v 2.6 1994/11/18 04:03:06 peter Exp $";
/*
 * meshfn.c - a set of mesh functions for use in mkpolyh.

 * Copyright 1994, 16th November.
 * By Peter Chang. All rights reserved.
 * peterc@v2.ph.man.ac.uk

 * See notice in misc.h for details of permissions and warranty of this
 * software.

 *
 * $Log: meshfn.c,v $
 * Revision 2.6  1994/11/18  04:03:06  peter
 * xpgs-2.5-patch 01: Changes in header files for clean compile
 *
 * Revision 2.5.1.1  1994/11/17  03:34:25  peter
 * Import of actual public release of xpgs-2.5: lots of cosmetic changes to docs
 *
 * Revision 2.5  1994/11/16  09:19:23  peter
 * Putting xpgs-2.5 in trunk.
 *
 * Revision 2.0.1.1  1994/11/16  09:10:31  peter
 * Import of xpgs-2.5: archive of new release to alt.sources (11/94)
 *
 *
 */

#include <stdio.h>
#include <math.h>

#include "polyh.h"
#include "meshfn.h"

#if __GNUC__ == 2
USE(rcsid);
#endif

/*
 * a mesh function is called with three values:
 *  two floats x, y;
 *  one integer mode.
 * It should behave as follows:
 *  mode = 0
 *   x, y are values taken from a square of side 2, centred on the origin
 *   ie. -1 <= x,y <= 1
 *   a z value is returned also within -1 <= z <= 1
 *  mode = 1
 *   internal parameters are set using the integers supplied
 *   by parameter[]
 *  mode = 2
 *   a help string is set
 */

/* --------------list of mesh functions available-------------- */
/*
 * Put your function declaration here & add the name to the flist array
 * between the last name and NULL.
 * Then write your function below the notice at line 73.
 */
float gaussian(float x, float y, int mode);
float mandel(float x, float y, int mode);
float sinc(float x, float y, int mode);
float swaves(float x, float y, int mode);

float (*flist[])(float , float , int ) = {
   gaussian,
   mandel,
   sinc,
   swaves,
   NULL
};


const char *meshhelp = NULL;

/*
 * three types of parameters
 * A] 0-2) bounding box size x,y,z (floats)
 * B] 0-1) gridding numbers xdiv, ydiv (integers)
 * C] 0-) function specific numbers (floats)
 */
float bbx = 1.0, bby = 1.0, bbz = 1.0;
int xdiv = 30, ydiv = 30;

/* a value close to zero */
#define AZERO 0.001

/* --------------Roll your own mesh functions below------------ */

/*
 * Gaussian (or Normal probability density) function,
 * var = 2*sigma^2
 */
float gaussian(float x, float y, int mode)
{
   static const char lhelp[] =
   "Gaussian (or bell-shaped) function: var= 2*sigma^2";
   static float var = 1.0;

   switch(mode) {
    case 1:
      if (parno>=1) {
	 parno = 1;
	 var = parameter[0];
      }
      printf("gaussian %f\n", var);
      break;
    case 2:
      meshhelp = lhelp;
      break;
    case 0: default:
      return exp(-(x*x+y*y)/var);
   }
   return 0.0;
}


/*
 * Mandelbrot set generating function, which returns the iteration
 * value as a fraction of the iteration limit
 */
float mandel(float cx, float cy, int mode)
{
   static const char lhelp[] =
   "Mandelbrot set potential: iteration; (x,y)center; (x,y)size";
   int i;
   float nx = 0.0, ny = 0.0, tx = 0.0, ty = 0.0, tz;
   static int max = 10;
   static float xc = 0.0, yc = 0.0, xs = 1.0, ys = 1.0;

   switch(mode) {
    case 1:
      if (parno>5) parno = 5;
      switch(parno) {
	 /*@ -casebreak */
       case 5:
	 ys = parameter[4];
       case 4:
	 xs = parameter[3];
       case 3:
	 yc = parameter[2];
       case 2:
	 xc = parameter[1];
       case 1:
	 max = (int) parameter[0];
       default:
	 break;
	 /*@ =casebreak */
      }
      printf("mandel %d %f %f %f %f\n", max, xc, yc, xs, ys);
    case 2:
      meshhelp = lhelp;
      break;
    case 0: default:
      cx -= xc;
      cy -= yc;
      cx *= xs;
      cy *= ys;
      i = -1;
      do {
	 tz = tx - ty + cx;
	 ny = 2.0*nx*ny + cy;
	 nx = tz;
	 tx = nx*nx;
	 ty = ny*ny;
	 i++;
      } while((tx + ty) < 4.0 && i<max);

      return (float) i / max;
   }
   return 0.0;
}


/*
 * Sinc (or zeroth order Bessel) function,
 *  wavevector
 */
float sinc(float x, float y, int mode)
{
   static const char lhelp[] =
   "Sinc (or zeroth order Bessel) function: k = wavevector";
   static float k = 1.0;

   float r;

   switch(mode) {
    case 1:
      if (parno>1) parno = 1;
      switch(parno) {
       case 1:
	 k = parameter[0];
	 if (k < AZERO) k = 1.0;
       default:
	 break;
      }
      printf("sinc %f\n", k);
      break;
    case 2:
      meshhelp = lhelp;
      break;
    case 0: default:
      r = x*x+y*y;
      if (r < AZERO) {
	 return 1.0;
      } else {
	 r = k*sqrt(r);
	 return (sin(r)/r);
      }
   }
   return 0.0;
}


/*
 * Sinusoidal waves,
 * xl, yl wavelength in x, y direction
 * phase is relative phase between x, y
 */
float swaves(float x, float y, int mode)
{
   static const char lhelp[] = "Sinusoidal waves: (x,y)wavelength; phase";
   static float xl = 1.0, yl = 1.0, phase = 0.0;

   switch(mode) {
    case 1:
      if (parno>3) parno = 3;
      switch(parno) {
	 /*@ -casebreak */
       case 3:
	 phase = parameter[2];
       case 2:
	 yl = parameter[1];
       case 1:
	 xl = parameter[0];
       default:
	 break;
	 /*@ =casebreak */
      }
      printf("swaves %f %f %f\n", xl, yl, phase);
      break;
    case 2:
      meshhelp = lhelp;
      break;
    case 0: default:
      return (sin(2*PI*x/xl)*sin(2*PI*(y/yl+phase)));
   }
   return 0.0;
}
