static char rcsid[] = "@(#)$Id: polyh.c,v 2.6.2.1 1995/01/26 04:47:12 peter Exp $";
/*
 * polyh.c - a set of subroutines to input/output POLYH and XSS formats
 *           and manipulate object structures.

 * Copyright 1994 and 1995, 26th January.
 * By Peter Chang
 * peterc@a3.ph.man.ac.uk

 * See notice in misc.h for details of permissions and warranty of this
 * software.

 *
 * $Log: polyh.c,v $
 * Revision 2.6.2.1  1995/01/26  04:47:12  peter
 * Fixed bug in detecting copy in addpolyh().
 *
 * Guess #segments in loadpolyh().
 *
 * Much much faster edge finding routine: ~66-97% less time taken.
 *
 * New function magsaffine() to be used in xdraw.c.
 *
 * Revision 2.6  1994/11/18  04:03:09  peter
 * xpgs-2.5-patch 01: Changes in header files for clean compile
 *
 * Revision 2.5.1.1  1994/11/17  03:34:22  peter
 * Import of actual public release of xpgs-2.5: lots of cosmetic changes to docs
 *
 * Revision 2.5  1994/11/16  09:19:25  peter
 * Putting xpgs-2.5 in trunk.
 *
 * Revision 2.0.1.1  1994/11/16  09:10:29  peter
 * Import of xpgs-2.5: archive of new release to alt.sources (11/94)
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "polyh.h"

#if __GNUC__ == 2
USE(rcsid);
#endif


/* delete all of 'shapes' structure */
void killallshapes(void)
{
   int i;

   for (i=0; i<nobj; i++) killshape(i);
   free(shapes);
}


void killshape(int current)
{
   Object *tshape = &shapes[current];

/*   if (tshape->hidden == 0) free(tshape->rn); */
   if (tshape->copy == current) {
      free(tshape->ri);
      free(tshape->polygon);
      if (tshape->edge != NULL) free(tshape->edge);
   }
}


/* 
 * load .polyh file
 * return npoly or 0 if fail
 */
int loadpolyh(Object *pobj)
{
   FILE  *fi;
   int   nver, j, k;
   int   nline, npoly;
   char  oname[NLEN], tline[LLEN];
   char *path = NULL;

   if ((fi = fopen(pobj->fname, "r")) == NULL) {
      if ((path = getenv("XPGS_DIR")) != NULL) {
#ifdef VMS
	 sprintf(oname, "%s%s", path, pobj->fname);
#else
	 sprintf(oname, "%s/%s", path, pobj->fname);
#endif
	 fi = fopen(oname, "r");
      }
   }
   if (fi == NULL) {
      fprintf(stderr, "Unable to open file: %s\n", pobj->fname);
      return 0;
   }

   fgets(tline, LLEN-1, fi);
   if (*tline == '#')
      fscanf(fi, "%59s", oname);

   fscanf(fi, "%d", &pobj->ipts);
   pobj->ipts++;

   if (pobj->ri != NULL) free(pobj->ri);
   if (pobj->polygon != NULL) free(pobj->polygon);
   DIEIF((pobj->ri = (Point *) calloc(pobj->ipts, sizeof(Point))) == NULL,
         "Too many points, unable to allocate memory\n");
   DIEIF((pobj->polygon = (usint *) calloc(100,sizeof(int))) == NULL,
               "Unable to allocate memory for polygons\n");

   for (j=1; j<pobj->ipts; j++)
      fscanf(fi, "%f %f %f", &pobj->ri[j].x, &pobj->ri[j].y, &pobj->ri[j].z);

   j = 0;
   npoly = 0;
   nline = 0;
   while (fscanf(fi, "%d", &nver)==1) {
      pobj->polygon[j++] = nver;
      if ((j % 100)==0) {
	 pobj->polygon = (usint *) realloc(pobj->polygon,(j+100)*sizeof(int));
	 DIEIF(pobj->polygon == NULL,
               "Unable to reallocate memory for polygons\n");
      }
      for (k=0; k<nver; k++) {
         fscanf(fi, "%d", &pobj->polygon[j++]);
         if ((j % 100)==0) {
	    pobj->polygon=(usint *) realloc(pobj->polygon,(j+100)*sizeof(int));
	    DIEIF(pobj->polygon == NULL,
		  "Unable to reallocate memory for polygons\n");
         }
      }
      nline += nver;
      npoly++;
   }

   pobj->nindex = j;
   pobj->nlines = (nline+1)/2; /* initial guess */
   fclose(fi);
   return npoly;
}


/* save .polyh file */
void savepolyh(Object *pobj, int cflag)
{
   int  i, j;
   int  nver;
   char tline[LLEN];
   FILE *fo;

   DIEIF3((fo = fopen(pobj->fname, "w")) == NULL,
      "Unable to open file: %s\n", pobj->fname);

   if (cflag) {
      printf("\nEnter comment line: ");
      fgets(tline, LLEN-1, stdin);
      printf("\n");
   } else tline[0] = '\n';

   fprintf(fo, "#POLYH %s", tline);
   fprintf(fo, "facet\n");
   fprintf(fo, "%d\n", pobj->ipts-1);
   for (i=1; i<pobj->ipts; i++) {
      fprintf(fo, "%8.5f %8.5f %8.5f\n",
          pobj->ri[i].x, pobj->ri[i].y, pobj->ri[i].z);
   }

   for (i=0; i<pobj->nindex; i++) {
      nver = pobj->polygon[i++];
      fprintf(fo, "%d\n", nver);
      for (j=1; j<nver; j++)
         fprintf(fo, "%d ", pobj->polygon[i++]);
      fprintf(fo, "%d\n", pobj->polygon[i]);
   }

   fclose(fo);
   printf("Saved %s\n", pobj->fname);
}


/* 
 * load a given name, trying it as an .xss first
 * return 1 for xss, 0 for polyh and -1 if fail
 */
int loadshapes(char name[NLEN], int verbose)
{
   char sname[NLEN], tname[NLEN];
   char *tptr;
   int i;
   FILE *ff;

   tptr = strstr(name, ".xss");
   i = 0;
   if (tptr != NULL) { /* specified a .xss file */
      i = 1;
   } else {
      tptr = strstr(name, ".polyh");
      if (tptr != NULL) { /* specified a .polyh file */
	 i = 2;
      }
   }
   if (i > 0) {
      *tptr = '\0';
   } else {
      sprintf(sname, "%s.xss", name);
      if ((ff = fopen(sname, "r")) == NULL) {
	 i = 2;
      } else {
	 i = 1;
	 fclose(ff);
      }
   }
   strcpy(tname, name);

   switch(i) {
    case 1:
      sprintf(sname, "%s.xss", tname);
      i = loadxss(sname, verbose);
      if (verbose) {
	 if (i<0)
	    fprintf(stderr, "ERROR: couldn't load %s correctly\n", sname);
	 else {
	    printf("Loaded %s\n", sname);
	    i = 1;
	 }
      }
      break;
    case 2:
      nobj = 0;
      DIEIF((shapes = (Object *) calloc(5, sizeof(Object))) == NULL,
	    "Unable to allocate memory for shapes array\n");
      sprintf(sname, "%s.polyh", tname);
      i = addpolyh(sname, verbose);
      initobj(shapes);
      phi   = (2.0*PI*rand())/RAND_MAX;
      theta = (PI*rand())/RAND_MAX;
      break;
    default:
      i = -1;
      break;
   }

   return i;
}


/* 
 * append a polyh object to 'shapes' structure
 * return cobj or -1 if fail
 */
int addpolyh(char name[NLEN], int verbose)
{
   int cobj, i;
   Object *cshape, *tshape;
   int (*pfun)(Object *);

   int (*getbuiltin(const char *, int ))(Object *);

   cobj = nobj++;
   if ((nobj % 5) == 0) {
      shapes = (Object *) realloc(shapes, (nobj+5)*sizeof(Object));
      DIEIF(shapes == NULL, "Unable to reallocate memory for shapes array\n");
   }

   cshape = &shapes[cobj];
   strcpy(cshape->fname, name);

   tshape = &shapes[0];
   for (i=0; i<cobj; i++, tshape++) {
      if (strcmp(cshape->fname, tshape->fname) == 0
	  && tshape->copy == i) break;
   }
   if (i < cobj) {
      cshape->copy = i;
      cshape->hidden = 0;
      cshape->bbox = -1;
      cshape->ipts = tshape->ipts;
      cshape->nindex = tshape->nindex;
      cshape->ri = tshape->ri;
      cshape->polygon = tshape->polygon;
      if (verbose) printf("      %s has already been loaded (%d)\n", name, i);
   } else {
      cshape->copy = cobj;
      cshape->hidden = 0;
      cshape->bbox = -1;
      cshape->edge = NULL;
      /* try builtin first */
      if ((pfun = (*getbuiltin)(cshape->fname, verbose)) == NULL) {
	 i = loadpolyh(cshape);
      } else {
	 i = (*pfun)(cshape);
      }
      if (verbose) {
	 if (i>0) printf("      %s: %d points %d indices %d polygons\n",
			 name, tshape->ipts-1, tshape->nindex, i);
	 else fprintf(stderr, "ERROR: couldn't load %s\n", name);
      }
      if (i <= 0) {
	 nobj--;
	 return -1;
      }
   }

   return cobj;
}


/*
 * the xss file format is as follows:
 *    1) comment line beginning with a hash '#XSS2' or '#XSS1'
 *    2) global theta, phi;
 *    3) number of objects;
 *    4) each object consists of
 *         - polyh file name with extension
 *         - translation vector
 *         - transformed triad of vectors that
 *           form columns of rotation & magnification matrix
 * now redundant -magnification vector
 */

/*
 * load .xss file
 * return (number of new objects) or -1 if fail
 */
int loadxss(char name[NLEN], int verbose)
{
   DIEIF((shapes=(Object *) calloc(5,sizeof(Object))) == NULL,
         "Unable to allocate memory for shapes array\n");

   nobj = 0;
   return insertxss(name, verbose);
}


/*
 * insert .xss file
 * return tobj (number of new objects) or -1 if fail
 */
int insertxss(char name[NLEN], int verbose)
{
   int j, cobj, tobj;
   char  tline[LLEN], oname[NLEN];
   Object *tshape;
   FILE *fi;
   int verno = -1;
   float mag;

   if((fi = fopen(name, "r")) == NULL) {
      fprintf(stderr, "Unable to open file: %s\n", name);
      return -1;
   }

   fgets(tline, LLEN-1, fi);
   if (strstr(tline, "#XSS1") != NULL) verno = 0;
   if (strstr(tline, "#XSS2") != NULL) verno = 1;
   if (verno < 0) {
      fclose(fi);
      fprintf(stderr, "Wrong magic in file: %s\n", name);
      return -1;
   }
   fscanf(fi, "%f %f", &theta, &phi);
   fscanf(fi, "%d", &tobj);

   if (verbose) {
      if (nobj > 0) printf("Inserting...\n");
      else printf("Loading...\n");
   }
   for (j=0; j<tobj; j++) {
      fscanf(fi, "%59s", oname);
      cobj = addpolyh(oname, verbose);
      if (cobj < 0) {
	 fclose(fi);
	 fprintf(stderr, "ERROR: something wrong in %s\n", name);
	 return -1;
      }
      tshape = &shapes[cobj];
      fscanf(fi, "%f %f %f", &tshape->ot.rsh.x, &tshape->ot.rsh.y,
	     &tshape->ot.rsh.z);
      fscanf(fi, "%f %f %f", &tshape->ot.ro[0].x, &tshape->ot.ro[0].y,
             &tshape->ot.ro[0].z);
      fscanf(fi, "%f %f %f", &tshape->ot.ro[1].x, &tshape->ot.ro[1].y,
             &tshape->ot.ro[1].z);
      fscanf(fi, "%f %f %f", &tshape->ot.ro[2].x, &tshape->ot.ro[2].y,
             &tshape->ot.ro[2].z);
      if (verno == 0) {
	 fscanf(fi, "%f", &mag);
	 magaffine(&tshape->ot, mag, mag, mag);
      }
      reinitobj(tshape);
   }
   fclose(fi);
   return tobj;
}


void savexss(char name[NLEN], char comment[LLEN])
{
   int  i, j;
   FILE *fo;
   Object *tshape;
#ifndef VMS
   /* stuff to backup old xss file */
   char tname[NLEN];

   if ((fo = fopen(name, "r")) != NULL) {
      fclose(fo);
      strcpy(tname, name);
      strcat(tname, "~");
      DIEIF(rename(name, tname) != 0, "Error renaming file\n");
   }
#endif
   DIEIF3((fo = fopen(name, "w")) == NULL, "Unable to open file: %s\n", name);

   fprintf(fo, "#XSS2 %s\n", comment);
   fprintf(fo, "%f %f\n", theta, phi);
   tshape = &shapes[0];
   for (i=0, j=0; j<nobj; j++, tshape++) {
      if (tshape->hidden == 0) i++;
   }
   fprintf(fo, "%d\n", i);
   tshape = &shapes[0];
   for (j=0; j<nobj; j++, tshape++) {
      if (tshape->hidden == 0) {
         fprintf(fo, "\n%s\n", tshape->fname);
         fprintf(fo, "%f %f %f\n", tshape->wt.rsh.x, tshape->wt.rsh.y,
                 tshape->wt.rsh.z);
         fprintf(fo, "%f %f %f\n", tshape->wt.ro[0].x, tshape->wt.ro[0].y,
                 tshape->wt.ro[0].z);
         fprintf(fo, "%f %f %f\n", tshape->wt.ro[1].x, tshape->wt.ro[1].y,
                 tshape->wt.ro[1].z);
         fprintf(fo, "%f %f %f\n", tshape->wt.ro[2].x, tshape->wt.ro[2].y,
                 tshape->wt.ro[2].z);
/*         fprintf(fo, "%f %f %f\n", tshape->wt.rm.x, tshape->wt.rm.y,
                 tshape->wt.rm.z); */
      }
   }

   fclose(fo);
   printf("Saved %s\n", name);
}


void get_coords(Point rbb[8], Point pbb[2])
{
   int i;
   const int bbox_perm[8][3] = {
      {0,0,0}, {1,0,0}, {0,1,0}, {0,0,1},
      {1,1,0}, {0,1,1}, {1,0,1}, {1,1,1}
   };

   for (i=0; i<8; i++) {
      rbb[i].x = pbb[bbox_perm[i][0]].x;
      rbb[i].y = pbb[bbox_perm[i][1]].y;
      rbb[i].z = pbb[bbox_perm[i][2]].z;
   }
}


/* find the corners of the bounding box in the objects to speed up drawing */
void findbbox(int current, Object *pobj, int verbose)
{
   int i;
   Point pt[2];
   Point *a, *b, *c;

   a = pt;
   b = a+1;
   if (pobj->copy == current) {
      a->x = 0.0;
      a->y = 0.0;
      a->z = 0.0;
      b->x = 0.0;
      b->y = 0.0;
      b->z = 0.0;
      c = pobj->ri + 1;
      for (i=1; i<pobj->ipts; i++, c++) {
         MYMIN(a->x,c->x);
         MYMAX(b->x,c->x);
         MYMIN(a->y,c->y);
         MYMAX(b->y,c->y);
         MYMIN(a->z,c->z);
         MYMAX(b->z,c->z);
      }
      pobj->bbox = 0;
      get_coords(pobj->rbb, pt);
      if (verbose) {
	 printf("      %s: bbox is (%7.5f, %7.5f, %7.5f)\n", pobj->fname,
		a->x, a->y, a->z);
	 printf("\t\t to (%7.5f, %7.5f, %7.5f)\n", b->x, b->y, b->z);
      }
   } else {
      for (i=0; i<8; i++) {
	 pobj->rbb[i] = shapes[pobj->copy].rbb[i];
      }
      a = pobj->rbb;
      b = a+7;
      pobj->bbox = shapes[pobj->copy].bbox;
      if (verbose) {
	 printf("      %s (copy of %d): bbox is (%7.5f, %7.5f, %7.5f)\n",
		pobj->fname, pobj->copy, a->x, a->y, a->z);
	 printf("\t\t to (%7.5f, %7.5f, %7.5f)\n", b->x, b->y, b->z);
      }
   }
}


/* find the edges in the objects to speed up drawing */
/*
 * speed this rountine up by doing a sort (quick or shell or bubble)
 * of the segment list and rationalise new list by skipping over
 * identical segments
 * so instead of O(n^2) we get O(n log n) growth in execution time
 */

/* returns +ve if a > b, -ve if a < b and 0 if a == b */

#if defined(__DECC) || defined(__osf)
#pragma inline (segcmp)
#endif

static __inline__ int segcmp(const void *sa, const void *sb)
{
   const Segment *ta, *tb;
   int tt;

   ta = sa; tb = sb;
   if ((tt = ta->f - tb->f) == 0) return ta->t - tb->t; /* second field */
   else return tt;
}


void findedge(int current, Object *pobj, int verbose)
{
   int i, k, tot;
   int pna, pnb, pnc;
   usint *poly, *epoly, *fpoly;
   Segment *tedge, *sedge, *pedge;

   if (pobj->copy != current) {
      pobj->edge = shapes[pobj->copy].edge;
      pobj->nlines = shapes[pobj->copy].nlines;
      if (verbose) printf("      %s (copy of %d): total of %d segments\n",
			  pobj->fname, pobj->copy, pobj->nlines);
      return;
   }

   /* make a temporary list of segments */
   DIEIF((pedge = (Segment *) calloc(2*pobj->nlines, sizeof(Segment))) == NULL,
	 "Unable to allocate temp memory for segments\n");
   tot = 0;
   tedge = pedge;

   poly = pobj->polygon;
   epoly = poly + pobj->nindex;
   do {
      fpoly = poly + 1; /* two stages to avoid nasty precedence */
      fpoly += *poly++;
      pnc = pna = *poly++;
      do {
	 pnb = *poly++;
	 if (pnb > pna) {
	    tedge->f = pna;
	    tedge->t = pnb;
	 } else {
	    tedge->f = pnb;
	    tedge->t = pna;
	 }
	 tedge++;
	 tot++;
	 pna = pnb;
      } while (poly < fpoly);
      if (pnc > pna) {
	 tedge->f = pna;
	 tedge->t = pnc;
      } else {
	 tedge->f = pnc;
	 tedge->t = pna;
      }
      tedge++;
      tot++;
   } while (poly<epoly);
   DIEIF(tot > 2*pobj->nlines, "Oops, gone pass end of temp edge array\n");

   /* sort segments */
   qsort(pedge, tot, sizeof(Segment), segcmp);

   /* make proper list, picking out unique segments only */
   DIEIF((pobj->edge = (Segment *) calloc(100, sizeof(Segment))) == NULL,
	 "Unable to allocate memory for segments\n");
   k = 0;
   tedge = pedge;
   sedge = pobj->edge;
   i = 0;
   do {
      sedge->f = tedge->f;
      sedge->t = tedge->t;
      tedge++;
      i++;
      /* find next good segment */
      while (segcmp(sedge, tedge) >= 0) {
	 tedge++;
	 i++;
	 if (i>=tot) break; 
      }
      sedge++;
      k++;
      if ((k % 100)==0) {
	 pobj->edge = (Segment *) realloc(pobj->edge,
					  (k+100)*sizeof(Segment));
	 DIEIF(pobj->edge == NULL,
	       "Unable to reallocate memory for segments\n");
	 sedge = pobj->edge + k;
      }
   } while (i<tot);
   if (verbose) printf("      %s: total of %d segments\n", pobj->fname, k);
   pobj->nlines = k;
   free(pedge);
#if 0
   tedge = pobj->edge;
   for (i=0; i<k; i++, tedge++) {
      printf("  %4d to %4d\n", tedge->f, tedge->t);
   }
#endif
}

/* initialise the shift, rotation and magnification of the object */
void initobj(Object *pobj)
{
   initaffine(&pobj->ot);
   reinitobj(pobj);
}


void reinitobj(Object *pobj)
{
   pobj->wt = pobj->ot;
}


/* rotate all points */
void rotate(Object *pobj, int num, float ltheta, float lphi, float lalpha)
{
   rotaffine(&pobj->wt, ltheta, lphi, num*lalpha);
}

/* magnify all points */
void magnify(Object *pobj, int num, int dirn, float factor)
{
   int   i;
   float xmag, ymag, zmag, tmag;

   tmag = 1.0;
   for (i=0; i<num; i++) {
      tmag *= factor;
   }
   switch(dirn) {
    case 1:
      xmag = tmag;
      ymag = 1.0;
      zmag = 1.0;
      break;
    case 2:
      xmag = 1.0;
      ymag = tmag;
      zmag = 1.0;
      break;
    case 3:
      xmag = 1.0;
      ymag = 1.0;
      zmag = tmag;
      break;
    case 0: default:
      xmag = tmag;
      ymag = tmag;
      zmag = tmag;
      break;
   }

   magaffine(&pobj->wt, xmag, ymag, zmag);
}

void translate(Object *pobj, int num, int dirn, float inch)
{
   float xs, ys, zs;

   inch *= num;
   switch(dirn) {
    case 1:
      xs = inch;
      ys = 0.0;
      zs = 0.0;
      break;
    case 2:
      xs = 0.0;
      ys = inch;
      zs = 0.0;
      break;
    case 3:
      xs = 0.0;
      ys = 0.0;
      zs = inch;
      break;
    default:
      return;
   }

   transaffine(&pobj->wt, xs, ys, zs);
}


void rotaffine(Affine *at, float ltheta, float lphi, float lalpha)
{
   int   i;
   float kl, km, kn, rac, rat, ras;
   float temp, xr, yr;

   kl = sin(ltheta) * cos(lphi);
   km = sin(ltheta) * sin(lphi);
   kn = cos(ltheta);

   while(lalpha > TWO_PI) lalpha -= TWO_PI;
   rac = cos(lalpha);
   rat = 1.0-rac;
   ras = sin(lalpha);

   for (i=0; i<3; i++) {
      temp = kl*at->ro[i].x + km*at->ro[i].y + kn*at->ro[i].z;
      xr = at->ro[i].x*rac + rat*temp*kl + ras*(km*at->ro[i].z-kn*at->ro[i].y);
      yr = at->ro[i].y*rac + rat*temp*km + ras*(kn*at->ro[i].x-kl*at->ro[i].z);
      at->ro[i].z = at->ro[i].z*rac + rat*temp*kn
                    + ras*(kl*at->ro[i].y-km*at->ro[i].x);
      at->ro[i].x = xr;
      at->ro[i].y = yr;
   }
}


void magaffine(Affine *at, float xmag, float ymag, float zmag)
{
   int i;

   for (i=0; i<3; i++) {
      at->ro[i].x *= xmag;
      at->ro[i].y *= ymag;
      at->ro[i].z *= zmag;
   }
   at->rsh.x *= xmag;
   at->rsh.y *= ymag;
   at->rsh.z *= zmag;
}


/* shiftless version */
void magsaffine(Affine *at, float xmag, float ymag, float zmag)
{
   int i;

   for (i=0; i<3; i++) {
      at->ro[i].x *= xmag;
      at->ro[i].y *= ymag;
      at->ro[i].z *= zmag;
   }
}


void transaffine(Affine *at, float xinch, float yinch, float zinch)
{
   at->rsh.x += xinch;
   at->rsh.y += yinch;
   at->rsh.z += zinch;
}


/* initialise affine */
void initaffine(Affine *at)
{
   at->rsh.x = 0.0;
   at->rsh.y = 0.0;
   at->rsh.z = 0.0;

   at->ro[0].x = 1.0;
   at->ro[0].y = 0.0;
   at->ro[0].z = 0.0;
   at->ro[1].x = 0.0;
   at->ro[1].y = 1.0;
   at->ro[1].z = 0.0;
   at->ro[2].x = 0.0;
   at->ro[2].y = 0.0;
   at->ro[2].z = 1.0;
}


/*
 * multiply two affine transformations:
 *  at = bt * at
 */
void mulaffine(Affine *at, Affine *bt)
{
   int i;
   Point *pt;

   pt = &at->ro[0];
   for (i=0; i<3; i++, pt++) {
      applysaffine(pt, pt, bt);
   }
   pt = &at->rsh;
   applyaffine(pt, pt, bt);
}


/*
 * apply affine to a point, includes setting mode for shift:
 *  p = a * q
 */

#if defined(__DECC) || defined(__osf)
#pragma inline (applyaffine,applysaffine)
#endif

__inline__ void applyaffine(Point *pt, Point *qt, Affine *at)
{
   register float tx, ty, tz;

   tx = qt->x;
   ty = qt->y;
   tz = qt->z;
   pt->x = at->rsh.x + tx*at->ro[0].x + ty*at->ro[1].x + tz*at->ro[2].x;
   pt->y = at->rsh.y + tx*at->ro[0].y + ty*at->ro[1].y + tz*at->ro[2].y;
   pt->z = at->rsh.z + tx*at->ro[0].z + ty*at->ro[1].z + tz*at->ro[2].z;
}


/* version without shift */
__inline__ void applysaffine(Point *pt, Point *qt, Affine *at)
{
   register float tx, ty, tz;

   tx = qt->x;
   ty = qt->y;
   tz = qt->z;
   pt->x = tx*at->ro[0].x + ty*at->ro[1].x + tz*at->ro[2].x;
   pt->y = tx*at->ro[0].y + ty*at->ro[1].y + tz*at->ro[2].y;
   pt->z = tx*at->ro[0].z + ty*at->ro[1].z + tz*at->ro[2].z;
}


float parameter[PARMAX];
int parno;

/* 
 * parse name and argument list
 * format:-
 *   function(p1,..,pn)
 * white spaces are allowed
 */
void pparse(const char *command, char fname[NLEN])
{
   char *tmp;
   char wsa[LLEN];

   parno = 0;

   /* clean up */
   tmp = strchr(command, '\n');
   if (tmp != NULL) *tmp = '\0';

   /* get name */
   strncpy(wsa, command, LLEN-1);
   tmp = strchr(wsa, '(');
   if (tmp != NULL) *tmp = '\0';
   if (strlen(wsa)>0) strncpy(fname, wsa, NLEN-1);

   if (tmp != NULL) {
      strcpy(wsa, tmp+1);
      /* find numbers delimited by commas */
      while ((tmp = strchr(wsa, ',')) != NULL) {
	 *tmp = '\0';
	 if (tmp == wsa) break;
	 parameter[parno++] = atof(wsa);
	 strcpy(wsa, tmp+1);
      }
      /* couldn't find another comma, so look for closing parenthesis */
      if ((tmp = strchr(wsa, ')')) != NULL) {
	 *tmp = '\0';
	 if (tmp != wsa) parameter[parno++] = atof(wsa);
      }
   }

}

