#ifndef POLYH_H
#define POLYH_H
/*
 * polyh.h - header file for object structures and routines to manipulate,
 *           save and load them.

 * Copyright 1994 and 1995, 26th January.
 * By Peter Chang
 * peterc@a3.ph.man.ac.uk

 * See notice in misc.h for details of permissions and warranty of this
 * software.

 *
 * $Log: polyh.h,v $
 * Revision 2.6.2.1  1995/01/26  04:45:24  peter
 * New function magsaffine() declared.
 *
 * Revision 2.6  1994/11/18  04:03:09  peter
 * xpgs-2.5-patch 01: Changes in header files for clean compile
 *
 * Revision 2.5.1.1  1994/11/17  03:34:31  peter
 * Import of actual public release of xpgs-2.5: lots of cosmetic changes to docs
 *
 * Revision 2.5  1994/11/16  09:19:26  peter
 * Putting xpgs-2.5 in trunk.
 *
 * Revision 2.0.1.1  1994/11/16  09:10:31  peter
 * Import of xpgs-2.5: archive of new release to alt.sources (11/94)
 *
 *
 */

#include "misc.h"


#define MYMAX(A, B) if ((A) < (B)) (A) = (B)
#define MYMIN(A, B) if ((A) > (B)) (A) = (B)

typedef struct  {
   float x;
   float y;
   float z;
} Point;

typedef struct {
   usint f;
   usint t;
} Segment;

typedef struct {
   Point ro[3];               /* orientation and deformation */
/*   Point rm;                   magnification (now redundant) */
   Point rsh;                 /* translation */
} Affine; /* Affine transformation (in components) */

typedef struct {
   char    fname[NLEN];         /* polyh file name */
   int     ipts, nindex;        /* # pts + 1 and # polygon indexes */
   int     nlines;              /* # edges */
   int     copy, hidden;        /* copy and hidden flags */
   int     bbox;                /* bbox flag */
   Point   rbb[8];              /* array of bounding box corners */
   Point   *ri;                 /* ptr to points */
   usint   *polygon;            /* ptr to polygons */
   Segment *edge;               /* ptr to edges */
   Affine  ot;                  /* original transformation */
   Affine  wt;                  /* working transformation */
   int     sibling;             /* younger bro' or sis' */
   int     child;               /* sprog of thy own */
   int     frames;              /* animation frames */
   Affine  *nt;                 /* ptr to transformations from original */
} Object;

/*
 * description of .polyh format and the object and its associate structures
 * 
 * 1) the first line may contain a comment (starting with a '#')
 * 2) the number of vertices contained in the polyhedra
 * 3) a list of (floating point) numbers
 *      each line holds the three spatial coordinates of each vertex
 * 4) a list of polygons
 *      each entry includes the number of vertices and
 *      a list of the vertices (index numbers to vertex list -
 *      first pt is 1, etc)
 * 
 * Each polygon need not be planar nor convex, the only requirement
 * is that subdivision of the polygon into triangles is possible
 * (and consistent) from the first vertex.
 * In other words, for a polygon with vertices {v1, v2,..., vn},
 * we can span the polygon with diagonals from v1 to each of
 * {v2,..., vn} to form the (n-2) triangles (v1,v2,v3), (v1,v3,v4),
 * ..., and (v1,v(n-1),vn).
 * 
 * The Point structure just holds three floats.
 * 
 * The Segment structure holds the indices of the two points of an edge.
 * 
 * The Affine structure contains three (Point) substructures:
 *     rot[3] holds the rot & mag matrix to be applied to the object;
 *     rm     is the set of x,y,z magnification factors -
 *            (now redundant);
 * and rsh    holds the translation vector
 * these operations are applied in the listed order.
 * 
 * The Object structure contains
 *     fname    the polyh file name or a builtin name and parameter list
 *     ipts     number of points + 1 in object
 *     nindex   number of polygon indices
 *     nlines   number of edges (for drawing in xss)
 *     copy     index of original in shapes array (if original then = itself)
 *     hidden   flag to indicate whether object should be drawn
 *     *ri      ptr to array of Point, holds the initial vertex positions
 *     *polygon ptr to polygon list
 *     *edge    ptr to edge list (for drawing in xss)
 *     ot       original Affine transformation
 *     wt       working Affine transformation
 *     sibling  index to an object with the same parent
 *     child    index to child of current object
 *     frames   number of additional Affine tranformations
 * and *nt      ptr to array of Affine transformations
 *
 */

extern Object *shapes;
extern usint nobj;
extern float theta, phi;

void killallshapes(void);
void killshape(int cuurent);
int loadshapes(char name[NLEN], int verbose);
int addpolyh(char name[NLEN], int verbose);
int loadpolyh(Object *ptrobj);
void savepolyh(Object *ptrobj, int cflag);
int loadxss(char name[NLEN], int verbose);
int insertxss(char name[NLEN], int verbose);
void savexss(char name[NLEN], char comment[LLEN]);

void findbbox(int current, Object *ptrobj, int verbose);
void findedge(int current, Object *ptrobj, int verbose);

void initobj(Object *ptrobj);
void reinitobj(Object *ptrobj);
void rotate(Object *ptrobj, int times, float ltheta, float lphi, float lalpha);
void magnify(Object *ptrobj, int num, int dirn, float factor);
void translate(Object *ptrobj, int num, int dirn, float inch);

void rotaffine(Affine *at, float ltheta, float lphi, float lalpha);
void magaffine(Affine *at, float xmag, float ymag, float zmag);
void magsaffine(Affine *at, float xmag, float ymag, float zmag);
void transaffine(Affine *at, float xinch, float yinch, float zinch);
void initaffine(Affine *at);
void mulaffine(Affine *at, Affine *bt);
__inline__ void applyaffine(Point *pt, Point *qt, Affine *at);
__inline__ void applysaffine(Point *pt, Point *qt, Affine *at);

/* parsing stuff */
#define PARMAX 12
extern float parameter[PARMAX];
extern int parno;

void pparse(const char *command, char fname[NLEN]);

#endif
