static char rcsid[] = "@(#)$Id: sisgen.c,v 2.6.2.1 1995/01/26 04:41:59 peter Exp $";
/*
 * sisgen - a program to generate and output a SIS.

 * It creates a Single Image Stereogram from an XSS or P*M file and,
 * optionally, a colour pattern given in P*M format and outputs
 * the result as a file or data stream in stdout, again in P*M.

 * Copyright 1994 and 1995, 26th January.
 * By Peter Chang. All rights reserved.
 * peterc@a3.ph.man.ac.uk

 * See notice in misc.h for details of permissions and warranty of this
 * software.

 *
 * $Log: sisgen.c,v $
 * Revision 2.6.2.1  1995/01/26  04:41:59  peter
 * Fixed bug with non-initialization of oname[].
 *
 * Comment changed in ppm output.
 *
 * Revision 2.6  1994/11/18  04:03:11  peter
 * xpgs-2.5-patch 01: Changes in header files for clean compile
 *
 * Revision 2.5.1.1  1994/11/17  03:34:25  peter
 * Import of actual public release of xpgs-2.5: lots of cosmetic changes to docs
 *
 * Revision 2.5  1994/11/16  09:19:28  peter
 * Putting xpgs-2.5 in trunk.
 *
 * Revision 2.0.1.1  1994/11/16  09:10:29  peter
 * Import of xpgs-2.5: archive of new release to alt.sources (11/94)
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#define SIRDS_STUFF
#include "pgs.h"
#include "polyh.h"
#include "pxmio.h"
#include "zbuffer.h"
#include "lookup.h"

#if __GNUC__ == 2
USE(rcsid);
#endif

Object *shapes = NULL;
usint nobj = 0;
float theta, phi;

int width = -1, height = -1;
int buf_h = ZBLINES, buf_top, buf_bot;
int fineness = 0;
int reverse = 1;
float background = -1.0;
float pmm = 4.0;
float enlarge = 1.0;

Myrgb  *palette = NULL;
int    nocols, zmax, coloured = 1;
int    tw, th;

static int tolerance = 1;
static int randomness = 0, tiling = 0;
static int density = 127;

static int output = 1;
static int input = 1;
static int xssload = 1;
static int xstep = 1, ystep = 1;
static float mag = 1.0;

int main(int argc, char **argv)
{
   float bv;
   int x, y, sep;
   int **ztopl = NULL, **ztopr = NULL;
   int dmax;
   int *zmap = NULL;
   int *zll, *zlr;
   Llist *same;
   usint *shift;
   Myrgb *pixarray;
   float xratio, yratio = -1.0;
   int offset, dmp;
   int otw = 0, oth = 0;
   usshort *tile = NULL;
   FILE  *fo;
   Affine globalaf;

   char fname[NLEN], oname[NLEN], tname[NLEN];

   int  sirdsnew(int *, int *, Llist *, int *, usint *);
   void tileline(Myrgb *, usshort *, float, float, int , int );
   void colourint(Myrgb *, Llist *, usint *, int );
   void pcl(int, char **, char *, char *, char *);

   oname[0] = '\0';

   pcl(argc, argv, fname, tname, oname);
   if (xssload) {
      DIEIF3(loadshapes(fname, output) < 0,
	     "ERROR: something wrong with %s\n", fname);
      if (width <= 0 || height <= 0) {
	 width = 770;
         height = 445;
      }
   } else {
      /* load and rescale contents of p?m file */
      zmap = (int *)loadpxm(fname, LDPXMDMAP);
      if (output && input) printf("input size %d %d\n", tw, th);
      if (width <= 0 || height <= 0) {
         width = tw;
         height = th;
      }
      otw = tw;
      oth = th;
      dmax = zmax+1;
      if (coloured) dmax *= dmax*dmax;
      rescaledmap(zmap, (height / (float) dmax), height/2);
   }
   ztopl = zmema();
   ztopr = zmema();

   bv = OS - reverse*OY*background;
   sep = (int) (enlarge*ES*bv/(VD+bv) + 0.5);

   if (randomness != 3) {
      tw = sep;
      th = height;
      srand(time(NULL));
   }
   switch (randomness) {
    case 3:
      tile = (usshort *)loadpxm(tname, LDPXMTILE);
      break;
    case 2:
      coloured = 1;
      tile = randppm();
      break;
    case 1:
      coloured = 0;
      tile = randpgm();
      break;
    case 0: default:
      coloured = 0;
      tile = randpbm(density);
      break;
   }

   fprintf(stderr,"Separation %d, eye %d tile %d %d\n", sep,
	   (int) ES, tw, th);
   fprintf(stderr, "Fineness %d, mask %d, tolerance %d\n",
	   fineness, (1 << fineness) - 1, tolerance);

   if (strlen(oname) < 1) {
      if (output && input) {
	 printf("Enter name for output ppm file: ");
	 scanf("%50s", oname);
	 printf("\n");
      } else if (input == 0) output = 0;
   }
#ifndef VMS
   if (output) {
      DIEIF3((fo = fopen(oname, "w")) == NULL,
	    "Unable to open file : %s\n", oname);
   } else {
      fo = stdout;
   }
#else
   DIEIF3((fo = fopen(oname, "w")) == NULL,
	  "Unable to open file : %s\n", oname);
#endif

   if (output) printf("output size %d %d\n", width, height);
   pxmstart(fo);
   if (output) fprintf(fo, "# %s made by sisgen from %s\n", oname, fname);
   fprintf(fo, "%d %d\n%d\n",width, height, zmax);

   DIEIF((same = (Llist *) malloc(width*sizeof(Llist))) == NULL,
	 "Unable to allocate memory for array\n");
   DIEIF((shift = (usint *) malloc(width*sizeof(usint))) == NULL,
	 "Unable to allocate memory for array\n");
   DIEIF((pixarray = (Myrgb *) malloc(width*sizeof(Myrgb))) == NULL,
	 "Unable to allocate memory for array\n");


   switch (tiling/3) {
    case 2:
      xratio = -1.0;
      yratio = tw / (float) sep;
      offset = sep;
      break;
    case 1:
      xratio = tw / (float) sep;
      offset = sep;
      break;
    case 0: default:
      xratio = 1.0;
      offset = tw;
      break;
   }
   switch (tiling%3) {
    case 2: /* from the right */
      offset = offset - width%offset;
      dmp = width - 1;
      break;
    case 1: /* from the centre */
      offset = offset - (width - offset)/2%offset;
      dmp = width/2;
      break;
    case 0: default: /* from the left */
      offset = 0;
      dmp = 0;
      break;
   }
   fprintf(stderr, "Tiling method %d (%f), offset %d starting at %d\n",
	   tiling, (xratio > 0) ? xratio : yratio, offset, dmp);

   lookup(reverse);
   buf_bot = 0;
   if (xssload) initaffine(&globalaf);

   for (y=0;y<height;y++) {
      int mins = (width << fineness);

      if (y == buf_bot) {
	 if (xssload) sfillall(y, ztopl, ztopr, mag, &globalaf);
	 else zsfillall(y, zmap, ztopl, ztopr, otw, oth, xstep, ystep);
      }
      zll = ztopl[y%buf_h];
      zlr = ztopr[y%buf_h];
      for (x=0; x<width; x++) {
	 same[x].t = x;
	 same[x].f = x;
	 shift[x] = 0;
      }
      if (xratio < 0.0) {
	 int tmp;

	 dmp = sirdsnew(zll, zlr, same, &mins, shift);
	 tmp = mins >> fineness;
	 if (tmp == 0) fprintf(stderr, "Oh no, dividing by zero\n");
	 xratio = tw / (float) tmp;
	 switch (tiling%3) {
	  case 2:
	    offset = tmp - width%tmp;
	    break;
	  case 1:
	    offset = tmp - (width - tmp)/2%tmp;
	    break;
	  case 0: default:
	    offset = 0;
	    break;
	 }
	 tileline(pixarray, tile, xratio, yratio, y, offset);
	 xratio = -1.0;
      } else {
	 sirdsnew(zll, zlr, same, &mins, shift);
	 tileline(pixarray, tile, xratio, xratio, y, offset);
      }
      colourint(pixarray, same, shift, dmp);
      pxmline(fo, pixarray);
   }

   pxmline(fo, NULL);

   if (output) {
      fclose(fo);
      printf("\nSaved %s\n", oname);
   }
   free(zplookup);
   free(tile);
   free(palette);
   free(shift);
   free(same);
   free(pixarray);
   zmemf(ztopl);
   zmemf(ztopr);
   if (xssload) killallshapes();

   return EXIT_SUCCESS;
}


void colourint(Myrgb *pix, Llist *same, usint *shift, int midpt)
{
   int fn, ww;
   int ll, rr;
   usint ss, tt;
   Myrgb *pa, *pb, *pg;

   fn = 1<<fineness;
   for (rr=midpt; rr < width; rr++) {
      ll = same[rr].f;
      if (ll != rr) {
	 ss = shift[rr];
	 pg = &pix[rr];
	 pb = &pix[ll];
	 if (ss == 0 || ll == 0) {
	    pg->r = pb->r;
	    pg->g = pb->g;
	    pg->b = pb->b;
	 } else {
	    pa = pb - 1;
	    tt = fn - ss;
	    pg->r = (ss*pa->r + tt*pb->r) >> fineness;
	    pg->g = (ss*pa->g + tt*pb->g) >> fineness;
	    pg->b = (ss*pa->b + tt*pb->b) >> fineness;
	 }
      }
   }

   ww = width - 1;
   for (ll=midpt-1; ll >= 0; ll--) {
      rr = same[ll].t;
      if (rr != ll) {
	 ss = shift[rr];
	 pg = &pix[ll];
	 pa = &pix[rr];
	 if (ss == 0 || rr == ww) {
	    pg->r = pa->r;
	    pg->g = pa->g;
	    pg->b = pa->b;
	 } else {
	    pb = pa + 1;
	    tt = fn - ss;
	    pg->r = (tt*pa->r + ss*pb->r) >> fineness;
	    pg->g = (tt*pa->g + ss*pb->g) >> fineness;
	    pg->b = (tt*pa->b + ss*pb->b) >> fineness;
	 }
      }
   }
}


int sirdsnew(int *zll, int *zlr, Llist *same, int *mins, usint *shift)
{
   register int s, x, mask, midpt;
   register int left, right;
   register int st, sl, sr;

   mask = (1 << fineness) - 1;
   midpt = width/2;

   for (left=0; left<width; left++) {
      s = *zll++;
      right = left + (s >> fineness);
      if (right < width) {
	 x = s - zlr[right];
	 if (x <= tolerance && x >= -tolerance) {
	    if (s < *mins) {
	       *mins = s;
	       midpt = left;
	    }
	    sl = left; sr = right;
	    for (st=same[sr].f; st!=sl && st!=sr; st=same[sr].f) {
	       if (st > sl) {
		  sr = st;
	       } else {
		  same[sl].t = sr;
		  same[sr].f = sl;
		  sr = sl;
		  sl = st;
	       }
	    }
	    same[sl].t = sr;
	    same[sr].f = sl;
	    shift[right] = (usint) ((s&mask) + shift[right])>>1;
	 }
      }
   }
   return midpt;
}


/*
 * apply tile to current line, where the colour is either 
 * a grey scale value or a palette number
 */
void tileline(Myrgb *pline, usshort *ttile, float xratio, float yratio,
	      int y, int offset)
{
   usshort *cline;
   int i, x;

   y = (int) (yratio * y + 0.5);
   cline = &ttile[(y % th) * tw];

   if (coloured) {
      Myrgb *pa;

      for (i=0; i<width; i++, pline++) {
	 x = (int) ((i+offset) * xratio + 0.5);
	 pa = palette + cline[x % tw];
	 pline->r = pa->r;
	 pline->g = pa->g;
	 pline->b = pa->b;
      }
   } else  {
      for (i=0; i<width; i++, pline++) {
	 x = (int) ((i+offset) * xratio + 0.5);
	 pline->r = cline[x % tw];
	 pline->g = 0;
	 pline->b = 0;
      }
   }
}


/*
 * Parse the command line set the filename and the appropriate flags
 */
void pcl(int argc, char **argv, char *xname, char *tname, char *oname)
{
   static const char *options[] = {
      "-help                     print this message",
      "-geometry <wxh>             picture geometry",
      "-background <b>             background plane (-1.0)",
      "-reverse               set crosseyed viewing",
      "-enlarge <ef>           enlargement of depth (1.0)",
      "-mag <mf>             magnification of scene (1.0)",
      "-pmm <res>           resolution in pixels/mm (4.0)",
      "-fine <0-8> <tol>      fineness of smoothing (0 1)",
      "-tile <0-8>                    tiling method (0)",
      "-image <pxm file>                tiling file",
      "-xss                  load xss or polyh file (default)",
      "-dmap                     load depthmap file",
      "-step <XxY>               steps used in dmap (1x1)",
      "-RANDOM <0-2>         kind of random pattern (0)",
      "-zblines <zb>  lines to use in each z buffer (256)",
#ifndef VMS
      "-input      use stdin for depth map p*m file",
      "-output    use stdout for resultant p*m file",
#endif
      "-DENSITY <0-255>       density of black dots (127)",
      NULL
   };

   static const char usage1[] =
      "usage:  sisgen [options] <input (.xss or .p*m) file> [output p*m file]";
   static const char usage2[] = " try sisgen -help for a list of option";

   int i, optno, fvalid = 0, olen;
   const char **sptr;

   for (i = 1; i < argc; i++) {
      olen = strlen(argv[i]);
      for (sptr = options, optno = 0; *sptr != NULL; sptr++, optno++)
      	 if (strncmp(argv[i], *sptr, olen) == 0) break;

      switch(optno) {
       case 0:
         printf("%s\n", usage1);
         printf("Options (defaults in parenthesis):\n");
	 for (sptr = options; *sptr != NULL; sptr++) printf(" %s\n", *sptr);
	 printf("\n");
         exit(EXIT_SUCCESS);
         break;

       case 1:
	 sscanf(argv[++i], "%dx%d", &width, &height);
         break;

       case 2:
         background = atof(argv[++i]);
         break;

       case 3:
         reverse = -1;
         break;

       case 4:
         enlarge = atof(argv[++i]);
         break;

       case 5:
	 mag = atof(argv[++i]);
         break;

       case 6:
         pmm = atof(argv[++i]);
         break;

       case 7:
         fineness = atoi(argv[++i]);
         tolerance = atoi(argv[++i]);
         break;

       case 8:
         tiling = atoi(argv[++i]);
         break;

       case 9:
	 strcpy(tname, argv[++i]);
	 randomness = 3;
         break;

       case 10:
         xssload = 1;
         break;

       case 11:
         xssload = 0;
         break;

       case 12:
	 sscanf(argv[++i], "%dx%d", &xstep, &ystep);
         break;

       case 13:
         randomness = atoi(argv[++i]);
         break;

       case 14:
         buf_h = atoi(argv[++i]);
         break;

       case 15:
#ifndef VMS
         input = 0;
	 strcpy(xname, "-");
	 fvalid++;
	 xssload = 0;
	 break;

       case 16:
         output = 0;
	 break;

       case 17:
#endif
         density = atoi(argv[++i]) & 0xff;
         break;

       default:
         if (fvalid  < 2 && *argv[i] != '-') {
	    switch (fvalid) {
	     case 0:
	       strcpy(xname, argv[i]);
	       break;
	     case 1:
	       strcpy(oname, argv[i]);
	       break;
	    }
            fvalid++;
         } else {
            DIE4("%s\n%s\n", usage1, usage2);
         }
         break;
      }
   }

   DIEIF4(fvalid < 1, "%s\n%s\n", usage1, usage2);
}

