static char rcsid[] = "@(#)$Id: zbuffer.c,v 2.6.2.1 1995/01/26 04:38:44 peter Exp $";
/*
 * zbuffer.c - routines for the z-buffer(s).

 * Copyright 1994 and 1995, 26th January.
 * By Peter Chang
 * peterc@a3.ph.man.ac.uk

 * See notice in misc.h for details of permissions and warranty of this
 * software.

 *
 * $Log: zbuffer.c,v $
 * Revision 2.6.2.1  1995/01/26  04:38:44  peter
 * Changed triangle() completely: Now convert triangle edges to scanline
 * values (x & z), ie build up two edgelists using Bresenham type stepping.
 * Then with another Bresenham algo, interpolate across each line in z values
 * and paint in a la z-buffer.
 *
 * Knock-on changes to zmem*() and further minor enhancements to *fillobj().
 *
 * Revision 2.6  1994/11/18  04:03:18  peter
 * xpgs-2.5-patch 01: Changes in header files for clean compile
 *
 * Revision 2.5.1.1  1994/11/17  03:34:24  peter
 * Import of actual public release of xpgs-2.5: lots of cosmetic changes to docs
 *
 * Revision 2.5  1994/11/16  09:19:39  peter
 * Putting xpgs-2.5 in trunk.
 *
 * Revision 2.0.1.1  1994/11/16  09:10:30  peter
 * Import of xpgs-2.5: archive of new release to alt.sources (11/94)
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>

#define SIRDS_STUFF
#include "pgs.h"
#include "polyh.h"
#include "zbuffer.h"
#include "lookup.h"

#if __GNUC__ == 2
USE(rcsid);
#endif


/* Defines all objects using discretised height fields
 * contains all relatives scales that depend on window height */
int sfillall(int yy, int **zbufa, int **zbufb, float mf, Affine *gt)
{
   int i;
   int x, y;
   int *zll, *zlr;
   float z;
   int bv, iv;

   void sfillobj(Object *, float , float , float , int **zta, int **ztb,
		 Affine *gt);

   iv = (reverse > 0) ? MYHIGH : MYLOW;

   buf_top = yy;
   buf_bot = yy + buf_h;
   for (y=0; y<buf_h; y++) { /* set background to const initially */
      zll = zbufa[y];
      zlr = zbufb[y];
      for (x=0; x<width; x++) {
         *zll++ = iv;
         *zlr++ = iv;
      }
   }

   for (i=0; i<nobj; i++) sfillobj(&shapes[i], mf, mf, mf, zbufa, zbufb, gt);

   z = OS - reverse*background*OY;
   bv = (int) ((enlarge*(1 << fineness))*ES*z/(VD+reverse*z));

   for (y=0; y<buf_h; y++) { /* reset background */
      zll = zbufa[y];
      zlr = zbufb[y];
      for (x=0; x<width; x++, zll++, zlr++) {
         if (*zll == iv) *zll = bv;
         if (*zlr == iv) *zlr = bv;
      }
   }
   return bv;
}


int zfillall(int yy, int **zbuf, float mf, Affine *gt)
{
   int i;
   int x, y;
   int *zl;
   int bv, iv;

   void zfillobj(Object *, float , float , float , int **zt, Affine *gt);

   iv = MYLOW;

   buf_top = yy;
   buf_bot = yy + buf_h;
   for (y=0; y<buf_h; y++) { /* set background to const initially */
      zl = zbuf[y];
      for (x=0; x<width; x++) {
         *zl++ = iv;
      }
   }

   for (i=0; i<nobj; i++) zfillobj(&shapes[i], mf, mf, mf, zbuf, gt);

   bv = (int) ((enlarge*(1 << fineness))*OY*background);

   for (y=0; y<buf_h; y++) { /* reset background */
      zl = zbuf[y];
      for (x=0; x<width; x++, zl++) {
         if (*zl == iv) *zl = bv;
      }
   }
   return bv;
}


/*
 * Dumps height data to two 's'-buffers
 */
int zsfillall(int yy, int *zmap, int **zbufa, int **zbufb, int ttw, int tth,
	      int xs, int ys)
{
   int x, y;
   int *zll, *zlr;
   float z;
   int bv, iv;
   int half;

   void zsdrawobj(int *, int **, int **, int , int , int , int , int );

   iv = (reverse > 0) ? MYHIGH : MYLOW;
   buf_top = yy;
   buf_bot = yy + buf_h;
   for (y=0; y<buf_h; y++) { /* set background to const initially */
      zll = zbufa[y];
      zlr = zbufb[y];
      for (x=0; x<width; x++) {
         *zll++ = iv;
         *zlr++ = iv;
      }
   }

   half = (buf_bot < height/2) ? 0 : 1;
   if (half == 1 && yy < height/2) half = 2;
#ifdef MYDEBUG
   printf("yy %d buf t %d %d %d : half %d\n", yy,
	  buf_top, buf_bot, buf_h, half);
#endif
   zsdrawobj(zmap, zbufa, zbufb, ttw, tth, xs, ys, half);

   z = OS - reverse*background*OY;
   bv = (int) ((enlarge*(1 << fineness))*ES*z/(VD+reverse*z));
   for (y=0; y<buf_h; y++) { /* reset background */
      zll = zbufa[y];
      zlr = zbufb[y];
      for (x=0; x<width; x++, zll++, zlr++) {
         if (*zll == iv) *zll = bv;
         if (*zlr == iv) *zlr = bv;
      }
   }
   return bv;
}


/*
 * a set of routines to fill a triangle
 *
 * new improved algorithms:
 *   use two edge arrays for left and right side
 *   multiple Bresenham lines: two for edges of the triangle
 *                              and one for each scanline
 */

#if defined(__DECC) || defined(__osf)
#pragma inline (lbres3dln,rbres3dln,lnrender)
#endif

#define MYSGN(x) (x>=0) ? 1 : -1

/* structure to hold edge coords */
typedef struct {
   int x;
   int z;
} Edgelist;

static Edgelist *lelist = NULL, *relist = NULL;

#define PPP 0

/* render a scanline */
static __inline__ void lnrender(const int y, const Edgelist *leptr,
				const Edgelist *reptr, int **zbuffer,
				const int comp)
{
   int i = reptr->x - leptr->x, z = reptr->z-leptr->z;
   int bx = abs(i) << 1, bz = abs(z) << 1, sz = MYSGN(z);
   int dz, end;
   int *zline;

   z = leptr->z;
   /* horizontal clip */
   end = (reptr->x >= width) ? (width-1) : reptr->x;
   i = leptr->x;
   if (i < width && end >= 0) { /* line fits buffer */
      zline = zbuffer[y%buf_h] + i;
      /* do scanline */
      if (bx > bz) { /* x dominant */
	 dz = bz - (bx>>1);
	 while (i<0) { /* make sure i is +ve */
	    if (dz >= 0) {
	       z += sz;
	       dz -= bx;
	    }
	    dz += bz;
	    i++; zline++;
	 }
	 for (; i<=end; i++, zline++) {
	    if ((z > *zline) ^ comp) *zline = z;
	    if (dz >= 0) {
	       z += sz;
	       dz -= bx;
	    }
	    dz += bz;
	 }
      } else { /* z dominant */
	 if (bx != 0) {
	    dz = bx - (bz>>1);
	    while (i<0) {
	       if (dz >= 0) dz -= bz;
	       do { /* go to next x */
		  z += sz;
		  dz += bx;
	       } while (dz < 0);
	       i++; zline++;
	    }
	    for (; i<=end; i++, zline++) {
	       if ((z > *zline) ^ comp) *zline = z;
	       if (dz >= 0) dz -= bz;
	       do {
		  z += sz;
		  dz += bx;
	       } while (dz < 0);
	    }
	 } else { /* no change in x */
	    if (i>=0 && ((z > *zline) ^ comp)) *zline = z;
	 }
      }
   }
#if 0
   /* lines on edges */
   zline = zbuffer[y%buf_h];
   z = leptr->z;
   if ((z > *(zline+leptr->x)) ^ comp) *(zline+leptr->x) = z;
   z = reptr->z;
   if ((z > *(zline+reptr->x)) ^ comp) *(zline+reptr->x) = z;
#if PPP
fprintf(stderr, "l %d : %d %f ", y, xl, zl);
fprintf(stderr, "| r %d : %d %f\n", y, xr, zr);
#endif
#endif
}


/*
 * The two Bresenham line traversal routines are _heavily_ adapted from
 * the C code from the article "Voxel Traversal along a 3D Line"
 * by Daniel Cohen, danny@bengus.bgu.ac.il in "Graphics Gems IV",
 * Academic Press, 1994
 */
static __inline__ void lbres3dln(int x, int y, int z,
				 const int xd, const int yd, const int zd)
{
   int n, sz, exy, exz, ezy, ax, ay, az;
   Edgelist *elent;

#if PPP
fprintf(stderr, "L: %d %d %d + %d %d %d -> %d %d %d\n", x, y, z, xd, yd, zd,
	x+xd, y-yd, z+zd);
#endif
   sz = MYSGN(zd);
   ax = abs(xd); ay = yd;      az = abs(zd);
   exy = ay-ax;  exz = az-ax;  ezy = ay-az;
   n = ax+ay+az;
   ax <<= 1; ay <<= 1; az <<= 1;
   elent = &lelist[y%buf_h];
   elent->x = x;
   elent->z = z;
   while (n--) {
      if (exy < 0) {
	 if (exz < 0) {
	    if (xd > 0) {
	       x++;
	    } else { /* need to store leading left edge */
	       elent->x = x;
	       elent->z = z;
	       x--;
	    }
	    exy += ay; exz += az;
	 } else {
	    z += sz;
	    exz -= ax; ezy += ay;
	 }
      } else {
	 if (ezy < 0) {
	    z += sz;
	    exz -= ax; ezy += ay;
	 } else { /* start new line */
	    y--;
	    if (y < buf_top) return;
	    exy -= ax; ezy -= az;
	    elent--;
	    elent->x = x;
	    elent->z = z;	    
	 }
      }
   }
   if (xd < 0) {
      elent->x = x;
      elent->z = z;
   }
#if PPP
fprintf(stderr, "L: %d, %d, %d -> %d %d\n", x, y, z, elent->x, elent->z);
#endif
}


static __inline__ void rbres3dln(int x, int y, int z,
				 const int xd, const int yd, const int zd)
{
   int n, sz, exy, exz, ezy, ax, ay, az;
   Edgelist *elent;

#if PPP
fprintf(stderr, "R: %d %d %d + %d %d %d -> %d %d %d\n", x, y, z, xd, yd, zd,
	x+xd, y-yd, z+zd);
#endif
   sz = MYSGN(zd);
   ax = abs(xd); ay = yd;      az = abs(zd);
   exy = ay-ax;  exz = az-ax;  ezy = ay-az;
   n = ax+ay+az;
   ax <<= 1; ay <<= 1; az <<= 1;
   elent = &relist[y%buf_h];
   elent->x = x;
   elent->z = z;
   while (n--) {
      if (exy < 0) {
	 if (exz < 0) {
	    if (xd > 0) { /* need to store trailing right edge */
	       elent->x = x;
	       elent->z = z;
	       x++;
	    } else {
	       x--;
	    }
	    exy += ay; exz += az;
	 } else {
	    z += sz;
	    exz -= ax; ezy += ay;
	 }
      } else {
	 if (ezy < 0) {
	    z += sz;
	    exz -= ax; ezy += ay;
	 } else {
	    y--;
	    if (y < buf_top) return;
	    exy -= ax; ezy -= az;
	    elent--;
	    elent->x = x;
	    elent->z = z;	    
	 }
      }
   }
   if (xd > 0) {
      elent->x = x;
      elent->z = z;
   }
#if PPP
fprintf(stderr, "R: %d, %d, %d -> %d %d\n", x, y, z, elent->x, elent->z);
#endif
}


/*
 * take the canonical triangle
 *
 *                     --3
 *                ----' '
 *            2--'   --'
 *           /    --'
 *          /  --'
 *         /--'
 *         1
 *
 * and split it into two
 *
 *            2------                  --3
 *           /    --'             ----' /  B
 *          /  --'  A         2--'-----/
 *         /--'
 *         1
 *
 * then scan from point 1 upwards line by line to fill A
 * and do triangle B in the same manner to point 3.
 */

/*
 * coords relative to centre of window and in pixels
 * comp = 1 -> reversed buffer
 * comp = 0 -> normal buffer
 */
#define ZFLOAT 0
/* cannot decide which is faster */
#if ZFLOAT
static void triangle(int x1, int y1, float z1, int x2, int y2, float z2,
		     int x3, int y3, float z3, int **zbuffer, const int comp)
#else
static void triangle(int x1, int y1, int z1, int x2, int y2, int z2,
		     int x3, int y3, int z3, int **zbuffer, const int comp)
#endif
{
   int i, j, ox, oy, y, swapped;
   int axl, axr, ayl, ayr, azl, azr;
   int x12, x13, x23, y12, y13, y23;
   int xl, xr, zl, zr;
   float temp;
   Edgelist *elent, *eelent;


   /* sort points in increasing y */
#if ZFLOAT
   if (y1 > y2) {
      axr=x1; ayr=y1; temp=z1;
      x1=x2; y1=y2; z1=z2;
      x2=axr; y2=ayr; z2=temp;
   }
   if (y1 > y3) {
      axr=x1; ayr=y1; temp=z1;
      x1=x3; y1=y3; z1=z3;
      x3=axr; y3=ayr; z3=temp;
   }
   if (y2 > y3) {
      axr=x3; ayr=y3; temp=z3;
      x3=x2; y3=y2; z3=z2;
      x2=axr; y2=ayr; z2=temp;
   }
#else
   if (y1 > y2) {
      axr=x1; ayr=y1; azr=z1;
      x1=x2; y1=y2; z1=z2;
      x2=axr; y2=ayr; z2=azr;
   }
   if (y1 > y3) {
      axr=x1; ayr=y1; azr=z1;
      x1=x3; y1=y3; z1=z3;
      x3=axr; y3=ayr; z3=azr;
   }
   if (y2 > y3) {
      axr=x3; ayr=y3; azr=z3;
      x3=x2; y3=y2; z3=z2;
      x2=axr; y2=ayr; z2=azr;
   }
#endif

   x12 = x2-x1;
   x13 = x3-x1;
   x23 = x3-x2;
   y12 = y2-y1;
   y13 = y3-y1;
   y23 = y3-y2;

   /* work out the cross-product in the z-dir */
   temp = ((float) x12) * y13 - y12 * ((float) x13);
   if (temp == 0.0) return;       /* points project on to a line */

   /* make sure the beginning of the line is on the left */
   /* and initialise variables */
   ox = width/2;
   oy = height/2;
#if PPP
fprintf(stderr, "x: %d %d %d\n", x1+ox, x2+ox, x3+ox);
fprintf(stderr, "y: %d %d %d\n", oy-y1, oy-y2, oy-y3);
#if ZFLOAT
   fprintf(stderr, "z: %f %f %f\n", z1, z2, z3);
#else
   fprintf(stderr, "z: %d %d %d\n", z1, z2, z3);
#endif
#endif
   swapped = 0;
   y = oy - y1;
   if (y < buf_top) return; /* entire thing out of view */
   j = oy - y2;
   if (y12 == 0) {
      /* no triangle A */
      if (x12 > 0) {
	 swapped = 1;
	 xl = x1 + ox;
	 zl = z1;
      } else {
	 xr = x1 + ox;
	 zr = z1;
      }
   } else {
      /* clip triangle A in vertical direction, if necessary */
      if (temp > 0.0) swapped = 1;

      if (j < buf_bot) { /* point 2 is above buffer bottom */
	 if (y >= buf_bot) { /* pt 1 below buffer */
	    /* set the correct values for x?, y & z? */
	    y -= buf_bot - 1; /* height of omitted strip */
	    if (swapped) {
	       temp = ((float)y)/y13;
	       xl = (int)(x13*temp) + x1 + ox;
	       zl = (int)((z3-z1)*temp) + z1;
	       temp = ((float)y)/y12;
	       xr = (int)(x12*temp) + x1 + ox;
	       zr = (int)((z2-z1)*temp) + z1;
	       axr = x2 - xr + ox;
	       ayr = y12 - y;
	       azr = z2 - zr;
	    } else {
	       temp = ((float)y)/y12;
	       xl = (int)(x12*temp) + x1 + ox;
	       zl = (int)((z2-z1)*temp) + z1;
	       axl = x2 - xl + ox;
	       ayl = y12 - y;
	       azl = z2 - zl;
	       temp = ((float)y)/y13;
	       xr = (int)(x13*temp) + x1 + ox;
	       zr = (int)((z3-z1)*temp) + z1;
	    }
	    y = buf_bot - 1;
	 } else {
	    xl = xr = x1 + ox;
	    zl = zr = z1;
	    if (swapped) {
	       axr = x12;
	       ayr = y12;
	       azr = z2-z1;
	    } else {
	       axl = x12;
	       ayl = y12;
	       azl = z2-z1;
	    }
	 }
	 if (swapped)
      	    rbres3dln(xr, y, zr, axr, ayr, azr);
	 else
      	    lbres3dln(xl, y, zl, axl, ayl, azl);
      }
   }

   i = y;
   y = j;
   j = oy - y3;
   if (y < buf_top) { /* A too tall */
      if (swapped) {
	 ayl = i - j;
	 axl = x3 - xl + ox;
	 azl = z3 - zl;
	 lbres3dln(xl, i, zl, axl, ayl, azl);
      } else {
	 ayr = i - j;
	 axr = x3 - xr + ox;
	 azr = z3 - zr;
	 rbres3dln(xr, i, zr, axr, ayr, azr);
      }
      j = buf_top;
   } else {
      /* clip triangle B in vertical direction, if necessary */
      if (j >= buf_bot) return; /* pt 3 below buffer */
      if (y >= buf_bot) { /* pt 2 below buffer */
	 /* set the correct values for d?, x?, y & z? */
	 y -= buf_bot - 1;
	 ayl = y23 - y; /* height of remaining strip */
	 ayr = ayl;
	 if (swapped) {
	    temp = -((float)ayl)/y13;
	    xl = (int)(x13*temp) + x3 + ox;
	    zl = (int)((z3-z1)*temp) + z3;
	    temp = -((float)ayl)/y23;
	    xr = (int)(x23*temp) + x3 + ox;
	    zr = (int)((z3-z2)*temp) + z3;
	 } else {
	    temp = -((float)ayl)/y23;
	    xl = (int)(x23*temp) + x3 + ox;
	    zl = (int)((z3-z2)*temp) + z3;
	    temp = -((float)ayl)/y13;
	    xr = (int)(x13*temp) + x3 + ox;
	    zr = (int)((z3-z1)*temp) + z3;
	 }
	 y = buf_bot - 1;
	 i = y;
      } else {
	 if (swapped) {
	    ayl = i - j;
	    xr = x2 + ox;
	    zr = z2;
	    ayr = y - j;
	 } else {
	    xl = x2 + ox;
	    zl = z2;
	    ayl = y - j;
	    ayr = i - j;
	 }
      }
      axl = x3 - xl + ox;
      azl = z3 - zl;
      axr = x3 - xr + ox;
      azr = z3 - zr;
      if (j < buf_top) j = buf_top; /* B too tall */
      if (swapped) {
	 lbres3dln(xl, i, zl, axl, ayl, azl);
	 if (ayr != 0) rbres3dln(xr, y, zr, axr, ayr, azr);
	 else {
	    elent = &relist[y%buf_h];
	    elent->x = xr;
	    elent->z = zr;
	 }
      } else {
	 if (ayl != 0) lbres3dln(xl, y, zl, axl, ayl, azl);
	 else {
	    elent = &lelist[y%buf_h];
	    elent->x = xl;
	    elent->z = zl;
	 }
	 rbres3dln(xr, i, zr, axr, ayr, azr);
      }
   }

   /* render the triangle */
   elent = &lelist[i%buf_h]; /* two edge lists */
   eelent = &relist[i%buf_h];
   for (; i>=j; i--) {
#if 0
#if PPP
fprintf(stderr, "y=%d\tx: %d %d\t", i, elent->x, eelent->x);
fprintf(stderr, "z: %d %d\n", elent->z, eelent->z);
#endif
#endif
      lnrender(i, elent, eelent, zbuffer, comp);
      elent--;
      eelent--;
   }
}


/*
 * an object,
 * made from its polygonal faces
 * hidden surfaces are removed using a z-buffer
 * polygons are constructed from triangles
 */

void sfillobj(Object *pobj, float xm, float ym, float zm,
	      int **zbufa, int **zbufb, Affine *gt)
{
   int xa, ya, xb, yb, xc, yc;
#if ZFLOAT
   int xar, xbr, xcr;
   float sa, sb, sc;
#else
   int xar, xbr, xcr, sa, sb, sc;
#endif
   float efn, es, oy, os, vd, vdos, zt;
   int cflag;
   usint *polygon, *epoly, *fpoly;
   float *lkup;
   Affine taf;
   Point tpt;

   es = ES;
   es *= reverse;
   os = OS;
   lkup = zplookup;
   lkup += (int) ((reverse>0) ? ZMUL*height : ZMUL*os);
   os *= reverse;
   vd = VD;
   vdos = vd + os;
   efn = enlarge*(1 << fineness);

   /*
    * points are given in normalised coordinates,
    * we scale it all up relative to the centre of the window and
    * window height
    */

   oy = height/2.0;
   xm *= oy;
   ym *= oy;
   zm *= oy;
   taf = pobj->wt;
   mulaffine(&taf, gt);
   magaffine(&taf, xm, ym, zm);

   cflag = (1+reverse)/2; /* = 1 for wide, = 0 for cross-eyed */
   polygon = pobj->polygon;
   epoly = polygon + pobj->nindex;
   do {
      fpoly = polygon + 1; /* two stages to avoid nasty precedence */
      fpoly += *polygon++;
      applyaffine(&tpt, &pobj->ri[*polygon++], &taf);
      zt = tpt.z;
      switch(cflag) {
       case 0:
	 if (zt < os) zt = os; /* os is actually -os */
	 else if (zt >= vdos) zt = vdos - 0.01*oy;
	 break;
       case 1: default:
	 if (zt > os) zt = os;
	 else if (zt < -height) zt = (float) -height;
	 break;
      }
/*      zt = vd/(vdos - zt); */
      zt = *(lkup+(int)(ZMUL*zt));
      xa = (int) (zt*tpt.x);
      ya = (int) (zt*tpt.y);
      zt = es*(1 - zt);
      xa -= zt/2.0;
      xar = (int)(xa + zt);
#if ZFLOAT
      sa = zt*efn;
#else
      sa = (int)(efn*zt);
#endif
      applyaffine(&tpt, &pobj->ri[*polygon++], &taf);
      zt = tpt.z;
      switch(cflag) {
       case 0:
	 if (zt < os) zt = os;
	 else if (zt >= vdos) zt = vdos - 0.01*oy;
	 break;
       case 1: default:
	 if (zt > os) zt = os;
	 else if (zt < -height) zt = (float) -height;
	 break;
      }
      zt = *(lkup+(int)(ZMUL*zt));
      xb = (int) (zt*tpt.x);
      yb = (int) (zt*tpt.y);
      zt = es*(1 - zt);
      xb -= zt/2.0;
      xbr = (int)(xb + zt);
#if ZFLOAT
      sb = efn*zt;
#else
      sb = (int)(efn*zt);
#endif
      do {
	 applyaffine(&tpt, &pobj->ri[*polygon++], &taf);
	 zt = tpt.z;
	 switch(cflag) {
	    case 0:
	    if (zt < os) zt = os;
	    else if (zt >= vdos) zt = vdos - 0.01*oy;
	    break;
	    case 1: default:
	    if (zt > os) zt = os;
	    else if (zt < -height) zt = (float) -height;
	    break;
	 }
	 zt = *(lkup+(int)(ZMUL*zt));
	 xc = (int) (zt*tpt.x);
	 yc = (int) (zt*tpt.y);
	 zt = es*(1 - zt);
	 xc -= zt/2.0;
	 xcr = (int)(xc + zt);
#if ZFLOAT
	 sc = efn*zt;
#else
	 sc = (int)(efn*zt);
#endif
         triangle(xa, ya, sa, xb, yb, sb, xc, yc, sc, zbufa, cflag);
         triangle(xar, ya, sa, xbr, yb, sb, xcr, yc, sc, zbufb, cflag);
	 xbr = xcr;
	 xb = xc;
	 yb = yc;
	 sb = sc;
      } while (polygon<fpoly);
   } while (polygon<epoly);
}


void zfillobj(Object *pobj, float xm, float ym, float zm, int **zbuf,
	      Affine *gt)
{
   int xa, ya, xb, yb, xc, yc;
#if ZFLOAT
   float za, zb, zc;
#else
   int za, zb, zc;
#endif
   float efn, oy, os, zt;
   usint *polygon, *epoly, *fpoly;
   Affine taf;
   Point tpt;

   os = OS;
   efn = enlarge*(1 << fineness);

   oy = height/2.0;
   xm *= oy;
   ym *= oy;
   zm *= oy;
   taf = pobj->wt;
   mulaffine(&taf, gt);
   magaffine(&taf, xm, ym, zm);

   polygon = pobj->polygon;
   epoly = polygon + pobj->nindex;
   do {
      fpoly = polygon + 1;
      fpoly += *polygon++;
      applyaffine(&tpt, &pobj->ri[*polygon++], &taf);
      zt = tpt.z;
      if (zt > os) zt = os;
      else if (zt < -oy) zt = -oy;
      xa = (int)(tpt.x);
      ya = (int)(tpt.y);
#if ZFLOAT
      za = efn*zt;
#else
      za = (int)(efn*zt);
#endif
      applyaffine(&tpt, &pobj->ri[*polygon++], &taf);
      zt = tpt.z;
      if (zt > os) zt = os;
      else if (zt < -oy) zt = -oy;
      xb = (int)(tpt.x);
      yb = (int)(tpt.y);
#if ZFLOAT
      zb = efn*zt;
#else
      zb = (int)(efn*zt);
#endif
      do {
	 applyaffine(&tpt, &pobj->ri[*polygon++], &taf);
	 zt = tpt.z;
	 if (zt > os) zt = os;
	 else if (zt < -oy) zt = -oy;
	 xc = (int)(tpt.x);
	 yc = (int)(tpt.y);
#if ZFLOAT
	 zc = efn*zt;
#else
	 zc = (int)(efn*zt);
#endif
         triangle(xa, ya, za, xb, yb, zb, xc, yc, zc, zbuf, 0);
#if 0
#if ZFLOAT
fprintf(stderr, " vertices (%d %d %f) (%d %d %f) (%d %d %f)\n",
	xa, ya, za, xb, yb, zb, xc, yc, zc);
#else
fprintf(stderr, " vertices (%d %d %d) (%d %d %d) (%d %d %d)\n",
	xa, ya, za, xb, yb, zb, xc, yc, zc);
#endif
#endif
	 xb = xc;
	 yb = yc;
	 zb = zc;
      } while (polygon<fpoly);
   } while (polygon<epoly);
}


/*
 * scan z-buffer and use a MxN square of z-values to
 * fill 's'-buffers
 */
void zsdrawobj(int *zmap, int **zbufl, int **zbufr,
	       int ttw, int tth, int xstep, int ystep, int half)
{
   int xa, ya, xb, yb, xc, yc, xd, yd;
   float sa, sb, sc, sd;
   float efn, es, os;
   int cflag;
   int z[4];
   int *zuline, *zdline;
   int l, m, n, p;
   int zl, zm, zn;
   float zt;
   int x, y, yt, rx, ry;
   int ox, oy;
   float *lkup;

   es = ES;
   es *= reverse;
/*   vd = VD; */
   os = OS;
/*   vdos = vd + os; */
   efn = enlarge*(1 << fineness);
   oy = tth/2;
   ox = ttw/2;
   lkup = zplookup;
   lkup += (int) ((reverse>0) ? ZMUL*height : ZMUL*os);

   cflag = (1+reverse)/2;
   switch(half) {
    case 0:
      ry = oy;
      y = ystep;
      yt = tth/2;
      break;
    case 1:
      ry = -ystep;
      y = tth/2;
      yt = tth;
      break;
    case 2: default:
      ry = oy;
      y = ystep;
      yt = tth;
      break;
   }
   for (; y<yt; ry-=ystep, y+=ystep) {
      rx = -ox;
      zuline = &zmap[(y-ystep)*ttw];
      zdline = &zmap[y*ttw];
      z[0] = *zuline;
      zuline += xstep;
/*      zt = vd/(vdos - z[0]); */
      zt = *(lkup+(ZMUL*z[0]));
      xa = (int)(zt*rx);
      ya = (int)(zt*ry);
      sa = es*(1 - zt);
      xa -= sa/2;
      z[3] = *zdline;
      zdline += xstep;
      zt = *(lkup+(ZMUL*z[3]));
      xd = (int)(zt*rx);
      yd = (int)(zt*(ry-ystep));
      sd = es*(1 - zt);
      xd -= sd/2;
      for (x=xstep; x<ttw; x+=xstep) {
	 rx += xstep;
	 z[1] = *zuline;
	 zuline += xstep;
	 zt = *(lkup+(ZMUL*z[1]));
	 xb = (int)(zt*rx);
	 yb = (int)(zt*ry);
	 sb = es*(1 - zt);
	 xb -= sb/2;
	 z[2] = *zdline;
	 zdline += xstep;
	 zt = *(lkup+(ZMUL*z[2]));
	 xc = (int)(zt*rx);
	 yc = (int)(zt*(ry-ystep));
	 sc = es*(1 - zt);
	 xc -= sc/2;

	 if ((z[0]-z[1]+z[2]-z[3]) == 0) {
            /* planar case, 4 vertices */
	    m = 1;
	 } else {
	    /* non-planar case -> two triangle */
            /* find most and least */
            l = m = 0;
            zl = zm = z[0];
            for (n=1; n<4; n++) {
               zn = z[n];
               if (zn > zm) {
                  zm = zn;
                  m = n;
               } else if (zn < zl) {
                  zl = zn;
                  l = n;
               }
	    }
            /* check if there are common heights */
            switch(l-m) {
             case -3: case -1: /* min clockwise adjacent from max */
               p = (m+1) & 3;
               if (zm == z[p]) m = l;
               break;
             case -2: case 2:  /* min opposite to max */
               break;
             case 1: case 3:   /* min clockwise adjacent from max */
               p = (l+1) & 3;
               if (z[p] != zl) m = l;
               break;
             case 0: default:
               break;
            }
	 }

	 if ((m&1) == 1) { /* start at corner a */
#if ZFLOAT
	    triangle(xa, ya, efn*sa, xb, yb, efn*sb,
		     xc, yc, efn*sc, zbufl, cflag);
	    triangle(xa, ya, efn*sa, xc, yc, efn*sc,
		     xd, yd, efn*sd, zbufl, cflag);
	    triangle((int)(xa+sa), ya, efn*sa, (int)(xb+sb), yb, efn*sb,
		     (int)(xc+sc), yc, efn*sc, zbufr, cflag);
	    triangle((int)(xa+sa), ya, efn*sa, (int)(xc+sc), yc, efn*sc,
		     (int)(xd+sd), yd, efn*sd, zbufr, cflag);
         } else {  /* end at corner a */
	    triangle(xd, yd, efn*sd, xa, ya, efn*sa,
		     xb, yb, efn*sb, zbufl, cflag);
	    triangle(xd, yd, efn*sd, xb, yb, efn*sb,
		     xc, yc, efn*sc, zbufl, cflag);
	    triangle((int)(xd+sd), yd, efn*sd, ((int)xa+sa), ya, efn*sa,
		     (int)(xb+sb), yb, efn*sb, zbufr, cflag);
	    triangle((int)(xd+sd), yd, efn*sd, (int)(xb+sb), yb, efn*sb,
		     (int)(xc+sc), yc, efn*sc, zbufr, cflag);
         }
#else
	    triangle(xa, ya, (int)(efn*sa), xb, yb, (int)(efn*sb),
		     xc, yc, (int)(efn*sc), zbufl, cflag);
	    triangle(xa, ya, (int)(efn*sa), xc, yc, (int)(efn*sc),
		     xd, yd, (int)(efn*sd), zbufl, cflag);
	    triangle((int)(xa+sa), ya, (int)(efn*sa), 
		     (int)(xb+sb), yb, (int)(efn*sb),
		     (int)(xc+sc), yc, (int)(efn*sc), zbufr, cflag);
	    triangle((int)(xa+sa), ya, (int)(efn*sa),
		     (int)(xc+sc), yc, (int)(efn*sc),
		     (int)(xd+sd), yd, (int)(efn*sd), zbufr, cflag);
         } else {  /* end at corner a */
	    triangle(xd, yd, (int)(efn*sd), xa, ya, (int)(efn*sa),
		     xb, yb, (int)(efn*sb), zbufl, cflag);
	    triangle(xd, yd, (int)(efn*sd), xb, yb, (int)(efn*sb),
		     xc, yc, (int)(efn*sc), zbufl, cflag);
	    triangle((int)(xd+sd), yd, (int)(efn*sd),
		     (int)(xa+sa), ya, (int)(efn*sa),
		     (int)(xb+sb), yb, (int)(efn*sb), zbufr, cflag);
	    triangle((int)(xd+sd), yd, (int)(efn*sd),
		     (int)(xb+sb), yb, (int)(efn*sb),
		     (int)(xc+sc), yc, (int)(efn*sc), zbufr, cflag);
         }
#endif

	 z[0] = z[1];
	 xa = xb;
	 ya = yb;
	 sa = sb;
	 z[3] = z[2];
	 xd = xc;
	 yd = yc;
	 sd = sc;
      }
   }
}


/* memory allocation for zbuffer */
int **zmema(void)
{
   int i;
   int **zbuf;

   DIEIF((zbuf = (int **) malloc(buf_h*sizeof(int *))) == NULL,
         "Unable to allocate memory for z-buffer\n");
   for (i=0; i<buf_h; i++) {
      DIEIF((zbuf[i] = (int *) malloc(width*sizeof(int))) == NULL,
            "Unable to allocate memory for z-buffer\n");
   }
   if (lelist == NULL)
      DIEIF((lelist = (Edgelist *) malloc(buf_h*sizeof(Edgelist))) == NULL,
	    "Unable to allocate memory for edge-list\n");
   if (relist == NULL)
      DIEIF((relist = (Edgelist *) malloc(buf_h*sizeof(Edgelist))) == NULL,
	    "Unable to allocate memory for edge-list\n");
   return zbuf;
}


void zmemf(int **zbuf)
{
   int i;

   if (zbuf != NULL) {
      for(i=0; i<buf_h; i++) free(zbuf[i]);
      free(zbuf);
   }
   if (lelist != NULL) {
      free(lelist);
      lelist = NULL;
   }
   if (relist != NULL) {
      free(relist);
      relist = NULL;
   }
}
