/*
 *
SURF: an X application to fine-tune, monitor, and control numerical simulations.
 *		Version 1.02
 *    Copyright (C) 1994 Weimin Zhao
 *
 * $Id$
 *
*/
/*
    This program and the library built upon it are free software;
    you can redistribute it and/or modify it under the terms of the GNU
    General Public License (GPL) and Library General Public License (LGPL)
    as published by the Free Software Foundation; either version 2 of the
    License, or any later version.

    This program and the library are distributed in the hope that it will
    be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
    of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*
 *	Bug reports to 		wzhao@mcs.kent.edu
 *		Weimin Zhao
 *		Liquid Crystal Institute, Kent State University, Kent, OH 44242, USA
*/
/******************************************************************************
*
*    This program calls a Fortran subroutine, fortransub(i, &x, &y, &z), which
* returns the 3D cooridinate (x, y, z) and then plot it on a 2D, Cartisian or
* polar coordinate system while respresenting z with the hue of color shades.
*
*******************************************************************************/

#include	"surf.h"
#include	<sys/times.h>
#include	<sys/types.h>
#include	<sys/wait.h>
#define		MAX_POP_STR		20
Steps		steps;
char		*progname;
Frame		frame;
int			nColors;
double		hueColor, chromColor, minColorV, maxColorV;
double		hotV;
double		pntrX, pntrY;
int			popWidth, popHeight;
Window		infoWin;
Position3D	maxPnt, minPnt;
#ifdef	HPUX
clock_t		timeU;
#else
time_t		timeU;
#endif
extern void	fortransub(int*, double*, double*, double*, int*, double*, double*, double*);


/******************************************************************************
*    The followings are file-scope variables.
******************************************************************************/
static int			start_x, start_y, sizeChunk, pop_width, pop_height;
static char			pop_name, pop_str[MAX_POP_STR], popVarName[10];
static Bool			NeedsRefresh = False, popped = False, saveLastChunk;
static Position3D	*copy = NULL;
static Steps		refresh;
static double		*xChunk = NULL, *yChunk = NULL, *zChunk = NULL;
static KeySym		keysym;
static int			bufsize=20;
static char			buffer[20];
static char			intPName[22][15];
static char			dblPName[22][15];
static int			intParm[22];
static double		dblParm[22], *hotA0 = NULL, *hotA = NULL;
static XEvent		report;
static int			isFuncKey, indxVar;
static IntParm		optNamParm[5];
static int			NumIntParm, NumDblParm, NumOptParm=5;
static Window		pop_win;
static Bool			infoWinPopped = False, popInfoAgain = False;
static Window		quitWin;
static Bool			newPos = False, quitWinPopped = False;
static double		oldR;
static int			oldX, oldY, newX, newY;
static Bool			getFileName = False;
static int			infoWinContent;
static	Bool		firstTime = True;
/*****************************************************************************/
extern void			updateParm(void);


void	Refresh(Bool ReDraw, int start, int end) {
int		i, Connect;

	/*
XEvent	dummy;
	while(XCheckTypedWindowEvent (display, win, Expose, &dummy));
	while(XCheckTypedWindowEvent (display, win, GraphicsExpose, &dummy));
	*/

	PlaceFrame();
	UpdateSteps('a');

	if(ReDraw) {
		if(frame.type == 15)
			for (i=start; i<end; i++)
				DrawPoint(copy[i], i%frame.x_div );
		else if(frame.type == 16)
			for (i=start; i<end; i++) {
				if(i%frame.x_div == 0 || i%frame.x_div == 1)
					Connect = 0;
				else if(i % 2 == 0)
				Connect = 2;
				else
				Connect = 1;
				DrawPoint(copy[i], Connect );
			}
		else
			for (i=start; i<end; i++)
				DrawPoint(copy[i], Connect );
	}
	XSetForeground(display, gc, pxlBW[0]);
	PlaceFrame();
	colorNow = -2;
	newPos = False;
}

Bool stopFortsub() {
int		bufsize=20, tmpCharCount;
KeySym	tmpKeysym;
XComposeStatus	tmpCompose;
char	tmpBuffer[20];
XEvent	tmpReport;
	if(XCheckWindowEvent(display, win, KeyPressMask, &tmpReport))
		tmpCharCount=XLookupString(&tmpReport.xkey, tmpBuffer,
			bufsize,&tmpKeysym,&tmpCompose);
		if(tmpBuffer[0] == 'x')
			return True;
	return False;
}

void	processGo(Bool stepWise) {
int			ii, i, j, jStart;
int			Connect, Begin, End;
struct tms	timeBuf;
#ifdef	HPUX
static clock_t		timeU0;
#else
static time_t		timeU0;
#endif

	refresh.from = (steps.from >= 0) ? steps.from : 0;
	times(&timeBuf);
	if(!stepWise) {
		XClearArea(display, win, (int)(0.05*width), (int)(0.05*height),
			(int)(0.9*width), (int)(0.9*height), False);
		NeedsRefresh = False;
		Refresh(NeedsRefresh, refresh.from, refresh.to);
		hotV = 0.0;
		timeU0 = timeBuf.tms_utime;
	}
	NeedsRefresh = True;
	newPos = False;

	if(stepWise)
		Begin = End = steps.now + 1;
	else {
		Begin = steps.from;
		End = steps.to;
	}
	for (i=Begin; i<=End; i++) {
	wait((void*)0 );
		fortransub(&i, xChunk, yChunk, zChunk, intParm, dblParm, &hotV, hotA);
		times(&timeBuf);
		timeU = timeBuf.tms_utime - timeU0;
		steps.now = i;
		UpdateSteps('n');
		if( steps.from == -1) break;
		if(!saveLastChunk) {
			for (j=0; j<sizeChunk; j++) {
				jStart = (i-refresh.from)*sizeChunk+j;
				copy[jStart].x = xChunk[j];
				copy[jStart].y = yChunk[j];
				if(frame.type == 15) {
					copy[jStart].z = steps.from+(steps.to-steps.from+1)
						*floor(1.*jStart/frame.x_div)*frame.x_div
						/((steps.to-steps.from+1)*sizeChunk) ;
					Connect = (jStart%frame.x_div != 0) ? 1 : 0;
				}
				else if(frame.type == 16) {
					copy[jStart].z = steps.from+(steps.to-steps.from+1)
						*floor(1.*jStart/frame.x_div)*frame.x_div
						/((steps.to-steps.from+1)*sizeChunk) ;
					if(jStart%frame.x_div == 0
						|| jStart%frame.x_div == 1)
						Connect = 0;
					else if(jStart % 2 == 0)
						Connect = 2;
					else
						Connect = 1;
				}
				else
					copy[jStart].z = zChunk[j];
				DrawPoint(copy[jStart], Connect );
				if(i==refresh.from && j==0)
					for(ii=0; ii<intParm[7]; ii++) {
						hotA0[ii] = hotA[ii];
					}
			}
			refresh.from = 0;
			refresh.to = (i+1)*sizeChunk;
		}
		else {
			for (j=0; j<sizeChunk; j++) {
				copy[j].x = xChunk[j];
				copy[j].y = yChunk[j];
				copy[j].z = zChunk[j];
				DrawPoint(copy[j], False);
			}
			refresh.from = 0;
			refresh.to = steps.now;
			if(i==refresh.from)
				for(ii=0; ii<intParm[7]; ii++) {
					hotA0[ii] = hotA[ii];
				}
		}
		if(stopFortsub()) break;
	}
	XSetForeground(display, gc, pxlBW[0]);
	PlaceFrame();
	colorNow = -2;
}

void	saveFile(FILE *file) {
int	i;

	fprintf(file, "# START of fortsub.rc.  Integer paramer follows.\n");
	for(i=0; i<NumIntParm; i++)
		fprintf(file, "%-12s\t%d\n", intPName[i], intParm[i]);
	fprintf(file, "# The following are double precision parameters.\n");
	for(i=0; i<NumDblParm; i++)
		fprintf(file, "%-12s\t%g\n", dblPName[i], dblParm[i]);
	fprintf(file, "# The following are configuration parameters.\n");
	for(i=0; i<NumOptParm; i++)
		fprintf(file, "%-12s\t%d\n", optNamParm[i].name, optNamParm[i].value);
	fprintf(file, "# END of the parameter file.\n");

}

void	getKeyInput(void) {
FILE	*file;
int		maxSteps;

	switch (pop_name) {
		case 'f':
			steps.from = atoi(pop_str);
			if(steps.from < -1) {
				XBell(display, 50);
				steps.from = steps.to = -1;
				}
			else if (steps.from == -1)
				steps.to = -1;
			UpdateSteps(pop_name);
			break;
			/*
		case 'n':
			steps.to = steps.from = atoi(pop_str);
			UpdateSteps(pop_name); break;
			processGo(True);
			break;
			*/
		case 'Y': maxPnt.y = atof(pop_str);
				NeedsRefresh = True;
				XClearArea(display, win, 0, 0, width, height, False);
				Refresh(NeedsRefresh, refresh.from, refresh.to);
				break;
		case 'y': minPnt.y = atof(pop_str); 
				NeedsRefresh = True;
				XClearArea(display, win, 0, 0, width, height, False);
				Refresh(NeedsRefresh, refresh.from, refresh.to);
				break;
		case 'Z': maxPnt.z = atof(pop_str);
			NeedsRefresh = True;
			XClearArea(display, win, 0, 0, width, height, False);
			Refresh(NeedsRefresh, refresh.from, refresh.to);
			break;
		case 'z': minPnt.z = atof(pop_str);
			NeedsRefresh = True;
			XClearArea(display, win, 0, 0, width, height, False);
			Refresh(NeedsRefresh, refresh.from, refresh.to);
			break;
		case 't':
			maxSteps = rndInt(frame.x_div*frame.y_div/sizeChunk)+steps.from-1;
			if(frame.type < 15 && atoi(pop_str) > maxSteps && !saveLastChunk ) {
				XBell(display, 50);
				steps.to=maxSteps;
			}
			else 
				steps.to = atoi(pop_str);
				updateParm();
			UpdateSteps(pop_name);
			break;
		case 'r':
			if(!(file = fopen(pop_str, "r"))) {
				processQuit("Cannot open file for read!");
			}
			else {
			initParm(file);
			fclose(file);
			NeedsRefresh = False;
			XClearWindow(display, win);
			getFileName = False;
			}
			break;
		case 's':
			if(!(file = fopen(pop_str, "w"))) {
				processQuit("Cannot open file for write!");
			}
			else {
			saveFile(file);
			fclose(file);
			getFileName = False;
			}
			break;
		default:  break;
	}
}

void	getFuncInput(void) {
int		tmp;

	if(isFuncKey == 1) {
		tmp = atoi(pop_str);
		if(frame.type == 16 && ((indxVar == 5 || indxVar == 2) && tmp%2 != 0)) {
			XBell(display, 100);
			return;
		}
		sprintf(popVarName, "%s: ", intPName[indxVar]);
		intParm[indxVar] = tmp;
		return;
	}
	else if(isFuncKey == 2) {
		sprintf(popVarName, "%s: ", dblPName[indxVar]);
		dblParm[indxVar] = atof(pop_str);
		/*
		printf("%s is set to %.2f\n", dblPName[indxVar], dblParm[indxVar]);
		*/
		return;
	}
}

void	updateParm(void) {
static	int nArray=0, nArray1=0, oldStepsTo = 0;
int		tmp;

	NumIntParm = intParm[0];
	NumDblParm = intParm[1];
	frame.type = intParm[4];
	nArray1 = intParm[8];
	maxPnt.x = dblParm[0];
	maxPnt.y = dblParm[1];
	minPnt.x = dblParm[3];
	minPnt.y = dblParm[4];

	if(nArray != intParm[7]) {
#ifdef HPUX  /* Just can't understand why I can't use realloc with HPUX! */
		if(!(hotA = (double *)malloc(sizeof(double )* intParm[7])))
			DIE("Can't malloc for the hot array!\n");
		if(!(hotA0 = (double *)malloc(sizeof(double )* intParm[7])))
			DIE("Can't malloc for the hot array0!\n");
#else
		if(!(hotA = (double *)realloc(hotA, sizeof(double )* intParm[7])))
			DIE("Can't realloc for the hot array!\n");
		if(!(hotA0 =  (double *)realloc(hotA0, intParm[7]* sizeof(double ))))
			DIE("Can't realloc for the hot array0!\n");
#endif
		nArray = intParm[7];
	}
	if(firstTime) {
		hotV = 0.0;
		steps.now = steps.from;
		maxPnt.z = dblParm[2];
		minPnt.z = dblParm[5];
		if(nArray1 > nArray)
			DIE("surf: Size of first array is too large.\n");
	}
	else {
		if(nArray1 > nArray) {
			XBell(display, 50);
			intParm[8] = nArray1 = nArray;
		}
	}

	saveLastChunk = (frame.type < 15) ? intParm[6] : 0;
	if(sizeChunk != intParm[5] || frame.x_div != intParm[2]
		|| frame.y_div != intParm[3]
		|| (frame.type >= 15 && oldStepsTo != steps.to) ) {
		frame.x_div = intParm[2];
		frame.y_div = intParm[3];
		sizeChunk = intParm[5];

		if(frame.type < 15) {
			/*
			if(steps.from < 0) 
				steps.to = rndInt(frame.x_div*frame.y_div/sizeChunk) - 1;
			else
			*/
			if(steps.from >=0)
				steps.to = rndInt(frame.x_div*frame.y_div/sizeChunk)
				+ steps.from - 1;
		}
		else if(firstTime) {
			steps.from = rndInt(minPnt.z);
			steps.to = rndInt(maxPnt.z);
		}
		tmp	= (!saveLastChunk) ? sizeChunk*(steps.to+1): sizeChunk ;
		oldStepsTo = steps.to;
		if(!(xChunk = (double *)realloc(xChunk, sizeof(double )*
			sizeChunk))) DIE("Can't realloc for xChunk!\n");
		if(!(yChunk = (double *)realloc(yChunk, sizeof(double )*
			sizeChunk))) DIE("Can't realloc for yChunk!\n");
		if(!(zChunk = (double *)realloc(zChunk, sizeof(double )*
			sizeChunk))) DIE("Can't realloc for zChunk!\n");
		if(!(copy = (Position3D *)realloc(copy, sizeof(Position3D) * 
			tmp ))) DIE("Can't allocate memory for copy!\n");
	}
	firstTime = False;

}

void	processInput(void) {
int		pop_len;

	if ((keysym==XK_Return)||(keysym==XK_KP_Enter)
		||(keysym==XK_Linefeed)) {
		XClearArea(display, pop_win, 0, 0, pop_width, pop_height, False);
		XUnmapWindow(display, pop_win);
		popped = False;
		if(strlen(pop_str) >=1) 
			if(!isFuncKey) getKeyInput();
			else {
				getFuncInput();
				updateParm();
			}
		return;
	}
	else if (keysym == XK_Escape) {
	/*
		XClearArea(display, win, (int)(0.05*width), (int)(0.05*height),
			(int)(0.9*width), (int)(0.9*height), False);
		XClearWindow(display, win);
	*/
		NeedsRefresh = False;
		XUnmapWindow(display, pop_win);
		popped = False;
		return;
	}
	else if ( (keysym >= XK_0 && keysym <= XK_9 )
		|| (keysym >= XK_KP_0 && keysym <= XK_KP_9 )
		|| keysym == XK_period || buffer[0] == '-') {
		if( (strlen(pop_str) + strlen(buffer)) >= MAX_POP_STR )
			XBell(display, 100);
		else
			strcat(pop_str, buffer);
	}
	else if ( getFileName && ( (keysym >= XK_A && keysym <= XK_Z)
		|| (keysym >= XK_a && keysym <= XK_z) || (keysym == XK_underscore))) {
		if( (strlen(pop_str) + strlen(buffer)) >= MAX_POP_STR )
			XBell(display, 100);
		else
			strcat(pop_str, buffer);
	}
	else if ((keysym==XK_BackSpace) || (keysym==XK_Delete)) {
		if((pop_len=strlen(pop_str)) > 0 ) {
			pop_str[pop_len-1] = '\0';
			XClearWindow(display, pop_win);
			XDrawString(display, pop_win, gc, 4, start_y,
				popVarName,strlen(popVarName));
		}
		else	XBell(display, 100);
	}
	/*
	else XBell(display, 50);
	*/
	XDrawString(display,pop_win,gc,start_x,start_y,pop_str,strlen(pop_str));
}

void	popInfoWin(void) {

	if(!infoWin) {
		popWidth=(int)(0.75*width); popHeight=(int)(0.75*height);
		infoWin = XCreateSimpleWindow(display, win,
			width-popWidth, height-popHeight,
			popWidth, popHeight, 2, pxlBW[0], pxlBW[1]);
		XSelectInput(display,infoWin, ExposureMask|KeyPressMask);
	}
	if(infoWinPopped) XClearArea(display, infoWin, 0, 0,
		popWidth, popHeight, False);
	else {
		XMapWindow(display, infoWin);
		infoWinPopped = True;
	}
	NeedsRefresh = False;
}

void	popMonitor(void) {
	popInfoWin();
	quadPlot(intParm[7], intParm[8], hotA, hotA0);
}

void	popInfo(char what) {

static char	*helpStr[] = {		/* set to auto to reduce mem usage! */
		"f:    set the FROM point of the steps.",
		"t:    set the TO point of the steps.",
		"g:    start the simulation steps.",
		"n:    go to the next step (stepwise mode).",
		"x:    stop the simulation while it is progressing.",
		"c:    clear the screen.",
		"d:    draw the results again.",
		"r:    read a new init file.",
		"s:    save the current setting to a file.",
		"z:    set the minimum value of z (# of steps for 2D case).",
		"Z:    set the maximum value of z (# of steps for 2D case).",
		"y:    set the minimum value of y, for 2D case only.",
		"Y:    set the maximum value of Y, for 2D case only.",
		"q:    quit the surf program.",
		"",
		"F1:        pop up help window.",
		"F2:        pop up status window.",
		"Shft-F1    pop up window displaying select results.",
		"Shft-F2:   pop up window displaying the hot array(s).",
		"",
		"Ctrl-F[1-12]:  reset integer parameters, intp(1-12).",
		"Alt-F[1-12]:   reset real*8 (double) parameters, dblp(1-12).",
		"F[3-12]:       reset integer parameters, intp(13-22).",
		"Shft-F[3-12]:  reset real*8 (double) parameters, dblp(13-22).",
		"",
		"Escape:        get rid of any pop-up windows."
		};
char 		statusStr[22][72], *zeroSpace='\0';
int			i, widthTitle;
static	int	linSpacing;
char		title[40];

	popInfoWin();
	linSpacing = font->max_bounds.ascent+font->max_bounds.descent + 2;
	if(what == 's') {
		sprintf(title, "PRESENT STATUS OF PARAMETERS");
		for (i=0; i< NumIntParm || i < NumDblParm; i++) {
			if(i<NumIntParm && i < NumDblParm)
				sprintf(statusStr[i],
					"Ctrl-F%-2d:  %-12s=%-15d  Alt-F%-2d:  %-12s=%-10g",
					i+1, intPName[i], intParm[i], i+1, dblPName[i], dblParm[i]);
			else if (NumIntParm > NumDblParm)
				sprintf(statusStr[i], "Ctrl-F%-2d:  %-12s=%-20d",
					i+1, intPName[i], intParm[i]);
			else
				sprintf(statusStr[i], "%38s  Alt-F%-2d:%-12s=%-10g", zeroSpace,
					i+1, dblPName[i], dblParm[i]);
			XDrawString(display, infoWin, gc, 50, 60+i*linSpacing,
				statusStr[i], strlen(statusStr[i]));
		}
	}
	else if (what == 'h') {
		sprintf(title, "SUMMARY OF COMMANDS");
		for(i=0; i< 26; i++)
			XDrawString(display, infoWin, gc, 50, 60+i*linSpacing,
				helpStr[i], strlen(helpStr[i]));
	}
	widthTitle=XTextWidth(font, title, strlen(title));
	XDrawString(display, infoWin, gc, (popWidth-widthTitle)/2, 25,
		title, strlen(title));
}

void displayResult(void) {
double	*arrayX, *arrayY;
int		iPhi, iR, i, iX, iY, incr, iFirst, iSecond;

	if(!(arrayX = (double *) malloc(sizeof(double)*frame.x_div)))
		DIE("Cannot malloc for arrayX in displayResult");
	if(!(arrayY = (double *) malloc(sizeof(double)*frame.y_div)))
		DIE("Cannot malloc for arrayY in displayResult");
		iPhi = rndInt(pntrY*frame.y_div/(maxPnt.y-minPnt.y));
		iR = rndInt(pntrX*frame.x_div/(frame.x_max-frame.x_min));
		iX = rndInt((pntrX-minPnt.x)*frame.x_div/(maxPnt.x-minPnt.x));
		iY = rndInt((pntrY-minPnt.y)*frame.y_div/(maxPnt.y-minPnt.y));
	if(frame.type == 0 || frame.type == 10) {
		incr = frame.y_div;
		iFirst = (frame.type == 0) ? iPhi : iY;
		iSecond = (frame.type == 0) ? iR : iX;
	}
	else if (frame.type == 1 || frame.type == 11) {
		incr = frame.x_div;
		iFirst = (frame.type == 1) ? iR : iX;
		iSecond = (frame.type == 1) ? iPhi : iY;
	}
	
	if(frame.type == 0 || frame.type == 10) {
	for(i=0; i< frame.x_div; i++)
		arrayX[i] = copy[i*incr + iFirst].z ;
	for(i=0; i< frame.y_div; i++)
		arrayY[i] = copy[iSecond*incr + i].z;
	}
	else if(frame.type == 1 || frame.type == 11) {
		for(i=0; i< frame.x_div; i++)
			arrayX[i] = copy[iSecond*incr + i].z;
		for(i=0; i<frame.y_div; i++)
			arrayY[i] = copy[i*incr + iFirst].z;
	}
	dblPlot(frame.x_div, frame.y_div, arrayX, arrayY);
	free(arrayX);	free(arrayY);
}

void handleFuncKeys() {

	indxVar = (int)(keysym - XK_F1);

	if (report.xkey.state & ControlMask) {
		if(indxVar >= NumIntParm) {
			XBell(display, 100);
			indxVar = -1;
			return;
		}
		sprintf(popVarName, "%s: ", intPName[indxVar]);
		isFuncKey = 1;
	}
	else if (report.xkey.state & Mod1Mask) {
		if(indxVar >= NumDblParm) {
			XBell(display, 100);
			indxVar = -1;
			return;
		}
		sprintf(popVarName, "%s: ", dblPName[indxVar]);
		isFuncKey = 2;
	}
	else if (report.xkey.state & ShiftMask)
		switch(keysym) {
			case XK_F1:  popInfoAgain = False; displayResult();
				indxVar = -1; infoWinContent = 21; break;
			case XK_F2:  popInfoAgain = False; indxVar = -1;
				popMonitor(); infoWinContent = 22; break;
			default:
				if(indxVar+10 >= NumDblParm) {
					XBell(display, 100);
					indxVar = -1;
					return;
				}
				sprintf(popVarName, "%s: ", dblPName[indxVar+10]);
				isFuncKey = 2;
				break;
		}
	else 
		switch(keysym) {
			case XK_F1:  popInfoAgain = False;
				popInfo('h'); indxVar = -1; infoWinContent = 11; break;
			case XK_F2:  popInfoAgain = False;
				popInfo('s'); indxVar = -1; infoWinContent = 12; break;
			default:
				if(indxVar+10 >= NumIntParm) {
					XBell(display, 100);
					indxVar = -1;
					return;
				}
				sprintf(popVarName, "%s: ", intPName[indxVar+10]);
				isFuncKey = 1;
				break;
		}
}

void	displayPopWin(Bool reDraw) {
	if(!popped) {
		XMapWindow(display, pop_win);
		popped = True;
	}
	start_x = 4+XTextWidth(font, popVarName, strlen(popVarName));
	start_y = font->max_bounds.ascent +4;
	XDrawString(display,pop_win,gc,4,start_y,popVarName,strlen(popVarName));
	if(reDraw)
		XDrawString(display,pop_win,gc,start_x,start_y,pop_str,strlen(pop_str));
	else 
		pop_str[0] = '\0';
	XSetInputFocus(display,pop_win,RevertToParent,CurrentTime);
}

/* Check if popped and set popVarName for keyed input. */
void	handlePop(void) {
	if(!pop_win) {
		pop_width = MAX_POP_STR*font->max_bounds.width + 8;
		pop_height= font->max_bounds.ascent+font->max_bounds.descent + 8;
		pop_win = XCreateSimpleWindow(display,win,2*pop_height+4,
			2*pop_height+4, pop_width,pop_height,4, pxlBW[0], pxlBW[1]);
		XSelectInput(display,pop_win, ExposureMask|KeyPressMask);
		/*
		popped = True;
		*/
	}

	if(!isFuncKey) {
		pop_name = buffer[0];
		switch(buffer[0]) {
			case 'f': strcpy(popVarName, "From: "); break;
			case 't': strcpy(popVarName, "To: "); break;
/*
			case 'n': strcpy(popVarName, "Now: "); break;
*/
			case 'r': strcpy(popVarName, "ReadFrom: ");
				firstTime = True; getFileName=True; break;
			case 's': strcpy(popVarName, "SaveTo: "); getFileName=True; break;
			case 'Y': strcpy(popVarName, "TopY: "); break;
			case 'y': strcpy(popVarName, "BottomY: "); break;
			case 'Z': strcpy(popVarName, "Zmax: "); break;
			case 'z': strcpy(popVarName, "Zmin: "); break;
			default: break;
		}
		indxVar=99;
	}
	else  handleFuncKeys();
	if(indxVar == -1) return;
	else 
		displayPopWin(False);
}

void processQuit(char *msg) {
int		widthMSG;

	if(!quitWin) {
		quitWin = XCreateSimpleWindow(display, win, (width-450)/2, height/2-20,
			450, 40, 2, pxlBW[0], pxlBG[2]);
		XSelectInput(display,quitWin, ExposureMask|KeyPressMask);
	}
	XMapWindow(display, quitWin);
	quitWinPopped = True;
	XSetForeground(display, gc, pxlBW[1]);
	widthMSG = XTextWidth(font, msg, strlen(msg));
	XDrawString(display, quitWin, gc, (450-widthMSG)/2, 
		20+font->ascent-8, msg, strlen(msg));
	XSetInputFocus(display,quitWin,RevertToParent,CurrentTime);
	XSetForeground(display, gc, pxlBW[0]);
	XSetBackground(display, gc, pxlBG[0]);
}

void processKey(void) {

	if(report.xkey.window == pop_win) {
		processInput();
		NeedsRefresh = False;
		return;
	}
	switch (keysym) {
		case XK_q:
			NeedsRefresh = False;
			processQuit("Press 'q' again to quit or 'Esc' to continue.");
			break;
		case XK_c:
			NeedsRefresh = False;
			XClearWindow(display, win);
			newPos = False;
			Refresh(NeedsRefresh, refresh.from, refresh.to);
			break;
		case XK_d:
			NeedsRefresh = True;
			Refresh(NeedsRefresh, refresh.from, refresh.to);
			newPos = False;
			break;
		case XK_g:
			processGo(False);
			break;
		case XK_n:
			if(steps.now >= steps.to)
				XBell(display, 100);
			else
				processGo(True);
			break;
		case XK_f:	case XK_t:
		case XK_r:	case XK_s:
		case XK_Z:	case XK_z:
			isFuncKey = 0;
			NeedsRefresh = False;
			handlePop();
			break;
		case XK_Y:	case XK_y:
			if(frame.type == 15 || frame.type == 16) {
				isFuncKey = 0;
				NeedsRefresh = False;
				handlePop();
			}
			break;
		case XK_F1:	case XK_F2: case XK_F3: case XK_F4:
		case XK_F5: case XK_F6: case XK_F7: case XK_F8:
		case XK_F9: case XK_F10: case XK_F11: case XK_F12:
			isFuncKey = 1;
			NeedsRefresh = False;
			handlePop();
			break;
		case XK_Escape:
			if(popped) {
				XUnmapWindow(display, pop_win);
				popped = False;
			}
			else if(infoWinPopped) {
				XUnmapWindow(display, infoWin);
				infoWinPopped = False;
				NeedsRefresh = True;
			}
			getFileName = False;
			break;
		default:
			break;
	}
}

void initParm(FILE *file) {
char	line[80], name[15];
int		i;


	fgets(line, sizeof(line), file);
	if(line[0] != '#') DIE("Error occured in the first line of fortsub.rc!");

	fgets(line, sizeof(line), file);
	if(sscanf(line, "%s %d", name, &NumIntParm) ==2) {
		strcpy(intPName[0], name);
		intParm[0] = NumIntParm;
	}
	else DIE("Error in reading the second line of fortsub.rc!\n");

	fgets(line, sizeof(line), file);
	if(sscanf(line, "%s %d", name, &NumDblParm) ==2) {
		strcpy(intPName[1], name);
		intParm[1] = NumDblParm;
	}
	else DIE("Error in reading the third line of fortsub.rc!\n");

	for(i=2; i<NumIntParm; i++) {
		(void)fgets(line, sizeof(line), file);
		if(sscanf(line, "%s %d", intPName[i], &intParm[i])!=2)
			DIE("Error in getting intNamParm!\n");
	}
	fgets(line, sizeof(line), file);
	if(line[0] != '#')
	DIE("Error occurred in integer parameter of fortsub.rc!\n");
	for(i=0; i<NumDblParm; i++) {
		(void)fgets(line, sizeof(line), file);
		if(sscanf(line, "%s %lf", dblPName[i], &dblParm[i])!=2)
			DIE("Error in getting dblNamParm!\n");
	}
	fgets(line, sizeof(line), file);
	if(line[0] != '#')
	DIE("Error occurred in double precision parameter of fortsub.rc!\n");
	for(i=0; i<NumOptParm; i++) {
		(void)fgets(line, sizeof(line), file);
		if(sscanf(line, "%s %d", optNamParm[i].name, &optNamParm[i].value)!=2)
			DIE("Error in getting optParm!\n");
			/*
		printf("%d: %s, %d\n", i, optNamParm[i].name, optNamParm[i].value);
		*/
	}
	fgets(line, sizeof(line), file);
	if(line[0] != '#')
	DIE("Error occurred in config option parameter of fortsub.rc!\n");

	updateParm();

	nColors = optNamParm[0].value;
	hueColor = (double)optNamParm[1].value;
	chromColor = (double)optNamParm[2].value;
	minColorV = (double)optNamParm[3].value;
	maxColorV = (double)optNamParm[4].value;

}

void infoEvent(void) {
	if(keysym == XK_Escape) {
		XClearWindow(display, infoWin);
		XUnmapWindow(display, infoWin);
		infoWinPopped = False;
	}
	else XBell(display, 50);
	return;
}

void realQuit(void) {
	if (getFileName) {
		XClearWindow(display, quitWin);
		XUnmapWindow(display, quitWin);
		quitWinPopped = False;
		getFileName = False;
		return;
	}
	else if (keysym==XK_q) {
		XUnloadFont(display,font->fid);
		XFreeGC(display,gc);
		XCloseDisplay(display);
		exit(0);
	}
	else if(keysym == XK_Escape) {
		XClearWindow(display, quitWin);
		XUnmapWindow(display, quitWin);
		quitWinPopped = False;
		return;
	}
	else XBell(display, 50);
	return;
}


void main(int argc, char **argv) {
GC			gcx;
FILE		*file;

	progname = argv[0];

	steps.from = steps.now = 0;
	if(argc == 2) file = fopen(argv[1], "r");
	else file = fopen("fortsub.rc", "r");
	if(!file) DIE("Cannot open the init file.\n");
	initParm(file);
	fclose(file);
	ScreenSetup(argc, argv);

	gcx = XCreateGC(display, win, 0, NULL);
	XSetFunction(display, gcx, GXxor);
	XSetForeground(display, gcx, pxlBG[2]);
	while(1) {
	int				charcount;
	XComposeStatus	compose;
	double			tmp1, tmp2, tmp3;
		XNextEvent(display, &report);
		switch(report.type) {
			case Expose:
				if(report.xexpose.count != 0) break;
				Refresh(NeedsRefresh, refresh.from, refresh.to);
				if(report.xexpose.window == win && newPos) {
					XSetFunction(display, gcx, GXcopy);
					if(frame.type == 0 || frame.type == 1) {
						XDrawLine(display, win, gcx, width/2, height/2,
							newX, newY);
						XDrawArc(display, win, gcx,
							(width*.5-pntrX), (int)(height*.5-pntrX),
							(int)(2.*pntrX), (int)(2.*pntrX), 0, 360*64);
					}
					else if(frame.type == 10 || frame.type == 11
						|| frame.type >= 15) {
						XDrawLine(display, win, gcx, frame.x_min, newY,
							frame.x_max, newY);
						XDrawLine(display, win, gcx, newX, frame.y_min,
							newX, frame.y_max);
					}
					oldX = newX;	oldY = newY;
					XSetFunction(display, gcx, GXxor);
				}
				if(infoWinPopped && popInfoAgain)
					switch(infoWinContent) {
						case 11: popInfo('h');	popInfoAgain = False; break;
						case 12: popInfo('s');	popInfoAgain = False; break;
						case 21: displayResult();	popInfoAgain = False; break;
						case 22: popMonitor();	popInfoAgain = False; break;
						default: break;
					}
				else {
					popInfoAgain = True;
					NeedsRefresh = True;
				}
				if (popped) displayPopWin(True);
				newPos = False;
				break;
			case ConfigureNotify:
				width = report.xconfigure.width;
				height = report.xconfigure.height;
				NeedsRefresh = False;
				if(infoWinPopped) {
					XUnmapWindow(display, infoWin);
					infoWinPopped = False;
				}
				infoWin = (Window)NULL;
				if(popped) {
					XUnmapWindow(display, pop_win);
					popped = False;
				}
				pop_win = (Window)NULL;
				quitWin = (Window)NULL;
				break;
			case MotionNotify:
				if(frame.type == 0 || frame.type == 1) {
					tmp1 = (double)(report.xbutton.x-width*0.5);
					tmp2 = (double)(height*0.5-report.xbutton.y);
					pntrX = hypot(fabs(tmp1), fabs(tmp2));
					tmp3 = atan2(tmp2, tmp1);
					newX = (int)(width*.5 +(frame.x_max-frame.x_min)*cos(tmp3));
					newY = (int)(height*.5-(frame.x_max-frame.x_min)*sin(tmp3));
					pntrY = (tmp3 >= 0.0) ? (int)(tmp3*180.0/M_PI)
						: (int)(tmp3*180.0/M_PI+360.0);
					if(newPos) {
						XDrawLine(display, win, gcx, width/2, height/2,
							oldX, oldY);
						XDrawArc(display, win, gcx, (int)(width*.5-oldR),
							(int)(height*.5-oldR),
							(int)(2.*oldR), (int)(2.*oldR), 0, 360*64);
						}
					XDrawArc(display, win, gcx, (int)(width*.5-pntrX),
						(int)(height*.5-pntrX), (int)(2.*pntrX), (int)(2.*pntrX),
						0, 360*64);
					oldR = pntrX;
					XDrawLine(display, win, gcx, width/2, height/2,
						newX, newY);
				}
				else if(frame.type == 10 || frame.type == 11
					|| frame.type >= 15 ) {
					newX = report.xbutton.x;
					newY = report.xbutton.y;
					pntrX = minPnt.x + (maxPnt.x-minPnt.x)*(newX-frame.x_min)
						/(frame.x_max-frame.x_min);
					pntrY = minPnt.y + (maxPnt.y-minPnt.y)*(newY-frame.y_min)
						/(frame.y_max-frame.y_min);
					if(newPos) {
						XDrawLine(display, win, gcx, frame.x_min, oldY,
							frame.x_max, oldY);
						XDrawLine(display, win, gcx, oldX, frame.y_min,
							oldX, frame.y_max);
					}
					XDrawLine(display, win, gcx, frame.x_min, newY,
						frame.x_max, newY);
					XDrawLine(display, win, gcx, newX, frame.y_min,
						newX, frame.y_max);
				}
					oldX = newX; 	oldY = newY;
				newPos = True;
				UpdateSteps('p');
				break;
			case KeyPress:
				charcount=XLookupString(&report.xkey,buffer,bufsize,
					&keysym,&compose);
					processKey();
				if(report.xkey.window == quitWin)
					realQuit();
				break;
			default:
				break;
		}
	}
}
