/* this file contains code to calculate the libration in Lunar lat and long.
 * it is based entirely on code, reused here almost entirely as-is except
 * where necessary to make it more lint free, supplied by Steve Moshier, who in
 * turn relied heavily on the following references:
 * 
 *  Eckhardt, Donald H., "Theory of the Libration of the Moon,"
 *  The Moon and the Planets 25, pp. 3-41 (1981)
 * 
 *  Eckhardt, Donald H., "Planetary and Earth Figure Perturbations
 *  in the Librations of the Moon," in O. Calame, ed.,
 *  _High-Precision Earth Rotation and Earth-Moon Dynamics_,
 *  pp. 193-198 (1982) (IAU Symposium No. 63)
 * 
 * Any errors in adapting this code to xephem are strictly my own.
 * Elwood Downey.
*/

#include <math.h>

#if defined(__STDC__) || defined(__cplusplus)
#define P_(s) s
#else
#define P_(s) ()
#endif

void llibration P_((double JD, double *llatp, double *llonp));
static void librate P_((double J, double eul[], double p[]));
static double mod360 P_((double x));
static double modtp P_((double x));
static void libmain P_((double *atau, double *ap1, double *ap2, double *arho,
    double *asigma));
static void libpp P_((double JD, double *tau, double *q1, double *q2));
static void mlongs P_((double J, int first, int nobjs, double L[]));
static void seleng P_((double JD, double u[]));

#undef P_

/* given a Julian date, return the lunar libration in lat and long.
 */
void
llibration (JD, llatp, llonp)
double JD;
double *llatp;
double *llonp;
{
	double e[3], p[3], u[3];
	double s;

	librate( JD, e, p ); /* See below */

	seleng( JD, u ); /* Calculate u(2), u(3) from series  */
	*llatp = asin(u[2]);
	s = cos( *llatp );
	*llonp = asin( u[1]/s );
}

static double DTR = (double )1.7453292519943295769e-2;
static double STR = (double )4.8481368110953599359e-6; /* radians per arc sec */
#ifndef PI
static double PI = (double ) 3.14159265358979323846;
#endif
static double J2000 = 2451545.0;	/* 2000 January 1.5 */
static double J1900 = 2415020.0;	/* 1900 January 0, 12h UT */


/* Inclination of Moon's equator to the ecliptic:
 * IAU value = 1 d 32' 32.7" = 5552.7"
 * Eckhardt's value = 5552.857"
 */
static double Ie = 5552857.0;
static double Delaunay[4]; /* arguments of main theory terms */
static double Args[10]; /* arguments of planetary perturbations */
static double OM;
static double F;

static void
librate( J, eul, p )
double J;
double eul[], p[];
{
double f, g, T, L;
double phi, theta, psi;
double rho, sigma, tau, dtau, dq1, dq2, p1, p2;
double x, c, s;

/* Calculate mean longitudes of Mercury through Saturn.
 * See mlongs.c.
 */
mlongs( J, 1, 6, Args );

/* Julian centuries from 2000 January 1.5,
 * barycentric dynamical time
 */
T = (J-2451545.0)/36525.0;

/* Fundamental arguments in the FK5 reference system.
 *
 * Delaunay[0]: l or MM
 * mean longitude of the Moon minus the
 * mean longitude of the Moon's perigee
 *
 * Delaunay[1]: l' or MS
 * mean longitude of the Sun minus the
 * mean longitude of the Sun's perigee
 *
 * Delaunay[2]: F or FF
 * mean longitude of the Moon minus the
 * mean longitude of the Moon's node
 *
 * Delaunay[3]: D or DD
 * mean elongation of the Moon from the Sun.
 */

/* Expressions from the 1980 IAU Theory of Nutation: */
f = ((1.778e-5*T + 0.0086972)*T + 477198.8673981)*T + 134.9629814;
Delaunay[0] = DTR * mod360(f);
f = ((-3.33e-6*T - 0.0001603)*T + 35999.0503400)*T + 357.5277233;
Delaunay[1] = DTR * mod360(f);
f  = ((3.056e-6*T - 0.00368250)*T + 483202.0175381)*T + 93.2719103;
F = mod360(f);
if( F > 180.0 )
	F -= 360.0;
F *= DTR;
Delaunay[2] = F;
f  = (( 5.278e-6*T - 0.0019142)*T + 445267.1114800)*T + 297.8503631;
Delaunay[3] = DTR * mod360(f);

/* The following alternate expressions use Eckhardt's rates:
f = 477198.86745492*T + 134.9629814;
Delaunay[0] = DTR * mod360(f);
f = 35999.049443904*T + 357.5277233;
Delaunay[1] = DTR * mod360(f);
f  = 483202.0186848*T + 93.2719103;
F = mod360(f);
if( F > 180.0 )
	F -= 360.0;
F *= DTR;
Delaunay[2] = F;
f  = 445267.11134504*T + 297.8503631;
Delaunay[3] = DTR * mod360(f);
*/

/* longitude of the mean ascending node of the lunar orbit
 * on the ecliptic, measured from the mean equinox of date
 */
OM = ((2.22e-6*T + 0.00207833)*T - 1934.1362608)*T + 125.0445222;
OM = mod360(OM);
if( OM > 180.0 )
	OM -= 360.0;
OM *= DTR;


/* L = mean longitude of the Moon referred to equinox of date */
L = OM + F;

Args[6] = L;
Args[7] = Delaunay[3]; /* D */
Args[8] = Delaunay[0]; /* l */
Args[9] = Delaunay[2]; /* F */

/* Sum up the planetary perturbations
 */
libpp( J, &dtau, &dq1, &dq2  );

/* Main theory expansions
 */
libmain( &tau, &p1, &p2, &rho, &sigma );

x = STR/1000.0;
rho *= x;
sigma /= Ie;
tau *= x;
dtau *= x;
dq1 *= x;
dq2 *= x;

/* p[] = selenographic unit vector toward pole of ecliptic
 */
c = cos(F);
s = sin(F);
p1 *= x;
p2 *= x;
p1 += dq2*s + dq1*c;
p2 += dq2*c - dq1*s;
p[0] = p1;
p[1] = p2;
p[2] = sqrt( 1.0 - p1*p1 -p2*p2 );

/* Incorporate the planetary perturbations into the Euler angles
 */
f = tau - sigma;
c = cos(f);
s = sin(f);
theta = Ie*x + rho;
f = sin(theta);
g = dtau - (c/f)*dq1 + (s/f)*dq2;
sigma += g;

f = cos(theta);
g = (s/f)*dq1 + (c/f)*dq2;
rho += g;
theta += g;

tau += dtau;


/* The final Euler angles
 */
eul[1] = theta;

phi = PI + OM + sigma;
phi = modtp( phi );
if( phi > PI )
	phi -= 2.0*PI;
eul[0] = phi;

psi = F + tau - sigma;
psi = modtp( psi );
if( psi > PI )
	psi -= 2.0*PI;
eul[2] = psi;
}

/* Reduce x modulo 360 degrees
 */
static
double mod360(x)
double x;
{
double y;

y = x - (double )360.0 * floor( x/((double )360.0) );
return(y);
}


/* Reduce x modulo 2 pi
 */
#define TPI (2.0*PI)
static
double modtp(x)
double x;
{
double y;

y = x - TPI * floor( x/TPI );
return(y);
}


/* Main theory expansion of the Lunar libration
 */

struct lpert {
	char harm[4];
	long terms[10];
	};
#define NPLUN 86


static struct lpert plun[NPLUN] = {
/*
l  l' F  D  TAUs  TAUc p(1)s p(1)c p(2)s p(2)c THETAs THETAc ISIGMAs ISIGMAc
0, 0, 0, 0,    0,214170,   0,-80724,   0,  392,    0,5552857,  0,    0, */
0, 0, 0, 1,  100,    0,    0,    0,    0,    0,    0,   98,  112,    0,
0, 0, 0, 2, -487,    0,    0,    0,    0,    0,    0, -100, -259,    0,
0, 0, 1,-2,    0,    0, 2910,   -2,   -1,-3198,  -20,    1,    0,    0,
0, 0, 1,-1,    0,    0,  118,    0,    0,   91,    0,    0,    0,    0,
0, 0, 1, 0,  -10, 1085,5562462,5746,-5769,5540334,-80830,299,295,80788,
0, 0, 1, 2,    0,    0,  119,    0,    0,    0,    0,    0,    0,    0,
0, 0, 2,-2, 1647,    1,    0,    0,    0,    0,    2,-3074,-3001,   -1,
0, 0, 2,-1,    0,    0,    0,    0,    0,    0,    0,  -14,  -14,    0,
0, 0, 2, 0,  -27,    0,    0,    0,    0,    0,    1,-10533,-10050, -1,
0, 0, 2, 2,    0,    0,    0,    0,    0,    0,    0,  -57,  -59,    0,
0, 0, 3,-2,    0,    0,  -13,    0,    0,    0,  -24,    0,    0,   46,
0, 0, 3, 0,    0,    0,   13,    0,    0,   11,  -73,    0,    1,  142,
0, 1,-2, 0,    0,    0,    0,    0,    0,    0,    0,  -17,   17,    0,
0, 1,-2, 1,    0,    0,    0,    0,    0,    0,    0,  122, -122,    0,
0, 1,-2, 2,  142,    0,    0,    0,    0,    0,    0,  122, -120,    0,
0, 1,-1, 0,    0,    0, 1021,   -2,   -2,-1059,  -16,    3,   -3,  -17,
0, 1,-1, 1,    0,    0,  306,    0,    0,  -62,    0,    0,    0,    0,
0, 1,-1, 2,    0,    0,  118,    0,    0,  127,    0,    0,    0,    0,
0, 1, 0,-2,   38,    0,    0,    0,    0,    0,    0,    0,    0,    0,
0, 1, 0, 0,90705,    6,    0,    0,    0,    0,   -1,  226,  138,    1,
0, 1, 0, 1,  -15,    0,    0,    0,    0,    0,    0, -184, -186,    0,
0, 1, 1,-2,    0,    0,   81,    0,    0,  -86,    0,    0,    0,    0,
0, 1, 1, 0,    0,    0, 1244,    1,   -1, 1284,  -18,    3,    3,   17,
0, 1, 2,-2,   32,    0,    0,    0,    0,    0,    0,  -85,  -83,    0,
0, 1, 2, 0,    0,    0,    0,    0,    0,    0,    0,   18,   19,    0,
0, 2,-2, 2,  -25,    0,    0,    0,    0,    0,    0,   18,  -18,    0,
0, 2,-1, 2,    0,    0,   18,    0,    0,   18,    0,    0,    0,    0,
0, 2, 0, 0,  226,    0,    0,    0,    0,    0,    0,    0,    0,    0,
1,-2, 0,-2,    0,    0,    0,    0,    0,    0,    0,  -51,   51,    0,
1,-2, 1,-2,    0,    0,  -46,    0,    0,  -56,    0,    0,    0,    0,
1,-1,-1,-1,    0,    0,  -13,    0,    0,   13,    0,    0,    0,    0,
1,-1,-1, 0,    0,    0,  128,    0,    0, -138,    0,    0,    0,    0,
1,-1, 0,-2,  -31,    0,    0,    0,    0,    0,    0,   33,  -36,    0,
1,-1, 0,-1,-1148,    0,    0,    0,    0,    0,    0,  -21,   19,    0,
1,-1, 0, 0, -164,    0,    0,    0,    0,    0,    0, -106, -120,    0,
1,-1, 1,-2,    0,    0,   38,    0,    0,   29,    0,    0,    0,    0,
1,-1, 1,-1,    0,    0,  -37,    0,    0,  -36,    0,    0,    0,    0,
1,-1, 1, 0,    0,    0,   11,    0,    0,    0,    0,    0,    0,    0,
1, 0,-3, 0,    0,    0,    0,    0,    0,    0, -179,    0,    1, -355,
1, 0,-2,-2,    0,    0,    0,    0,    0,    0,    0, -136,  138,    0,
1, 0,-2, 0, -426,    2,  -11,  -77,  -77,   11,  133,24628,-24559, 138,
1, 0,-2, 2,    0,    0,    0,    0,    0,    0,    0,   59,  -23,    0,
1, 0,-1,-2,    0,    0, -350,    0,    1,   79,   17,   -1,    1,   30,
1, 0,-1,-1,    0,    0,  -47,    0,    0,   47,    0,    0,    0,    0,
1, 0,-1, 0,-1394,-6594,124492, -123,   82,-75458,-160,-711,  695,  715,
1, 0,-1, 2,    0,    0,   36,    0,    0,   19,    0,    0,    0,    0,
1, 0, 0,-4,   13,    0,    0,    0,    0,    0,    0,    0,    0,    0,
1, 0, 0,-2, 4130,   -2,   10,   -8,  -11,  -12,   -2,-1933, 2469,   -2,
1, 0, 0,-1,-3460,   -3,    0,    0,    0,    0,    0,    0,    0,    0,
1, 0, 0, 0,-16795,  -9, -717, -824,  836, -723,   88,-99080,-101389,-85,
1, 0, 0, 2,  -63,    0,    0,    0,    0,    0,    0,    0,  -19,    0,
1, 0, 1,-2,   -6,   38,-2677,   -2,   -1,-1614,   11,  -11,  -11,    7,
1, 0, 1,-1,    0,    0,  -52,    0,    0,  -54,    0,    0,    0,    0,
1, 0, 1, 0,    0,    0, 1575,    1,   -2,  299, -734,    0,    1, 1449,
1, 0, 1, 2,    0,    0,   13,    0,    0,    0,    0,    0,    0,    0,
1, 0, 2,-2,   22,    0,    0,    0,    0,    0,   -2,  506,  479,    2,
1, 0, 2, 0,    0,    0,    0,    0,    0,    0,    0, -727, -810,    0,
1, 0, 3, 0,    0,    0,    0,    0,    0,    0,    0,    0,    0,   14,
1, 1,-2, 0,    0,    0,    0,    0,    0,    0,    0,  -28,   28,    0,
1, 1,-1,-2,    0,    0,  -17,    0,    0,    0,    0,    0,    0,    0,
1, 1,-1, 0,    0,    0,  116,    0,    0, -161,    0,    0,    0,    0,
1, 1, 0,-2,  231,    0,    0,    0,    0,    0,    0,  -58,   84,    0,
1, 1, 0, 0,  102,    0,    0,    0,    0,    0,    0, -163, -150,    0,
1, 1, 1,-2,    0,    0,  -83,    0,    0,  -53,    0,    0,    0,    0,
1, 1, 2,-2,    0,    0,    0,    0,    0,    0,    0,   14,   13,    0,
1, 2, 0,-2,   10,    0,    0,    0,    0,    0,    0,    0,    0,    0,
2,-2, 0,-2,  406,    0,    0,    0,    0,    0,    0,    0,    0,    0,
2,-1,-1,-2,    0,    0,   13,    0,    0,  -13,    0,    0,    0,    0,
2,-1, 0,-2,  957,    0,    0,    0,    0,    0,    0,    0,    0,    0,
2,-1, 1,-2,    0,    0,   11,    0,    0,   12,    0,    0,    0,    0,
2, 0,-3, 0,    0,    0,  231,    5,    5, -232,    0,    0,   -3,  -10,
2, 0,-2,-2,    0,    0,    0,    0,    0,    0,    0,   13,  -16,    0,
2, 0,-2, 0,17014,  495,    0,    0,    0,    0,   -9,  -23, -200,   -6,
2, 0,-1,-2,    0,    0,  231,    0,    0, -218,    0,    0,    0,    0,
2, 0,-1, 0,    0,    0,  379,   14,  -14,  482,  -18,    7,    0,    0,
2, 0, 0,-4,   22,    0,    0,    0,    0,    0,    0,    0,    0,    0,
2, 0, 0,-2, 9940,   22,    0,    0,    0,    0,    0,  -31, -125,    0,
2, 0, 0, 0, -445,   -1,    0,    0,    0,    0,    1, -377, -893,   -1,
2, 0, 1,-2,    0,    0,  152,    0,    0,  192,    0,    0,    0,    0,
2, 0, 1, 0,    0,    0,   73,    0,    0,    0,    0,    0,    0,   25,
2, 0, 2,-2,    0,    0,    0,    0,    0,    0,    0,   24,   28,    0,
2, 0, 2, 0,    0,    0,    0,    0,    0,    0,    0,  -48,  -55,    0,
2, 1,-2, 0,   12,    0,    0,    0,    0,    0,    0,    0,    0,    0,
2, 1, 0,-2,  155,    1,    0,    0,    0,    0,    0,    0,    0,    0,
3, 0, 0,-2,   34,    0,    0,    0,    0,    0,    0,    0,    0,    0,
3, 0, 0, 0,  -22,    0,    0,    0,    0,    0,    0,    0,  -11,    0,
};

static void
libmain( atau, ap1, ap2, arho, asigma )
double *atau, *ap1, *ap2, *arho, *asigma;
{
double tau, p1, p2, rho, sigma;
double c, s, x;
char *h;
long *pterm;
int i, j, k;

/* Constant term of tau = longitude offset from mean Moon-Earth vector
 * to x axis of principal moment coordinate system.
 */
tau = 214170.0;
p1 = -80724.;
p2 =    392.;
rho = 0.0; /* inclination, theta = Ie + rho */
sigma = 0.0; /* node */

for( i=0; i<NPLUN; i++ )
	{
	h = plun[i].harm;
	x = 0.0;
	for( j=0; j<4; j++ )
		{
		k = *h++;
		if( k )
			x += k * Delaunay[j];
		}
	c = cos( x );
	s = sin( x );
	pterm = plun[i].terms;
	tau += *pterm++ * s;
	tau += *pterm++ * c;
	p1 += *pterm++ * s;
	p1 += *pterm++ * c;
	p2 += *pterm++ * s;
	p2 += *pterm++ * c;
	rho += *pterm++ * s;
	rho += *pterm++ * c;
	sigma += *pterm++ * s;
	sigma += *pterm++ * c;
	}
*atau = tau;
*ap1 = p1;
*ap2 = p2;
*arho = rho;
*asigma = sigma;
}

/* Planetary and Earth figure lunar libration perturbations.
 */

struct ppert {
	char harm[10];
	double phase;
	long terms[3];
	};
#define NPEF 77


/* Note, L = mean longitude of the Moon referred to mean equinox
 *         = OM + FF
 * phase is in degrees, amplitudes are in units of .001".
 */

static struct ppert pef[NPEF] = {
/*
M   V   E   M   J   S   L   D   l   F  phase   dtau dq1 dq2 */
0,-10,  3,  0,  0,  0,  0,  0,  1,  0,  26.69, -579,-15,  0,
0,  0, -4,  8, -3,  0,  0,  0,  0,  0, 254.01,  936,  0,  0, 
0,  0,  0,  0, -2,  5,  0,  0,  0,  0, 192.23,  175,  0,  0,
0,  0, -3,  7,  0,  0,  0,  1, -2,  1, 215.56,  -18,  0,  0,
0,  3, -7,  4,  0,  0,  0,  0,  0,  0, 328.23,  -33,  0,  0,
0, 18,-16,  0,  0,  0,  0,  0, -1,  0,  26.54,14403,391,-35,
0,  8,-13,  0,  0,  0,  0,  0,  0,  0, 235.75,  233,  0,  0,
0,  0,  2,  0, -3,  1,  0,  2, -2,  0,   1.30,  -11,  0,  0,
0, 26,-29,  0,  0,  0,  0,  0, -1,  0, 255.00, -105, -3,  1,
0,  0,  0,  2,  0,  0,  0,  1,  0, -1, 343.72,  -64,-10,-10,
0, -3,  4,  0,  0,  0,  0, -1,  1,  0,  89.66,  109,  0,  0,
0,  0, -3,  4,  0,  0,  0, -1,  1,  0, 208.75,   14,  0,  0,
0, 15,-12,  0,  0,  0,  0, -1,  0,  0, 279.70,  -15, -1,  0,
0,-24, 24,  0,  0,  0,  0, -2,  3,  0,   0.57,   10,  0,  0,
0, 23,-25,  0,  0,  0,  0, -1,  0,  0, 168.18,   13,  0,  0,
0,  5, -6,  0,  0,  0,  0,  2,  0, -2,  92.30,  -53,-14,-14,
0, -6,  8,  0,  0,  0,  0, -2,  2,  0,  12.87,   66,  0,  0,
0, 12, -8,  0,  0,  0,  0, -2,  1,  0, 228.51,   35,  2, -1,
0,  0,  8,-15,  0,  0,  0,  0,  0,  0, 335.31,  -26,  0,  0,
0,-20, 20,  0,  0,  0,  0,  1, -1,  1, 345.97,   43,  2, -2,
-3, 0,  1,  0,  0,  0,  0,  2, -1,  0, 263.44,  -29, -1,  1,
0,  0, -2,  0,  3,  0,  0, -2,  2,  0, 169.96,  320,  9, -6,
0,  0, -6,  8,  0,  0,  0, -2,  2,  0,  57.64,  -23,  0,  0,
0,  0,  1,  0, -1,  0,  0,  1, -1,  0, 180.70,  -18,  0,  0,
0, 20,-21,  0,  0,  0,  0, -2,  1,  0,  91.80, -108, -6,  4,
0,  0,  1,  0,  1,  0,  0,  1,  0, -1,  75.47,   26,  6,  6,
0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  80.21,  -19,  0,  0,
0,  0,  5, -6,  0,  0,  0,  2, -2,  0, 331.68,   13,  0,  0,
0, -8, 12,  0,  0,  0,  0, -1,  0,  1, 206.52,    2, 17,-17,
0,  0,  0,  0,  0,  0, -1,  0,  0,  1, 180.00, 8183,8188,-8095,
0,  0, -1,  0,  0,  0,  0, -1,  0,  1,  84.55,  739,7484,-7478,
0,  0, -1,  0, -2,  5,  0, -1,  0,  1, 300.94,   -1,-13, 13,
0,  0,  2,  0, -2,  0,  0,  2, -2,  0,   0.09,  159, -3,  1,
0,  0, -1,  2,  0,  0,  0,  0,  0,  0, 221.86,  347, -2,  2,
0,  5, -7,  0,  0,  0,  0,  1,  0, -1, 349.27,  -19, -4, -4,
0,  0,  0,  0,  1,  0,  0,  0,  0,  0, 106.70,  -99,  0,  0,
0,-15, 13,  0,  0,  0,  0,  2, -1,  0, 333.68,  -27,  0,  0,
0,  3, -3,  0,  0,  0,  0,  2, -2,  0, 180.04,  -23,  0,  0,
0,  0,  0,  0,  0,  0, -2,  0,  0,  2,   0.00,   43, 15,-15,
0,-21, 21,  0,  0,  0,  0,  0,  1,  0, 180.06,  -37,  0,  0,
0, -3,  5,  0,  0,  0,  0,  0,  0,  0, 195.47,   98,  0,  0,
0,  0, -2,  5,  0,  0,  0,  0,  0,  0, 239.86,  109,  0,  0,
0,  5, -8,  0,  0,  0,  0,  0,  0,  0,  70.24,   27,  0,  0,
0,  0,  0,  0,  2,  0,  0,  0,  0,  0, 316.18,   11,  0,  0,
0,  0, -3,  6,  0,  0,  0,  0,  0,  0,  88.66,  -24,  0,  0,
0,  0,  5, -9,  0,  0,  0,  0,  0,  0, 242.30,   11,  0,  0,
0,-18, 18,  0,  0,  0,  0,  2, -1,  0,   0.11,   16,  0,  0,
0,  0, -1,  0,  2,  0,  0, -1,  0,  1, 259.47,  -19,  4,  4,
0,  2, -3,  0,  0,  0,  0,  0,  0,  0,  89.95, -750,  0, -1,
0, -4,  5,  0,  0,  0,  0, -2,  2,  0, 268.89,  -13,  0,  0,
0,  0,  4, -7,  0,  0,  0,  0,  0,  0,  91.45,  -31,  0,  0,
0, 18,-16,  0,  0,  0,  0,  0, -3,  2,  26.54,  -11,  0,  0,
0,  0,  3, -5,  0,  0,  0,  0,  0,  0, 120.80, -847,  0, -1,
0,-21, 23,  0,  0,  0,  0,  2, -1,  0,  15.06, -274,  0, -1,
0,  1, -2,  0,  0,  0,  0, -1,  2, -1, 256.79, -228,  0,  0,
0, 19,-18,  0,  0,  0,  0, -1,  1, -1, 102.97,  -32,  0,  0,
0,-13, 10,  0,  0,  0,  0,  2, -1,  0, 259.64,  -11,  0,  0,
4,  0, -3,  0,  0,  0,  0,  0, -1,  0, 279.37,   19,  0,  0,
0,  5, -6,  0,  0,  0,  0,  2, -2,  0,  91.88,   37,  0,  0,
0, -1,  2,  0,  0,  0,  0,  0,  0,  0,  76.28, -269,  0, -1,
0,  4, -4,  0,  0,  0,  0,  2,  0, -2,   0.02,   12, -1, -1,
0,  0,  2, -3,  0,  0,  0,  0,  0,  0, 150.29,  112,  0,  0,
0,  4, -5,  0,  0,  0,  0,  1,  0, -1,  76.51,  -15,  0,  0,
0,  0,  1, -1,  0,  0,  0,  0,  0,  0,   0.88,  -13,  0,  0,
0,  4, -6,  0,  0,  0,  0,  0,  0,  0, 343.21,  -23,  0,  0,
0, -2,  2,  0,  0,  0,  0, -2,  2,  0, 180.03,  -13,  0,  0,
0,  1, -1,  0,  0,  0,  0,  0,  0,  0,   0.01,  372,  0,  1,
0,  0,  1,  0, -2,  0,  0,  0,  0,  0, 121.01,   34,  0,  0,
0,  3, -4,  0,  0,  0,  0,  0,  0,  0, 270.99,  -32,  0,  0,
0,  0,  1,  0, -1,  0,  0,  0,  0,  0,   1.229,-110,  0, -1,
0,  0,  2, -2,  0,  0,  0,  0,  0,  0, 179.77,   31,  0,  0,
0,  2, -2,  0,  0,  0,  0,  0,  0,  0,   0.20,  -25,  0,  0,
0,  0, -1,  0,  0,  0,  0, -1,  1,  1, 264.87,    0, 34,-34,
0,  0, -3,  4,  0,  0,  0, -1,  1,  1, 125.16,    0, 13,-13,
0, -3,  3,  0,  0,  0,  0, -2,  1,  2, 180.01,    0,-15, 15,
0,  0, -2,  0,  2,  0,  0, -2,  1,  2, 179.88,    0,-15, 15,
0,  0,  1,  0,  0,  0,  0,  1, -1,  1,  95.13,    0,-12, 12,
};

/* ARGSUSED */
static void
libpp( JD, tau, q1, q2  )
double JD;
double *tau, *q1, *q2;
{
double dtau, dq1, dq2, c, s, x;
char *h;
long *pterm;
int i, j, k;

/* Planetary perturbations of tau, q1, and q2.
 */
dtau = 0.0;
dq1 = 0.0;
dq2 = 0.0;
for( i=0; i<NPEF; i++ )
	{
	h = pef[i].harm;
	x = 0.0;
	for( j=0; j<10; j++ )
		{
		k = *h++;
		if( k )
			x += k * Args[j];
		}
	x += DTR * pef[i].phase;
	c = cos( x );
	s = sin( x );
	pterm = pef[i].terms;
	dtau += *pterm++ * s;
	dq1 += *pterm++ * s;
	dq2 += *pterm++ * c;
	}
*tau = dtau;
*q1 = dq1;
*q2 = dq2;
}


/* Coefficients for the mean longitudes of the planets, from:
 * Bretagnon, P., "Theorie du mouvement de l'ensemble des
 * planetes. Solution VSOP82," Astronomy and Astrophysics 114,
 * 278-288 (1982).  Pluto is *not* treated in that article.
 *
 * The highest degree term is listed first, the constant term
 * last. Time is in hundreds of years from J2000.  Angles
 * are in radians.
 */
static double mls[] = {
 3.100e-11,  -9.34290e-8,  2608.79031415742, 4.40260884240, /* Mercury */
-3.038e-11,   2.87555e-8,  1021.32855462110, 3.17614669689, /* Venus */
 7.3e-13,    -9.91890e-8,   628.30758491800, 1.75347031435, /* etc. */
-5.057e-11,   4.54761e-8,   334.06124314923, 6.20348091341,
 7.482e-11,  -1.4837133e-6,  52.96909650946, 0.59954649739,
-3.3330e-10,  3.6659741e-6,  21.32990954380, 0.87401675650,
 1.0450e-10, -8.48828e-8,     7.47815985673, 5.48129387159,
-4.340e-11,   1.66346e-8,     3.81330356378, 5.31188628676,
 5.94686e-7, -1.975706e-4,    2.533920996,   4.16934609, /* Pluto */
};


/* Program to calculate the mean longitudes using
 * the above coefficients.
 */
static void
mlongs( J, first, nobjs, L )
double J;
int first, nobjs;
double L[];
{
register double f;
double T;
double *p;
int i;


T = (J - J2000)/36525.0;
p = &mls[ 4*(first-1) ];

for(i=0; i<nobjs; i++ )
	{
	f = *p++;
	f = T * f + *p++;
	f = T * f + *p++;
	f = T * f + *p++;
	L[i] = f;
	}
}

/* Matrix and rotation subroutines
 */

#define N 3

/* Selenographic direction of the Earth.
 */

struct u23 {
	char harm[4];
	long cofs[4];
	};

#define NUTBL 423
static struct u23 utbl[NUTBL] = {
/*             u(2) s  u(2) c  u(3) s  u(3) c
 0, 0, 0, 0,      0,-212762,      0,  80183, */
 0, 0, 0, 1,-124202,      0,      0,      0,
 0, 0, 0, 2,2347535,   -280,      4,    113,
 0, 0, 0, 3,    326,      0,      0,      0,
 0, 0, 0, 4,  15413,    -12,      0,      0,
 0, 0, 0, 6,    178,      0,      0,      0,
 0, 0, 1,-6,      0,      0,     37,      0,
 0, 0, 1,-4,      0,      0,   3727,      0,
 0, 0, 1,-3,      0,      0,   -335,      0,
 0, 0, 1,-2,      0,      0, 646992,      1,
 0, 0, 1,-1,      0,      0,  -6597,      0,
 0, 0, 1, 0,    -31,     -7,-23954429,   -5,
 0, 0, 1, 1,      0,      0,   7026,      0,
 0, 0, 1, 2,      0,      0,-152813,      0,
 0, 0, 1, 3,      0,      0,     38,      0,
 0, 0, 1, 4,      0,      0,  -1562,      0,
 0, 0, 1, 6,      0,      0,    -19,      0,
 0, 0, 2,-4,    171,      0,      0,      0,
 0, 0, 2,-3,    238,      0,      0,      0,
 0, 0, 2,-2, -49498,     32,      0,    -12,
 0, 0, 2,-1,    631,      0,      0,      0,
 0, 0, 2, 0,-699300,   -722,      1,    270,
 0, 0, 2, 1,    218,      0,      0,      0,
 0, 0, 2, 2,  -4544,     -4,      0,      0,
 0, 0, 2, 4,    -48,      0,      0,      0,
 0, 0, 3,-4,      0,      0,     35,      0,
 0, 0, 3,-3,      0,      0,    -13,      0,
 0, 0, 3,-2,      0,      0,   3889,      0,
 0, 0, 3,-1,      0,      0,    -49,      0,
 0, 0, 3, 0,      0,      0,    279,      0,
 0, 0, 4,-2,     93,      0,      0,      0,
 0, 0, 4, 0,     18,      0,      0,      0,
 0, 1,-3,-2,      0,      0,     91,      0,
 0, 1,-2,-2,    303,      0,      0,      0,
 0, 1,-2, 0,   -586,      0,      0,      0,
 0, 1,-2, 2,  -1371,      0,      0,      0,
 0, 1,-1,-4,      0,      0,    207,      0,
 0, 1,-1,-2,      0,      0,  10404,      0,
 0, 1,-1,-1,      0,      0,    -30,      0,
 0, 1,-1, 0,      0,      0,  13834,      1,
 0, 1,-1, 1,      0,      0,  -1342,      0,
 0, 1,-1, 2,      0,      0,  12193,      0,
 0, 1,-1, 3,      0,      0,    -34,      0,
 0, 1,-1, 4,      0,      0,    114,      0,
 0, 1, 0,-6,    -35,      0,      0,      0,
 0, 1, 0,-4,  -2068,     -2,      0,      0,
 0, 1, 0,-3,    -59,      0,      0,      0,
 0, 1, 0,-2,-163827,    -18,      0,      0,
 0, 1, 0,-1,    559,      0,      0,      0,
 0, 1, 0, 0,-752860,      0,      0,      0,
 0, 1, 0, 1,  17850,      0,      0,      0,
 0, 1, 0, 2, -24568,      8,      0,      0,
 0, 1, 0, 3,    162,      0,      0,      0,
 0, 1, 0, 4,   -345,      0,      0,      0,
 0, 1, 1,-4,      0,      0,    422,      0,
 0, 1, 1,-3,      0,      0,    -24,      0,
 0, 1, 1,-2,      0,      0,  31383,      0,
 0, 1, 1,-1,      0,      0,    -17,      0,
 0, 1, 1, 0,      0,      0,  15380,      0,
 0, 1, 1, 1,      0,      0,  -1044,      0,
 0, 1, 1, 2,      0,      0,   1699,      0,
 0, 1, 1, 3,      0,      0,    -18,      0,
 0, 1, 1, 4,      0,      0,     34,      0,
 0, 1, 2,-3,     19,      0,      0,      0,
 0, 1, 2,-2,  -1833,      2,      0,      0,
 0, 1, 2,-1,     12,      0,      0,      0,
 0, 1, 2, 0,   -275,      0,      0,      0,
 0, 1, 2, 1,    -30,      0,      0,      0,
 0, 1, 2, 2,     47,      0,      0,      0,
 0, 1, 3,-2,      0,      0,    160,      0,
 0, 2,-1,-4,      0,      0,     17,      0,
 0, 2,-1,-2,      0,      0,    502,      0,
 0, 2,-1, 0,      0,      0,    124,      0,
 0, 2,-1, 2,      0,      0,    119,      0,
 0, 2,-2,-2,     14,      0,      0,      0,
 0, 2,-2, 0,    -10,      0,      0,      0,
 0, 2, 0,-4,   -168,      0,      0,      0,
 0, 2, 0,-2,  -8020,      0,      0,      0,
 0, 2, 0,-1,    -42,      0,      0,      0,
 0, 2, 0, 0,  -7624,      0,      0,      0,
 0, 2, 0, 1,    -40,      0,      0,      0,
 0, 2, 0, 2,   -178,      0,      0,      0,
 0, 2, 1,-4,      0,      0,     29,      0,
 0, 2, 1,-2,      0,      0,   1177,      0,
 0, 2, 1, 0,      0,      0,    151,      0,
 0, 2, 1, 2,      0,      0,     19,      0,
 0, 2, 2,-2,    -53,      0,      0,      0,
 0, 3,-1,-2,      0,      0,     21,      0,
 0, 3, 0,-4,    -11,      0,      0,      0,
 0, 3, 0,-2,   -339,      0,      0,      0,
 0, 3, 0, 0,   -102,      0,      0,      0,
 0, 3, 1,-2,      0,      0,     40,      0,
 0, 4, 0,-2,    -13,      0,      0,      0,
 1,-3, 0, 0,     52,      0,      0,      0,
 1,-3, 0, 2,     34,      0,      0,      0,
 1,-2,-2, 2,    -20,      0,      0,      0,
 1,-2,-1,-2,      0,      0,    -147,     0,
 1,-2,-1, 0,      0,      0,    -122,     0,
 1,-2,-1, 2,      0,      0,     -70,     0,
 1,-2, 1,-2,      0,      0,     -93,     0,
 1,-2, 1, 0,      0,      0,    -155,     0,
 1,-2, 1, 2,      0,      0,     -77,     0,
 1,-2, 0,-4,     23,      0,      0,      0,
 1,-2, 0,-2,   2495,      0,      0,      0,
 1,-2, 0, 0,   2585,      0,      0,      0,
 1,-2, 0, 2,    785,      0,      0,      0,
 1,-2, 0, 4,     32,      0,      0,      0,
 1,-2, 2,-2,     10,      0,      0,      0,
 1,-1,-3, 0,      0,      0,     -18,     0,
 1,-1,-3, 2,      0,      0,      28,     0,
 1,-1,-2,-2,    -51,      0,      0,      0,
 1,-1,-2, 0,     90,      0,      0,      0,
 1,-1,-2, 2,   -459,      0,      0,      0,
 1,-1,-1,-4,      0,      0,    -68,      0,
 1,-1,-1,-3,      0,      0,     27,      0,
 1,-1,-1,-2,      0,      0,  -1779,      0,
 1,-1,-1,-1,      0,      0,    -61,      0,
 1,-1,-1, 0,      0,      0,  -7127,      0,
 1,-1,-1, 1,      0,      0,     20,      0,
 1,-1,-1, 2,      0,      0,  -1787,      0,
 1,-1,-1, 4,      0,      0,    -58,      0,
 1,-1, 0,-6,     15,      0,      0,      0,
 1,-1, 0,-4,    704,      0,      0,      0,
 1,-1, 0,-3,   -269,      0,      0,      0,
 1,-1, 0,-2,  28053,      8,      0,      0,
 1,-1, 0,-1,     89,      0,      0,      0,
 1,-1, 0, 0, 146422,    -41,      1,     17,
 1,-1, 0, 1,   -133,      0,      0,      0,
 1,-1, 0, 2,  15101,    -11,      0,      0,
 1,-1, 0, 4,    371,      0,      0,      0,
 1,-1, 1,-4,      0,      0,   -173,      0,
 1,-1, 1,-3,      0,      0,     13,      0,
 1,-1, 1,-2,      0,      0,  -1116,      0,
 1,-1, 1,-1,      0,      0,     56,      0,
 1,-1, 1, 0,      0,      0,  -9164,      0,
 1,-1, 1, 1,      0,      0,     12,      0,
 1,-1, 1, 2,      0,      0,  -1485,      0,
 1,-1, 1, 4,      0,      0,    -39,      0,
 1,-1, 2,-2,     20,      0,      0,      0,
 1,-1, 2, 0,   -215,      0,      0,      0,
 1,-1, 2, 2,    -43,      0,      0,      0,
 1,-1, 4,-2,    -20,      0,      0,      0,
 1,-1, 4, 0,    -80,      0,      0,      0,
 1, 0,-3,-2,      0,      0,    -77,      0,
 1, 0,-3, 0,      2,     13,  -5041,      0,
 1, 0,-3, 2,      0,      0,    505,      0,
 1, 0,-2,-4,    125,      0,      0,      0,
 1, 0,-2,-2,   8275,     -7,      0,      0,
 1, 0,-2,-1,    -12,      0,      0,      0,
 1, 0,-2, 0,  42460,     34,     -5,    -26,
 1, 0,-2, 1,    -29,      0,      0,      0,
 1, 0,-2, 2,  -8098,     -3,      0,      0,
 1, 0,-2, 3,    -10,      0,      0,      0,
 1, 0,-2, 4,    -47,      0,      0,      0,
 1, 0,-1,-6,      0,      0,     55,      0,
 1, 0,-1,-4,      0,      0,   3909,      0,
 1, 0,-1,-3,      0,      0,   -269,      0,
 1, 0,-1,-2,      0,      0, 258489,     -1,
 1, 0,-1,-1,      0,      0,     26,      0,
 1, 0,-1, 0,   1351,   6503,-1426756,    44,
 1, 0,-1, 1,      0,      0,    581,      0,
 1, 0,-1, 2,      0,      0, -33526,      1,
 1, 0,-1, 3,      0,      0,     36,      0,
 1, 0,-1, 4,      0,      0,   -479,      0,
 1, 0, 0,-6,   -515,      0,      0,      0,
 1, 0, 0,-5,     25,      0,      0,      0,
 1, 0, 0,-4, -39304,    -29,      0,     11,
 1, 0, 0,-3,   3148,      2,      0,      0,
 1, 0, 0,-2,-4560228,   129,     -7,    -43,
 1, 0, 0,-1,  21959,     -5,      0,      0,
 1, 0, 0, 0,22544791,    90,    661,    703,
 1, 0, 0, 1,  -8346,      7,      0,      0,
 1, 0, 0, 2, 198148,   -145,      4,     59,
 1, 0, 0, 3,    -52,      0,      0,      0,
 1, 0, 0, 4,   2597,     -2,      0,      0,
 1, 0, 0, 6,     36,      0,      0,      0,
 1, 0, 1,-6,      0,      0,     95,      0,
 1, 0, 1,-5,      0,      0,    -10,      0,
 1, 0, 1,-4,      0,      0,   6694,      0,
 1, 0, 1,-3,      0,      0,   -323,      0,
 1, 0, 1,-2,      7,    -44, 232074,      1,
 1, 0, 1,-1,      0,      0,   -789,      0,
 1, 0, 1, 0,     39,     65,-1311863,    -1,
 1, 0, 1, 1,      0,      0,    869,      0,
 1, 0, 1, 2,      0,      0, -19745,      0,
 1, 0, 1, 4,      0,      0,   -279,      0,
 1, 0, 2,-4,    102,      0,      0,      0,
 1, 0, 2,-3,     22,      0,      0,      0,
 1, 0, 2,-2,  -3523,     10,      0,      0,
 1, 0, 2,-1,     64,      0,      0,      0,
 1, 0, 2, 0, -38403,    -40,      0,     15,
 1, 0, 2, 1,     28,      0,      0,      0,
 1, 0, 2, 2,   -585,     -1,      0,      0,
 1, 0, 3,-4,      0,      0,    -21,      0,
 1, 0, 3,-2,      0,      0,    745,      0,
 1, 0, 3, 0,      0,      0,     21,      0,
 1, 0, 4,-2,     20,      0,      0,      0,
 1, 1,-3, 0,      0,      0,     13,      0,
 1, 1,-2,-4,     13,      0,      0,      0,
 1, 1,-2,-2,    375,      0,      0,      0,
 1, 1,-2, 0,    -68,      0,      0,      0,
 1, 1,-2, 2,    122,      0,      0,      0,
 1, 1,-1,-4,      0,      0,    442,      0,
 1, 1,-1,-3,      0,      0,    -18,      0,
 1, 1,-1,-2,      0,      0,  11563,      0,
 1, 1,-1, 0,      0,      0,   5772,      0,
 1, 1,-1, 1,      0,      0,    -32,      0,
 1, 1,-1, 2,      0,      0,    823,      0,
 1, 1,-1, 4,      0,      0,     18,      0,
 1, 1, 0,-6,    -93,      0,      0,      0,
 1, 1, 0,-4,  -4487,     -3,      0,      0,
 1, 1, 0,-3,    230,      0,      0,      0,
 1, 1, 0,-2,-204059,      0,      0,      0,
 1, 1, 0,-1,    121,      0,      0,      0,
 1, 1, 0, 0,-108999,     44,     -1,    -17,
 1, 1, 0, 1,   1262,     -1,      0,      0,
 1, 1, 0, 2,  -3289,      3,      0,      0,
 1, 1, 0, 3,     30,      0,      0,      0,
 1, 1, 0, 4,   -122,      0,      0,      0,
 1, 1, 1,-6,      0,      0,     15,      0,
 1, 1, 1,-4,      0,      0,    610,      0,
 1, 1, 1,-3,      0,      0,    -23,      0,
 1, 1, 1,-2,      0,      0,  10259,      0,
 1, 1, 1,-1,      0,      0,     11,      0,
 1, 1, 1, 0,      0,      0,   7275,      0,
 1, 1, 1, 1,      0,      0,   -132,      0,
 1, 1, 1, 2,      0,      0,    316,      0,
 1, 1, 2,-4,     13,      0,      0,      0,
 1, 1, 2,-2,    -73,      0,      0,      0,
 1, 1, 2, 0,    174,      0,      0,      0,
 1, 1, 3,-2,      0,      0,     27,      0,
 1, 2,-2,-2,     13,      0,      0,      0,
 1, 2,-1,-4,      0,      0,     31,      0,
 1, 2,-1,-2,      0,      0,    412,      0,
 1, 2,-1, 0,      0,      0,     71,      0,
 1, 2, 0,-4,   -317,      0,      0,      0,
 1, 2, 0,-3,     12,      0,      0,      0,
 1, 2, 0,-2,  -7321,      0,      0,      0,
 1, 2, 0, 0,  -1120,      0,      0,      0,
 1, 2, 0, 1,    -10,      0,      0,      0,
 1, 2, 0, 2,    -16,      0,      0,      0,
 1, 2, 1,-4,      0,      0,     36,      0,
 1, 2, 1,-2,      0,      0,    368,      0,
 1, 2, 1, 0,      0,      0,     74,      0,
 1, 3,-1,-2,      0,      0,     14,      0,
 1, 3, 1,-2,      0,      0,     12,      0,
 1, 3, 0,-4,    -18,      0,      0,      0,
 1, 3, 0,-2,   -246,      0,      0,      0,
 1, 3, 0, 0,    -17,      0,      0,      0,
 2,-2, 0,-4,     34,      0,      0,      0,
 2,-2, 0,-2,   -161,      0,      0,      0,
 2,-2, 0, 0,    214,      0,      0,      0,
 2,-2, 0, 2,     84,      0,      0,      0,
 2,-2, 1,-2,      0,      0,    -15,      0,
 2,-2, 1, 0,      0,      0,    -21,      0,
 2,-1,-2, 0,    -31,      0,      0,      0,
 2,-1,-2, 2,    -28,      0,      0,      0,
 2,-1,-1,-4,      0,      0,    -39,      0,
 2,-1,-1,-2,      0,      0,     93,      0,
 2,-1,-1,-1,      0,      0,     20,      0,
 2,-1,-1, 0,      0,      0,   -300,      0,
 2,-1,-1, 2,      0,      0,   -131,      0,
 2,-1, 0,-6,     18,      0,      0,      0,
 2,-1, 0,-4,    385,      0,      0,      0,
 2,-1, 0,-3,     51,      0,      0,      0,
 2,-1, 0,-2,  -3891,      0,      0,      0,
 2,-1, 0,-1,   -369,      0,      0,      0,
 2,-1, 0, 0,  10625,    -10,      0,      0,
 2,-1, 0, 1,    -19,      0,      0,      0,
 2,-1, 0, 2,   1525,     -1,      0,      0,
 2,-1, 0, 4,     50,      0,      0,      0,
 2,-1, 1,-2,      0,      0,    125,      0,
 2,-1, 1,-1,      0,      0,     23,      0,
 2,-1, 1, 0,      0,      0,  -1056,      0,
 2,-1, 1, 2,      0,      0,   -162,      0,
 2,-1, 2, 0,    -28,      0,      0,      0,
 2, 0,-3,-2,      0,      0,   -116,      0,
 2, 0,-3, 0,      0,      0,    249,      1,
 2, 0,-3, 2,      0,      0,     11,      0,
 2, 0,-2,-4,    117,      0,      0,      0,
 2, 0,-2,-2,    939,      0,      0,      0,
 2, 0,-2, 0, -19541,   -494,      0,      0,
 2, 0,-2, 2,   -427,      0,      0,      0,
 2, 0,-1,-6,      0,      0,     78,      0,
 2, 0,-1,-4,      0,      0,   3118,      0,
 2, 0,-1,-3,      0,      0,    -64,      0,
 2, 0,-1,-2,      0,      0,  -3096,     -2,
 2, 0,-1,-1,      0,      0,    -84,      0,
 2, 0,-1, 0,      0,      0, -29668,      5,
 2, 0,-1, 1,      0,      0,     38,      0,
 2, 0,-1, 2,      0,      0,  -2172,      0,
 2, 0,-1, 4,      0,      0,    -49,      0,
 2, 0, 0,-6,   -722,      0,      0,      0,
 2, 0, 0,-5,     66,      0,      0,      0,
 2, 0, 0,-4, -29221,    -26,      0,      0,
 2, 0, 0,-3,   1125,      0,      0,      0,
 2, 0, 0,-2,-223359,    243,     -5,   -109,
 2, 0, 0,-1,   2011,     -2,      0,      0,
 2, 0, 0, 0, 762516,   -639,     37,    280,
 2, 0, 0, 1,   -755,      0,      0,      0,
 2, 0, 0, 2,  18420,    -16,      0,      0,
 2, 0, 0, 3,    -11,      0,      0,      0,
 2, 0, 0, 4,    336,      0,      0,      0,
 2, 0, 1,-6,      0,      0,     82,      0,
 2, 0, 1,-4,      0,      0,    675,      0,
 2, 0, 1,-3,      0,      0,    -54,      0,
 2, 0, 1,-2,      0,      0,  21927,      0,
 2, 0, 1,-1,      0,      0,   -156,      0,
 2, 0, 1, 0,      0,      0, -80451,      0,
 2, 0, 1, 1,      0,      0,     83,      0,
 2, 0, 1, 2,      0,      0,  -1988,      0,
 2, 0, 1, 4,      0,      0,    -37,      0,
 2, 0, 2,-4,     93,      0,      0,      0,
 2, 0, 2,-2,   -289,      2,      0,      0,
 2, 0, 2, 0,  -2364,     -2,      0,      0,
 2, 0, 2, 2,    -59,      0,      0,      0,
 2, 0, 3,-2,      0,      0,     79,      0,
 2, 1,-2,-4,     10,      0,      0,      0,
 2, 1,-2,-2,     40,      0,      0,      0,
 2, 1,-2, 0,     22,      0,      0,      0,
 2, 1,-1,-6,      0,      0,     12,      0,
 2, 1,-1,-4,      0,      0,    277,      0,
 2, 1,-1,-2,      0,      0,   -126,      0,
 2, 1,-1, 0,      0,      0,    329,      0,
 2, 1,-1, 2,      0,      0,     63,      0,
 2, 1, 0,-6,   -115,      0,      0,      0,
 2, 1, 0,-4,  -2609,     -2,      0,      0,
 2, 1, 0,-3,     80,      0,      0,      0,
 2, 1, 0,-2,  -8426,     10,      0,      0,
 2, 1, 0, 0,  -8773,      8,      0,      0,
 2, 1, 0, 1,    120,      0,      0,      0,
 2, 1, 0, 2,   -403,      0,      0,      0,
 2, 1, 0, 4,    -11,      0,      0,      0,
 2, 1, 1,-6,      0,      0,     11,      0,
 2, 1, 1,-4,      0,      0,     56,      0,
 2, 1, 1,-2,      0,      0,    898,      0,
 2, 1, 1, 0,      0,      0,    854,      0,
 2, 1, 1, 1,      0,      0,    -13,      0,
 2, 1, 1, 2,      0,      0,     42,      0,
 2, 1, 2, 0,     23,      0,      0,      0,
 2, 2,-1,-4,      0,      0,     16,      0,
 2, 2, 0,-6,    -11,      0,      0,      0,
 2, 2, 0,-4,   -151,      0,      0,      0,
 2, 2, 0,-2,   -280,      0,      0,      0,
 2, 2, 0, 0,    -65,      0,      0,      0,
 2, 2, 1,-2,      0,      0,     31,      0,
 3,-2, 0,-2,     11,      0,      0,      0,
 3,-2, 0, 0,     22,      0,      0,      0,
 3,-1,-1, 0,      0,      0,    -26,      0,
 3,-1,-1, 2,      0,      0,    -10,      0,
 3,-1, 0,-4,    -33,      0,      0,      0,
 3,-1, 0,-2,   -261,      0,      0,      0,
 3,-1, 0,-1,    -23,      0,      0,      0,
 3,-1, 0, 0,    942,      0,      0,      0,
 3,-1, 0, 2,    145,      0,      0,      0,
 3,-1, 1,-2,      0,      0,     26,      0,
 3,-1, 1, 0,      0,      0,    -99,      0,
 3,-1, 1, 2,      0,      0,    -16,      0,
 3, 0,-2,-4,     18,      0,      0,      0,
 3, 0,-2,-2,    -35,      0,      0,      0,
 3, 0,-2, 0,    -15,      0,      0,      0,
 3, 0,-2, 2,    -44,      0,      0,      0,
 3, 0,-1,-6,      0,      0,     40,      0,
 3, 0,-1,-4,      0,      0,     13,      0,
 3, 0,-1,-2,      0,      0,    299,      0,
 3, 0,-1, 0,      2,     10,  -1707,      0,
 3, 0,-1, 2,      0,      0,   -149,      0,
 3, 0, 0,-6,   -366,      0,      0,      0,
 3, 0, 0,-5,     16,      0,      0,      0,
 3, 0, 0,-4,    276,      0,      0,      0,
 3, 0, 0,-3,     28,      0,      0,      0,
 3, 0, 0,-2, -20248,     22,      0,      0,
 3, 0, 0,-1,    185,      0,      0,      0,
 3, 0, 0, 0,  46992,    -43,      2,     19,
 3, 0, 0, 1,    -64,      0,      0,      0,
 3, 0, 0, 2,   1627,     -2,      0,      0,
 3, 0, 0, 4,     38,      0,      0,      0,
 3, 0, 1,-4,      0,      0,    -22,      0,
 3, 0, 1,-2,      0,      0,   2067,      0,
 3, 0, 1,-1,      0,      0,    -17,      0,
 3, 0, 1, 0,      0,      0,  -5180,      0,
 3, 0, 1, 2,      0,      0,   -181,      0,
 3, 0, 2,-4,     16,      0,      0,      0,
 3, 0, 2,-2,    -17,      0,      0,      0,
 3, 0, 2, 0,   -153,      0,      0,      0,
 3, 1,-1,-2,      0,      0,     11,      0,
 3, 1,-1, 0,      0,      0,     26,      0,
 3, 1, 0,-6,    -49,      0,      0,      0,
 3, 1, 0,-4,     24,      0,      0,      0,
 3, 1, 0,-2,   -692,      0,      0,      0,
 3, 1, 0, 0,   -791,      0,      0,      0,
 3, 1, 0, 1,     11,      0,      0,      0,
 3, 1, 0, 2,    -44,      0,      0,      0,
 3, 1, 1,-2,      0,      0,     74,      0,
 3, 1, 1, 0,      0,      0,     82,      0,
 3, 2, 0,-2,    -22,      0,      0,      0,
 4,-1, 0,-2,    -37,      0,      0,      0,
 4,-1, 0, 0,     79,      0,      0,      0,
 4,-1, 0, 2,     13,      0,      0,      0,
 4, 0,-2,-2,     13,      0,      0,      0,
 4, 0,-2, 0,    -27,      0,      0,      0,
 4, 0,-1,-2,      0,      0,     29,      0,
 4, 0,-1, 0,      0,      0,    -99,      0,
 4, 0,-1, 2,      0,      0,    -10,      0,
 4, 0, 0,-4,    147,      0,      0,      0,
 4, 0, 0,-2,  -1759,      2,      0,      0,
 4, 0, 0,-1,     17,      0,      0,      0,
 4, 0, 0, 0,   3051,     -3,      0,      0,
 4, 0, 0, 2,    138,      0,      0,      0,
 4, 0, 1,-4,      0,      0,    -11,      0,
 4, 0, 1,-2,      0,      0,    180,      0,
 4, 0, 1, 0,      0,      0,   -343,      0,
 4, 0, 1, 2,      0,      0,    -16,      0,
 4, 1, 0,-4,     11,      0,      0,      0,
 4, 1, 0,-2,    -51,      0,      0,      0,
 4, 1, 0, 0,    -66,      0,      0,      0,
 5, 0, 0,-4,     19,      0,      0,      0,
 5, 0, 0,-2,   -143,      0,      0,      0,
 5, 0, 0, 0,    203,      0,      0,      0,
 5, 0, 0, 2,     11,      0,      0,      0,
 5, 0, 1,-2,      0,      0,     15,      0,
 5, 0, 1, 0,      0,      0,    -23,      0,
 6, 0, 0,-2,    -12,      0,      0,      0,
 6, 0, 0, 0,     14,      0,      0,      0,
};

#define NU3T 39
#define NU2T 66
static double uptbl[3*NU3T+3*NU2T] = {
/* rot/day     rot    du() */
.0000285785, .45432,  -19., /* Start u(3) terms */
.0000285785, .38932,    4.,
.0000400379, .27701,   20.,
.0001757789, .37964,   32.,
.0002994687, .17302,   52.,
.0003093863, .19550,  -43.,
.0003094558, .42869,  -14.,
.0005302777, .31094,   20.,
.0344033074, .26768,  -21.,
.0345505078, .25800,   18.,
.0348884359, .36574,  -15.,
.0359921770, .14866,  -25.,
.0361158668, .44288,   42.,
.0361393774, .49731,  -17.,
.0362516078, .04550,   27.,
.0362630672, .36819,  -67.,
.0364540072, .47116,  -67.,
.0366009953, .05510,  -08.,
.0366010320, .01801, 5965.,
.0366011015, .25120,   74.,
.0366695504, .18670,   11.,
.0366744680, .42766,   12.,
.0367367363, .37661,   14.,
.0367380907, .06858,  842.,
.0367466595, .38903,  -12.,
.0367469630, .29761,  -18.,
.0367494284, .26489,  -18.,
.0367497319, .17347,  -12.,
.0367583007, .49392,  808.,   
.0367596551, .18589,   14.,
.0367744863, .06066,  -13.,
.0368031824, .26827,   85.,
.0368952899, .31130,  372.,
.0368953594, .04449,  -14.,
.0369389234, .37256,   86.,
.0370627194, .03649,   38.,
.0378284261, .11807,  -24.,
.0379757327, .11133,   26.,
.0383135546, .21613,  -28., /* Last u(3) term */
.0000101050, .46267,   19.,  /* Start u(2) terms */
.0000549867, .23702,    1.,
.0000737277, .10359,   38.,
.0000786453, .34455,    2.,
.0000982261, .21378,   -3.,
.0001470942, .28005, 1035.,
.0001471637, .01324, -426.,
.0001472004, .47615,   17.,
.0001570813, .03516,  -89.,
.0001734910, .47639,   11.,
.0001907277, .34131,    1.,
.0002308090, .46459,   -6.,
.0002707422, .36755,    3.,
.0002807711, .32910,   52.,
.0002941885, .06009,    7.,
.0003278992, .14341,    4.,
.0003379281, .07718,  -13.,
.0003469821, .38611,  -14.,
.0003493875, .23432,    4.,
.0004850223, .09612,  -06.,
.0005204731, .32862,    7.,
.0005881919, .20152,    5.,
.0006086703, .47251,    8.,
.0006873156, .31734,  387.,
.0007444726, .02625,    7.,
.0007616829, .13513,  -25.,
.0009351739, .07430,  822.,
.0009680867, .15172,   44.,
.0010252437, .32758, -286.,
.0011086650, .01791,  141.,
.0012275370, .07908, -22.,
.0012821560, .46097,   25.,
.0013746312, .08412,   50.,
.0014317883, .34609,  -13.,
.0017125594, .17520, 1184.,
.0020453762, .13595,   32.,
.0022173300, .03805,   17.,
.0022761852, .28554, -200.,
.0023908210, .47860,   57.,
.0023998750, .49587,  209.,
.0025069941, .11902, -750.,
.0025519156, .08405,  -15.,
.0025643121, .42138, -226.,
.0026448594, .03758,  -48.,
.0029686121, .12875,  -11.,
.0034251187, .35095,  332.,
.0036729770, .11041,  -11.,
.0038464681, .39124,  -15.,
.0044503625, .18353,   17.,
.0047831793, .09428,   54.,
.0050139882, .23248, -194.,
.0051376781, .02753,  -44.,
.0052193903, .32869, -173.,
.0052447972, .15179,  -19.,
.0061629218, .33094,   10.,
.0068502374, .20078,   47.,
.0073639011, .31772,   11.,
.0075209824, .32761,   10.,
.0085627968, .37598,  -33.,
.0102753561, .05117,  -24.,
.0280096200, .27453,  157.,
.0289277445, .00590,  211.,
.0312776574, .09114,  190.,
.0362179180, .21892, -436.,
.0364487270, .35823,-1138.,
.0365724168, .15162, -662. /* Last u(2) term */
};


static void
seleng( JD, u )
double JD;
double u[];
{
double u2, u3, c, s, tpi, T, x;
double *pd;
long *pl;
char *ph;
int i, j, k;

tpi = 2.0*PI;
T = JD - J1900;
pd = &uptbl[0];
u2 = 0.0;
u3 = 0.0;
/* Planetary perturbations
 */
for( i=0; i<NU3T; i++ )
	{
	x = *pd++ * T;
	x += *pd++;
	x -= floor(x);
	u3 += *pd++ * cos( tpi * x );
	}

for( i=0; i<NU2T; i++ )
	{
	x = *pd++ * T;
	x += *pd++;
	x -= floor(x);
	u2 += *pd++ * sin( tpi * x );
	}

/* Main theory terms
 */
for( i=0; i<NUTBL; i++ )
	{
	ph = utbl[i].harm;
	x = 0.0;
	for( j=0; j<4; j++ )
		{
		if( (k = *ph++) != 0 )
			x += k * Delaunay[j];
		}
	pl = utbl[i].cofs;
	s = sin(x);
	c = cos(x);
	u2 += *pl++ * s;
	u2 += *pl++ * c;
	u3 += *pl++ * s;
	u3 += *pl++ * c;
	}
u2 -= 212762.; /* constant terms */
u3 += 80183.;
c = STR / 1000.0;
u2 *= c;
u3 *= c;
u[1] = u2;
u[2] = u3;
u[0] = sqrt( 1.0 - u2*u2 - u3*u3 );
}
