/* code to manage the stuff on the sky view display.
 * the filter menu is in filtermenu.c.
 */

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#if defined(__STDC__)
#include <stdlib.h>
#endif
#include <X11/Xlib.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/DrawingA.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/CascadeB.h>
#include <Xm/RowColumn.h>
#include <Xm/ToggleB.h>
#include <Xm/Text.h>
#include <Xm/Scale.h>
#include <Xm/Separator.h>
#include "astro.h"
#include "circum.h"

/* we keep a linked-list of Objects we want trails for. these in turn "contain"
 * a variable-length array of TSkys for each location in the trail. The TSkys
 * are kept in increasing-time order as they are inserted for later plotting.
 * an item is added when the popup says so; it is removed if we are told the
 * object has been changed via sv_newobj(); all are removed if the whole db
 * changes.
 * also, we add to sky[] in multiples of TRAILCHUNKS for memory efficiency.
 */
typedef struct {
    unsigned flags;	/* OBJF_ things for this particular point */
    double ts_mjd;	/* mjd when this Obj was valid */
    Obj o;		/* copy of the object at ts_mjd */
} TSky;
struct trailObj {
    struct trailObj *ntop;	/* pointer to next, or NULL */
    int on;		/* use to mark on/off; actually remved on next update */
    Obj *op;		/* object being trailed */
    int nsky;		/* number of items in use within sky[] */
    int nskymem;	/* total space of sky[] */
    TSky sky[1];	/* basicaly a variable-length array of Objs */
};
typedef struct trailObj TrailObj;

#if defined(__STDC__) || defined(__cplusplus)
#define P_(s) s
#else
#define P_(s) ()
#endif

extern Now *mm_get_now P_((void));
extern Obj *db_basic P_((int id));
extern void db_scaninit P_((DBScan *sp, HowScan how));
extern Obj *db_scan P_((DBScan *sp));
extern void db_del_cp P_((void));
extern void all_newdb P_((int appended));
extern char *obj_description P_((Obj *op));
extern char *syserrstr P_((void));
extern double mjd_hr P_((double jd));
extern int f_ison P_((void));
extern int lc P_((int cx, int cy, int cw, int x1, int y1, int x2, int y2,
    int *sx1, int *sy1, int *sx2, int *sy2));
extern int svf_filter_ok P_((Obj *op));
extern int tickmarks P_((double min, double max, int numdiv, double ticks[]));
extern int open_wfifores P_((char *fifores, char **fnp, FILE **fpp));
extern int openh P_((char *name, int flags));
extern int obj_cir P_((Now *np, Obj *op));
extern void aa_hadec P_((double lati, double alt, double az, double *ha,
    double *dec));
extern void confnd P_((double r, double d, double e, char **name));
extern void db_update P_((Obj *op));
extern void ecl_eq P_((double Mjd, double Lat, double Lng, double *ra,
    double *dec));
extern void eq_ecl P_((double Mjd, double ra, double dec, double *Lat,
    double *Lng));
extern void f_showit P_((Widget w, char *s));
extern void fs_pangle P_((char out[], double a));
extern void fs_date P_((char out[], double jd));
extern void fs_ra P_((char out[], double ra));
extern void fs_time P_((char out[], double t));
extern void get_something P_((Widget w, char *resource, char *value));
extern void get_xmlabel_font P_((Widget w, XFontStruct **f));
extern void gk_mag P_((double g, double k, double rp, double rho, double *mp));
extern void hadec_aa P_((double lati, double ha, double dec, double *alt,
    double *az));
extern void hlp_dialog P_((char *tag, char *deflt[], int ndeflt));
extern void mjd_year P_((double Mjd, double *yr));
extern void now_lst P_((Now *np, double *lst));
extern void obj_pickgc P_((Obj *op, Widget w, GC *gcp));
extern void obj_set P_((Obj *op));
extern void precess P_((double mjd1, double mjd2, double *ra, double *dec));
extern void range P_((double *v, double r));
extern void refract P_((double pr, double tr, double ta, double *aa));
extern void set_something P_((Widget w, char *resource, char *value));
extern void set_xmstring P_((Widget w, char *resource, char *txt));
extern void svf_create P_((void));
extern void svf_manage_toggle P_((void));
extern void svf_unmanage P_((void));
extern void timestamp P_((Now *np, Widget w));
extern void unrefract P_((double pr, double tr, double aa, double *ta));
extern void watch_cursor P_((int want));
extern void xe_msg P_((char *msg, int app_modal));
extern void year_mjd P_((double y, double *Mjd));

void sv_manage P_((void));
int sv_ison P_((void));
void sv_newobj P_((int dbidx));
void sv_newdb P_((int appended));
void sv_update P_((Now *np, int how_much));
void sv_point P_((Obj *op));
void sv_id P_((Obj *op));
void sv_cursor P_((Cursor c));
void sv_all P_((Now *np, int preclr));
void sv_draw_obj P_((Display *dsp, Drawable win, GC gc, Obj *op, int x, int y,
    int diam, int dotsonly));
static void sv_copy_sky P_((void));
static void sv_create_svform P_((void));
static void sv_create_find P_((Widget parent));
static void sv_close_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_help_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_aa_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_all_labels_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_filter_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_grid_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_out_fifo_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_in_fifo_cb P_((Widget w, XtPointer client, XtPointer call));
static void skyinfifo_cb P_((XtPointer client, int *fdp, XtInputId *idp));
static void sv_loc_fifo_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_track_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_set_scales P_((void));
static void sv_da_exp_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_da_input_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_da_motion_cb P_((Widget w, XtPointer client, XEvent *ev,
    Boolean *continue_to_dispatch));
static void sv_changed_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_ecliptic_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_justdots_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_finding_cb P_((Widget wid, XtPointer client, XtPointer call));
static void sv_find_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_magdrag_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_getcircle P_((unsigned int *wp, unsigned int *hp,
    unsigned int *rp, unsigned int *xbp, unsigned int *ybp));
static void sv_popup P_((XEvent *ev, Obj *op, TSky *tsp));
static void sv_create_popup P_((void));
static void sv_pu_activate_cb P_((Widget w, XtPointer client, XtPointer call));
static void sv_pu_trail_cb P_((Widget wid, XtPointer client, XtPointer call));
static void sv_pu_label_cb P_((Widget wid, XtPointer client, XtPointer call));
static void sv_pu_track_cb P_((Widget wid, XtPointer client, XtPointer call));
static void sv_read_scale P_((int which));
static void sv_draw_track_coords P_((Now *np, unsigned int w, unsigned int h,
    double alt, double az, double ra, double dec));
static int sv_mark P_((Obj *op, int in_center, int out_center, int mark));
static void tobj_rmoff P_((void));
static void tobj_rmobj P_((Obj *op));
static TrailObj *tobj_addobj P_((Obj *op));
static TrailObj *tobj_growsky P_((TrailObj *top));
static void tobj_newdb P_((void));
static TrailObj *tobj_find P_((Obj *op));
static TrailObj *tobj_addsky P_((TrailObj *top, double jd, Obj *op));
static void tobj_display_all P_((unsigned r, unsigned xb, unsigned yb));
static sv_dbobjloc P_((Obj *op, int r, int *xp, int *yp));
static sv_trailobjloc P_((TSky *tsp, int r, int *xp, int *yp));
static sv_precheck P_((Obj *op));
static sv_bright_ok P_((Obj *op));
static sv_infov P_((Obj *op));
static sv_loc P_((int rad, double altdec, double azra, int *xp, int *yp));
static sv_unloc P_((int rad, int x, int y, double *altdecp, double *azrap));
static void sv_fullwhere P_((Now *np, double altdec, double azra, double *altp,
    double *azp, double *rap, double *decp));
static void draw_ecliptic P_((Display *dsp, Window win, GC gc, unsigned int r,
    unsigned int xb, unsigned int yb));
static void draw_grid P_((Display *dsp, Window win, GC gc, unsigned int r,
    unsigned int xb, unsigned int yb));
static magdiam P_((double m));
static void sv_mk_gcs P_((Display *dsp, Window win));
static void draw_label P_((Display *dsp, Window win, GC gc, char label[],
    int x, int y));
static void skyloc_fifo P_((double ra, double dec, double yr));
static void skyin_fifo P_((void));
static void skyout_fifo P_((Now *np));

#undef P_

extern Widget toplevel_w;
extern char *myclass;
extern XtAppContext xe_app;
#define XtD XtDisplay(toplevel_w)

#define	NGRID		20	/* max grid lines each dimension */
#define	NSEGS		5	/* piecewise segments per arc */
#define	ECL_TICS	2	/* period of points for ecliptic, pixels */

#define	MARKR		20	/* pointer marker half-width, pixels */

#define	BMAGLIMIT	(-28)	/* brightest setting for the Mag scales */
#define	FMAGLIMIT	20	/* faintest setting for the Mag scales */
#define	TRAILCHUNKS	50	/* we malloc these many at a time */

static Widget svform_w;			/* main sky view form dialog */
static Widget svda_w;			/* sky view drawing area */
static Widget fov_w, altdec_w, azra_w;	/* scale widgets */
static Widget fmag_w, bmag_w;		/* magnitude scale widgets */
static Widget hgrid_w, vgrid_w;		/* h and v grid spacing labels */
static Widget aa_w, rad_w;		/* altaz/radec toggle buttons */
static Widget dt_w;			/* the date/time stamp label widget */
static Widget find_w[2];		/* cascade buttons for "find" objx/y */
static Pixmap sv_pm;			/* off-screen pixmap we *really* draw */
static XFontStruct *sv_f;		/* font for the tracking coord display*/

/* pixels and GCs
 */
static Pixel fg_p, bg_p, sky_p;	/* default fg, background, and sky colors */
static GC sv_gc;		/* the default GC */
static GC sv_strgc;		/* gc for use in drawing tracking text */

static int aa_mode = -1;	/* 1 for alt/az or 0 for ra/dec */
static double sv_altdec;	/* view center alt or dec, rads */
static double sv_azra;		/* view center az or ra, rads */
static double sv_fov;		/* field of view, rads */
static int fmag, bmag;		/* faintest/brightest magnitude to display */
static int justdots;		/* set when only want to use dots on the map */
static int want_ecliptic;	/* set when want to see the ecliptic */
static TrailObj *trailobj;	/* head of a malloced linked-list -- 0 when
				 * empty
				 */
static Obj *track_op;		/* object to track, when not 0 */
static Widget tracktb_w;	/* toggle button indicating tracking is on */
static int want_grid;		/* set when we want to draw the coord grid */
static all_labels;		/* !0 when want to display all labels */
static int sv_ournewobj;	/* used to inhibit useless redrawing */

static int want_outfifo;	/* set to send center coords to fifo */
static Widget outfifo_w;	/* toggle button controlling out fifo */
static int want_infifo;		/* set to accept aiming coord on fifo */
static Widget infifo_w;		/* toggle button controlling in fifo */
static int want_locfifo;	/* set to send pointer coord to fifo */
static Widget locfifo_w;	/* toggle button controlling loc fifo */

static char null_grid_str[] = "             ";

/* info about the popup widget.
 * we make one once and keep reusing it -- seems to be a bit faster that way.
 */
typedef struct {
    Widget pu_w;
    Widget name_w;
    Widget desc_w;
    Widget spect_w;
    Widget ud_w;
    Widget ut_w;
    Widget ra_w;
    Widget dec_w;
    Widget alt_w;
    Widget az_w;
    Widget mag_w;
    Widget trail_w;
    Widget label_w;
    Widget make_w;
    Widget track_w;
    Obj *op;	/* real database pointer, or possibly svobj */
    TSky *tsp;	/* this is used if we are displaying a trailed object */
} Popup;
static Popup pu;
static Obj svobj;	/* used to point when far from any real object */

enum {AIM, MK_OBJX};		/* popup button activate codes */

enum SCALES {			/* scale changed codes */
    FMAG_S, BMAG_S, FOV_S, ALTDEC_S, AZRA_S
};

/* Obj.flags or TSky flags values */
#define	OBJF_ONSCREEN	0x1	/* bit set if obj is on screen */
#define	OBJF_LABEL	0x2	/* set if label is to be on */

/* called when the sky view is activated via the main menu pulldown.
 * if never called before, create and manage all the widgets as a child of a
 * form. otherwise, just toggle whether the form is managed.
 * also manage the corresponding filter dialog along with.
 * freeing the pixmap will force a fresh update on the next expose.
 */
void
sv_manage ()
{
	if (!svform_w)
	    sv_create_svform(); /* also creates the filter */
	
	if (XtIsManaged(svform_w)) {
	    XtUnmanageChild (svform_w);
	    svf_unmanage();
	    if (sv_pm) {
		XFreePixmap (XtDisplay(svda_w), sv_pm);
		sv_pm = (Pixmap) NULL;
	    }
	} else {
	    XtManageChild (svform_w);
	    /* rely on expose to a fresh update */
	}
}

sv_ison()
{
	return (svform_w && XtIsManaged(svform_w));
}

/* called when a user-defined object has changed.
 * take it off the trailobj list, if it's there (it's ok if it's not).
 * then since we rely on knowing our update will be called we need do nothing
 *   more to redisplay without the object.
 */
void
sv_newobj(dbidx)
int dbidx;		/* OBJX or OBJY */
{
	Obj *op = db_basic(dbidx);

	tobj_rmobj (op);
}

/* called when the db (beyond NOBJ) has changed.
 * if it was appended to we can just redraw else it was reduced and we need to
 * discard any trails we are keeping for objects that no longer exist.
 */
void
sv_newdb(appended)
int appended;
{
	if (!appended)
	    tobj_newdb();
	sv_update (mm_get_now(), 1);
}

/* called when we are to update our view.
 * don't bother if we are unmanaged unless there are trails to be saved.
 * remove any trails that have been turned off.
 * reaim if we are tracking an object.
 * add any objects in the trails list then redraw the screen if we are up.
 * also don't bother if we are the reason for an all_newobj() or if fields are
 *   off.
 */
/* ARGSUSED */
void
sv_update (np, how_much)
Now *np;
int how_much;
{
	TrailObj *top;
	int up;

	if (sv_ournewobj)
	    return;

	up = svform_w && XtIsManaged(svform_w);
	if (!up && !trailobj)
	    return;

	/* remove trails no longer wanted. */
	tobj_rmoff();

	/* add an entry to each trailed object for this time. */
	for (top = trailobj; top; top = top->ntop) {
	    db_update (top->op);
	    top = tobj_addsky (top, mjd, top->op);
	}

	if (up && f_ison()) {
	    if (track_op) {
		db_update (track_op);
		if (sv_mark (track_op, 1, 1, 1) < 0)
		    sv_all(np, 1);
	    } else 
		sv_all(np, 1);
	}
}

/* point at the given object and mark it.
 * N.B. we do *not* update the s_ fields of op.
 */
void
sv_point (op)
Obj *op;
{
	if (!sv_ison() || !op || op->type == UNDEFOBJ)
	    return;

	(void) sv_mark (op, 1, 1, 1);
}

/* show a marker at the location of the given object if it's in fov now.
 * N.B. we do *not* update the s_ fields of op.
 */
void
sv_id (op)
Obj *op;
{
	if (!sv_ison() || !op || op->type == UNDEFOBJ)
	    return;

	(void) sv_mark (op, 0, 0, 1);
}

/* called to put up or remove the watch cursor.  */
void
sv_cursor (c)
Cursor c;
{
	Window win;

	if (svform_w && (win = XtWindow(svform_w))) {
	    Display *dsp = XtDisplay(svform_w);
	    if (c)
		XDefineCursor (dsp, win, c);
	    else
		XUndefineCursor (dsp, win);
	}
}

/* draw everything subject to any filtering.
 * also emit the new center coords to the SKYOUTFIFO.
 */
/* ARGSUSED */
void
sv_all(np, preclr)
Now *np;
int preclr;
{
	Display *dsp = XtDisplay(svda_w);
	DBScan dbs;
	Window win = sv_pm;
	unsigned int w, h, r, xb, yb;
	Obj *op;

	watch_cursor(1);

	/* put up the timestamp */
	timestamp (np, dt_w);

	sv_getcircle (&w, &h, &r, &xb, &yb);

	/* rebuild the clean circle */
	XSetForeground (dsp, sv_gc, bg_p);
	XFillRectangle (dsp, win, sv_gc, 0, 0, w, h);
	XSetForeground (dsp, sv_gc, sky_p);
	XFillArc (dsp, win, sv_gc, xb+1, yb+1, 2*r-2, 2*r-2, 0, 360*64);
	XSetForeground (dsp, sv_gc, fg_p);
	XDrawArc (dsp, win, sv_gc, xb, yb, 2*r-1, 2*r-1, 0, 360*64);

	/* go through the database and display what we want */
	for (db_scaninit(&dbs, SCAN_ALL); op = db_scan(&dbs); ) {
	    GC gc;
	    int x, y;
	    obj_pickgc(op, svda_w, &gc);
	    if (!sv_dbobjloc(op, r, &x, &y))
		op->o_flags &= ~OBJF_ONSCREEN;
	    else {
		/* yup, it's really supposed to be on the screen */
		db_update (op);
		x += xb;
		y += yb;
		sv_draw_obj (dsp, win, gc, op, x, y,
				    magdiam(op->s_mag/MAGSCALE), justdots);
		op->o_flags |= OBJF_ONSCREEN;
		if (all_labels || (op->o_flags & OBJF_LABEL))
		    draw_label (dsp, win, gc, op->o_name, x, y);
	    }
	}

	sv_draw_obj (dsp, win, (GC)0, NULL, 0, 0, 0, 0);	/* flush */

	/* go through the trailobj list and display that stuff too. */
	tobj_display_all(r, xb, yb);

	/* draw a grid on top if desired */
	if (want_grid) {
	    XSetForeground (dsp, sv_gc, fg_p);
	    draw_grid(dsp, win, sv_gc, r, xb, yb);
	}

	/* draw the ecliptic on top if desired */
	if (want_ecliptic) {
	    XSetForeground (dsp, sv_gc, fg_p);
	    draw_ecliptic (dsp, win, sv_gc, r, xb, yb);
	}

	/* emit our center coords */
	skyout_fifo (np);

	/* and we're done */
	sv_copy_sky();

	watch_cursor(0);
}

/* draw the given object so it has a nominal diameter of diam pixels.
 * we maintain a static cache of common X drawing objects for efficiency.
 * (mallocing seemed to keep the memory arena too fragmented).
 * to force a flush, call with op == NULL.
 */
void
sv_draw_obj (dsp, win, gc, op, x, y, diam, dotsonly)
Display *dsp;
Drawable win;
GC gc;
Obj *op;
int x, y;
int diam;
int dotsonly;
{
#define	CACHE_SZ	100
#define	CACHE_PAD	10	/* most we ever need in one call */
#define	CACHE_HWM	(CACHE_SZ - CACHE_PAD)	/* hi water mark */
	static XPoint   xpoints[CACHE_SZ],   *xp  = xpoints;
	static XArc     xdrawarcs[CACHE_SZ], *xda = xdrawarcs;
	static XArc     xfillarcs[CACHE_SZ], *xfa = xfillarcs;
	static XSegment xsegments[CACHE_SZ], *xs  = xsegments;
	static GC cache_gc;
	int force;
	int t;

	/* for sure if no op or different gc */
	force = !op || gc != cache_gc;

	if (force || xp >= xpoints + CACHE_HWM) {
	    int n;
	    if ((n = xp - xpoints) > 0) {
		XDrawPoints (dsp, win, cache_gc, xpoints, n, CoordModeOrigin);
		xp = xpoints;
#ifdef SVTRACE
		printf ("points=%d\n", n);
#endif
	    }
	}
	if (force || xda >= xdrawarcs + CACHE_HWM) {
	    int n;
	    if ((n = xda - xdrawarcs) > 0) {
		XDrawArcs (dsp, win, cache_gc, xdrawarcs, n);
		xda = xdrawarcs;
#ifdef SVTRACE
		printf ("drawarcs=%d\n", n);
#endif
	    }
	}
	if (force || xfa >= xfillarcs + CACHE_HWM) {
	    int n;
	    if ((n = xfa - xfillarcs) > 0) {
		XFillArcs (dsp, win, cache_gc, xfillarcs, n);
		xfa = xfillarcs;
#ifdef SVTRACE
		printf ("fillarcs=%d\b", n);
#endif
	    }
	}
	if (force || xs >= xsegments + CACHE_HWM) {
	    int n;
	    if ((n = xs - xsegments) > 0) {
		XDrawSegments (dsp, win, cache_gc, xsegments, n);
		xs = xsegments;
#ifdef SVTRACE
		printf ("segments=%d\n", n);
#endif
	    }
	}

	cache_gc = gc;

	if (!op)
	    return;	/* just flushing, thanks */

	/* one-pixel wide anything is just a dot */
	if (diam <= 1) {
	    xp->x = x;
	    xp->y = y;
	    xp++;
	    return;
	}

	/* if dotsonly is on don't use the fancy symbols, just use dots */
	if (dotsonly) {
	    xfa->x = x - diam/2;
	    xfa->y = y - diam/2;
	    xfa->width = diam;
	    xfa->height = diam;
	    xfa->angle1 = 0;
	    xfa->angle2 = 360*64;
	    xfa++;
	    return;
	}

	switch (op->type) {
	case PLANET:
	    /* filled circle */
	    xfa->x = x - diam/2;
	    xfa->y = y - diam/2;
	    xfa->width = diam;
	    xfa->height = diam;
	    xfa->angle1 = 0;
	    xfa->angle2 = 360*64;
	    xfa++;
	    break;
	case FIXED:
	    switch (op->f_class) {
	    case 'G': case 'H': case 'A': /* galaxy */
		/* filled ellipse */
		xfa->x = x - diam/2;
		xfa->y = y - diam/4;
		xfa->width = diam;
		xfa->height = diam/2;
		xfa->angle1 = 0;
		xfa->angle2 = 360*64;
		xfa++;
		break;
	    case 'C': case 'U': /* globular clusters */
		/* same as cluster but with a central dot */
		xfa->x = x - 1;
		xfa->y = y - 1;
		xfa->width = 3;
		xfa->height = 3;
		xfa->angle1 = 0;
		xfa->angle2 = 360*64;
		xfa++;
		/* fall through */
		/* FALLTHROUGH */
	    case 'O': /* open cluster */
		/* open circle of dots */
		xp->x = x;	xp->y = y-3;	xp++;
		xp->x = x+3;	xp->y = y-1;	xp++;
		xp->x = x+3;	xp->y = y+1;	xp++;
		xp->x = x;	xp->y = y+3;	xp++;
		xp->x = x-3;	xp->y = y+1;	xp++;
		xp->x = x-3;	xp->y = y-1;	xp++;
		break;
	    case 'P': /* planetary nebula */
		/* open ellipse */
		xda->x = x - diam/2;
		xda->y = y - diam/4;
		xda->width = diam;
		xda->height = diam/2;
		xda->angle1 = 0;
		xda->angle2 = 360*64;
		xda++;
		break;
	    case 'S': /* stars */
		/* filled circle */
		xfa->x = x - diam/2;
		xfa->y = y - diam/2;
		xfa->width = diam;
		xfa->height = diam;
		xfa->angle1 = 0;
		xfa->angle2 = 360*64;
		xfa++;
		break;
	    case 'B': case 'D': /* binary and double stars */
		/* filled circle with one horizontal line through it */
		xfa->x = x - diam/2;
		xfa->y = y - diam/2;
		xfa->width = diam;
		xfa->height = diam;
		xfa->angle1 = 0;
		xfa->angle2 = 360*64;
		xfa++;
		t = 3*diam/4;
		xs->x1 = x - t;
		xs->y1 = y;
		xs->x2 = x + t;
		xs->y2 = y;
		xs++;
		break;
	    case 'M': /* multiple stars */
		/* filled circle with two horizontal lines through it */
		xfa->x = x - diam/2;
		xfa->y = y - diam/2;
		xfa->width = diam;
		xfa->height = diam;
		xfa->angle1 = 0;
		xfa->angle2 = 360*64;
		xfa++;
		t = 3*diam/4;
		xs->x1 = x - t;
		xs->y1 = y - diam/6;
		xs->x2 = x + t;
		xs->y2 = y - diam/6;
		xs++;
		xs->x1 = x - t;
		xs->y1 = y + diam/6;
		xs->x2 = x + t;
		xs->y2 = y + diam/6;
		xs++;
		break;
	    case 'V': /* variable star */
		/* central dot with concentric circle */
		xfa->x = x - diam/4;
		xfa->y = y - diam/4;
		xfa->width = diam/2;
		xfa->height = diam/2;
		xfa->angle1 = 0;
		xfa->angle2 = 360*64;
		xfa++;
		xda->x = x - diam/2;
		xda->y = y - diam/2;
		xda->width = diam;
		xda->height = diam;
		xda->angle1 = 0;
		xda->angle2 = 360*64;
		xda++;
		break;
	    case 'F': case 'K': /* diffuse and dark nebulae */
		/* open diamond */
		xs->x1 = x-diam/2;	xs->y1 = y;
		    xs->x2 = x;		    xs->y2 = y-diam/2;	xs++;
		xs->x1 = x;		xs->y1 = y-diam/2;
		    xs->x2 = x+diam/2;	    xs->y2 = y;		xs++;
		xs->x1 = x+diam/2;	xs->y1 = y;
		    xs->x2 = x;		    xs->y2 = y+diam/2;	xs++;
		xs->x1 = x;		xs->y1 = y+diam/2;
		    xs->x2 = x-diam/2;	    xs->y2 = y;		xs++;
		break;
	    case 'N': /* bright nebulae */
		/* filled diamond */
		{   XPoint p[5];
		    p[0].x = x-diam/2;	p[0].y = y;
		    p[1].x = x;		p[1].y = y-diam/2;
		    p[2].x = x+diam/2;	p[2].y = y;
		    p[3].x = x;		p[3].y = y+diam/2;
		    p[4].x = x-diam/2;	p[4].y = y;
		    XFillPolygon (dsp, win, gc, p, 5, Convex, CoordModeOrigin);
		}
		break;
	    case 'Q': /* Quasar */
		/* plus sign */
		xs->x1 = x-diam/2;	xs->y1 = y;
		    xs->x2 = x+diam/2;	    xs->y2 = y;		xs++;
		xs->x1 = x;		xs->y1 = y-diam/2;
		    xs->x2 = x;		    xs->y2 = y+diam/2;	xs++;
		break;
	    case 'T':	/* stellar object */
	    case '\0':	/* undefined type */
	    default:	/* unknown type */
		/* an X */
		xs->x1 = x-diam/3;	xs->y1 = y-diam/3;
		    xs->x2 = x+diam/3;	    xs->y2 = y+diam/3;	xs++;
		xs->x1 = x-diam/3;	xs->y1 = y+diam/3;
		    xs->x2 = x+diam/3;    	xs->y2 = y-diam/3;	xs++;
		break;
	    }
	    break;
	case HYPERBOLIC:
	case PARABOLIC:
	    /* usually cometose -- draw a filled circle with tail */
	    t = 3*diam/4;
	    xfa->x = x - diam/4;
	    xfa->y = y - diam/4;
	    xfa->width = diam/2;
	    xfa->height = diam/2;
	    xfa->angle1 = 0;
	    xfa->angle2 = 360*64;
	    xfa++;
	    xfa->x = x - t;
	    xfa->y = y - t;
	    xfa->width = 2*t;
	    xfa->height = 2*t;
	    xfa->angle1 = 120*64;
	    xfa->angle2 = 30*64;
	    xfa++;
	    break;
	case ELLIPTICAL:
	    /* filled square */
	    XFillRectangle (dsp, win, gc, x-diam/2, y-diam/2, diam, diam);
	    break;
	case EARTHSAT:
	    /* open circle */
	    xda->x = x - diam/2;
	    xda->y = y - diam/2;
	    xda->width = diam;
	    xda->height = diam;
	    xda->angle1 = 0;
	    xda->angle2 = 360*64;
	    xda++;
	    break;
	default:
	    printf ("Bad type to sv_draw_obj: %d\n", op->type);
	    exit(1);
	}
}

/* create the main skyview form */
static void
sv_create_svform()
{
	static struct {
	    char *name;
	    void (*cb)();
	} cbs[] = {
	    {"Filter", sv_filter_cb},
	    {"Close", sv_close_cb},
	    {"Help", sv_help_cb},
	};
	Widget rc_w;
	Widget w;
	Widget rb_w;
	Widget sep_w;
	Widget ctlfr_w, ctl_w;
	XmString str;
	Arg args[20];
	EventMask mask;
	int i, n;

	/* create form */
	n = 0;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	XtSetArg (args[n], XmNdefaultPosition, False); n++;
	XtSetArg (args[n], XmNresizePolicy, XmRESIZE_NONE); n++;
	svform_w = XmCreateFormDialog (toplevel_w, "SkyView", args, n);
	XtAddCallback (svform_w, XmNhelpCallback, sv_help_cb, 0);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "xephem Sky view"); n++;
	XtSetValues (XtParent(svform_w), args, n);

	/* make a form across the bottom in a frame for the control buttons */

	n = 0;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	ctlfr_w = XmCreateFrame (svform_w, "CtlFr", args, n);
	XtManageChild (ctlfr_w);

	n = 0;
	XtSetArg (args[n], XmNfractionBase, 10); n++;
	ctl_w = XmCreateForm (ctlfr_w, "CtlF", args, n);
	XtManageChild (ctl_w);

	    /* make the control buttons */

	    for (i = 0; i < XtNumber(cbs); i++) {
		n = 0;
		XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
		XtSetArg (args[n], XmNtopOffset, 10); n++;
		XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
		XtSetArg (args[n], XmNbottomOffset, 10); n++;
		XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
		XtSetArg (args[n], XmNleftPosition, 1+i*3); n++;
		XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
		XtSetArg (args[n], XmNrightPosition, 3+i*3); n++;
		w = XmCreatePushButton (ctl_w, cbs[i].name, args, n);
		XtAddCallback (w, XmNactivateCallback, cbs[i].cb, 0);
		XtManageChild (w);
	    }

	/* make a RowColumn down the left side for the misc controls */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, ctlfr_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNisAligned, False); n++;
	XtSetArg (args[n], XmNadjustMargin, False); n++;
	rc_w = XmCreateRowColumn (svform_w, "SVRC", args, n);
	XtManageChild (rc_w);

	    /* make the radio box for alt/az vs ra/dec */

	    n = 0;
	    rb_w = XmCreateRadioBox (rc_w, "SVModeRB", args, n);
	    XtManageChild (rb_w);

		str = XmStringCreate ("Alt-Az", XmSTRING_DEFAULT_CHARSET);
		n = 0;
		XtSetArg (args[n], XmNlabelString, str); n++;
		XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
		aa_w = XmCreateToggleButton (rb_w, "AltAzMode", args, n);
		XtAddCallback (aa_w, XmNvalueChangedCallback, sv_aa_cb, NULL);
		XtManageChild (aa_w);
		XmStringFree (str);

		str = XmStringCreate ("RA-Dec", XmSTRING_DEFAULT_CHARSET);
		n = 0;
		XtSetArg (args[n], XmNlabelString, str); n++;
		XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
		rad_w = XmCreateToggleButton(rb_w, "RADecMode", args, n);
		XtManageChild (rad_w);
		XmStringFree (str);

		if(XmToggleButtonGetState(aa_w)==XmToggleButtonGetState(rad_w)){
		    xe_msg("Sky View display mode conflicts -- using Alt/Az",0);
		    aa_mode = 1;	/* default to aa if they conflict */
		    XmToggleButtonSetState (aa_w, aa_mode, False); 
		    XmToggleButtonSetState (rad_w, !aa_mode, False); 
		}
		aa_mode = XmToggleButtonGetState(aa_w);

	    n = 0;
	    w = XmCreateSeparator (rc_w, "RCSep1", args, n);
	    XtManageChild (w);

	    /* make the "grid" toggle button and calibration labels */

	    n = 0;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    w = XmCreateToggleButton (rc_w, "Grid", args, n);
	    XtAddCallback (w, XmNvalueChangedCallback, sv_grid_cb, 0);
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    vgrid_w = XmCreateLabel (rc_w, "SVGridV", args, n);
	    set_xmstring (vgrid_w, XmNlabelString, null_grid_str);
	    XtManageChild (vgrid_w);

	    n = 0;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    hgrid_w = XmCreateLabel (rc_w, "SVGridH", args, n);
	    set_xmstring (hgrid_w, XmNlabelString, null_grid_str);
	    XtManageChild (hgrid_w);

	    n = 0;
	    w = XmCreateSeparator (rc_w, "RCSep2", args, n);
	    XtManageChild (w);

	    /* make the faint magnitude slider scale */

	    str = XmStringCreate ("Faintest Mag", XmSTRING_DEFAULT_CHARSET);
	    n = 0;
	    XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
	    XtSetArg (args[n], XmNminimum, BMAGLIMIT); n++;
	    XtSetArg (args[n], XmNmaximum, FMAGLIMIT); n++;
	    XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_LEFT); n++;
	    XtSetArg (args[n], XmNscaleMultiple, 1); n++;
	    XtSetArg (args[n], XmNshowValue, True); n++;
	    XtSetArg (args[n], XmNtitleString, str); n++;
	    fmag_w = XmCreateScale (rc_w, "FaintMagScale", args, n);
	    XtAddCallback(fmag_w, XmNvalueChangedCallback, sv_changed_cb, 
							    (XtPointer)FMAG_S);
	    XtAddCallback(fmag_w, XmNdragCallback, sv_magdrag_cb, NULL);
	    XtManageChild (fmag_w);
	    XmStringFree (str);

	    /* make the bright magnitude slider scale */

	    str = XmStringCreate ("Brightest Mag", XmSTRING_DEFAULT_CHARSET);
	    n = 0;
	    XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
	    XtSetArg (args[n], XmNminimum, BMAGLIMIT); n++;
	    XtSetArg (args[n], XmNmaximum, FMAGLIMIT); n++;
	    XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_LEFT); n++;
	    XtSetArg (args[n], XmNscaleMultiple, 1); n++;
	    XtSetArg (args[n], XmNshowValue, True); n++;
	    XtSetArg (args[n], XmNtitleString, str); n++;
	    bmag_w = XmCreateScale (rc_w, "BrightMagScale", args, n);
	    XtAddCallback(bmag_w, XmNvalueChangedCallback, sv_changed_cb,
							    (XtPointer)BMAG_S);
	    XtAddCallback(bmag_w, XmNdragCallback, sv_magdrag_cb, NULL);
	    XtManageChild (bmag_w);
	    XmStringFree (str);

	    n = 0;
	    w = XmCreateSeparator (rc_w, "RCSep3", args, n);
	    XtManageChild (w);

	    /* make the "just dots" toggle button */

	    str = XmStringCreate ("Just Dots", XmSTRING_DEFAULT_CHARSET);
	    n = 0;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    XtSetArg (args[n], XmNlabelString, str); n++;
	    w = XmCreateToggleButton (rc_w, "JustDots", args, n);
	    XtAddCallback (w, XmNvalueChangedCallback, sv_justdots_cb, NULL);
	    XtManageChild (w);
	    justdots = XmToggleButtonGetState (w);
	    XmStringFree(str);

	    /* make the "ecliptic" toggle button */

	    str = XmStringCreate ("Ecliptic", XmSTRING_DEFAULT_CHARSET);
	    n = 0;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    XtSetArg (args[n], XmNlabelString, str); n++;
	    w = XmCreateToggleButton (rc_w, "Ecliptic", args, n);
	    XtAddCallback (w, XmNvalueChangedCallback, sv_ecliptic_cb, NULL);
	    XtManageChild (w);
	    want_ecliptic = XmToggleButtonGetState (w);
	    XmStringFree(str);

	    /* make the "All labels" toggle button */

	    str = XmStringCreate ("All labels", XmSTRING_DEFAULT_CHARSET);
	    n = 0;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    XtSetArg (args[n], XmNlabelString, str); n++;
	    w = XmCreateToggleButton (rc_w, "AllLabels", args, n);
	    XtAddCallback (w, XmNvalueChangedCallback, sv_all_labels_cb, NULL);
	    XtManageChild (w);
	    all_labels = XmToggleButtonGetState (w);
	    XmStringFree(str);

	    /* create the tracking tb */

	    n = 0;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    XtSetArg (args[n], XmNset, False); n++;	/* insure not overrode*/
	    tracktb_w = XmCreateToggleButton (rc_w, "Tracking", args, n);
	    XtAddCallback (tracktb_w, XmNvalueChangedCallback, sv_track_cb, 0);
	    XtSetSensitive (tracktb_w, False);
	    XtManageChild (tracktb_w);

	    n = 0;
	    w = XmCreateSeparator (rc_w, "RCSep4", args, n);
	    XtManageChild (w);

	    /* make the "Out FIFO" toggle button */

	    str = XmStringCreate ("Out FIFO", XmSTRING_DEFAULT_CHARSET);
	    n = 0;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    XtSetArg (args[n], XmNlabelString, str); n++;
	    outfifo_w = XmCreateToggleButton (rc_w, "OutFIFO", args, n);
	    XtAddCallback (outfifo_w, XmNvalueChangedCallback, sv_out_fifo_cb,
									NULL);
	    XtManageChild (outfifo_w);
	    want_outfifo = XmToggleButtonGetState (outfifo_w);
	    skyout_fifo ((Now *)0);
	    XmStringFree(str);

	    /* make the "In FIFO" toggle button */

	    str = XmStringCreate ("In FIFO", XmSTRING_DEFAULT_CHARSET);
	    n = 0;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    XtSetArg (args[n], XmNlabelString, str); n++;
	    infifo_w = XmCreateToggleButton (rc_w, "InFIFO", args, n);
	    XtAddCallback (infifo_w, XmNvalueChangedCallback, sv_in_fifo_cb,
									NULL);
	    XtManageChild (infifo_w);
	    want_infifo = XmToggleButtonGetState (infifo_w);
	    /* TODO: react to initial state */
	    XmStringFree(str);

	    /* make the "Loc FIFO" toggle button */

	    str = XmStringCreate ("Loc FIFO", XmSTRING_DEFAULT_CHARSET);
	    n = 0;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    XtSetArg (args[n], XmNlabelString, str); n++;
	    locfifo_w = XmCreateToggleButton (rc_w, "LocFIFO", args, n);
	    XtAddCallback (locfifo_w, XmNvalueChangedCallback, sv_loc_fifo_cb,
									NULL);
	    XtManageChild (locfifo_w);
	    want_locfifo = XmToggleButtonGetState (locfifo_w);
	    XmStringFree(str);

	    n = 0;
	    w = XmCreateSeparator (rc_w, "RCSep5", args, n);
	    XtManageChild (w);

	    /* make the "find" cascade button */
	    sv_create_find (rc_w);

	/* make a vertical sep to the right of the left row/col */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, ctlfr_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNleftWidget, rc_w); n++;
	XtSetArg (args[n], XmNorientation, XmVERTICAL); n++;
	sep_w = XmCreateSeparator (svform_w, "Sep1", args, n);
	XtManageChild(sep_w);

	/* make a timestamp label just above the control panel's frame */

	n = 0;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNleftWidget, sep_w); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, ctlfr_w); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_CENTER); n++;
	dt_w = XmCreateLabel (svform_w, "TimeStamp", args, n);
	XtManageChild(dt_w);

	/* make the bottom scale above the time stamp */

	n = 0;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNleftWidget, sep_w); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, dt_w); n++;
	XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
	XtSetArg (args[n], XmNscaleMultiple, 1); n++;
	XtSetArg (args[n], XmNshowValue, True); n++;
	azra_w = XmCreateScale (svform_w, "AzRAScale", args, n);
	XtAddCallback (azra_w, XmNvalueChangedCallback, sv_changed_cb,
							    (XtPointer)AZRA_S);
	XtManageChild (azra_w);

	/* make the left scale */

	str = XmStringCreate ("FOV", XmSTRING_DEFAULT_CHARSET);
	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNleftWidget, sep_w); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, azra_w); n++;
	XtSetArg (args[n], XmNorientation, XmVERTICAL); n++;
	XtSetArg (args[n], XmNminimum, 1); n++;
	XtSetArg (args[n], XmNmaximum, 180); n++;
	XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_BOTTOM); n++;
	XtSetArg (args[n], XmNscaleMultiple, 1); n++;
	XtSetArg (args[n], XmNshowValue, True); n++;
	XtSetArg (args[n], XmNtitleString, str); n++;
	fov_w = XmCreateScale (svform_w, "FOVScale", args, n);
	XtAddCallback (fov_w, XmNvalueChangedCallback, sv_changed_cb, 
							    (XtPointer)FOV_S);
	XtManageChild (fov_w);
	XmStringFree (str);

	/* make the right scale */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, azra_w); n++;
	XtSetArg (args[n], XmNorientation, XmVERTICAL); n++;
	XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_TOP); n++;
	XtSetArg (args[n], XmNscaleMultiple, 1); n++;
	XtSetArg (args[n], XmNshowValue, True); n++;
	altdec_w = XmCreateScale (svform_w, "AltDecScale", args, n);
	XtAddCallback (altdec_w, XmNvalueChangedCallback, sv_changed_cb,
							(XtPointer)ALTDEC_S);
	XtManageChild (altdec_w);

	/* make a drawing area inside the sliders for drawing the sky */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, azra_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNleftWidget, fov_w); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNrightWidget, altdec_w); n++;
	svda_w = XmCreateDrawingArea (svform_w, "Map", args, n);
	XtAddCallback (svda_w, XmNexposeCallback, sv_da_exp_cb, 0);
	XtAddCallback (svda_w, XmNinputCallback, sv_da_input_cb, 0);
	mask = Button1MotionMask | ButtonPressMask | ButtonReleaseMask
						   | PointerMotionHintMask,
	XtAddEventHandler (svda_w, mask, False, sv_da_motion_cb,0);
	XtManageChild (svda_w);

	/* create the filter dialog.
	 * it's not managed yet, but its state info is used right off.
	 */
	svf_create();

	/* create the popup */
	sv_create_popup();

	/* read the widget to init the real values.
	 * best to do these as a group near the end because some may rely on
	 * other widgets too.
	 */
	sv_read_scale(BMAG_S);
	sv_read_scale(FMAG_S);
	sv_read_scale(AZRA_S);
	sv_read_scale(FOV_S);
	sv_read_scale(ALTDEC_S);

	/* then read back the raw values to form a consistent set of widgets. */
	sv_set_scales();
}

/* make the "Locate" cascade button */
static void
sv_create_find (parent)
Widget parent;
{
	Arg args[20];
	Widget mb_w, menu_w, p_w;
	Widget w;
	int n;
	int i;

	/* create menu bar for the cascade button */

	n = 0;
	mb_w = XmCreateMenuBar (parent, "PointMB", args, n);
	XtManageChild (mb_w);

	/* create pulldown managed by the cascade button */
	n = 0;
	menu_w = XmCreatePulldownMenu (parent, "PointPD", args, n);

	    /* create the pushbuttons forming the cascade menu.
	     * go ahead and fill in and manage the planet names now.
	     * we do the user objects just as we are cascading.
	     */
	    for (i = 0; i < NOBJ; i++) {
		n = 0;
		w = XmCreatePushButton (menu_w, "PointPB", args, n);
		XtAddCallback (w, XmNactivateCallback, sv_find_cb,
								(XtPointer)i);
		if (i >= MERCURY && i <= MOON) {
		    Obj *op = db_basic (i);
		    set_xmstring (w, XmNlabelString, op->o_name);
		    XtManageChild (w);
		} else if (i == OBJX)
		    find_w[0] = w;
		else if (i == OBJY)
		    find_w[1] = w;
	    }

	n = 0;
	XtSetArg (args[n], XmNsubMenuId, menu_w); n++;
	XtSetArg (args[n], XmNmarginHeight, 0); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_CENTER); n++;
	p_w = XmCreateCascadeButton (mb_w, "PointCB", args, n);
	set_xmstring (p_w, XmNlabelString, "Locate...");
	XtAddCallback (p_w, XmNcascadingCallback, sv_finding_cb, 0);
	XtManageChild (p_w);
}

/* callback from the main Close button.
 */
/* ARGSUSED */
static void
sv_close_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XtUnmanageChild (svform_w);
	svf_unmanage ();
	if (sv_pm) {
	    XFreePixmap (XtDisplay(svda_w), sv_pm);
	    sv_pm = (Pixmap) NULL;
	}
}

/* callback from the Help button.
 */
/* ARGSUSED */
static void
sv_help_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	static char *msg[] = {
"This displays all database objects currently in memory onto the sky. The view",
"may be Alt/Az or RA/Dec. The three sliders adjust the field of fov, the",
"azimuth (or RA), and the altitude (or Dec). Objects may be filtered out by",
"type and magnitude."
};

	hlp_dialog ("Sky View", msg, sizeof(msg)/sizeof(msg[0]));
}

/* callback when the alt/az button changes state.
 */
/* ARGSUSED */
static void
sv_aa_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	double alt, az, ra, dec;
	Now *np;

	np = mm_get_now();

	sv_fullwhere (np, sv_altdec, sv_azra, &alt, &az, &ra, &dec);

	if (aa_mode) {
	    /* change from alt/az to ra/dec mode */
	    sv_azra = ra;
	    sv_altdec = dec;
	} else {
	    /* change from ra/dec to alt/az mode if center is above horizon */
	    if (alt < 0.0) {
		xe_msg ("View center would go below horizon.", 1);
		XmToggleButtonSetState (aa_w, False, False);
		XmToggleButtonSetState (rad_w, True, False);
		return;
	    }
	    sv_altdec = alt;
	    sv_azra = az;
	}

	aa_mode ^= 1;

	sv_set_scales();

	if (track_op) {
	    if (sv_mark (track_op, 1, 1, 1) < 0)
		sv_all(np, 1);
	} else 
	    sv_all(np, 1);
}

/* callback from the filter button.
 * just toggle the filter dialog.
 * always restore it to the real state before bringing it up.
 */
/* ARGSUSED */
static void
sv_filter_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	svf_manage_toggle ();
}

/* callback from the grid button.
 * if adding the grid just draw it, but if taking it away copy from pixmap.
 */
/* ARGSUSED */
static void
sv_grid_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	want_grid = XmToggleButtonGetState(w);

	sv_all(mm_get_now(), 1);

	if (!want_grid) {
	    set_xmstring (vgrid_w, XmNlabelString, null_grid_str);
	    set_xmstring (hgrid_w, XmNlabelString, null_grid_str);
	}
}

/* callback from the "Out FIFO" toggle button.
 */
/* ARGSUSED */
static void
sv_out_fifo_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	want_outfifo = XmToggleButtonGetState(w);
	skyout_fifo (mm_get_now());
}

/* callback from the "In FIFO" toggle button.
 */
/* ARGSUSED */
static void
sv_in_fifo_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	want_infifo = XmToggleButtonGetState(w);
	skyin_fifo ();
}

/* called whenever there is input waiting from the SKYINFIFO.
 * we gulp a lot and only use the last whole line we see that looks right.
 * N.B. do EXACTLY ONE read -- don't know that more won't block.
 */
/* ARGSUSED */
static void
skyinfifo_cb (client, fdp, idp)
XtPointer client;       /* file name */
int *fdp;               /* pointer to file descriptor */
XtInputId *idp;         /* pointer to input id */
{
	static char fmt[] = "RA:%9lf Dec:%9lf Epoch:%9lf";
	char buf[8096];
	Now *np = mm_get_now();
	char *fn = (char *)client;
	char msg[256];
	double ra, dec, y;
	char *bp, *lp;
	int gotone;
	int n;

	n = read (*fdp, buf, sizeof(buf));
	if (n < 0) {
	    (void) sprintf (msg, "Error reading from %.100s: %.100s\n", fn,
								syserrstr());
	    xe_msg (msg, 0);
	    return;
	}
	if (n == 0) {
	    /* EOF on a R/W open must mean it's really a plain file.  */
	    (void) sprintf (msg, "Closing %.200s at EOF.\n", fn);
	    xe_msg (msg, 0);
	    XmToggleButtonSetState (infifo_w, False, True); /* closes all */
	    return;
	}

	gotone = 0;
	for (bp = lp = buf; --n >= 0; )
	    if (*bp++ == '\n') {
		bp[-1] = '\0';
		if (sscanf (lp, fmt, &ra, &dec, &y) == 3) {
		    if (ra < 0 || ra >= 2*PI || dec > PI/2 || dec < -PI/2) {
			(void) sprintf (msg, "Bad %s coords: RA=%g Dec=%g\n",
								fn, ra, dec);
			xe_msg (msg, 0);
		    } else
			gotone = 1;
		}
		lp = bp;
	    }

	if (gotone) {
	    Obj obj;
	    double m;

	    obj.type = FIXED;
	    (void) strcpy (obj.o_name, "<Anon>");
	    obj.f_RA = ra;
	    obj.f_dec = dec;
	    year_mjd (y, &m);
	    obj.f_epoch = m;
	    if (obj_cir (np, &obj) == 0)
		(void) sv_mark (&obj, 1, 1, 0);
	}
}

/* callback from the "Loc FIFO" toggle button.
 */
/* ARGSUSED */
static void
sv_loc_fifo_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	want_locfifo = XmToggleButtonGetState(w);
	skyloc_fifo (0.0, 0.0, 0.0);
}

/* callback when the permanent tracking tb down the left r/c is toggled.
 * N.B. also used to implement turning tracking off from the popup too.
 */
/* ARGSUSED */
static void
sv_track_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (!XmToggleButtonGetState (w)) {
	    track_op = (Obj *)0;
	    XtSetSensitive (w, False);
	    sv_all (mm_get_now(), 1);	/* erase the mark */
	}
}

/* set the altdec_w and azra_w scale values, depending on aa_mode and sv_altdec
 * and sv_azra.
 */
static void
sv_set_scales()
{
	Arg args[20];
	int n;

	if (aa_mode) {
	    int altitude, azimuth;

	    altitude = raddeg(sv_altdec) + 0.5;
	    azimuth = raddeg(sv_azra) + 0.5;
	    if (azimuth >= 360) azimuth -= 360;
	    if (azimuth < 0) azimuth += 360;

	    n = 0;
	    XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_RIGHT); n++;
	    XtSetArg (args[n], XmNmaximum, 359); n++;
	    XtSetArg (args[n], XmNminimum, 0); n++;
	    XtSetArg (args[n], XmNdecimalPoints, 0); n++;
	    XtSetArg (args[n], XmNvalue, azimuth); n++;
	    XtSetValues (azra_w, args, n);

	    n = 0;
	    XtSetArg (args[n], XmNmaximum, 90); n++;
	    XtSetArg (args[n], XmNminimum, 0); n++;
	    XtSetArg (args[n], XmNvalue, altitude); n++;
	    XtSetValues (altdec_w, args, n);

	} else {
	    int ra, dec;

	    ra = radhr(sv_azra)*10.0 + 0.5;
	    if (ra >= 240) ra -= 240;
	    dec = raddeg(sv_altdec) + (sv_altdec >= 0 ? 0.5 : -0.5);

	    n = 0;
	    XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_LEFT); n++;
	    XtSetArg (args[n], XmNmaximum, 239); n++;
	    XtSetArg (args[n], XmNminimum, 0); n++;
	    XtSetArg (args[n], XmNdecimalPoints, 1); n++;
	    XtSetArg (args[n], XmNvalue, ra); n++;
	    XtSetValues (azra_w, args, n);

	    n = 0;
	    XtSetArg (args[n], XmNmaximum, 90); n++;
	    XtSetArg (args[n], XmNminimum, -90); n++;
	    XtSetArg (args[n], XmNvalue, dec); n++;
	    XtSetValues (altdec_w, args, n);
	}
}

/* expose event of sky view drawing area.
 * if same size just copy from pixmap, else recompute all (it's resized).
 * N.B. we set bit_gravity to ForgetGravity so we can just use Expose events.
 * N.B. we turn off backing store since we effectively do it in the pixmap.
 */
/* ARGSUSED */
static void
sv_da_exp_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	static unsigned int wid_last, hei_last;
	XmDrawingAreaCallbackStruct *c = (XmDrawingAreaCallbackStruct *)call;
	Display *dsp = XtDisplay(w);
	Window win = XtWindow(w);
	Window root;
	int x, y;
	unsigned int bw, d;
	unsigned wid, hei;


	switch (c->reason) {
	case XmCR_EXPOSE: {
	    static before;
	    XExposeEvent *e = &c->event->xexpose;
	    if (!before) {
		XSetWindowAttributes swa;
		unsigned long mask = CWBitGravity | CWBackingStore;
		swa.bit_gravity = ForgetGravity;
		swa.backing_store = NotUseful;
		XChangeWindowAttributes (e->display, e->window, mask, &swa);
		before = 1;
	    }
	    /* wait for the last in the series */
	    if (e->count != 0)
		return;
	    break;
	    }
	default:
	    printf ("Unexpected svda_w event. type=%d\n", c->reason);
	    exit(1);
	}

	XGetGeometry (dsp, win, &root, &x, &y, &wid, &hei, &bw, &d);

	if (!sv_pm || wid != wid_last || hei != hei_last) {
	    if (sv_pm)
		XFreePixmap (dsp, sv_pm);

	    if (!sv_gc)
		sv_mk_gcs(dsp, win);

	    sv_pm = XCreatePixmap (dsp, win, wid, hei, d);
	    wid_last = wid;
	    hei_last = hei;

	    sv_all(mm_get_now(), 1);
	} else {
	    /* same size; just copy from the pixmap */
	    sv_copy_sky();
	}
}

/* copy the pixmap to the drawing area
 */
static void
sv_copy_sky()
{
	Display *dsp = XtDisplay (svda_w);
	Window win = XtWindow (svda_w);
	unsigned wid, hei;	/* overall width and height */
	unsigned int r;		/* circle radius */
	unsigned int xb, yb;	/* x and y border */

	sv_getcircle (&wid, &hei, &r, &xb, &yb);

	XCopyArea (dsp, sv_pm, win, sv_gc, 0, 0, wid, hei, 0, 0);
}

/* called on receipt of a MotionNotify while Button1 is down or any
 * ButtonPress/ButonRelease while the cursor is over the svda_w window.
 */
/* ARGSUSED */
static void
sv_da_motion_cb (w, client, ev, continue_to_dispatch)
Widget w;
XtPointer client;
XEvent *ev;
Boolean *continue_to_dispatch;
{
	unsigned int wide, h, r, xb, yb;
	double altdec, azra;
	Window root, child;
	int rx, ry, wx, wy;
	unsigned mask;
	int evt = ev->type;
	int m1, b1p, b1r, b2p;
	int inside;

	/* what happened? */
	m1  = evt == MotionNotify  && ev->xmotion.state  == Button1Mask;
	b1p = evt == ButtonPress   && ev->xbutton.button == Button1;
	b1r = evt == ButtonRelease && ev->xbutton.button == Button1;
	b2p = evt == ButtonPress   && ev->xbutton.button == Button2;

	/* do we care? */
	if (!m1 && !b1p && !b1r && !b2p)
	    return;

	/* where are we? */
	XQueryPointer (XtDisplay(w), XtWindow(w),
			    &root, &child, &rx, &ry, &wx, &wy, &mask);
	sv_getcircle (&wide, &h, &r, &xb, &yb);
	inside = sv_unloc (r, wx-xb, wy-yb, &altdec, &azra);

	if (m1 || b1p || b1r) {
	    static dirty;

	    if (inside && (m1 || b1p)) {
		/* display the coordinates in the corners */
		double alt, az, ra, dec;
		Now *np = mm_get_now();

		sv_fullwhere (np, altdec, azra, &alt, &az, &ra, &dec);
		sv_draw_track_coords (np, wide, h, alt, az, ra, dec);
		dirty = 1;
	    } else {
		if (dirty) {
		    /* clean off the coordinates */
		    sv_copy_sky();
		    dirty = 0;
		}
	    }
	}
	if (b2p && inside) {
	    /* send the current coords to the SKYLOCFIFO. */
	    double alt, az, ra, dec, y;
	    Now *np = mm_get_now();

	    sv_fullwhere (np, altdec, azra, &alt, &az, &ra, &dec);
	    mjd_year (epoch == EOD ? mjd : epoch, &y);
	    skyloc_fifo (ra, dec, y);
	}
}

/* a dot has been picked:
 * find what it is and report stuff about it in a popup.
 */
/* ARGSUSED */
static void
sv_da_input_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
#define	PICKRANGE	100	/* sqr of dist allowed from pointer */
	XmDrawingAreaCallbackStruct *c = (XmDrawingAreaCallbackStruct *)call;
	DBScan dbs;
	XEvent *ev;
	TrailObj *top;
	unsigned int wide, h, r, xb, yb;
	int x, y;
	int mind = 1000000;	/* min distance to far */
	TSky *mintsp = NULL;	/* set to a TSky if find one close */
	Obj *minop = NULL;	/* set to an Obj if find one closer */
	Obj *op;
	int cx, cy;
	int d;
	int i;

	if (c->reason != XmCR_INPUT)
	    return;
	ev = c->event;
	if (ev->xany.type != ButtonPress || ev->xbutton.button != 3)
	    return;

	sv_getcircle (&wide, &h, &r, &xb, &yb);

	/* search everything based on cursor loc within the circle */
	cx = ev->xbutton.x - xb;
	cy = ev->xbutton.y - yb;

	/* don't bother if outside the circle */
	if ((cx-r)*(cx-r) + (cy-r)*(cy-r) >= r*r)
	    return;

	/* ok, get to work */
	watch_cursor(1);

	/* search the trailed stuff first because
	 * the first entry in a trailing list will be both in the db and in
	 * the trails list -- we want to find the latter when we can.
	 */
	for (top = trailobj; top; top = top->ntop) {
	    if (!top->on)
		continue;
	    for (i = 0; i < top->nsky; i++) {
		if (!sv_trailobjloc (&top->sky[i], r, &x, &y))
		    continue;
		d = (cx - x)*(cx - x) + (cy - y)*(cy - y);
		if (d < mind) {
		    mintsp = &top->sky[i];
		    mind = d;
		}
	    }
	}

	/* search the database too -- might be something closer */
	for (db_scaninit(&dbs, SCAN_ALL); op = db_scan (&dbs); ) {
	    if (!(op->o_flags & OBJF_ONSCREEN))
		continue;
	    if (!sv_dbobjloc (op, r, &x, &y))
		continue;	/* not in the circle now */
	    d = (cx - x)*(cx - x) + (cy - y)*(cy - y);
	    if (d < mind) {
		minop = op;
		mind = d;
	    }
	}

	if (mind <= PICKRANGE)
	    sv_popup (ev, minop, minop ? NULL : mintsp);
	else
	    sv_popup (ev, NULL, NULL);

	watch_cursor(0);
}

/* callback when any of the scales change value.
 * client is a SCALES enum to tell us which.
 */
/* ARGSUSED */
static void
sv_changed_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	int which = (int) client;

	/* read the widget to set the real value */
	sv_read_scale (which);

	/* update the scene */
	sv_all(mm_get_now(), 1);
}

/* callback when the "just dots" toggle button changes state.
 */
/* ARGSUSED */
static void
sv_justdots_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	justdots = XmToggleButtonGetState (w);
	sv_all(mm_get_now(), 1);
}

/* callback when the "ecliptic" toggle button changes state.
 */
/* ARGSUSED */
static void
sv_ecliptic_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	want_ecliptic = XmToggleButtonGetState (w);
	sv_all(mm_get_now(), 1);
}

/* callback when the "all labels" toggle button changes state.
 */
/* ARGSUSED */
static void
sv_all_labels_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	all_labels = XmToggleButtonGetState (w);
	sv_all(mm_get_now(), 1);
}

/* callback when the "Find" cascade button is activated.
 * check the used defined objects and adjust the cascade buttons accordingly.
 */
/* ARGSUSED */
static void
sv_finding_cb (wid, client, call)
Widget wid;
XtPointer client;
XtPointer call;
{
	int i;

	for (i = 0; i < 2; i++) {
	    Obj *op = db_basic (i == 0 ? OBJX : OBJY);
	    Widget w = find_w[i];
	    if (op->type == UNDEFOBJ)
		XtUnmanageChild (w);
	    else {
		set_xmstring (w, XmNlabelString, op->o_name);
		XtManageChild (w);
	    }
	}
}

/* callback when one of the "locate" cascade buttons changes state.
 * object dbidx is in client.
 */
/* ARGSUSED */
static void
sv_find_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	Obj *op = db_basic((int)client);

	(void) sv_mark (op, 0, 1, 1);
}

/* callback as either magnitude scale is dragged.
 * enforce at least a 1 mag window.
*/
/* ARGSUSED */
static void
sv_magdrag_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	int f, b;

	XmScaleGetValue (fmag_w, &f);
	XmScaleGetValue (bmag_w, &b);
	if (f - b < 1)
	    XmScaleSetValue (w, w == fmag_w ? b+1 : f-1);
}

/* given one of the SCALES enums, read it and compute the corresponding 
 * real value.
 */
static void
sv_read_scale (which)
int which;
{
	int i;

	switch (which) {
	case FMAG_S:
	    XmScaleGetValue (fmag_w, &fmag);
	    break;
	case BMAG_S:
	    XmScaleGetValue (bmag_w, &bmag);
	    break;
	case FOV_S:
	    XmScaleGetValue (fov_w, &i);
	    sv_fov = degrad(i);
	    break;
	case ALTDEC_S:
	    XmScaleGetValue (altdec_w, &i);
	    sv_altdec = degrad(i);
	    break;
	case AZRA_S:
	    XmScaleGetValue (azra_w, &i);
	    sv_azra = aa_mode ? degrad(i) : hrrad((double)i/10.0);
	    break;
	default:
	    printf ("sv_read_scale: bad which: %d\n", which);
	    exit (1);
	}
}

/* draw all the stuff in the coord tracking areas in the corners of the drawing
 * area.
 * N.B. draw directly to the window, *NOT* the pixmap, so it can be erased by
 *   just copying the pixmap to the window again.
 */
static void
sv_draw_track_coords (np, w, h, alt, az, ra, dec)
Now *np;
unsigned int w, h;
double alt, az, ra, dec;
{
#define	CON_GAP		4	/* pixels between Alt and constellation */
	static last_con_w;
	Window win = XtWindow(svda_w);
	Display *dsp = XtDisplay(svda_w);
	char alt_buf[64], az_buf[64];
	char ra_buf[64], dec_buf[64];
	char *constellation;
	XCharStruct all;
	int len;
	int dir, asc, des;
	Pixel fg, bg;
	unsigned gcm;
	XGCValues gcv;

	(void) strcpy (alt_buf, "Alt: ");
	fs_pangle (alt_buf+5, alt);
	(void) strcpy (az_buf, "Az: ");
	fs_pangle (az_buf+4, az);
	(void) strcpy (ra_buf, "RA: ");
	fs_ra (ra_buf+4, ra);
	(void) strcpy (dec_buf, "Dec: ");
	fs_pangle (dec_buf+5, dec);

	confnd (ra, dec, epoch == EOD ? mjd : epoch, &constellation);
	constellation += 5;	/* skip the abbreviation */

	/* draw Alt in the upper left */
	len = strlen (alt_buf);
	XTextExtents (sv_f, alt_buf, len, &dir, &asc, &des, &all);
	XDrawImageString (dsp, win, sv_strgc, 0, asc, alt_buf, len); 

	/* draw the constellation name in upper left below Alt.
	 * clear out previous in case it was longer.
	 */
	len = strlen (constellation);
	XTextExtents (sv_f, constellation, len, &dir, &asc, &des, &all);
	gcm = GCForeground | GCBackground;
	XGetGCValues (dsp, sv_strgc, gcm, &gcv);
	fg = gcv.foreground;
	bg = gcv.background;
	XSetForeground (dsp, sv_strgc, bg);
	XFillRectangle (dsp, win, sv_strgc, 0, asc+des+CON_GAP, last_con_w,
								    asc+des);
	XSetForeground (dsp, sv_strgc, fg);
	XDrawImageString (dsp, win, sv_strgc, 0, 2*asc+des+CON_GAP,
							constellation, len); 
	last_con_w = all.rbearing;

	/* draw Az in upper right */
	len = strlen (az_buf);
	XTextExtents (sv_f, az_buf, len, &dir, &asc, &des, &all);
	XDrawImageString (dsp, win, sv_strgc, w-all.rbearing, asc, az_buf, len);

	/* draw RA in lower left */
	len = strlen (ra_buf);
	XTextExtents (sv_f, ra_buf, len, &dir, &asc, &des, &all);
	XDrawImageString (dsp, win, sv_strgc, 0, h-des, ra_buf, len); 

	/* draw Dec in lower right */
	len = strlen (dec_buf);
	XTextExtents (sv_f, dec_buf, len, &dir, &asc, &des, &all);
	XDrawImageString (dsp, win, sv_strgc, w-all.rbearing, h-des, dec_buf,
									len);
}

/* mark the given object, possibly changing fov depending on the center flags.
 * return 0 if did all that was asked else generate a message using xe_msg(*,1)
 *   and return -1.
 * N.B. we do *not* update the s_ fields of op.
 */
static int
sv_mark (op, in_center, out_center, mark)
Obj *op;
int in_center;	/* whether to center an object already within the fov */
int out_center;	/* whether to center an object not already within the fov */
int mark;	/* whether to mark the object if it ends up visible */
{
	char msg[128];
	unsigned int wide, h, r, xb, yb;
	double altdec, azra;
	int infov;
	int x, y;

	/* never show something below the horizon if we're in alt/az mode */
	if (aa_mode && op->s_alt < 0.0) {
	    (void) sprintf (msg, "`%.100s' is below the horizon now.\n",
								    op->o_name);
	    xe_msg (msg, 1);
	    return (-1);
	}

	/* see if it's within the current fov.
	 * complain if it's not and we have no permission to reaim.
	 */
	sv_getcircle (&wide, &h, &r, &xb, &yb);
	altdec = aa_mode ? op->s_alt : op->s_dec;
	azra   = aa_mode ? op->s_az  : op->s_ra;
	infov = sv_loc (r, altdec, azra, &x, &y);
	if (!infov && !out_center) {
	    (void) sprintf (msg, "`%.100s' is outside the field of view.\n",
								    op->o_name);
	    xe_msg (msg, 1);
	    return (-1);
	}

	/* center the thing if we should */
	if (infov && in_center || !infov && out_center) {
	    sv_altdec = altdec;
	    sv_azra = azra;
	    sv_set_scales();
	    sv_all(mm_get_now(), 1);
	    if (sv_loc (r, altdec, azra, &x, &y) < 0) {
		printf ("sv_mark: object disappeared from view?!\n");
		exit (1);
	    }
	}

	/* and mark if asked to */
	if (mark) {
	    Display *dsp = XtDisplay (svda_w);
	    Window win = XtWindow (svda_w);

	    x += xb;
	    y += yb;

	    XSetForeground (dsp, sv_gc, fg_p);
	    XDrawLine (dsp, win, sv_gc, x-MARKR, y-MARKR, x+MARKR, y+MARKR);
	    XDrawLine (dsp, win, sv_gc, x+MARKR, y-MARKR, x-MARKR, y+MARKR);
	}

	return (0);
}

/* return the current circle info about svda_w */
static void
sv_getcircle (wp, hp, rp, xbp, ybp)
unsigned int *wp, *hp;		/* overall width and height */
unsigned int *rp;		/* circle radius */
unsigned int *xbp, *ybp;	/* x and y border */
{
	Display *dsp = XtDisplay(svda_w);
	Window win = XtWindow(svda_w);
	Window root;
	int x, y;
	unsigned int w, h;
	unsigned int bw, d;

	XGetGeometry (dsp, win, &root, &x, &y, wp, hp, &bw, &d);
	w = *wp/2;
	h = *hp/2;
	*rp = w > h ? h : w;	/* radius */
	*xbp = w - *rp;		/* x border */
	*ybp = h - *rp;		/* y border */

}

/* we are called when the right mouse button is pressed because the user wants
 * to identify and possibly control an object in some way. we fill in the popup
 * with goodies and manage it.
 * three cases:
 *   1) we are called with op set and tsp == 0. this means we were called with
 *      a real db object so just use *op;
 *   2) we are called with op == 0 and tsp set. this means we were called with
 *      a trailed object and so use *tsp->op;
 *   3) we are called with neither op nor tsp set. this means we were called
 *      far from any object; compute the location and display it. also set
 *      pu.op to a static object with at least ra/dec/alt/az value for possible
 *      pointing.
 * position the popup as indicated by ev and display it.
 * it goes down by itself.
 */
static void
sv_popup (ev, op, tsp)
XEvent *ev;
Obj *op;
TSky *tsp;
{
	char buf[32], buf2[64];
	int noobj;
	int trail;
	int label;
	int track;
	double jd;

	if (tsp) {
	    /* we were given a trailed Obj.
	     * pu.op is the top object, pu.tsp is tsp.
	     */
	    TrailObj *top;
	    op = &tsp->o;
	    top = tobj_find (op);
	    jd = tsp->ts_mjd;
	    trail = top->on;
	    label = !!(tsp->flags & OBJF_LABEL);
	    track = top->op == track_op;
	    pu.op = top->op;
	    pu.tsp = tsp;
	    noobj = 0;
	} else if (op) {
	    /* no tsp trail -- just use the db
	     * pu.op is op, pu.tsp is NULL.
	     */
	    Now *np = mm_get_now();
	    jd = mjd;
	    trail = 0;
	    label = !!(op->o_flags & OBJF_LABEL);
	    noobj = 0;
	    track = op == track_op;
	    pu.op = op;
	    pu.tsp = NULL;
	} else {
	    /* nothing -- compute from ev and use svobj
	     * pu.op and op are &svobj, pu.tsp is NULL.
	     */
	    Now *np = mm_get_now();
	    double altdec, azra, alt, az, ra, dec;
	    unsigned int wide, h, r, xb, yb;
	    int wx, wy;
	    wx = ev->xbutton.x;
	    wy = ev->xbutton.y;
	    sv_getcircle (&wide, &h, &r, &xb, &yb);
	    if (!sv_unloc (r, wx-xb, wy-yb, &altdec, &azra))
		return;	/* outside the circle */
	    sv_fullwhere (np, altdec, azra, &alt, &az, &ra, &dec);
	    jd = mjd;
	    svobj.s_ra = ra;
	    svobj.s_dec = dec;
	    svobj.s_az = az;
	    svobj.s_alt = alt;
	    svobj.type = FIXED;
	    pu.op = op = &svobj;
	    pu.tsp = NULL;
	    noobj = 1;
	}

	if (noobj) {
	    XtUnmanageChild (pu.name_w);
	} else {
	    XtManageChild (pu.name_w);
	    (void) sprintf (buf2, "   Name: %.20s", op->o_name);
	    set_xmstring (pu.name_w, XmNlabelString, buf2);
	}

	if (noobj) {
	    XtUnmanageChild (pu.desc_w);
	} else {
	    XtManageChild (pu.desc_w);
	    (void) sprintf (buf2, "   Type: %.20s", obj_description(op));
	    set_xmstring (pu.desc_w, XmNlabelString, buf2);
	}

	if (!noobj && op->type == FIXED && op->f_spect[0]) {
	    (void) sprintf (buf2, "  Spect: %.*s", sizeof(op->f_spect),
								op->f_spect);
	    set_xmstring (pu.spect_w, XmNlabelString, buf2);
	    XtManageChild (pu.spect_w);
	} else
	    XtUnmanageChild (pu.spect_w);

	if (noobj) {
	    XtUnmanageChild (pu.ud_w);
	} else {
	    XtManageChild (pu.ud_w);
	    fs_date (buf, jd);
	    (void) sprintf (buf2, "UT Date: %s", buf);
	    set_xmstring (pu.ud_w, XmNlabelString, buf2);
	}

	if (noobj) {
	    XtUnmanageChild (pu.ut_w);
	} else {
	    XtManageChild (pu.ut_w);
	    fs_time (buf, mjd_hr(jd));
	    (void) sprintf (buf2, "UT Time: %s", buf);
	    set_xmstring (pu.ut_w, XmNlabelString, buf2);
	}

	fs_ra (buf, op->s_ra);
	(void) sprintf (buf2, "     RA: %s", buf);
	set_xmstring (pu.ra_w, XmNlabelString, buf2);

	fs_pangle (buf, op->s_dec);
	(void) sprintf (buf2, "    Dec: %s", buf);
	set_xmstring (pu.dec_w, XmNlabelString, buf2);

	fs_pangle (buf, op->s_alt);
	(void) sprintf (buf2, "    Alt: %s", buf);
	set_xmstring (pu.alt_w, XmNlabelString, buf2);

	fs_pangle (buf, op->s_az);
	(void) sprintf (buf2, "     Az: %s", buf);
	set_xmstring (pu.az_w, XmNlabelString, buf2);

	if (noobj) {
	    XtUnmanageChild (pu.mag_w);
	} else {
	    XtManageChild (pu.mag_w);
	    (void) sprintf (buf2, "    Mag: %.3g", op->s_mag/MAGSCALE);
	    set_xmstring (pu.mag_w, XmNlabelString, buf2);
	}

	if (noobj) {
	    XtUnmanageChild (pu.trail_w);
	    XtUnmanageChild (pu.label_w);
	    XtUnmanageChild (pu.make_w);
	    XtUnmanageChild (pu.track_w);
	} else {
	    XtManageChild (pu.trail_w);
	    XtManageChild (pu.label_w);
	    XtManageChild (pu.make_w);
	    XtManageChild (pu.track_w);
	    XmToggleButtonSetState (pu.trail_w, trail, False);
	    XmToggleButtonSetState (pu.label_w, label, False);
	    XmToggleButtonSetState (pu.track_w, track, False);
	}

	XmMenuPosition (pu.pu_w, (XButtonPressedEvent *)ev);
	XtManageChild (pu.pu_w);
}

/* create the id popup */
static void
sv_create_popup()
{
	static Widget *puw[] = {
	    &pu.name_w,
	    &pu.desc_w,
	    &pu.spect_w,
	    &pu.ud_w,
	    &pu.ut_w,
	    &pu.ra_w,
	    &pu.dec_w,
	    &pu.alt_w,
	    &pu.az_w,
	    &pu.mag_w,
	};
	Arg args[20];
	XmString str;
	Widget w;
	int i, n;

	/* create the outer form */
	n = 0;
	XtSetArg (args[n], XmNisAligned, True); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_CENTER); n++;
	pu.pu_w = XmCreatePopupMenu (toplevel_w, "SVPopup", args, n);

	/* create the label widgets */

	for (i = 0; i < XtNumber(puw); i++) {
	    n = 0;
	    *puw[i] = XmCreateLabel (pu.pu_w, "SVPopValueL", args, n);
	    XtManageChild (*puw[i]);
	}

	/* add a nice separator */
	n = 0;
	w = XmCreateSeparator (pu.pu_w, "SVSep", args, n);
	XtManageChild(w);

	/* make the command buttons */

	str = XmStringCreateLtoR ("Point", XmSTRING_DEFAULT_CHARSET);
	n = 0;
	XtSetArg (args[n], XmNlabelString, str); n++;
	w = XmCreatePushButton (pu.pu_w, "SVPopPoint", args, n);
	XtAddCallback (w, XmNactivateCallback, sv_pu_activate_cb,
							    (XtPointer)AIM);
	XtManageChild (w);
	XmStringFree (str);

	str = XmStringCreateLtoR ("Make ObjX/Y", XmSTRING_DEFAULT_CHARSET);
	n = 0;
	XtSetArg (args[n], XmNlabelString, str); n++;
	pu.make_w = XmCreatePushButton (pu.pu_w, "SVPopMakeObj", args, n);
	XtAddCallback (pu.make_w, XmNactivateCallback, sv_pu_activate_cb,
							(XtPointer)MK_OBJX);
	XtManageChild (pu.make_w);
	XmStringFree(str);

	str = XmStringCreateLtoR ("Track", XmSTRING_DEFAULT_CHARSET);
	n = 0;
	XtSetArg (args[n], XmNlabelString, str); n++;
	XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
	pu.track_w = XmCreateToggleButton (pu.pu_w, "SVPopTrack", args, n);
	XtAddCallback (pu.track_w, XmNvalueChangedCallback, sv_pu_track_cb, 0);
	XtManageChild (pu.track_w);
	XmStringFree (str);

	str = XmStringCreateLtoR ("Leave Trail", XmSTRING_DEFAULT_CHARSET);
	n = 0;
	XtSetArg (args[n], XmNlabelString, str); n++;
	XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
	pu.trail_w = XmCreateToggleButton (pu.pu_w, "SVPopTrail", args, n);
	XtAddCallback(pu.trail_w, XmNvalueChangedCallback, sv_pu_trail_cb,NULL);
	XtManageChild (pu.trail_w);
	XmStringFree(str);

	str = XmStringCreateLtoR ("Persistent Label", XmSTRING_DEFAULT_CHARSET);
	n = 0;
	XtSetArg (args[n], XmNlabelString, str); n++;
	XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
	pu.label_w = XmCreateToggleButton (pu.pu_w, "SVPopLabel", args, n);
	XtAddCallback(pu.label_w, XmNvalueChangedCallback, sv_pu_label_cb,NULL);
	XtManageChild (pu.label_w);
	XmStringFree(str);
}

/* called when any of the popup's pushbuttons are activated.
 * client is a code to indicate which.
 */
/* ARGSUSED */
static void
sv_pu_activate_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	int code = (int)client;

	switch (code) {
	case AIM:
	    (void) sv_mark (pu.tsp ? &pu.tsp->o : pu.op, 1, 1, 0);
	    break;
	case MK_OBJX:
	    if (pu.op->type == PLANET)
		xe_msg ("User objects can not be of type PLANET.", 0);
	    else {
		sv_ournewobj = 1;
		obj_set (pu.op);
		sv_ournewobj = 0;
		break;
	    }
	    break;
	default:
	    printf ("sv_pu_activate_db(): code=%d\n", code);
	    exit (1);
	    break;
	}
}

/* called when the Trail popup toggle button changes.
 * clean up the trailobj stuff.
 */
/* ARGSUSED */
static void
sv_pu_trail_cb (wid, client, call)
Widget wid;
XtPointer client;
XtPointer call;
{
	if (XmToggleButtonGetState(wid)) {
	    /* trail is being turned on.
	     * if it already has a trail just turn it back on and draw it, else
	     *   add a new db obj to trailobj list.
	     */
	    TrailObj *top = tobj_find (pu.op);
	    if (top) {
		unsigned int w, h, r, xb, yb;
		sv_getcircle (&w, &h, &r, &xb, &yb);
		top->on = 1;
		tobj_display_all(r, xb, yb);
		sv_copy_sky();
	    } else {
		Now *np = mm_get_now();
		top = tobj_addobj (pu.op);
		(void) tobj_addsky (top, mjd, pu.op);
		/* no need to redraw since a trail of lenth 1 won't *look*
		 * any different.
		 */
	    }
	} else {
	    /* trailing is being turned off. mark it as being off.
	     * it will get removed at the next update if it's still off.
	     * redraw sky so it disappears.
	     */
	    TrailObj *top = tobj_find (pu.op);
	    top->on = 0;
	    sv_all (mm_get_now(), 1);
	}
}

/* called when the Label popup toggle button changes.
 * we get all context from the pu structure.
 */
/* ARGSUSED */
static void
sv_pu_label_cb (wid, client, call)
Widget wid;
XtPointer client;
XtPointer call;
{
	unsigned int w, h, r, xb, yb;
	int x, y;
	int label;
	char *name;

	/* if this is a trailed item then its TSky will be in pu.tsp
	 * otherwise it is a plain db object.
	 * either way, toggle the corresponding OBJF_LABEL bit too.
	 */
	if (pu.tsp) {
	    label = (pu.tsp->flags ^= OBJF_LABEL) & OBJF_LABEL;
	    if (label) {
		sv_getcircle (&w, &h, &r, &xb, &yb);
		(void) sv_trailobjloc (pu.tsp, r, &x, &y);
		name = pu.tsp->o.o_name;
	    }
	} else {
	    label = (pu.op->o_flags ^= OBJF_LABEL) & OBJF_LABEL;
	    if (label) {
		sv_getcircle (&w, &h, &r, &xb, &yb);
		(void) sv_dbobjloc (pu.op, r, &x, &y);
		name = pu.op->o_name;
	    }
	}

	if (label) {
	    /* label is being turned on so draw it */
	    Display *dsp = XtDisplay(svda_w);
	    Window win = sv_pm;
	    GC gc;
	    x += xb;
	    y += yb;
	    obj_pickgc (pu.op, svda_w, &gc);
	    draw_label (dsp, win, gc, name, x, y);
	    sv_copy_sky();
	} else {
	    /* label is being turned off so redraw */
	    sv_all (mm_get_now(), 1);
	}
}

/* called when the Track popup toggle button changes.
 * we get all context from the pu structure.
 */
/* ARGSUSED */
static void
sv_pu_track_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (XmToggleButtonGetState (w)) {
	    track_op = pu.op;
	    XtSetSensitive (tracktb_w, True);
	    XmToggleButtonSetState (tracktb_w, True, True);
	    (void) sv_mark (track_op, 0, 0, 1);
	} else
	    XmToggleButtonSetState (tracktb_w, False, True);
}

/* remove trails no longer wanted. */
static void
tobj_rmoff()
{
	TrailObj **topp;	/* address to be changed if we decide to 
				 * remove *topp
				 */
	TrailObj *top;		/* handy *topp */

	for (topp = &trailobj; (top = *topp) != NULL; ) {
	    if (top->on) {
		topp = &top->ntop;
	    } else {
		*topp = top->ntop;
		XtFree ((char *)top);
	    }
	}
}

/* remove the trailobj list that contains the given pointer.
 * we have to search each trail list to find the one with this pointer.
 * it might be the one on TrailObj itself or one of the older ones on
 *    the TSky list.
 * it's no big deal if op isn't really on any trail list.
 */
static void
tobj_rmobj (op)
Obj *op;
{
	TrailObj **topp;	/* address to be changed if we decide to 
				 * remove *topp
				 */
	TrailObj *top;		/* handy *topp */

	for (topp = &trailobj; (top = *topp) != NULL; ) {
	    int i;
	    if (top->op == op)
		goto out;
	    for (i = 0; i < top->nsky; i++)
		if (&top->sky[i].o == op)
		    goto out;
	    topp = &top->ntop;
	}

    out:

	if (!top)
	    return;	/* oh well */

	*topp = top->ntop;
	XtFree ((char *)top);
}

/* add a new empty entry to the trailobj list for db object op.
 * return a pointer to the new TrailObj.
 */
static TrailObj *
tobj_addobj (op)
Obj *op;
{
	TrailObj *top;

	/* don't forget there is inherently room for one TSky in a TrailObj */
	top = (TrailObj *)
	    XtMalloc (sizeof(TrailObj) + (TRAILCHUNKS-1)*sizeof(TSky));
	top->nskymem = TRAILCHUNKS; /* there is room for TRAILCHUNKS TSkys .. */
	top->nsky = 0;	  /* though none are in use in our new one */
	top->op = op;
	top->on = 1;

	/* link directly off trailobj -- order is unimportant */
	top->ntop = trailobj;
	trailobj = top;

	return (top);
}

/* increase the number of Sky's top can hold.
 * since we may have to move top to do it, return the new pointer.
 */
static TrailObj *
tobj_growsky (top)
TrailObj *top;
{
	TrailObj *ltop;
	TrailObj *newtop;
	int newn;

	/* set ltop to the TrailObj in the list just before top */
	if (trailobj == top)
	    ltop = NULL;
	else {
	    for (ltop=trailobj; ltop; ltop=ltop->ntop)
		if (ltop->ntop == top)
		    break;
	    if (!ltop) {
		printf ("tobj_growsky(): top not found!\n");
		exit (1);
	    }
	}

	/* don't forget there is already one TSky within a TrailObj. */
	newn = top->nskymem + TRAILCHUNKS;
	newtop = (TrailObj *) XtRealloc ((char *)top,
				sizeof(TrailObj) + (newn-1)*sizeof(TSky));
	if (ltop)
	    ltop->ntop = newtop;
	else
	    trailobj = newtop;
	newtop->nskymem = newn;
	return (newtop);
}

#if 0
/* empty the trailobj list and reclaim all space.
 */
static void
tobj_reset()
{
	TrailObj *top, *ntop;

	for (top = trailobj; top; top = ntop) {
	    ntop = top->ntop;
	    XtFree ((char *)top);
	}

	trailobj = NULL;
	track_op = NULL;
}
#endif

/* remove each trail that refers to a db object no longer in the db.
 * also reset the tracked object if it's gone now too.
 * this is done when the db is reduced.
 */
static void
tobj_newdb()
{
	TrailObj *top;
	DBScan dbs;
	Obj *op;

	for (top = trailobj; top; top = top->ntop) {
	    for (db_scaninit (&dbs, SCAN_ALL); op = db_scan (&dbs); )
		if (top->op == op)
		    break;
	    if (!op)
		tobj_rmobj (top->op);
	}

	if (track_op) {
	    for (db_scaninit (&dbs, SCAN_ALL); op = db_scan (&dbs); )
		if (track_op == op)
		    break;
	    if (!op)
		track_op = NULL;
	}
}

/* find the TrailObj that contains op.
 * return NULL if don't find it.
 */
static TrailObj *
tobj_find (op)
Obj *op;
{
	TrailObj *top;

	for (top = trailobj; top; top = top->ntop) {
	    int i;
	    if (top->op == op)
		return (top);
	    for (i = 0; i < top->nsky; i++)
		if (&top->sky[i].o == op)
		    return (top);
	}

	return (NULL);
}

/* add an Obj to the given trailobj, top.
 * mark it as being valid as of jd (which is really an mjd).
 * these are maintained in increasing order of time, ie, the first is the
 * earliest. we take care not to add one for an identical time already in
 * the list. since we might have to move top to grow it, we return a (possibly
 * different) pointer.
 */
static TrailObj *
tobj_addsky (top, jd, op)
TrailObj *top;
double jd;
Obj *op;
{
	int i;

	/* make sure there is room for one more */
	if (top->nsky == top->nskymem)
	    top = tobj_growsky (top);

	/* add op to top->sky in ascending time order.
	 * exit loop with i as the index of the cell to use.
	 */
	for (i = top->nsky; i > 0; --i)
	    if (top->sky[i-1].ts_mjd < jd)
		break;
	    else if (top->sky[i-1].ts_mjd == jd)
		return (top);	/* don't add a dup */
	    else
		top->sky[i] = top->sky[i-1];

	top->sky[i].flags = 0;
	top->sky[i].ts_mjd = jd;
	top->sky[i].o = *op;
	top->nsky++;

	return (top);
}

/* display everything in the trailobj list that is marked on onto sv_pm
 * clipped to a circle of radius r, offset by xb and yb borders.
 */
static void
tobj_display_all(r, xb, yb)
unsigned r, xb, yb;
{
	Display *dsp = XtDisplay(svda_w);
	Window win = sv_pm;
	TrailObj *top;

	for (top = trailobj; top; top = top->ntop) {
	    int x1, y1, x2, y2;
	    int before;
	    Obj *op;
	    GC gc;
	    int i;

	    if (!top->on)
		continue;

	    obj_pickgc(top->op, svda_w, &gc);
	    before = 0;

	    for (i = 0; i < top->nsky; i++) {
		op = &top->sky[i].o;
		if (sv_trailobjloc (&top->sky[i], r, &x2, &y2)) {
		    sv_draw_obj (dsp, win, gc, op, x2+xb, y2+yb,
				magdiam(op->s_mag/MAGSCALE), justdots);
		    if (all_labels || (top->sky[i].flags & OBJF_LABEL))
			draw_label (dsp, win, gc, op->o_name, x2+xb, y2+yb);
		}
		if (before++) {
		    int sx1, sy1, sx2, sy2;
		    if (lc(0,0,r*2,x1,y1,x2,y2,&sx1,&sy1,&sx2,&sy2) == 0)
			XDrawLine (dsp, win, gc, sx1+xb, sy1+yb, sx2+xb,sy2+yb);
		}
		x1 = x2;
		y1 = y2;
	    }

	}

	sv_draw_obj (dsp, win, (GC)0, NULL, 0, 0, 0, 0);	/* flush */
}

/* determine if the given object is visible and within a circle of radius r
 * pixels. if so, return 1 and compute the location in *xp / *yp, else return 0.
 * N.B. only call this for bona fide db objects -- *not* for objects in the
 *   TrailObj lists -- it will destroy their history.
 */
static
sv_dbobjloc (op, r, xp, yp)
Obj *op;
int r;
int *xp, *yp;
{
	double altdec, azra;

	if (!sv_precheck(op))
	    return (0);

	/* remaining things need accurate s_* fields */
	db_update(op);

	if (aa_mode && op->s_alt < 0.0)
	    return(0);	/* it's below horizon and we're in alt-az mode */

	if (op->s_mag/MAGSCALE > fmag || op->s_mag/MAGSCALE < bmag)
	    return(0);	/* it's not within mag range after all */

	altdec = aa_mode ? op->s_alt : op->s_dec;
	azra   = aa_mode ? op->s_az  : op->s_ra;
	if (!sv_loc (r, altdec, azra, xp, yp))
	    return(0); 	/* it's outside the fov after all */

	return (1);	/* yup, really within r */
}

/* given a TSky find its location in a circle of radius r, all in pixels.
 * return 1 if the resulting x/y is in fact within the circle, else 0 if it is
 *   outside or otherwise should not be shown now (but return the x/y anyway).
 * N.B. we take care to not change the tsp->o in any way.
 */
static
sv_trailobjloc (tsp, r, xp, yp)
TSky *tsp;
int r;
int *xp, *yp;
{
	Obj *op = &tsp->o;
	double altdec, azra;
	int infov;

	altdec = aa_mode ? op->s_alt : op->s_dec;
	azra   = aa_mode ? op->s_az  : op->s_ra;
	infov = sv_loc (r, altdec, azra, xp, yp);
	return (infov && (!aa_mode || op->s_alt >= 0.0) && sv_precheck(op)?1:0);
}

/* do as much as possible to pre-check whether op is on screen now
 * WITHOUT computing it's actual coordinates. put another way, we are not to
 * use and s_* fields in these tests.
 * return 0 if we know it's definitely not on screen, or 1 if it might be.
 */
static
sv_precheck (op)
Obj *op;
{
	if (op->type == UNDEFOBJ)
	    return(0);

	if (!svf_filter_ok(op) || !sv_bright_ok(op) || !sv_infov(op))
	    return(0);	/* wrong type, wrong brightness or outside for sure */
	
	return (1);
}

/* return 1 if db object is ever possibly within the fmag/bmag range; else 0.
 */
static
sv_bright_ok(op)
Obj *op;
{
	switch (op->type) {
	case PLANET:
	    /* always go for the planets for now but ..
	     * TODO: work up a table of extreme planet magnitudes.
	     */
	    return (1);
	    /* break; */
	case HYPERBOLIC: return (1);	/* and interlopers */
	case PARABOLIC: return (1);
	case ELLIPTICAL: {
	    double mag;		/* magnitude */
	    double per, aph;	/* perihelion and aphelion distance */

	    per = op->e_a*(1.0 - op->e_e);
	    aph = op->e_a*(1.0 + op->e_e);
	    if (per <= 1.1 && aph >= 0.9)
		return (1); /* might be blazing in the back yard some day */
	    if (op->e_mag.whichm == MAG_HG)
		 mag = op->e_mag.m1 + 5*log10(per*fabs(per-1.0));
	    else
		 gk_mag(op->e_mag.m1, op->e_mag.m2,
						per, fabs(per-1.0), &mag);
	    return (mag <= fmag && mag >= bmag);
	    /* break; */
	    }
	case FIXED:
	    return (op->f_mag/MAGSCALE <= fmag && op->f_mag/MAGSCALE >= bmag);
	    /* break; */
	case EARTHSAT:
	    /* TODO: work on satellite magnitudes someday */
	    return (1);
	default: 
	    printf ("sv_bright_ok(): bad type: %d\n", op->type);
	    exit (1);
	    return (0);	/* for lint */
	}
}

/* return 1 if the object can potentially be within the current sv_fov, else 0.
 * N.B. this is meant to be cheap - we only do fixed objects and we don't
 *      precess. most importantly, we don't use any s_* fields.
 */
static
sv_infov (op)
Obj *op;
{
#define	DELEP	100		/* maximum epoch difference we dare go without
				 * precessing, years
				 */
#define	MARGIN	degrad(3.0)	/* border around fov still considered "in"
				 * in spite of having not precessed.
				 */
	double ra0, dec0;	/* ra/dec of our center of view */
	double a, sa, ca;	/* angle from viewpoint to pole */
	double b, sb, cb;	/* angle from object to pole */
	double c, cc;		/* diff of polar angles of obj and viewpoint */
	double d;		/* angular separation of object and viewpoint */
	Now *np = mm_get_now();

	if (op->type != FIXED)
	    return (1);
	if (fabs (mjd - op->f_epoch) > DELEP)
	    return (1);

	if (aa_mode) {
	    /* TODO: cache this -- it's the same for each obj! */
	    double ha, lst;
	    aa_hadec (lat, sv_altdec, sv_azra, &ha, &dec0);
	    now_lst (np, &lst);
	    ra0 = hrrad(lst) - ha;
	    range (&ra0, 2*PI);
	} else {
	    ra0 = sv_azra;
	    dec0 = sv_altdec;
	}

	a = PI/2 - dec0;
	sa = sin(a);
	ca = cos(a);
	b = PI/2 - op->f_dec;
	sb = sin(b);
	cb = cos(b);
	c = ra0 - op->f_RA;
	cc = cos(c);
	d = acos(ca*cb + sa*sb*cc);
	return (d < sv_fov/2 + MARGIN);
}

/* compute x/y loc of a point at azra/altdec on a circle of radius rad pixels
 *   as viewed from sv_azra/sv_altdec with sv_fov.
 * return 1 if fits on screen, else 0 (but still return x/y).
 */
static
sv_loc (rad, altdec, azra, xp, yp)
int rad;	/* radius of target display, pixels */
double altdec;	/* angle up from spherical equator, such as alt or dec; rad */
double azra;	/* angle around spherical pole, such as az or ra; rad */
int *xp, *yp;	/* return X coords within circle */
{
#define	LOCEPS	(1e-5)	/* an angle too small to see on screen, rads */
	static double last_sv_altdec = 123.0, last_sa, last_ca;
	double a,sa,ca;	/* angle from viewpoint to pole */
	double b,sb,cb;	/* angle from object to pole */
	double c,sc,cc;	/* difference in polar angles of obj and viewpoint */
	double d,sd,cd;	/* angular separation of object and viewpoint */
	double r;	/* proportion of d to desired field of view */
	double se, ce;	/* angle between (viewpoint,pole) and (viewpoint,obj) */

	a = PI/2 - sv_altdec;
	if (sv_altdec == last_sv_altdec) {
	    sa = last_sa;
	    ca = last_ca;
	} else {
	    last_sv_altdec = sv_altdec;
	    last_sa = sa = sin(a);
	    last_ca = ca = cos(a);
	}

	b = PI/2 - altdec;
	sb = sin(b);
	cb = cos(b);
	if (aa_mode)
	    c = azra - sv_azra;
	else
	    c = sv_azra - azra;
	cc = cos(c);
	d = acos(ca*cb + sa*sb*cc);
	if (d < LOCEPS) {
	    *xp = *yp = rad;
	    return (1);
	}

	r = d/(sv_fov/2.0);

	sc = sin(c);
	sd = sin(d);
	se = sc*sb/sd;
	*xp = rad*(1 + r*se) + 0.5;
	if (a < LOCEPS) {
	    /* as viewpoint approaches N pole, e approaches PI - c */
	    ce = -cc;
	} else if (a > PI - LOCEPS) {
	    /* as viewpoint approaches S pole, e approaches c */
	    ce = cc;
	} else {
	    cd = cos(d);
	    ce = (cb - cd*ca)/(sd*sa);
	}
	*yp = rad*(1 - r*ce) + 0.5;

	return (r >= 1.0 ? 0 : 1);
}

/* compute azra/altdec loc of a point at x/y on a circle of radius rad pixels
 *   as viewed from sv_azra/sv_altdec with sv_fov.
 * return 1 if x/y are within the circle, else 0.
 */
static
sv_unloc (rad, x, y, altdecp, azrap)
int rad;	/* radius of target display, pixels */
int x, y;	/* X coords within circle */
double *altdecp;/* angle up from spherical equator, such as alt or dec; rad */
double *azrap;	/* angle around spherical pole, such as az or ra; rad */
{
#define	UNLOCEPS (1e-4)	/* sufficiently close to pole to not know az/ra; rads */
	double a,sa,ca;	/* angle from viewpoint to pole */
	double r;	/* distance from center to object, pixels */
	double d,sd,cd;	/* angular separation of object and viewpoint */
	double e,se,ce;	/* angle between (viewpoint,pole) and (viewpoint,obj) */
	double b,sb,cb;	/* angle from object to pole */
	double c,sc,cc;	/* difference in polar angles of obj and viewpoint */

	if (x == rad && y == rad) {
	    /* at the center -- avoids cases where r == 0 */
	    *altdecp = sv_altdec;
	    *azrap = sv_azra;
	    return (1);
	}

	a = PI/2 - sv_altdec;
	sa = sin(a);
	ca = cos(a);

	r = sqrt ((double)((x-rad)*(x-rad) + (y-rad)*(y-rad)));
	if (r > rad)
	    return(0); /* outside the circle */

	d = r/rad*(sv_fov/2.0);
	sd = sin(d);
	cd = cos(d);
	ce = (rad - y)/r;
	se = (x - rad)/r;
	cb = ca*cd + sa*sd*ce;
	b = acos(cb);
	*altdecp = PI/2 - b;

	/* find c, the polar angle between viewpoint and object */
	if (a < UNLOCEPS) {
	    /* as viewpoint approaches N pole, c approaches PI - e */
	    c = acos(-ce);
	} else if (a > PI - UNLOCEPS) {
	    /* as viewpoint approaches S pole, c approaches e */
	    c = acos(ce);
	} else if (b < UNLOCEPS || b > PI - UNLOCEPS) {
	    /* as object approaches either pole, c becomes arbitary */
	    c = 0.0;
	} else {
	    sb = sin(b);
	    cc = (cd - ca*cb)/(sa*sb);
	    if (cc < -1.0) cc = -1.0;	/* heh, it happens ... */
	    if (cc >  1.0) cc =  1.0;	/* heh, it happens ... */
	    c = acos (cc);		/* 0 .. PI; next step checks if c
					 * should be > PI
					 */
	}
	if (se < 0.0) 		/* if e > PI */
	    c = PI + (PI - c);	/*     so is c */

	if (aa_mode)
	    *azrap = sv_azra + c;
	else
	    *azrap = sv_azra - c;
	range (azrap, 2*PI);

	return (1);
}

/* given an altdec/azra pair, find all coords for our current location.
 * all values will be topocentric is we are currently in Alt/Az display mode,
 * else all values will be geocentric.
 */
static void
sv_fullwhere (np, altdec, azra, altp, azp, rap, decp)
Now *np;
double altdec, azra;
double *altp, *azp;
double *rap, *decp;
{
	double ha;
	double lst;

	now_lst (np, &lst);
	if (aa_mode) {
	    /* need to make the ra/dec entries */
	    *altp = altdec;
	    *azp = azra;
	    unrefract (pressure, temp, altdec, &altdec);
	    aa_hadec (lat, altdec, azra, &ha, decp);
	    *rap = hrrad(lst) - ha;
	    range (rap, 2*PI);
	    if (epoch != EOD)
		precess (mjd, epoch, rap, decp);
	} else {
	    /* need to make the alt/az entries */
	    *rap = azra;
	    *decp = altdec;
	    if (epoch != EOD)
		precess (epoch, mjd, &azra, &altdec);
	    ha = hrrad(lst) - azra;
	    range (&ha, 2*PI);
	    hadec_aa (lat, ha, altdec, altp, azp);
	    refract (pressure, temp, *altp, altp);
	}
}

/* draw a nice grid on a circle of radius r, x and y borders xb and yb.
 */
static void
draw_grid(dsp, win, gc, r, xb, yb)
Display *dsp;
Window win;
GC gc;
unsigned int r;
unsigned int xb, yb;
{
	double vticks[NGRID], hticks[NGRID];
	double vmin, vmax, hmin, hmax;
	XSegment xsegments[NGRID*NSEGS], *xs;
	char msg[64];
	int nvt, nht;
	int i, j;
	int pole;

	/* set vertical min and max, and detect whether pole is within fov.
	 * since display is in degrees, use that unit for finding ticks.
	 */
	pole = 0;
	vmin = sv_altdec-sv_fov/2;
	if (vmin < 0 && aa_mode)
	    vmin = 0.0;
	if (vmin <= -PI/2) {
	    /* clamp at pole */
	    vmin = -PI/2;
	    pole = 1;
	}
	vmax = sv_altdec+sv_fov/2;
	if (vmax >= PI/2) {
	    /* clamp at pole */
	    vmax = PI/2;
	    pole = 1;
	}
	vmin = raddeg(vmin);
	vmax = raddeg(vmax);

	/* set horizontal min and max.
	 * if pole is visible, we go all the way around.
	 * else compute spherical angle spanned by fov "del" from pole.
	 * again, compute ticks in the units we display in.
	 */
	if (pole) {
	    /* pole is visible */
	    hmin = 0.0;
	    hmax = 2*PI;
	} else {
	    double del = PI/2 - fabs(sv_altdec);
	    double a= acos((cos(sv_fov) - cos(del)*cos(del))/sin(del)/sin(del));
	    hmin = sv_azra-a/2;
	    hmax = sv_azra+a/2;
	}
	hmin = aa_mode ? raddeg(hmin) : radhr(hmin);
	hmax = aa_mode ? raddeg(hmax) : radhr(hmax);

	/* find tick marks.
	 * generally get less than half the max, so insure it to be consistent.
	 * N.B. remember that tickmarks() can return up to 2 more than asked.
	 */
	nvt = tickmarks(vmin, vmax, NGRID-2, vticks);
	nht = tickmarks(hmin, hmax, NGRID-2, hticks);

	/* report the spacing */
	(void) sprintf (msg, "%s: %g Degs", aa_mode ? "Alt" : "Dec",
					    (vticks[nvt-1]-vticks[0])/(nvt-1));
	f_showit (vgrid_w, msg);
	(void) sprintf (msg, " %s: %g %s", aa_mode ? "Az" : "RA",
		(hticks[nht-1]-hticks[0])/(nht-1), aa_mode ? "Degs" : "Hrs");
	f_showit (hgrid_w, msg);

	/* convert back to rads */
	for (i = 0; i < nvt; i++)
	    vticks[i] = degrad(vticks[i]);
	for (i = 0; i < nht; i++)
	    hticks[i] = aa_mode ? degrad(hticks[i]) : hrrad(hticks[i]);

	/* for each horizontal tick mark
	 *   for each vertical tick mark
	 *     compute coord on screen
	 *     if we've at least 2 pts now
	 *       connect the points with what is visible within the circle.
	 */
	for (i = 0; i < nht; i += 1) {
	    double h = hticks[i];
	    int before = 0;
	    int vis1, vis2;
	    int x1, y1, x2, y2;
	    xs = xsegments;
	    for (j = 0; j < nvt; j++) {
		double v = vticks[j];
		vis2 = sv_loc(r, v, h, &x2, &y2);
		if (before++ && (vis1 || vis2)) {
		    int sx1, sy1, sx2, sy2;
		    if (lc(0,0,r*2,x1,y1,x2,y2,&sx1,&sy1,&sx2,&sy2) == 0) {
			xs->x1 = sx1+xb; xs->y1 = sy1+yb;
			xs->x2 = sx2+xb; xs->y2 = sy2+yb;
			xs++;
		    }
		}
		x1 = x2;
		y1 = y2;
		vis1 = vis2;
	    }
	    XDrawSegments (dsp, win, gc, xsegments, xs - xsegments);
	}

	/* for each vertical tick mark
	 *   for each horizontal tick mark
	 *     compute coord on screen
	 *     if we've at least 2 pts now
	 *       connect the points with what is visible within the circle.
	 *	 (break into smaller pieces because these lines tend to curve)
	 */
	for (i = 0; i < nvt; i+=1) {
	    double v = vticks[i];
	    double h1;
	    int before = 0;
	    int vis1, vis2;
	    int x1, y1, x2, y2;
	    xs = xsegments;
	    for (j = 0; j < nht; j++) {
		double h = hticks[j];
		vis2 = sv_loc(r, v, h, &x2, &y2);
		if (before++ && (vis1 || vis2)) {
		    /* last point is at (x1,y1) == (h1,v);
		     * this point is at (x2,y2) == (h, v);
		     * connect with NSEGS segments.
		     */
		    int sx1, sy1, sx2, sy2;
		    int xt, yt;
		    int vist, k;
		    for (k = 1; k <= NSEGS; k++) {
			if (k == NSEGS) {
			    xt = x2;
			    yt = y2;
			    vist = vis2;
			} else
			    vist = sv_loc(r, v, h1+k*(h-h1)/NSEGS, &xt, &yt);
			if ((vis1 || vist) &&
				lc(0,0,r*2,x1,y1,xt,yt,&sx1,&sy1,&sx2,&sy2)==0){
			    xs->x1 = sx1+xb; xs->y1 = sy1+yb;
			    xs->x2 = sx2+xb; xs->y2 = sy2+yb;
			    xs++;
			}
			x1 = xt;
			y1 = yt;
			vis1 = vist;
		    }
		}
		x1 = x2;
		y1 = y2;
		h1 = h;
		vis1 = vis2;
	    }
	    XDrawSegments (dsp, win, gc, xsegments, xs - xsegments);
	}
}


/* draw the ecliptic on a circle of radius r, x and y borders xb and yb.
 */
static void
draw_ecliptic(dsp, win, gc, r, xb, yb)
Display *dsp;
Window win;
GC gc;
unsigned int r;
unsigned int xb, yb;
{
#define	ECL_CACHE_SZ	100
	XPoint point_cache[ECL_CACHE_SZ];
	double elat0, elng0;	/* ecliptic lat and long at center of fov */
	double elngmin, elngmax;/* ecliptic long limits */
	double ra, dec;
	double elng;
	double lst;
	int ncache;
	Now *np;

	np = mm_get_now();
	now_lst (np, &lst);

	/* find ecliptic coords of center of view */
	if (aa_mode) {
	    double ha0;		/* local hour angle */
	    aa_hadec (lat, sv_altdec, sv_azra, &ha0, &dec);
	    ra = hrrad(lst) - ha0;
	} else {
	    ra = sv_azra;
	    dec = sv_altdec;
	}
	eq_ecl (mjd, ra, dec, &elat0, &elng0);

	/* no ecliptic visible if ecliptic latitude at center of view 
	 * is not less than fov/2
	 */
	if (fabs(elat0) >= sv_fov/2.0)
	    return;

	/* worst-case elong limits is center elong += half fov */
	elngmin = elng0 - sv_fov/2.0;
	elngmax = elng0 + sv_fov/2.0;

	/* put a mark at every ECL_TICS pixels */
	ncache = 0;
	for (elng = elngmin; elng <= elngmax; elng += sv_fov/(2.0*r/ECL_TICS)) {
	    int x, y;
	    double altdec, azra;

	    /* convert longitude along the ecliptic to ra/dec */
	    ecl_eq (mjd, 0.0, elng, &azra, &altdec);

	    /* if in aa mode, we need it in alt/az */
	    if (aa_mode) {
		hadec_aa (lat, hrrad(lst) - azra, altdec, &altdec, &azra);
		refract (pressure, temp, altdec, &altdec);
	    }

	    /* if visible, display point */
	    if ((!aa_mode || altdec >= 0) && sv_loc (r, altdec, azra, &x, &y)) {
		XPoint *xp = &point_cache[ncache++];
		xp->x = x + xb;
		xp->y = y + yb;
		if (ncache == XtNumber(point_cache)) {
		    XDrawPoints (dsp,win,gc,point_cache,ncache,CoordModeOrigin);
		    ncache = 0;
		}
	    }
	}

	if (ncache > 0)
	    XDrawPoints (dsp, win, gc, point_cache, ncache, CoordModeOrigin);
}

/* given the magnitude of an object, return its desired diameter, in pixels.
 * N.B. we assume it is at least as bright as fmag.
 */
static
magdiam(m)
double m;
{
	return (((int)(fmag-m)+3)/2);
}

/* make the general purpose sv_gc and learn the three colors.
 * also make the tracking text gc, sv_strgc and the XFontStruct sv_f.
 */
static void
sv_mk_gcs(dsp, win)
Display *dsp;
Window win;
{
	XGCValues gcv;
	unsigned gcm = 0;
	Pixel p;

	get_something (svda_w, XmNforeground, (char *)&fg_p);
	get_something (svda_w, XmNbackground, (char *)&sky_p);
	get_something (svform_w, XmNbackground, (char *)&bg_p);
	set_something (svda_w, XmNbackground, (char *)bg_p);

	sv_gc = XCreateGC (dsp, win, gcm, &gcv);

	gcm = GCForeground | GCBackground | GCFont;
	get_something (aa_w, XmNforeground, (char *)&p);
	gcv.foreground = p;
	get_something (aa_w, XmNbackground, (char *)&p);
	gcv.background = p;
	get_xmlabel_font (aa_w, &sv_f);
	gcv.font = sv_f->fid;
	sv_strgc = XCreateGC (dsp, win, gcm, &gcv);
}

/* draw a label for an object that is located at [x,y]
 */
static void
draw_label (dsp, win, gc, label, x, y)
Display *dsp;
Window win;
GC gc;
char label[];
int x, y;
{
	XDrawString (dsp, win, gc, x+4, y-4, label, strlen(label));
}

/* send current coords to the SKYOUTFIFO.
 * open/close fifo as required as per want_outfifo as well.
 * N.B. we assume SIGPIPE is being ignored.
 * N.B. may be called with np == 0 to mean just open the fifo.
 */
static void
skyout_fifo (np)
Now *np;
{
	static char fmt[]="RA:%9.6f Dec:%9.6f Epoch:%9.3f\n";
	static char fifores[] = "SKYOUTFIFO";	/* name of X resource */
	static char *fn;	/* file name */
	static FILE *fp;	/* FILE pointer */
	double alt, az, ra, dec;
	double y;

	if (!want_outfifo) {
	    if (fp) {
		(void) fclose (fp);
		fp = NULL;
	    }
	    return;
	}

	/* open fifo if not already */
	if (!fp && open_wfifores (fifores, &fn, &fp) < 0) {
	    XmToggleButtonSetState (outfifo_w, False, True);
	    return;
	}

	if (!np)
	    return;

	sv_fullwhere (np, sv_altdec, sv_azra, &alt, &az, &ra, &dec);
	mjd_year (epoch == EOD ? mjd : epoch, &y);

	if (fprintf (fp, fmt, ra, dec, y) < 0 || fflush (fp) != 0) {
	    char msg[256];
	    (void) sprintf (msg, "Closing %.100s due to write error: %.100s\n",
							    fn, syserrstr());
	    xe_msg (msg, 0);
	    (void) fclose (fp);
	    fp = NULL;	/* signal no fp for next time */
	    XmToggleButtonSetState (outfifo_w, False, True);
	}
}

/* set up whether to accept pointing coords from SKYINFIFO.
 * open/close fifo as required as per want_infifo as well.
 */
static void
skyin_fifo ()
{
	static char fifores[] = "SKYINFIFO";	/* name of X resource */
	static int infifo_fd;		/* file descriptor for SKYINFIFO */
	static XtInputId infifo_xid;	/* input id for SKYINFIFO */
	char *fn;

	if (!want_infifo) {
	    if (infifo_fd >= 0) {
		(void) close (infifo_fd);
		infifo_fd = -1;
	    }
	    if (infifo_xid) {
		XtRemoveInput (infifo_xid);
		infifo_xid = 0;
	    }
	    return;
	}

	fn = XGetDefault (XtD, myclass, fifores);
	if (!fn) {
	    char msg[256];
	    (void) sprintf (msg, "No `%.200s' resource.\n", fifores);
	    xe_msg (msg, 0);
	    return;
	}

	/* open for read/write. this assures open will never block, that
	 * reads (and hence select()) WILL block if it's empty, and let's
	 * processes using it come and go as they please.
	 */
	infifo_fd = openh (fn, 2);
	if (infifo_fd < 0) {
	    char msg[256];
	    (void) sprintf (msg, "Can not open %.150s: %.50s\n",fn,syserrstr());
	    xe_msg (msg, 0);
	    XmToggleButtonSetState (infifo_w, False, True);
	    return;
	}

	infifo_xid = XtAppAddInput (xe_app, infifo_fd,
		    (XtPointer)XtInputReadMask, skyinfifo_cb, (XtPointer)fn);
}

/* send these coords to the SKYLOCFIFO.
 * open/close fifo as required as per want_locfifo as well.
 * N.B. we assume SIGPIPE is being ignored.
 * N.B. may be called with y == 0 to mean just open the fifo.
 */
static void
skyloc_fifo (ra, dec, y)
double ra, dec, y;
{
	static char fmt[]="RA:%9.6f Dec:%9.6f Epoch:%9.3f\n";
	static char fifores[] = "SKYLOCFIFO";	/* name of X resource */
	static char *fn;	/* file name */
	static FILE *fp;	/* FILE pointer */

	if (!want_locfifo) {
	    if (fp) {
		(void) fclose (fp);
		fp = NULL;
	    }
	    return;
	}

	/* open fifo if not already */
	if (!fp && open_wfifores (fifores, &fn, &fp) < 0) {
	    XmToggleButtonSetState (locfifo_w, False, True);
	    return;
	}

	if (y == 0.0)
	    return;

	/* this output is often going to a db server process that sends us a
	 * set of near-by objects. in anticipation of doing this many times for
	 * different objects, delete back to the checkpoint first each time
	 * to keep our db from growing indefinitely.
	 */
	db_del_cp();
	all_newdb (0);

	if (fprintf (fp, fmt, ra, dec, y) < 0 || fflush (fp) != 0) {
	    char msg[256];
	    (void) sprintf (msg, "Error writing to %.100s: %.100s\n", fn,
								syserrstr());
	    xe_msg (msg, 0);
	    (void) fclose (fp);
	    fp = NULL;	/* signal no fp for next time */
	    XmToggleButtonSetState (locfifo_w, False, True);
	}
}
