/* code to manage the stuff on the solar system display.
 * functions and data to support the main display begin with ss_.
 * function and data to support the stereo display begin with st_.
 */

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#if defined(__STDC__)
#include <stdlib.h>
#endif
#include <X11/Xlib.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/DrawingA.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/ToggleB.h>
#include <Xm/Text.h>
#include <Xm/Scale.h>
#include "astro.h"
#include "circum.h"

/* heliocentric coordinates, and enough info to locate it on screen */
typedef struct {
    Obj o;		/* copy of Obj at the given moment */
    double smjd;	/* mjd when Obj was valid */
    int sx, sy;		/* main view window coords of object */
    int stx;		/* stereo view x coord (y is the same in both) */
    float x, y, z;	/* heliocentric cartesian coords */
} HLoc;

#if defined(__STDC__) || defined(__cplusplus)
#define P_(s) s
#define	Const const
#else
#define P_(s) ()
#define	Const
#endif

extern Now *mm_get_now P_((void));
extern Obj *db_basic P_((int id));
extern double mjd_hr P_((double jd));
extern void db_update P_((Obj *op));
extern void f_pangle P_((Widget w, double a));
extern void f_date P_((Widget w, double jd));
extern void f_double P_((Widget w, char *fmt, double f));
extern void f_ra P_((Widget w, double ra));
extern void f_string P_((Widget w, char *s));
extern void f_time P_((Widget w, double t));
extern void get_something P_((Widget w, char *resource, char *value));
extern void hlp_dialog P_((char *tag, char *deflt[], int ndeflt));
extern void obj_pickgc P_((Obj *op, Widget w, GC *gcp));
extern void set_something P_((Widget w, char *resource, char *value));
extern void set_xmstring P_((Widget w, char *resource, char *txt));
extern void timestamp P_((Now *np, Widget w));

void ss_manage P_((void));
void ss_newobj P_((int dbidx));
int ss_ison P_((void));
void ss_update P_((Now *np, int how_much));
void ss_cursor P_((Cursor c));
static void ss_create_form P_((void));
static void st_create_form P_((void));
static Widget ss_create_pot P_((Widget form_w, Widget bot_w));
static void ss_activate_cb P_((Widget w, XtPointer client, XtPointer call));
static void ss_obj_cb P_((Widget w, XtPointer client, XtPointer call));
static void ss_changed_cb P_((Widget w, XtPointer client, XtPointer call));
static void ss_close_cb P_((Widget w, XtPointer client, XtPointer call));
static void ss_help_cb P_((Widget w, XtPointer client, XtPointer call));
static void ss_da_exp_cb P_((Widget w, XtPointer client, XtPointer call));
static void ss_da_input_cb P_((Widget w, XtPointer client, XtPointer call));
static void st_parallax_cb P_((Widget w, XtPointer client, XtPointer call));
static void st_map_cb P_((Widget wid, XtPointer client, XtPointer call));
static void st_da_exp_cb P_((Widget w, XtPointer client, XtPointer call));
static void st_da_input_cb P_((Widget w, XtPointer client, XtPointer call));
static void ss_popup P_((XEvent *ev, HLoc *lp));
static void ss_create_popup P_((void));
static int cmpHLoc P_((Const void *lp1, Const void *lp2));
static void ss_all P_((int preclr));
static void solar_system P_((HLoc *lp, double scale, double selt, double celt,
    double selg, double celg, unsigned nx, unsigned ny));

#undef P_

extern Widget toplevel_w;
#define	XtD	XtDisplay(toplevel_w)

static Widget ssform_w;		/* main solar system form dialog */
static Widget hr_w, hlng_w, hlat_w; /* scales for heliocentric R, long, lat */
static Widget ssda_w;		/* solar system drawring area */
static Widget dt_w;		/* date/time stamp label widget */

static Widget stform_w;		/* main stereo form dialog */
static Widget parallax_w;	/* scale to set amount of parallax */
static Widget stda_w;		/* stereo solar system drawring area */

enum {TRAILS, BIGDOTS, CONNECT, ECLIPTIC, LEGS, TAGS, STEREO};	/* toggle ids */

#define	MINMAG	3.0	/* minimum mag factor, pixels/AU */
#define	MAXMAG	250.0	/* maximum mag factor, pixels/AU */

#define	NECLPT	10	/* approx number of circles in the ecliptic grid */
#define	NECLSEG	53	/* number of segments in each stereo ecliptic line */

/* whether each option is currently on */
static int trails;
static int bigdots;
static int connectdots;
static int ecliptic;
static int legs;
static int nametags;
static int stereo;

/* current value of desired parallax */
static int parallax;

static HLoc *points[NOBJ];	/* malloc'd set of points on screen now */
static int npoints[NOBJ];	/* number of points */

static char obj_on[NOBJ];	/* 1 of object is on */
static Widget obj_w[NOBJ];	/* toggle buttons for each object */

/* info about the popup widget */
typedef struct {
    Widget pu_w;
    Widget name_w;
    Widget ud_w, udl_w;
    Widget ut_w, utl_w;
    Widget ra_w, ral_w;
    Widget dec_w, decl_w;
    Widget hlong_w, hlongl_w;
    Widget hlat_w, hlatl_w;
    Widget eadst_w, eadstl_w;
    Widget sndst_w, sndstl_w;
    Widget elong_w, elongl_w;
} Popup;
static Popup pu;

static char earthname[] = "Earth";

/* called when the solar system view is activated via the main menu pulldown.
 * if never called before, create and manage all the widgets as a child of a
 * form. otherwise, just toggle whether the form is managed.
 */
void
ss_manage ()
{
	if (!ssform_w) {
	    ss_create_form();
	    ss_create_popup();
	    st_create_form();
	}
	
	if (XtIsManaged(ssform_w)) {
	    if (XtIsManaged(stform_w))
		XtUnmanageChild(stform_w);
	    XtUnmanageChild (ssform_w);
	} else {
	    XtManageChild (ssform_w);
	    if (stereo)
		XtManageChild (stform_w);
	}
}

/* called when one of the user defined objects has changed.
 * discard the points for that object.
 * if the object is now defined in the SS manage the toggle button.
 * N.B. no need to rebuild the scene -- ss_update() will be called for us.
 */
void
ss_newobj (dbidx)
int dbidx;
{
	static char me[] = "ss_newobj()";
	Obj *op;

	/* we might get called before we are ever brought up the first time */
	if (!ssform_w)
	    return;

	if (dbidx >= NOBJ) {
	    printf ("%s: dbidx=%d but NOBJ=%d\n", me, dbidx, NOBJ);
	    exit (1);
	}

	if (points[dbidx])
	    XtFree ((char *)points[dbidx]);
	points[dbidx] = NULL;
	npoints[dbidx] = 0;

	op = db_basic(dbidx);
	switch (op->type) {
	case ELLIPTICAL: case HYPERBOLIC: case PARABOLIC:
	    XtManageChild (obj_w[dbidx]);
	    XmToggleButtonSetState (obj_w[dbidx], True, False);
	    set_xmstring (obj_w[dbidx], XmNlabelString, op->o_name);
	    obj_on[dbidx] = 1;
	    break;
	default:
	    XtUnmanageChild (obj_w[dbidx]);
	    XmToggleButtonSetState (obj_w[dbidx], False, False);
	    obj_on[dbidx] = 0;
	    break;
	}
	    
}

ss_ison()
{
	return (ssform_w && XtIsManaged(ssform_w));
}

/* called when we are to update our view.
 * don't bother if we are unmanaged unless trails is on - in that case,
 * compute the new locations but don't display them.
 */
/* ARGSUSED */
void
ss_update (np, how_much)
Now *np;
int how_much;
{
	HLoc *lp;
	int up;
	int dbidx;

	up = ssform_w && XtIsManaged(ssform_w);
	if (!up && !trails)
	    return;

	/* tag earth's data (from SUN object) as object MOON */
	for (dbidx = 0; dbidx < NOBJ; dbidx++) {
	    Obj *op;
	    double sd;
	    int inss;

	    /* see if object is still a solar system object */
	    op = db_basic(dbidx);
	    switch (op->type) {
	    case PLANET: case ELLIPTICAL: case HYPERBOLIC: case PARABOLIC:
		inss = 1;
		break;
	    default:
		inss = 0;
		break;
	    }

	    /* discard previous set if not leaving trails or obj no longer 
	     * in solar system.
	     */
	    if (!inss || !trails) {
		if (points[dbidx])
		    XtFree ((char*)points[dbidx]);
		points[dbidx] = NULL;
		npoints[dbidx] = 0;
	    }

	    if (!inss)
		continue;

	    /* just one SUN will do */
	    if (dbidx == SUN && npoints[SUN] > 0)
		continue;

	    npoints[dbidx]++;
	    points[dbidx] = (HLoc *) XtRealloc ((char *)points[dbidx],
						npoints[dbidx]*sizeof(HLoc));
	    lp = &points[dbidx][npoints[dbidx]-1];
	    if (dbidx == MOON) {
		/* really want earth info here; get it from SUN */
		op = db_basic (SUN);
		db_update(op);
		sd = op->s_edist;
	    } else {
		sd = op->s_sdist;
	    }
	    lp->x = sd*cos(op->s_hlat)*cos(op->s_hlong);
	    lp->y = sd*cos(op->s_hlat)*sin(op->s_hlong);
	    lp->z = sd*sin(op->s_hlat);
	    lp->o = *op;
	    if (dbidx == MOON)
		lp->o.pl.code = MOON;
	    lp->smjd = mjd;

	    /* keep points array sorted in increasing time order */
	    qsort ((char *)points[dbidx], npoints[dbidx], sizeof(HLoc),
								    cmpHLoc);
	}

	if (up) {
	    ss_all(!trails);
	    timestamp (np, dt_w);
	}
	    
}

/* called to put up or remove the watch cursor.  */
void
ss_cursor (c)
Cursor c;
{
	Window win;

	if (ssform_w && (win = XtWindow(ssform_w))) {
	    Display *dsp = XtDisplay(ssform_w);
	    if (c)
		XDefineCursor (dsp, win, c);
	    else
		XUndefineCursor (dsp, win);
	}
}

/* create the main solarsystem form */
static void
ss_create_form()
{
	typedef struct {
	    char *name;		/* name of widget */
	    char *label;	/* label on toggle button */
	    int id;		/* one of the toggle ids */
	    int *state;		/* int we use to keep state */
	} DrCtrl;
	static DrCtrl drctrls[] = {
	    {"ConnectDots", "Connect Dots", CONNECT, &connectdots},
	    {"Ecliptic", "Ecliptic", ECLIPTIC, &ecliptic},
	    {"Names","Names", TAGS, &nametags},
	    {"BigDots", "Big Dots", BIGDOTS, &bigdots},
	    {"Legs", "Legs", LEGS, &legs},
	    {"Trails", "Leave Trails", TRAILS, &trails},
	};
	Widget frame_w;
	Widget form_w;
	Widget fr_w;
	Widget pot_w;
	Widget w;
	XmString str;
	Arg args[20];
	int i;
	int n;

	/* create form */
	n = 0;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	XtSetArg (args[n], XmNdefaultPosition, False); n++;
	XtSetArg (args[n], XmNresizePolicy, XmRESIZE_NONE); n++;
	ssform_w = XmCreateFormDialog (toplevel_w, "SolarSystem", args, n);
	XtAddCallback (ssform_w, XmNhelpCallback, ss_help_cb, 0);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "xephem Solar System"); n++;
	XtSetValues (XtParent(ssform_w), args, n);

	/* make a form for the bottom controls */

	n = 0;
	XtSetArg (args[n], XmNfractionBase, 13); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	form_w = XmCreateForm (ssform_w, "CtlForm", args, n);
	XtManageChild (form_w);

	    /* make the "close" push button */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 1); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNrightPosition, 4); n++;
	    w = XmCreatePushButton (form_w, "Close", args, n);
	    XtAddCallback (w, XmNactivateCallback, ss_close_cb, 0);
	    XtManageChild (w);

	    /* make the "stereo" toggle button in a frame */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 5); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNrightPosition, 8); n++;
	    fr_w = XmCreateFrame (form_w, "StereoFrame", args, n);
	    XtManageChild (fr_w);
	    w = XmCreateToggleButton (fr_w, "Stereo", args, n);
	    XtAddCallback (w, XmNvalueChangedCallback, ss_activate_cb, 
							    (XtPointer)STEREO);
	    XtManageChild (w);
	    stereo = XmToggleButtonGetState (w);

	    /* make the "help" push button */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 9); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNrightPosition, 12); n++;
	    w = XmCreatePushButton (form_w, "Help", args, n);
	    XtAddCallback (w, XmNactivateCallback, ss_help_cb, 0);
	    XtManageChild (w);

	/* make the planet on/off table */

	pot_w = ss_create_pot(ssform_w, form_w);
	XtManageChild (pot_w);

	/* make the drawing option toggle buttons in a form */

	n = 0;
	XtSetArg (args[n], XmNfractionBase, 16); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, pot_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	form_w = XmCreateForm (ssform_w, "DOptForm", args, n);
	XtManageChild (form_w);

	    for (i = 0; i < XtNumber(drctrls); i++) {
		DrCtrl *cp = &drctrls[i];

		str = XmStringCreate(cp->label, XmSTRING_DEFAULT_CHARSET);
		n = 0;
		XtSetArg (args[n], XmNtopAttachment, XmATTACH_POSITION); n++;
		XtSetArg (args[n], XmNtopPosition, i/3*8); n++;
		XtSetArg (args[n], XmNbottomAttachment, XmATTACH_POSITION); n++;
		XtSetArg (args[n], XmNbottomPosition, 8+i/3*8); n++;
		XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
		XtSetArg (args[n], XmNleftPosition, 1+(i%3)*5); n++;
		XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
		XtSetArg (args[n], XmNrightPosition, 5+(i%3)*5); n++;
		XtSetArg (args[n], XmNlabelString, str); n++;
		XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
		w = XmCreateToggleButton(form_w, cp->name, args, n);
		XmStringFree (str);
		XtManageChild (w);
		*(cp->state) = XmToggleButtonGetState (w);
		XtAddCallback(w, XmNvalueChangedCallback, ss_activate_cb,
							    (XtPointer)cp->id);
	    }

	/* make the time/date stamp label */

	n = 0;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, form_w); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_CENTER); n++;
	dt_w = XmCreateLabel (ssform_w, "DateStamp", args, n);
	XtManageChild (dt_w);

	/* make the bottom scale */

	n = 0;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, dt_w); n++;
	XtSetArg (args[n], XmNmaximum, 359); n++;
	XtSetArg (args[n], XmNminimum, 0); n++;
	XtSetArg (args[n], XmNscaleMultiple, 1); n++;
	XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
	XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_RIGHT); n++;
	XtSetArg (args[n], XmNshowValue, True); n++;
	hlng_w = XmCreateScale (ssform_w, "HLongScale", args, n);
	XtAddCallback (hlng_w, XmNdragCallback, ss_changed_cb, 0);
	XtAddCallback (hlng_w, XmNvalueChangedCallback, ss_changed_cb, 0);
	XtManageChild (hlng_w);

	/* make the left scale */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, hlng_w); n++;
	XtSetArg (args[n], XmNdecimalPoints, 1); n++;
	XtSetArg (args[n], XmNmaximum, 100); n++;
	XtSetArg (args[n], XmNminimum, 0); n++;
	XtSetArg (args[n], XmNscaleMultiple, 1); n++;
	XtSetArg (args[n], XmNorientation, XmVERTICAL); n++;
	XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_TOP); n++;
	/* XtSetArg (args[n], XmNshowValue, True); n++; */
	hr_w = XmCreateScale (ssform_w, "DistScale", args, n);
	XtAddCallback (hr_w, XmNdragCallback, ss_changed_cb, 0);
	XtAddCallback (hr_w, XmNvalueChangedCallback, ss_changed_cb, 0);
	XtManageChild (hr_w);

	/* make the right scale */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, hlng_w); n++;
	XtSetArg (args[n], XmNmaximum, 90); n++;
	XtSetArg (args[n], XmNminimum, -90); n++;
	XtSetArg (args[n], XmNscaleMultiple, 1); n++;
	XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_TOP); n++;
	XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_TOP); n++;
	XtSetArg (args[n], XmNshowValue, True); n++;
	hlat_w = XmCreateScale (ssform_w, "HLatScale", args, n);
	XtAddCallback (hlat_w, XmNdragCallback, ss_changed_cb, 0);
	XtAddCallback (hlat_w, XmNvalueChangedCallback, ss_changed_cb, 0);
	XtManageChild (hlat_w);

	/* make a frame for the drawing area */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, hlng_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNleftWidget, hr_w); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNrightWidget, hlat_w); n++;
	XtSetArg (args[n], XmNshadowType, XmSHADOW_ETCHED_OUT); n++;
	frame_w = XmCreateFrame (ssform_w, "SolarFrame", args, n);
	XtManageChild (frame_w);

	/* make a drawing area for drawing the solar system */

	n = 0;
	ssda_w = XmCreateDrawingArea (frame_w, "SolarDA", args, n);
	XtAddCallback (ssda_w, XmNexposeCallback, ss_da_exp_cb, 0);
	XtAddCallback (ssda_w, XmNinputCallback, ss_da_input_cb, 0);
	XtManageChild (ssda_w);
}

/* create the stereo solarsystem form */
static void
st_create_form()
{
	Widget frame_w;
	Arg args[20];
	int n;

	/* create form */
	n = 0;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	XtSetArg (args[n], XmNdefaultPosition, False); n++;
	XtSetArg (args[n], XmNnoResize, True); n++;	/* user can't resize */
	stform_w = XmCreateFormDialog (toplevel_w, "StereoSolarSystem", args,n);
	XtAddCallback (stform_w, XmNmapCallback, st_map_cb, NULL);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "xephem Stereo Solar System"); n++;
	XtSetValues (XtParent(stform_w), args, n);

	/* make the parallax scale at the bottom.
	 * the value is the offset of the sun (at a=0), in pixels.
	 */

	n = 0;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNmaximum, 20); n++;
	XtSetArg (args[n], XmNminimum, -20); n++;
	XtSetArg (args[n], XmNscaleMultiple, 2); n++;
	XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
	XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_RIGHT); n++;
	parallax_w = XmCreateScale (stform_w, "Parallax", args, n);
	XtAddCallback (parallax_w, XmNdragCallback, st_parallax_cb, 0);
	XtAddCallback (parallax_w, XmNvalueChangedCallback, st_parallax_cb, 0);
	XtManageChild (parallax_w);
	XmScaleGetValue (parallax_w, &parallax);

	/* make a frame for the drawing area */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, parallax_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNshadowType, XmSHADOW_ETCHED_OUT); n++;
	frame_w = XmCreateFrame (stform_w, "StereoFrame", args, n);
	XtManageChild (frame_w);

	/* make a drawing area for drawing the stereo solar system */

	n = 0;
	stda_w = XmCreateDrawingArea (frame_w, "StereoDA", args, n);
	XtAddCallback (stda_w, XmNexposeCallback, st_da_exp_cb, 0);
	XtAddCallback (stda_w, XmNinputCallback, st_da_input_cb, 0);
	XtManageChild (stda_w);
}

/* create the planet on/off table.
 * don't manage it yet; return the outtermost Widget.
 */
static Widget
ss_create_pot (form_w, bot_w)
Widget form_w;	/* form host */
Widget bot_w;	/* attach our bottom to this one */
{
	Widget frame_w;
	Widget rc_w;
	Arg args[20];
	int n;
	int i;

	n = 0;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, bot_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	frame_w = XmCreateFrame (form_w, "ObjTblF", args, n);

	n = 0;
	XtSetArg (args[n], XmNpacking, XmPACK_COLUMN); n++;
	XtSetArg (args[n], XmNnumColumns, 4); n++;
	rc_w = XmCreateRowColumn (frame_w, "ObjTblRC", args, n);
	XtManageChild (rc_w);

	/* make the toggle buttons.
	 * fill in the names we can now.
	 * default the planets to all on.
	 */
	for (i = 0; i < NOBJ; i++) {
	    Obj *op = db_basic(i);
	    n = 0;
	    obj_w[i] = XmCreateToggleButton (rc_w, "ObjTB", args, n);
	    XtAddCallback (obj_w[i], XmNvalueChangedCallback, ss_obj_cb,
							(XtPointer)i);
	    switch (op->type) {
	    case ELLIPTICAL: case PARABOLIC: case HYPERBOLIC: case PLANET:
		set_xmstring (obj_w[i], XmNlabelString,
		    is_planet(op,MOON)? earthname : op->o_name);
		XtManageChild (obj_w[i]);
		XmToggleButtonSetState (obj_w[i], True, False);
		break;
	    }
	    obj_on[i] = XmToggleButtonGetState (obj_w[i]);
	}

	return (frame_w);
}

/* callback from the control toggle buttons
 */
/* ARGSUSED */
static void
ss_activate_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	int what = (int) client;

	switch (what) {
	case TRAILS:
	    trails = XmToggleButtonGetState(w);
	    break;
	case BIGDOTS:
	    bigdots = XmToggleButtonGetState(w);
	    ss_all (1);
	    break;
	case CONNECT:
	    connectdots = XmToggleButtonGetState(w);
	    ss_all (1);
	    break;
	case ECLIPTIC:
	    ecliptic = XmToggleButtonGetState(w);
	    ss_all (1);
	    break;
	case LEGS:
	    legs = XmToggleButtonGetState(w);
	    ss_all (1);
	    break;
	case TAGS:
	    nametags = XmToggleButtonGetState(w);
	    ss_all (1);
	    break;
	case STEREO:
	    stereo = XmToggleButtonGetState(w);
	    if (stereo) {
		XtManageChild(stform_w);
		ss_all (1);
	    } else
		XtUnmanageChild(stform_w);
	    break;
	default:
	    printf ("solsysmenu.c: unknown toggle button\n");
	    exit(1);
	}
}

/* callback from the object on/off toggle buttons.
 * client is the dbidx.
 */
/* ARGSUSED */
static void
ss_obj_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	int who = (int) client;

	obj_on[who] = XmToggleButtonGetState(w);
	ss_all(1);
}

/* callback when any of the scales change value.
 */
/* ARGSUSED */
static void
ss_changed_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmScaleCallbackStruct *sp = (XmScaleCallbackStruct *)call;

	if (w != hr_w && w != hlng_w && w != hlat_w) {
	    printf ("solsysmenu.c: Unknown scaled callback\n");
	    exit(1);
	}

	ss_all(1);

	/* some of these get a bit lengthy, so discard remaining events.
	 * TODO: don't discard pending Expose events
	 */
	if (sp->reason == XmCR_DRAG)
	    XSync (XtDisplay(w), True);
}

/* callback from the Close button.
 */
/* ARGSUSED */
static void
ss_close_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (XtIsManaged(stform_w))
	    XtUnmanageChild (stform_w);
	XtUnmanageChild (ssform_w);
}

/* callback from the Help button.
 */
/* ARGSUSED */
static void
ss_help_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	static char *msg[] = {
"This displays the solar system. The sun is always at the center. The left",
"slider controls your distance from the sun - further up is closer. The",
"bottom slider controls your heliocentric longitude. The right slider controls",
"your heliocentric latitude - your angle above the ecliptic."
};

	hlp_dialog ("Solar System View", msg, sizeof(msg)/sizeof(msg[0]));
}

/* expose of solar system drawing area.
 */
/* ARGSUSED */
static void
ss_da_exp_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmDrawingAreaCallbackStruct *c = (XmDrawingAreaCallbackStruct *)call;

	switch (c->reason) {
	case XmCR_EXPOSE: {
	    /* turn off gravity so we get expose events for either shrink or
	     * expand.
	     */
	    static before;
	    XExposeEvent *e = &c->event->xexpose;

	    if (!before) {
		XSetWindowAttributes swa;
		swa.bit_gravity = ForgetGravity;
		XChangeWindowAttributes (e->display, e->window,
							    CWBitGravity, &swa);
		before = 1;
	    }
	    /* wait for the last in the series */
	    if (e->count != 0)
		return;
	    break;
	    }
	default:
	    printf ("Unexpected ssda_w event. type=%d\n", c->reason);
	    exit(1);
	}

	ss_update (mm_get_now(), 1);
}

/* a dot has been picked: find what it is and report it. */
/* ARGSUSED */
static void
ss_da_input_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmDrawingAreaCallbackStruct *c = (XmDrawingAreaCallbackStruct *)call;
	XEvent *ev;
	
#define	PICKRANGE	100	/* sqr of dist allowed from pointer */
	int x, y, mind;
	int dbidx;
	HLoc *lp;
	int i;

	if (c->reason != XmCR_INPUT)
	    return;
	ev = c->event;
	if (ev->xany.type != ButtonPress || ev->xbutton.button != Button3)
	    return;

	x = ((XButtonPressedEvent *)ev)->x;
	y = ((XButtonPressedEvent *)ev)->y;

	lp = NULL;
	for (dbidx = 0; dbidx < NOBJ; dbidx++)
	    for (i = 0; i < npoints[dbidx]; i++) {
		int d = (x-points[dbidx][i].sx)*(x-points[dbidx][i].sx) +
				(y-points[dbidx][i].sy)*(y-points[dbidx][i].sy);
		if (!lp || d < mind) {
		    lp = &points[dbidx][i];
		    mind = d;
		}
	    }

	if (lp && mind <= PICKRANGE)
	    ss_popup (ev, lp);
}

/* ARGSUSED */
static void
st_parallax_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmScaleCallbackStruct *sp = (XmScaleCallbackStruct *)call;

	XmScaleGetValue (w, &parallax);
	ss_all (1);

	/* some of these get a bit lengthy, so discard remaining events.
	 * TODO: don't discard pending Expose events
	 */
	if (sp->reason == XmCR_DRAG)
	    XSync (XtDisplay(w), True);
}

/* called whenever the stereo scene is mapped.
 * we set the size of the DrawingArea the same as the main window's.
 * we also try to position it just to the left, but it doesn't always work :-(
 */
/* ARGSUSED */
static void
st_map_cb (wid, client, call)
Widget wid;
XtPointer client;
XtPointer call;
{
	Dimension w, h;
	Position x, y;
	Arg args[20];
	int n;

	n = 0;
	XtSetArg (args[n], XmNwidth, &w); n++;
	XtSetArg (args[n], XmNheight, &h); n++;
	XtGetValues (ssda_w, args, n);

	n = 0;
	XtSetArg (args[n], XmNwidth, w); n++;
	XtSetArg (args[n], XmNheight, h); n++;
	XtSetValues (stda_w, args, n);

	n = 0;
	XtSetArg (args[n], XmNx, &x); n++;
	XtSetArg (args[n], XmNy, &y); n++;
	XtGetValues (ssform_w, args, n);

	n = 0;
	XtSetArg (args[n], XmNx, x-w); n++;
	XtSetArg (args[n], XmNy, y); n++;
	XtSetValues (stform_w, args, n);
}

/* expose of stereo solar system drawing area.
 */
/* ARGSUSED */
static void
st_da_exp_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmDrawingAreaCallbackStruct *c = (XmDrawingAreaCallbackStruct *)call;

	switch (c->reason) {
	case XmCR_EXPOSE: {
	    /* turn off gravity so we get expose events for either shrink or
	     * expand.
	     */
	    static before;
	    XExposeEvent *e = &c->event->xexpose;

	    if (!before) {
		XSetWindowAttributes swa;
		swa.bit_gravity = ForgetGravity;
		XChangeWindowAttributes (e->display, e->window,
							    CWBitGravity, &swa);
		before = 1;
	    }
	    /* wait for the last in the series */
	    if (e->count != 0)
		return;
	    break;
	    }
	default:
	    printf ("Unexpected stda_w event. type=%d\n", c->reason);
	    exit(1);
	}

	ss_update (mm_get_now(), 1);
}

/* a dot has been picked on the stereo map: find what it is and report it. */
/* ARGSUSED */
static void
st_da_input_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmDrawingAreaCallbackStruct *c = (XmDrawingAreaCallbackStruct *)call;
	XEvent *ev;
	
#define	PICKRANGE	100	/* sqr of dist allowed from pointer */
	int x, y, mind;
	int dbidx;
	HLoc *lp;
	int i;

	if (c->reason != XmCR_INPUT)
	    return;
	ev = c->event;
	if (ev->xany.type != ButtonPress || ev->xbutton.button != Button3)
	    return;

	x = ((XButtonPressedEvent *)ev)->x;
	y = ((XButtonPressedEvent *)ev)->y;

	lp = NULL;
	for (dbidx = 0; dbidx < NOBJ; dbidx++)
	    for (i = 0; i < npoints[dbidx]; i++) {
		int d = (x-points[dbidx][i].stx)*(x-points[dbidx][i].stx) +
				(y-points[dbidx][i].sy)*(y-points[dbidx][i].sy);
		if (!lp || d < mind) {
		    lp = &points[dbidx][i];
		    mind = d;
		}
	    }

	if (lp && mind <= PICKRANGE)
	    ss_popup (ev, lp);
}

/* fill in the popup with goodies from lp.
 * display fields the same way they are in main data menu.
 * position the popup as indicated by ev and display it.
 * it goes down by itself.
 */
static void
ss_popup (ev, lp)
XEvent *ev;
HLoc *lp;
{
	Obj *op = &lp->o;
	double d;

	if (is_planet(op, MOON)) {
	    /* MOON is used to denote Earth */
	    f_string (pu.name_w, earthname);
	    XtManageChild (pu.ud_w);
	    XtManageChild (pu.udl_w);
	    XtManageChild (pu.ut_w);
	    XtManageChild (pu.utl_w);
	    XtManageChild (pu.ra_w);
	    XtManageChild (pu.ral_w);
	    set_xmstring (pu.ral_w, XmNlabelString, "Sun RA:");
	    XtManageChild (pu.dec_w);
	    XtManageChild (pu.decl_w);
	    set_xmstring (pu.decl_w, XmNlabelString, "Sun Dec:");
	    XtManageChild (pu.hlong_w);
	    XtManageChild (pu.hlongl_w);
	    XtUnmanageChild (pu.hlat_w);
	    XtUnmanageChild (pu.hlatl_w);
	    XtUnmanageChild (pu.eadst_w);
	    XtUnmanageChild (pu.eadstl_w);
	    XtManageChild (pu.sndst_w);
	    XtManageChild (pu.sndstl_w);
	    XtUnmanageChild (pu.elong_w);
	    XtUnmanageChild (pu.elongl_w);
	    set_something (pu.pu_w, XmNnumColumns, (char *)7);
	} else {
	    f_string (pu.name_w, op->o_name);
	    if (is_planet (op, SUN)) {
		XtUnmanageChild (pu.ud_w);
		XtUnmanageChild (pu.udl_w);
		XtUnmanageChild (pu.ut_w);
		XtUnmanageChild (pu.utl_w);
		XtUnmanageChild (pu.ra_w);
		XtUnmanageChild (pu.ral_w);
		XtUnmanageChild (pu.dec_w);
		XtUnmanageChild (pu.decl_w);
		XtUnmanageChild (pu.hlong_w);
		XtUnmanageChild (pu.hlongl_w);
		XtUnmanageChild (pu.hlat_w);
		XtUnmanageChild (pu.hlatl_w);
		XtUnmanageChild (pu.eadst_w);
		XtUnmanageChild (pu.eadstl_w);
		XtUnmanageChild (pu.sndst_w);
		XtUnmanageChild (pu.sndstl_w);
		XtUnmanageChild (pu.elong_w);
		XtUnmanageChild (pu.elongl_w);
		set_something (pu.pu_w, XmNnumColumns, (char *)1);
	    } else {
		XtManageChild (pu.ud_w);
		XtManageChild (pu.udl_w);
		XtManageChild (pu.ut_w);
		XtManageChild (pu.utl_w);
		XtManageChild (pu.ra_w);
		XtManageChild (pu.ral_w);
		set_xmstring (pu.ral_w, XmNlabelString, "RA:");
		XtManageChild (pu.dec_w);
		XtManageChild (pu.decl_w);
		set_xmstring (pu.decl_w, XmNlabelString, "Dec:");
		XtManageChild (pu.hlong_w);
		XtManageChild (pu.hlongl_w);
		XtManageChild (pu.hlat_w);
		XtManageChild (pu.hlatl_w);
		XtManageChild (pu.eadst_w);
		XtManageChild (pu.eadstl_w);
		XtManageChild (pu.sndst_w);
		XtManageChild (pu.sndstl_w);
		XtManageChild (pu.elong_w);
		XtManageChild (pu.elongl_w);
		set_something (pu.pu_w, XmNnumColumns, (char *)10);
	    }
	}

	f_date (pu.ud_w, lp->smjd);
	f_time (pu.ut_w, mjd_hr(lp->smjd));
	f_ra (pu.ra_w, op->s_ra);
	f_pangle (pu.dec_w, op->s_dec);
	f_pangle (pu.hlong_w, op->s_hlong);
	f_pangle (pu.hlat_w, op->s_hlat);

	d = op->s_edist;
	f_double (pu.eadst_w, d >= 9.99995 ? "%6.3f" : "%6.4f", d);

	d = is_planet(op, MOON) ? op->s_edist
				: op->s_sdist;
	f_double (pu.sndst_w, d >= 9.99995 ? "%6.3f" : "%6.4f", d);

	f_double (pu.elong_w, "%6.1f", op->s_elong);

	XmMenuPosition (pu.pu_w, (XButtonPressedEvent *)ev);
	XtManageChild (pu.pu_w);
}

/* create the id popup */
static void
ss_create_popup()
{
	Arg args[20];
	Widget w;
	int n;

	/* create the outer form */
	n = 0;
	XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
	XtSetArg (args[n], XmNpacking, XmPACK_COLUMN); n++;
	XtSetArg (args[n], XmNisAligned, False); n++;
	pu.pu_w = XmCreatePopupMenu (toplevel_w, "SSPopup", args, n);

	/* create the widgets */

	/* name */
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	w = XmCreateLabel (pu.pu_w, "SSPopNameL", args, n);
	set_xmstring (w, XmNlabelString, "Name:");
	XtManageChild (w);
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
	pu.name_w = XmCreateLabel (pu.pu_w, "SSPopName", args, n);
	XtManageChild (pu.name_w);

	/* UT date */
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	pu.udl_w = XmCreateLabel (pu.pu_w, "SSPopupUTDateL", args, n);
	set_xmstring (pu.udl_w, XmNlabelString, "UT Date:");
	XtManageChild (pu.udl_w);
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
	pu.ud_w = XmCreateLabel (pu.pu_w, "SSPopUTDate", args, n);
	XtManageChild (pu.ud_w);

	/* UT time */
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	pu.utl_w = XmCreateLabel (pu.pu_w, "SSPopupUTTimeL", args, n);
	set_xmstring (pu.utl_w, XmNlabelString, "UT Time:");
	XtManageChild (pu.utl_w);
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
	pu.ut_w = XmCreateLabel (pu.pu_w, "SSPopUTTime", args, n);
	XtManageChild (pu.ut_w);

	/* ra */
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	pu.ral_w = XmCreateLabel (pu.pu_w, "SSPopRAL", args, n);
	XtManageChild (pu.ral_w);
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
	pu.ra_w = XmCreateLabel (pu.pu_w, "SSPopRA", args, n);
	XtManageChild (pu.ra_w);

	/* dec */
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	pu.decl_w = XmCreateLabel (pu.pu_w, "SSPopDecL", args, n);
	XtManageChild (pu.decl_w);
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
	pu.dec_w = XmCreateLabel (pu.pu_w, "SSPopDec", args, n);
	XtManageChild (pu.dec_w);

	/* hlong */
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	pu.hlongl_w = XmCreateLabel (pu.pu_w, "SSPopupHLongL", args, n);
	set_xmstring (pu.hlongl_w, XmNlabelString, "HeLong:");
	XtManageChild (pu.hlongl_w);
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
	pu.hlong_w = XmCreateLabel (pu.pu_w, "SSPopHLong", args, n);
	XtManageChild (pu.hlong_w);

	/* hlat */
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	pu.hlatl_w = XmCreateLabel (pu.pu_w, "SSPopupHLatL", args, n);
	set_xmstring (pu.hlatl_w, XmNlabelString, "HeLat:");
	XtManageChild (pu.hlatl_w);
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
	pu.hlat_w = XmCreateLabel (pu.pu_w, "SSPopHLat", args, n);
	XtManageChild (pu.hlat_w);

	/* earth dist */
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	pu.eadstl_w = XmCreateLabel (pu.pu_w, "SSPopupEaDstL", args, n);
	set_xmstring (pu.eadstl_w, XmNlabelString, "EaDst:");
	XtManageChild (pu.eadstl_w);
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
	pu.eadst_w = XmCreateLabel (pu.pu_w, "SSPopEaDst", args, n);
	XtManageChild (pu.eadst_w);

	/* sun dist */
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	pu.sndstl_w = XmCreateLabel (pu.pu_w, "SSPopupSnDstL", args, n);
	set_xmstring (pu.sndstl_w, XmNlabelString, "SnDst:");
	XtManageChild (pu.sndstl_w);
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
	pu.sndst_w = XmCreateLabel (pu.pu_w, "SSPopSnDst", args, n);
	XtManageChild (pu.sndst_w);

	/* elong */
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	pu.elongl_w = XmCreateLabel (pu.pu_w, "SSPopupElongL", args, n);
	set_xmstring (pu.elongl_w, XmNlabelString, "Elong:");
	XtManageChild (pu.elongl_w);
	n = 0;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
	pu.elong_w = XmCreateLabel (pu.pu_w, "SSPopElong", args, n);
	XtManageChild (pu.elong_w);
}

/* sort by time.
 */
static int
cmpHLoc (lp1, lp2)
Const void *lp1, *lp2;
{
	double dt = ((HLoc *)lp1)->smjd - ((HLoc *)lp2)->smjd;
	return (dt < 0 ? -1 : dt > 0 ? 1 : 0);
}

/* draw everything in the points arrays from the current vantage to the
 * current screen size.
 * always draw the main window; also the stereo one if it's up.
 */
/* ARGSUSED */
static void
ss_all(preclr)
int preclr;
{
#define	CACHE_SZ	100	/* collect these many X commands */
#define	CACHE_PAD	10	/* flush when only this many left */
#define	CACHE_HWM	(CACHE_SZ - CACHE_PAD)	/* hi water mark */
	static Pixmap ss_pm;
	static Pixmap st_pm;
	static GC ss_bgc;
	static GC st_bgc;
	static XFontStruct *ss_fs;
	static XFontStruct *st_fs;
	static unsigned ss_last_nx, ss_last_ny;
	static unsigned st_last_nx, st_last_ny;
	static int ss_cw;
	static int st_cw;
	Display *dsp = XtDisplay(ssda_w);
	Window ss_win = XtWindow(ssda_w);
	Window st_win = XtWindow(stda_w);
	Window root;
	unsigned int ss_nx, ss_ny;
	unsigned int st_nx, st_ny;
	int x, y;
	unsigned int bw, d;
	int dbidx;
	int sv;		/* ScaleValue tmp */
	double scale;		/* pixels per au */
	double elt, selt, celt;	/* heliocentric lat of eye, rads */
	double elg, selg, celg;	/* heliocentric lng of eye, rads */
	int n;

	if (!ss_bgc) {
	    XGCValues gcv;
	    unsigned int gcm;
	    Pixel p;

	    gcm = GCForeground;
	    get_something (ssda_w, XmNbackground, (char *)&p);
	    gcv.foreground = p;
	    ss_bgc = XCreateGC (dsp, ss_win, gcm, &gcv);
	    ss_fs = XQueryFont (dsp, XGContextFromGC (ss_bgc));
	    ss_cw = ss_fs->max_bounds.width;
	}

	if (!st_bgc && stereo) {
	    XGCValues gcv;
	    unsigned int gcm;
	    Pixel p;

	    gcm = GCForeground;
	    get_something (stda_w, XmNbackground, (char *)&p);
	    gcv.foreground = p;
	    st_bgc = XCreateGC (dsp, st_win, gcm, &gcv);
	    st_fs = XQueryFont (dsp, XGContextFromGC (st_bgc));
	    st_cw = st_fs->max_bounds.width;
	}

	XGetGeometry(dsp, ss_win, &root, &x, &y, &ss_nx, &ss_ny, &bw, &d);
	if (!ss_pm || ss_nx != ss_last_nx || ss_ny != ss_last_ny) {
	    if (ss_pm)
		XFreePixmap (dsp, ss_pm);
	    ss_pm = XCreatePixmap (dsp, ss_win, ss_nx, ss_ny, d);
	    ss_last_nx = ss_nx;
	    ss_last_ny = ss_ny;
	}

	if (stereo) {
	    XGetGeometry(dsp, st_win, &root, &x, &y, &st_nx, &st_ny, &bw, &d);
	    if (!st_pm || st_nx != st_last_nx || st_ny != st_last_ny) {
		if (st_pm)
		    XFreePixmap (dsp, st_pm);
		st_pm = XCreatePixmap (dsp, st_win, st_nx, st_ny, d);
		st_last_nx = st_nx;
		st_last_ny = st_ny;
	    }
	}

	XFillRectangle (dsp, ss_pm, ss_bgc, 0, 0, ss_nx, ss_ny);
	if (stereo)
	    XFillRectangle (dsp, st_pm, st_bgc, 0, 0, st_nx, st_ny);

	XmScaleGetValue (hr_w, &sv);
	scale = MINMAG * pow (MAXMAG/MINMAG, sv/100.);
	XmScaleGetValue (hlat_w, &sv);
	elt = degrad(sv);
	selt = sin(elt);
	celt = cos(elt);
	XmScaleGetValue (hlng_w, &sv);
	elg = degrad(sv);
	selg = sin(elg);
	celg = cos(elg);

	/* display each point and optionally connect with line segments. */
	for (dbidx = 0; dbidx < NOBJ; dbidx++) {
	    XPoint ss_xpoints[CACHE_SZ], *ss_xp = ss_xpoints; /* for planets */
	    XPoint st_xpoints[CACHE_SZ], *st_xp = st_xpoints;
	    XPoint ss_xlines[CACHE_SZ], *ss_xl = ss_xlines;   /* for orbits */
	    XPoint st_xlines[CACHE_SZ], *st_xl = st_xlines;
	    GC gc;
	    obj_pickgc (db_basic(dbidx), ssda_w, &gc); /*moonColor for Earth?*/
	    if (npoints[dbidx] > 0 && obj_on[dbidx]) {
		HLoc *flp, *lp;
		flp = points[dbidx];
		for (lp = flp; lp < flp + npoints[dbidx]; lp++) {
		    HLoc leghl;/* for projection of planet on ecliptic plane*/
		    solar_system(lp, scale, selt, celt, selg, celg,ss_nx,ss_ny);
		    ss_xp->x = lp->sx;		ss_xp->y = lp->sy;	ss_xp++;
		    if (bigdots) {
			ss_xp->x = lp->sx+1;	ss_xp->y = lp->sy;	ss_xp++;
			ss_xp->x = lp->sx;	ss_xp->y = lp->sy+1;	ss_xp++;
			ss_xp->x = lp->sx+1;	ss_xp->y = lp->sy+1;	ss_xp++;
		    }
		    if (connectdots) {
			ss_xl->x = lp->sx;
			ss_xl->y = lp->sy;
			ss_xl++;
		    }
		    if (legs) {
			leghl = *lp;
			leghl.z = 0;
			solar_system(&leghl, scale, selt, celt, selg, celg,
								ss_nx, ss_ny);
			XDrawLine(dsp,ss_pm,gc,lp->sx,lp->sy,leghl.sx,leghl.sy);
		    }
		    if (stereo) {
			st_xp->x = lp->stx;	st_xp->y = lp->sy;	st_xp++;
			if (bigdots) {
			    st_xp->x = lp->stx+1; st_xp->y = lp->sy;	st_xp++;
			    st_xp->x = lp->stx;   st_xp->y = lp->sy+1;	st_xp++;
			    st_xp->x = lp->stx+1; st_xp->y = lp->sy+1;	st_xp++;
			}
			if (connectdots) {
			    st_xl->x = lp->stx;
			    st_xl->y = lp->sy;
			    st_xl++;
			}
			if (legs) {
			    /* N.B. we assume leghl is already computed */
			    XDrawLine(dsp, st_pm, gc, lp->stx, lp->sy,
							leghl.stx, leghl.sy);
			}
		    }
		    if (dbidx == SUN) {
			XDrawArc (dsp, ss_pm, gc, lp->sx-3, lp->sy-3,
							    7, 7, 0, 64*360);
			if (stereo)
			    XDrawArc (dsp, st_pm, gc, lp->stx-3, lp->sy-3,
							    7, 7, 0, 64*360);
		    }
		    if ((n = ss_xp - ss_xpoints) >= CACHE_HWM) {
			XDrawPoints (dsp,ss_pm,gc,ss_xpoints,n,CoordModeOrigin);
			ss_xp = ss_xpoints;
		    }
		    if ((n = st_xp - st_xpoints) >= CACHE_HWM) {
			XDrawPoints (dsp,st_pm,gc,st_xpoints,n,CoordModeOrigin);
			st_xp = st_xpoints;
		    }
		    if ((n = ss_xl - ss_xlines) >= CACHE_HWM) {
			XDrawLines (dsp,ss_pm,gc,ss_xlines,n,CoordModeOrigin);
			ss_xlines[0] = ss_xlines[n-1];	/* last is now first */
			ss_xl = ss_xlines + 1;		/* already one */
		    }
		    if ((n = st_xl - st_xlines) >= CACHE_HWM) {
			XDrawLines (dsp,st_pm,gc,st_xlines,n,CoordModeOrigin);
			st_xlines[0] = st_xlines[n-1];	/* last is now first */
			st_xl = st_xlines + 1;		/* already one */
		    }
		}
		if (nametags) {
		    char *name;
		    if (dbidx == MOON)
			name = earthname;
		    else {
			Obj *op = db_basic (dbidx);
			name = op->o_name;
		    }
		    XDrawString (dsp, ss_pm, gc, flp->sx+ss_cw, flp->sy,
						    name, strlen(name));
		    if (stereo)
			XDrawString (dsp, st_pm, gc, flp->stx+st_cw, flp->sy,
						    name, strlen(name));
		}
	    }

	    /* once more to pick up any extra */

	    if ((n = ss_xp - ss_xpoints) > 0)
		XDrawPoints (dsp, ss_pm, gc, ss_xpoints, n, CoordModeOrigin);
	    if ((n = st_xp - st_xpoints) > 0)
		XDrawPoints (dsp, st_pm, gc, st_xpoints, n, CoordModeOrigin);
	    if ((n = ss_xl - ss_xlines) > 1)
		XDrawLines (dsp, ss_pm, gc, ss_xlines, n, CoordModeOrigin);
	    if ((n = st_xl - st_xlines) > 1)
		XDrawLines (dsp, st_pm, gc, st_xlines, n, CoordModeOrigin);
	}

	/* draw the ecliptic plane, if desired */
	if (ecliptic) {
	    static GC egc;
	    double minau, maxau;
	    int dir, asc, des;
	    XCharStruct all;
	    double aus[NECLPT+2];
	    char spacing[64];
	    int len;
	    int nau, i;

	    if (!egc) {
		XGCValues gcv;
		unsigned int gcm;
		Pixel p;

		gcm = GCForeground;
		get_something (ssda_w, XmNforeground, (char *)&p);
		gcv.foreground = p;
		egc = XCreateGC (dsp, ss_win, gcm, &gcv);
	    }

	    minau = 0;
	    maxau = ss_nx > ss_ny ? ss_nx/2.0/scale : ss_ny/2.0/scale;
	    nau = tickmarks (minau, maxau, NECLPT, aus);

	    /* draw tick mark spacing message in upper left corner */
	    (void) sprintf (spacing, "%g AU", aus[1] - aus[0]);
	    len = strlen (spacing);
	    XTextExtents (ss_fs, spacing, len, &dir, &asc, &des, &all);
	    XDrawString (dsp, ss_pm, egc, 0, asc, spacing, strlen(spacing));

	    /* draw each grid line.
	     * the main view are simple elipses; the stereo view is
	     * polylines drawn in world coords.
	     */
	    for (i = 0; i < nau; i++) {
		int arcx, arcy, arcw, arch;
		double au = aus[i];
		HLoc hhl, whl;		/* width and height */

		if (au <= 0.0)
		    continue;
		whl.x = au*celg;	/* a point to the right */
		whl.y = au*selg;
		whl.z = 0.0;
		solar_system(&whl, scale, selt, celt, selg, celg, ss_nx, ss_ny);
		hhl.x = -au*selg;	/* a point up */
		hhl.y = selt < 0 ? -au*celg : au*celg;
		hhl.z = 0.0;
		solar_system(&hhl, scale, selt, celt, selg, celg, ss_nx, ss_ny);

		arcx = ss_nx - whl.sx;
		arcy = hhl.sy;
		arcw = 2*whl.sx - ss_nx - 1;
		arch = ss_ny - 2*hhl.sy - 1;
		if (arcw*arcw + arch*arch > ss_nx*ss_nx + ss_ny*ss_ny)
		    continue;	/* avoid clipping -- some servers are slow */
		if (arch <= 0)
		    arch = 1;	/* avoid pushing our luck with XDrawArc */
		XDrawArc (dsp, ss_pm, egc, arcx, arcy, arcw, arch, 0,360*64);

		if (stereo) {
		    XPoint xps[NECLSEG+1];
		    int i;

		    for (i = 0; i < NECLSEG; i++) {
			double hlng = i*2*PI/NECLSEG;
			HLoc hloc;

			hloc.x = au*cos(hlng);
			hloc.y = au*sin(hlng);
			hloc.z = 0.0;
			solar_system(&hloc, scale, selt, celt, selg, celg,
								ss_nx, ss_ny);
			xps[i].x = hloc.stx;
			xps[i].y = hloc.sy;
		    }
		    xps[NECLSEG] = xps[0];	/* close */
		    XDrawLines (dsp, st_pm, egc, xps,NECLSEG+1,CoordModeOrigin);
		}
	    }
	}

	XCopyArea (dsp, ss_pm, ss_win, ss_bgc, 0, 0, ss_nx, ss_ny, 0, 0);

	if (stereo)
	    XCopyArea (dsp, st_pm, st_win, st_bgc, 0, 0, st_nx, st_ny, 0, 0);
}

/* compute location of HLoc in window of size [nx,ny].
 * N.B. others assume we only use lp->{x,y,z} and set lp->{sx,sy,sty}
 */
static void
solar_system(lp, scale, selt, celt, selg, celg, nx, ny)
HLoc *lp;
double scale;		/* mag factor */
double selt, celt;	/* sin/cos heliocentric lat of eye, rads */
double selg, celg;	/* sin/cos heliocentric lng of eye, rads */
unsigned nx, ny;	/* size of drawing area, in pixels */
{
	double x, y, z;	/* progressive transform values... */
	double xp, yp, zp;
	double xpp, ypp, zpp;
	double back;

	/* initial loc of points[i] */
	x = lp->x;
	y = lp->y;
	z = lp->z;

	/* rotate by -elg about z axis to get to xz plane.
	 * once we rotate up about x to the z axis (next step) that will put
	 * +x to the right and +y up.
	 * tmp = -elg;
	 * xp = x*cos(tmp) - y*sin(tmp);
	 * yp = x*sin(tmp) + y*cos(tmp);
	 */
	xp =  x*celg + y*selg;
	yp = -x*selg + y*celg;
	zp = z;

	/* rotate by -(PI/2-elt) about x axis to get to z axis.
	 * +x right, +y up, +z towards, all in AU.
	 * tmp = -(PI/2-elt);
	 * ypp = yp*cos(tmp) - zp*sin(tmp);
	 * zpp = yp*sin(tmp) + zp*cos(tmp);
	 */
	xpp = xp;
	ypp =  yp*selt + zp*celt;
	zpp = -yp*celt + zp*selt;

	/* now, straight ortho projection */
	lp->sx = nx/2 + xpp*scale;
	lp->sy = ny/2 - ypp*scale;

	/* back is y coord, in AU, behind which there is no parallax.
	 * parallax is the offset of the sun (at a=0), in pixels.
	 */
	back = (nx > ny ? nx : ny)/-2.0/scale;  /* based on screen size */
	if (zpp < back)
	    lp->stx = lp->sx;
	else
	    lp->stx = lp->sx + parallax*(back-zpp)/back;
}
