/*
 * Screen dimmer module for xmss.
 */

#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "xmss-module.h"
#include "colormap.h"

static XrmOptionDescRec dim_options[] =
{
    "-dimsteps",	".steps",	 XrmoptionSepArg, (caddr_t) 0,
    "-ds",		".steps",	 XrmoptionSepArg, (caddr_t) 0,
    "-dimpercent",	".percent",	 XrmoptionSepArg, (caddr_t) 0,
    "-dp",		".percent",	 XrmoptionSepArg, (caddr_t) 0,
    "-dimgrabs",	".serverGrabs",	 XrmoptionNoArg,  (caddr_t) "True",
    "+dimgrabs",	".serverGrabs",	 XrmoptionNoArg,  (caddr_t) "False",
};

extern Module dim_module;

static int fraction, dir, steps, cur_steps, op_steps, blocked, grabs, grabbing;
static XColor *Initial, *Final;
static double *Current, *Step;
static _Xconst Module *blank;

/*
 * Initialization:  check visual and colormap for compatibility, disqualifying
 * ourself if unusable, and set up for dimming.  Return True if we successfully
 * initialize, or False to disable our use.
 */

static int
#if NeedFunctionPrototypes
dim_init(int reinit)
#else
dim_init(reinit)
    int reinit;
#endif
{
    if (!(fraction = getNumRes("dim.percent", SCLASS, ".percent")))
	fraction = 70;
    if (!(steps = getNumRes("dim.steps", SCLASS, ".Steps")))
	steps = 46;
    grabs = getBoolRes("dim.serverGrabs", SCLASS, ".ServerGrabs");
    if (reinit)
	return True;
    if (!colormap_init())
	return False;
    if (!(Initial = malloc(vis->colormap_size * sizeof *Initial)))
	return False;
    if (!(Final = malloc(vis->colormap_size * sizeof *Final)))
    {
	free(Initial);
	return False;
    }
    if (!(Current = malloc(3 * vis->colormap_size * sizeof *Current)))
    {
	free(Initial);
	free(Final);
	return False;
    }
    if (!(Step = malloc(3 * vis->colormap_size * sizeof *Step)))
    {
	free(Initial);
	free(Final);
	free(Current);
	return False;
    }
    dir = 0;
    return True;
}

/*
 * Post-initialization:  croak if the blank module doesn't exist.
 */

static int
#if NeedFunctionPrototypes
dim_finit(int reinit)
#else
dim_finit(reinit)
    int reinit;
#endif
{
    return (blank = module_select("blank")) != 0;
}

/*
 * Start the dimming operation.  Returns the interaction delay, which in our
 * case is as short as possible until we finish dimming/undimming.
 */

static long
#if NeedFunctionPrototypes
dim_on(void)
#else
dim_on()
#endif
{
    Colormap *CMs;
    int z;

    CMs = XListInstalledColormaps(dpy, root, &z);
    if ((blocked = (z != 1)))
	return (*blank->startup)();
    for (z = vis->colormap_size; z--; )
	Colors[z].pixel = z;
    XQueryColors(dpy, CMs[0], Colors, vis->colormap_size);
    XFree(CMs);
    /* divide values by "fraction" and install new colors */
    cur_steps = steps;
    op_steps = steps;
    for (z = vis->colormap_size; z--; )
    {
	Initial[z] = Final[z] = Colors[z];
	Final[z].red *= (100.0 - fraction) / 100.0;
	Final[z].green *= (100.0 - fraction) / 100.0;
	Final[z].blue *= (100.0 - fraction) / 100.0;
	Current[z * 3] = Colors[z].red;
	Current[z * 3 + 1] = Colors[z].green;
	Current[z * 3 + 2] = Colors[z].blue;
	Step[z * 3] = (Final[z].red - Initial[z].red) / (double) steps;
	Step[z * 3 + 1] = (Final[z].green - Initial[z].green) / (double) steps;
	Step[z * 3 + 2] = (Final[z].blue - Initial[z].blue) / (double) steps;
    }
    if ((grabbing = grabs))
	XGrabServer(dpy);
    XStoreColors(dpy, blankCM, Colors, vis->colormap_size);
    XInstallColormap(dpy, blankCM);
    dir = 1;
    return 0;
}

/*
 * Begin undimming.  Return the interaction delay, which again is as short as
 * possible.
 */

static long
#if NeedFunctionPrototypes
dim_off(void)
#else
dim_off()
#endif
{
    XColor *tmpc;
    int z;

    if (blocked)
	return (*blank->shutdown)();
    tmpc = Initial;
    Initial = Final;
    Final = tmpc;
    for (z = vis->colormap_size * 3; z--; )
	Step[z] = - Step[z];
    dir = -1;
    cur_steps = op_steps - cur_steps;
    return 0;
}

/*
 * The work procedure is called on every pass through xmss's interaction loop.
 * In our case, its purpose is to blank or unblank the screen in steps.  It
 * returns the new delay, or -1 to return to the default.
 */

static long
#if NeedFunctionPrototypes
dim_work(int blanked)
#else
dim_work(blanked)
    int blanked;
#endif
{
    long delay;
    int nx, z;

    if (blocked)
	return (*blank->work)(blanked);
    if (!dir)
	return -1;
    delay = 0;
    nx = 0;
    for (z = vis->colormap_size; z--; )
    {
	Colors[z].red = (Current[z * 3] += Step[z * 3]) + 0.5;
	Colors[z].green = (Current[z * 3 + 1] += Step[z * 3 + 1]) + 0.5;
	Colors[z].blue = (Current[z * 3 + 2] += Step[z * 3 + 2]) + 0.5;
    }
    XStoreColors(dpy, blankCM, Colors, vis->colormap_size);
    if (!--cur_steps)
    {
	if (dir == -1)
	    XUninstallColormap(dpy, blankCM);
	delay = -1;
	dir = 0;
	if (grabbing)
	    XUngrabServer(dpy);
    }
    return delay;
}

Module dim_module =
{
    "dim",			/* module name */
    dim_options,		/* options */
    sizeof dim_options / sizeof dim_options[0],	/* number of options */
    dim_init,			/* initialization */
    dim_finit,			/* post-initialization */
    dim_on,			/* transition to "on" state */
    dim_off,			/* transition to "off" state */
    dim_work,			/* called on every pass */
};
