/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: action.c,v 2.5 94/10/29 17:30:05 nau Exp $";

/* action routines for output window
 */

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <sys/types.h>

#include "global.h"

#include "action.h"
#include "buffer.h"
#include "change.h"
#include "command.h"
#include "copy.h"
#include "create.h"
#include "crosshair.h"
#include "data.h"
#include "dialog.h"
#include "draw.h"
#include "error.h"
#include "file.h"
#include "fileselect.h"
#include "find.h"
#include "mymem.h"
#include "misc.h"
#include "mirror.h"
#include "move.h"
#include "pinout.h"
#include "polygon.h"
#include "printdialog.h"
#include "remove.h"
#include "rotate.h"
#include "search.h"
#include "select.h"
#include "set.h"
#include "undo.h"

#include <X11/cursorfont.h>

/* ---------------------------------------------------------------------------
 * some local defines
 */
#define	TAN_30_DEGREE		0.577350269		/* tan(30) */
#define	TAN_60_DEGREE		1.732050808		/* tan(60) */
#define	MODE_STACK_DEPTH	20

/* ---------------------------------------------------------------------------
 * some local types
 */
typedef enum {
	F_AddSelected,
	F_All,
	F_AllConnections,
	F_AllUnusedPins,
	F_Block,
	F_CanonicalName,
	F_Center,
	F_Clear,
	F_ClearAndRedraw,
	F_ClearList,
	F_Close,
	F_Connection,
	F_Copy,
	F_ElementConnections,
	F_ElementToBuffer,
	F_Find,
	F_Grid,
	F_InsertPoint,
	F_Layer,
	F_Layout,
	F_LayoutAs,
	F_LayoutToBuffer,
	F_Line,
	F_LineSize,
	F_Mirror,
	F_Move,
	F_NameOnPCB,
	F_None,
	F_Notify,
	F_Object,
	F_Package,
	F_PasteBuffer,
	F_Pin,
	F_Pinout,
	F_Polygon,
	F_PostScript,
	F_PreviousPoint,
	F_Rectangle,
	F_Redraw,
	F_Remove,
	F_RemovePoint,
	F_Reset,
	F_ResetLinesAndPolygons,
	F_ResetPinsAndVias,
	F_Restore,
	F_Rotate,
	F_Save,
	F_SelectedLines,
	F_SelectedObjects,
	F_SelectedPins,
	F_SelectedVias,
	F_Text,
	F_ToggleAllDirections,
	F_ToggleGrid,
	F_ToggleObject,
	F_Via,
	F_ViaDrillingHole,
	F_ViaSize,
	F_Zoom
} FunctionID;

typedef struct					/* used to identify subfunctions */
{
	char			*Identifier;
	FunctionID		ID;
} FunctionType, *FunctionTypePtr;

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	Boolean					IgnoreMotionEvents = 0;
static	FunctionType			Functions[] = {
	{ "AddSelected", F_AddSelected },
	{ "All", F_All },
	{ "AllConnections", F_AllConnections },
	{ "AllUnusedPins", F_AllUnusedPins },
	{ "Block", F_Block },
	{ "CanonicalName", F_CanonicalName },
	{ "Center", F_Center },
	{ "Clear", F_Clear},
	{ "ClearAndRedraw", F_ClearAndRedraw},
	{ "ClearList", F_ClearList },
	{ "Close", F_Close },
	{ "Connection", F_Connection },
	{ "Copy", F_Copy },
	{ "ElementConnections", F_ElementConnections },
	{ "ElementToBuffer", F_ElementToBuffer },
	{ "Find", F_Find },
	{ "Grid", F_Grid },
	{ "InsertPoint", F_InsertPoint },
	{ "Layer", F_Layer },
	{ "Layout", F_Layout },
	{ "LayoutAs", F_LayoutAs },
	{ "LayoutToBuffer", F_LayoutToBuffer },
	{ "Line", F_Line },
	{ "LineSize", F_LineSize },
	{ "Mirror", F_Mirror },
	{ "Move", F_Move },
	{ "NameOnPCB", F_NameOnPCB },
	{ "None", F_None },
	{ "Notify", F_Notify },
	{ "Object", F_Object },
	{ "Package", F_Package },
	{ "PasteBuffer", F_PasteBuffer },
	{ "Pin", F_Pin },
	{ "Pinout", F_Pinout },
	{ "Polygon", F_Polygon },
	{ "PostScript", F_PostScript },
	{ "PreviousPoint", F_PreviousPoint},
	{ "Rectangle", F_Rectangle },
	{ "Redraw", F_Redraw },
	{ "Remove", F_Remove },
	{ "RemovePoint", F_RemovePoint },
	{ "Reset", F_Reset },
	{ "ResetLinesAndPolygons", F_ResetLinesAndPolygons },
	{ "ResetPinsAndVias", F_ResetPinsAndVias },
	{ "Restore", F_Restore },
	{ "Rotate", F_Rotate },
	{ "Save", F_Save },
	{ "SelectedLines", F_SelectedLines },
	{ "SelectedObjects", F_SelectedObjects },
	{ "SelectedPins", F_SelectedPins },
	{ "SelectedVias", F_SelectedVias },
	{ "Text", F_Text },
	{ "ToggleAllDirections", F_ToggleAllDirections },
	{ "ToggleGrid", F_ToggleGrid },
	{ "ToggleObject", F_ToggleObject },
	{ "Via", F_Via },
	{ "ViaSize", F_ViaSize },
	{ "ViaDrillingHole", F_ViaDrillingHole },
	{ "Zoom", F_Zoom }};

/* ---------------------------------------------------------------------------
 * some local routines
 */
static	Boolean	PointerPosition(Drawable, int *, int *);
static	int		GetFunctionID(String);
static	void	AdjustAttachedLine(void);
static	void	AdjustAttachedBox(void);
static	void	AdjustAttachedObjects(void);
static	void	NotifyLine(void);
static	void	NotifyBlock(void);
static	void	NotifyMode(void);

/* ---------------------------------------------------------------------------
 * returns the pointer position relativ to the specified window 
 * funtion returns FALSE if everything's OK
 */
static Boolean PointerPosition(Drawable UseWindow, int *X, int *Y)
{
	int				root_x, root_y;
	unsigned int	mask;
	Window			root_window, child_window;

	return(!(XQueryPointer(Dpy, UseWindow, &root_window, &child_window,
		&root_x, &root_y, X, Y, &mask)));
}

/* ---------------------------------------------------------------------------
 * get function ID of passed string
 */
static int GetFunctionID(String Ident)
{
	int		i;

	i = ENTRIES(Functions);
	while (i)
		if (!strcmp(Ident, Functions[--i].Identifier))
			return((int) Functions[i].ID);
	return(-1);
}

/* ---------------------------------------------------------------------------
 * makes the 'marked line' fit into a 45 degree direction
 *
 * directions:
 *
 *           0
 *          7 1
 *         6   2
 *          5 3
 *           4
 */
static void AdjustAttachedLine(void)
{
	Position	dx, dy,
				min;
	BYTE		direction = 0;
	float		m;

		/* I need at least one point */
	if (Crosshair.AttachedLine.State == STATE_FIRST)
		return;

		/* no 45 degree lines required */
	if (TEST_FLAG(ALLDIRCETIONFLAG, PCB))
	{
		Crosshair.AttachedLine.X2 = Crosshair.X;
		Crosshair.AttachedLine.Y2 = Crosshair.Y;
		return;
	}

		/* first calculate direction of line */
	dx = Crosshair.X -Crosshair.AttachedLine.X1;
	dy = Crosshair.Y -Crosshair.AttachedLine.Y1;
	if (!dx)
	{
		if (!dy)
				/* zero length line, don't draw anything */
			return;
		else
			direction = dy > 0 ? 0 : 4;
	}
	else
	{
		m = (float) dy / (float) dx;
		direction = 2;
		if (m > TAN_30_DEGREE)
			direction = m > TAN_60_DEGREE ? 0 : 1;
		else
			if (m < -TAN_30_DEGREE)
				direction = m < -TAN_60_DEGREE ? 0 : 3;
	}
	if (dx < 0)
		direction += 4;

	dx = abs(dx);
	dy = abs(dy);
	min = MIN(dx, dy);

		/* now set up the second pair of coordinates */
	switch (direction)
	{
		case 0:
		case 4:
			Crosshair.AttachedLine.X2 = Crosshair.AttachedLine.X1;
			Crosshair.AttachedLine.Y2 = Crosshair.Y;
			break;

		case 2:
		case 6:
			Crosshair.AttachedLine.X2 = Crosshair.X;
			Crosshair.AttachedLine.Y2 = Crosshair.AttachedLine.Y1;
			break;

		case 1:
			Crosshair.AttachedLine.X2 = Crosshair.AttachedLine.X1 +min;
			Crosshair.AttachedLine.Y2 = Crosshair.AttachedLine.Y1 +min;
			break;

		case 3:
			Crosshair.AttachedLine.X2 = Crosshair.AttachedLine.X1 +min;
			Crosshair.AttachedLine.Y2 = Crosshair.AttachedLine.Y1 -min;
			break;

		case 5:
			Crosshair.AttachedLine.X2 = Crosshair.AttachedLine.X1 -min;
			Crosshair.AttachedLine.Y2 = Crosshair.AttachedLine.Y1 -min;
			break;

		case 7:
			Crosshair.AttachedLine.X2 = Crosshair.AttachedLine.X1 -min;
			Crosshair.AttachedLine.Y2 = Crosshair.AttachedLine.Y1 +min;
			break;
	}
}

/* ---------------------------------------------------------------------------
 * set new coordinates if in 'RECTANGLE' mode
 * the cursor shape is also adjusted
 */
static void AdjustAttachedBox(void)
{
	switch(Crosshair.AttachedBox.State)
	{
		case STATE_SECOND:		/* one corner is selected */
		{
			unsigned int	shape;

				/* update coordinates */
			Crosshair.AttachedBox.X2 = Crosshair.X;
			Crosshair.AttachedBox.Y2 = Crosshair.Y;

				/* set pointer shape depending on location relative
				 * to first corner
				 */
			if (Crosshair.Y <= Crosshair.AttachedBox.Y1)
				shape = (Crosshair.X >= Crosshair.AttachedBox.X1) ?
					XC_ur_angle : XC_ul_angle;
			else
				shape = (Crosshair.X >= Crosshair.AttachedBox.X1) ?
					XC_lr_angle : XC_ll_angle;
			if (Output.XCursorShape != shape)
				SetOutputXCursor(shape);
			break;
		}

		default:
				/* just reset the cursor shape if necessary */
			if (Output.XCursorShape != XC_ul_angle)
				SetOutputXCursor(XC_ul_angle);
			break;
	}	
}

/* ---------------------------------------------------------------------------
 * adjusts the objects which are to be created like attached lines...
 */
static void AdjustAttachedObjects(void)
{
	switch(Settings.Mode)
	{
			/* update at least an attached block (selection) */
		case NO_MODE:
			if (Crosshair.AttachedBox.State)
			{
				Crosshair.AttachedBox.X2 = Crosshair.X;
				Crosshair.AttachedBox.Y2 = Crosshair.Y;
			}
			break;

			/* rectangle creation mode */
		case RECTANGLE_MODE:
			AdjustAttachedBox();
			break;

			/* line or polygon creation mode */
		case LINE_MODE:
		case POLYGON_MODE:
			AdjustAttachedLine();
			break;
	}
}
/* ---------------------------------------------------------------------------
 * creates points of a line
 */
static void NotifyLine(void)
{
	switch(Crosshair.AttachedLine.State)
	{
		case STATE_FIRST:			/* first point */
			Crosshair.AttachedLine.State = STATE_SECOND;
			Crosshair.AttachedLine.X1 = Crosshair.AttachedLine.X2 = Crosshair.X;
			Crosshair.AttachedLine.Y1 = Crosshair.AttachedLine.Y2 = Crosshair.Y;
			break;

		default:					/* all following points */
			Crosshair.AttachedLine.State = STATE_THIRD;
			break;
	}
}

/* ---------------------------------------------------------------------------
 * create first or second corner of a marked block
 */
static void NotifyBlock(void)
{
	HideCrosshair(True);
	switch(Crosshair.AttachedBox.State)
	{
		case STATE_FIRST:		/* setup first point */
			Crosshair.AttachedBox.X1= Crosshair.AttachedBox.X2= Crosshair.X;
			Crosshair.AttachedBox.Y1= Crosshair.AttachedBox.Y2= Crosshair.Y;
			Crosshair.AttachedBox.State = STATE_SECOND;
			break;

		case STATE_SECOND:		/* setup second point */
			Crosshair.AttachedBox.State = STATE_THIRD;
			break;
	}
	RestoreCrosshair(True);
}

/* ---------------------------------------------------------------------------
 * does something appropriate for the current mode setting. This normaly
 * means creation of an object at the current crosshair location.
 */
static void NotifyMode(void)
{
	void	*ptr1, *ptr2;
	int		type;

	switch(Settings.Mode)
	{
		case VIA_MODE:
		{
			PinTypePtr	via;

			if ((via = CreateNewVia(PCB->Data, Crosshair.X, Crosshair.Y,
					Settings.ViaThickness, Settings.ViaDrillingHole,
					NULL, VIAFLAG)) != NULL)
			{
				DrawVia(via);
				SetChangedFlag(True);
			}
			break;
		}

		case LINE_MODE:
				/* do update of position */
			NotifyLine();

				/* create line if both ends are determined && length != 0 */
			if (Crosshair.AttachedLine.State == STATE_THIRD &&
				(Crosshair.AttachedLine.X1 != Crosshair.AttachedLine.X2 ||
				Crosshair.AttachedLine.Y1 != Crosshair.AttachedLine.Y2))
			{
				LineTypePtr	line;

				if ((line = CreateNewLineOnLayer(CURRENT,
						Crosshair.AttachedLine.X1, Crosshair.AttachedLine.Y1,
						Crosshair.AttachedLine.X2, Crosshair.AttachedLine.Y2,
						Settings.LineThickness, NOFLAG)) != NULL)
				{
					DrawLine(CURRENT, line);
					SetChangedFlag(True);
				}

					/* copy the coordinates */
				Crosshair.AttachedLine.X1 = Crosshair.AttachedLine.X2;
				Crosshair.AttachedLine.Y1 = Crosshair.AttachedLine.Y2;
			}
			break;

		case RECTANGLE_MODE:
				/* do update of position */
			NotifyBlock();

				/* create rectangle if both corners are determined 
				 * and width, height are != 0
				 */
			if (Crosshair.AttachedBox.State == STATE_THIRD &&
				Crosshair.AttachedBox.X1 != Crosshair.AttachedBox.X2 &&
				Crosshair.AttachedBox.Y1 != Crosshair.AttachedBox.Y2)
			{
				PolygonTypePtr	polygon;

				if ((polygon = CreateNewPolygonFromRectangle(CURRENT, 
						Crosshair.AttachedBox.X1, Crosshair.AttachedBox.Y1,
						Crosshair.AttachedBox.X2, Crosshair.AttachedBox.Y2,
						NOFLAG)) != NULL)
				{
					DrawPolygon(CURRENT, polygon);
					SetChangedFlag(True);
				}

					/* reset state to 'first corner' */
				Crosshair.AttachedBox.State = STATE_FIRST;
			}
			break;

		case TEXT_MODE:
		{
			char	*string;

			if ((string = GetUserInput("Enter text:", "")) != NULL)
			{
				TextTypePtr	text;

				if ((text = CreateNewText(CURRENT, &PCB->Font, Crosshair.X,
						Crosshair.Y, 0, 100, string, NOFLAG)) != NULL)
				{
					DrawText(CURRENT, text);
					SetChangedFlag(True);
				}

					/* free memory allocated by GetUserInput() */
				SaveFree(string);
			}
			break;
		}

		case POLYGON_MODE:
		{
			PolygonPointTypePtr	points = Crosshair.AttachedPolygon.Points;
			Cardinal			n = Crosshair.AttachedPolygon.PointN;
			
				/* do update of position; use the 'LINE_MODE' mechanism */
			NotifyLine();

				/* check if this is the last point of a polygon */
			if (n >= 3 &&
				points->X == Crosshair.AttachedLine.X2 &&
				points->Y == Crosshair.AttachedLine.Y2)
			{
				CopyAttachedPolygonToLayer();
				break;
			}

				/* create new point if it's the first one or if it's
				 * different to the last one
				 */
			if (!n ||
				points[n-1].X != Crosshair.AttachedLine.X2 ||
				points[n-1].Y != Crosshair.AttachedLine.Y2)
			{
				CreateNewPointInPolygon(&Crosshair.AttachedPolygon,
					Crosshair.AttachedLine.X2, Crosshair.AttachedLine.Y2);

					/* copy the coordinates */
				Crosshair.AttachedLine.X1 = Crosshair.AttachedLine.X2;
				Crosshair.AttachedLine.Y1 = Crosshair.AttachedLine.Y2;
			}
			break;
		}

		case PASTEBUFFER_MODE:	
			if (CopyPastebufferToLayout(Crosshair.X, Crosshair.Y))
				SetChangedFlag(True);
			break;

		case REMOVE_MODE:
			if ((type = SearchObjectByPosition(REMOVE_TYPES, &ptr1, &ptr2,
				Crosshair.X, Crosshair.Y)) != NO_TYPE)
			{
				RemoveObject(type, ptr1, ptr2);
				SetChangedFlag(True);
			}
			break;

		case MIRROR_MODE:
			if ((type = SearchObjectByPosition(MIRROR_TYPES, &ptr1, &ptr2,
				Crosshair.X, Crosshair.Y)) != NO_TYPE)
			{
				MirrorObject(type, ptr1, ptr2);
				SetChangedFlag(True);
			}
			break;

		case ROTATE_MODE:
			if ((type = SearchObjectByPosition(ROTATE_TYPES, &ptr1, &ptr2,
				Crosshair.X, Crosshair.Y)) != NO_TYPE)
			{
				RotateObject(type, ptr1, ptr2, Crosshair.X, Crosshair.Y, 1);
				SetChangedFlag(True);
			}
			break;

			/* both are almost the same */
		case COPY_MODE:
		case MOVE_MODE:
			switch(Crosshair.AttachedObject.State)
			{
					/* first notify, lookup object */
				case STATE_FIRST:
				{
					int	types = (Settings.Mode == MOVE_MODE) ?
									MOVE_TYPES : COPY_TYPES;

					Crosshair.AttachedObject.Type= SearchObjectByPosition(types,
						&Crosshair.AttachedObject.Ptr1,
						&Crosshair.AttachedObject.Ptr2,
						Crosshair.X, Crosshair.Y);
					if (Crosshair.AttachedObject.Type != NO_TYPE)
					{
						BoxTypePtr	box;

							/* change to next state and save cursor position */
						Crosshair.AttachedObject.X = Crosshair.X;
						Crosshair.AttachedObject.Y = Crosshair.Y;
						Crosshair.AttachedObject.State = STATE_SECOND;

							/* get bounding box of object an set cursor range */
						box= GetObjectBoundingBox(Crosshair.AttachedObject.Type,
							Crosshair.AttachedObject.Ptr1,
							Crosshair.AttachedObject.Ptr2);
						SetCrosshairRange(Crosshair.X -box->X1,
							Crosshair.Y -box->Y1,
							PCB->MaxWidth -(box->X2 -Crosshair.X),
							PCB->MaxHeight -(box->Y2 -Crosshair.Y));
					}
					break;
				}

					/* second notify, move or copy object */
				case STATE_SECOND:
					if (Settings.Mode == MOVE_MODE)
						MoveObject(Crosshair.AttachedObject.Type,
							Crosshair.AttachedObject.Ptr1,
							Crosshair.AttachedObject.Ptr2,
							Crosshair.X -Crosshair.AttachedObject.X,
							Crosshair.Y -Crosshair.AttachedObject.Y);
					else
						CopyObject(Crosshair.AttachedObject.Type,
							Crosshair.AttachedObject.Ptr1,
							Crosshair.AttachedObject.Ptr2,
							Crosshair.X -Crosshair.AttachedObject.X,
							Crosshair.Y -Crosshair.AttachedObject.Y);
					SetChangedFlag(True);

						/* reset identifiers */
					Crosshair.AttachedObject.Type = NO_TYPE;
					Crosshair.AttachedObject.State = STATE_FIRST;
					break;
			}
			break;
	}
}

/* ---------------------------------------------------------------------------
 * action routine to move to X pointer relative to the current position
 * syntax: MovePointer(deltax,deltay)
 */
void ActionMovePointer(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	int		x, y;

	if (*Num == 2)
	{
		HideCrosshair(False);

		MoveCrosshairRelative(atoi(*Params) *PCB->Grid,
			atoi(*(Params+1)) *PCB->Grid);

			/* now get pointer position and move it */
		PointerPosition(Output.OutputWindow, &x, &y);
		x = TO_SCREEN_X(Crosshair.X) -x;
		y = TO_SCREEN_Y(Crosshair.Y) -y;
		XWarpPointer(Dpy, Output.OutputWindow, None, 0, 0, 0, 0, x, y);

			/* XWarpPointer creates Motion events normally bound to
			 * EventMoveCrosshair.
			 * We don't do any updates when EventMoveCrosshair
			 * is called the next time to prevent from rounding errors
			 */
		IgnoreMotionEvents = True;

			/* update object position and cursor location */
		AdjustAttachedObjects();
		SetCursorStatusLine();
		RestoreCrosshair(False);
	}
}

/* ---------------------------------------------------------------------------
 * !!! no action routine !!!
 *
 * event handler to set the cursor according to the X pointer position
 * called from inside main.c
 */
void EventMoveCrosshair(XMotionEvent *Event)
{
		/* ignore events that are probably caused by ActionMovePointer */
	if (!IgnoreMotionEvents)
	{
		HideCrosshair(False);

			/* correct the values with zoom factor */
		MoveCrosshairAbsolute(TO_PCB_X(Event->x) +PCB->Grid/2,
			TO_PCB_Y(Event->y) +PCB->Grid/2);

			/* update object position and cursor location */
		AdjustAttachedObjects();
		SetCursorStatusLine();
		RestoreCrosshair(False);
	}
	else
		IgnoreMotionEvents = False;
}

/* ---------------------------------------------------------------------------
 * action routine to change the grid, zoom and sizes
 * the first the type of object and the second is passed to
 * the specific routine
 * the value of the second determines also if it is absolute (without a sign)
 * or relative to the current one (with sign + or -)
 * syntax: SetValue(Grid|Zoom|LineSize|ViaDrillingHole|ViaSize, value)
 */
void ActionSetValue(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	Boolean	r;		/* flag for 'relative' value */
	int		value;

	if (*Num == 2)
	{
		HideCrosshair(True);

			/* if the first character is a sign we have to add the
			 * value to the current one
			 */
		r = !isdigit(**(Params+1));
		value = atoi(*(Params+1));
		switch(GetFunctionID(*Params))
		{
			case F_ViaDrillingHole:
				SetViaDrillingHole(r ? value +Settings.ViaDrillingHole : value);
				break;

			case F_Grid:
				SetGrid(r ? value +PCB->Grid : value);
				break;

			case F_Zoom:
				SetZoom(r ? value +PCB->Zoom : value);
				break;

			case F_LineSize:
				SetLineSize(r ? value +Settings.LineThickness : value);
				break;

			case F_ViaSize:
				SetViaSize(r ? value +Settings.ViaThickness : value);
				break;
		}
		RestoreCrosshair(True);
	}
}

/* ---------------------------------------------------------------------------
 * quits user input, the translations of the text widget handle it 
 * syntax: FinishInput(OK|Cancel)
 */
void ActionFinishInputDialog(Widget W, XEvent *Event,
	String *Params, Cardinal *Num)
{
	if (*Num == 1)
		FinishInputDialog(!strcmp("OK", *Params));
}

/* ---------------------------------------------------------------------------
 * quits application
 * syntax: Quit()
 */
void ActionQuit(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (*Num == 0 && (!PCB->Changed || ConfirmDialog("OK to loose data ?")))
		QuitApplication();
}

/* ---------------------------------------------------------------------------
 * searches connections of the pin or via at the cursor position
 * syntax: Connection(Find|ResetLinesAndPolygons|ResetPinsAndVias|Reset)
 */
void ActionConnection(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (*Num == 1)
	{
		HideCrosshair(True);
		switch(GetFunctionID(*Params))
		{
			case F_Find:
			{
				Cursor	oldCursor;

				oldCursor = SetOutputXCursor(XC_watch);
				LookupConnection(Crosshair.X, Crosshair.Y);
				SetOutputXCursor(oldCursor);
				break;
			}

			case F_ResetLinesAndPolygons:
				ResetFoundLinesAndPolygons();
				break;

			case F_ResetPinsAndVias:
				ResetFoundPinsAndVias();
				break;

			case F_Reset:
				ResetFoundPinsAndVias();
				ResetFoundLinesAndPolygons();
				break;
		}
		RestoreCrosshair(True);
	}
}

/* ---------------------------------------------------------------------------
 * starts input of user commands
 * syntax: Command()
 */
void ActionCommand(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
			char	*command;
	static	char	*previous = NULL;

	if (*Num == 0)
	{
		HideCrosshair(True);
		command = GetUserInput("Enter command:", previous ? previous : "");
		if (command != NULL)
		{
				/* copy new comand line to save buffer */
			if (Settings.SaveLastCommand)
			{
				SaveFree(previous);
				previous = MyStrdup(command, "ActionCommand()");
			}
			ExecuteUserCommand(command);
			SaveFree(command);
		}
		RestoreCrosshair(True);
	}
}

/* ---------------------------------------------------------------------------
 * several display related actions
 * syntax: Display(NameOnPCB|CanonicalName)
 *         Display(Center|ClearAndRedraw|Redraw)
 *         Display(Grid|Toggle45Degree|ToggleGrid)
 *         Display(Pinout)
 */
void ActionDisplay(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
			int		id;
	static	String	command[] = { "MovePointer", "0", "0" };

	if (*Num == 1)
	{
		HideCrosshair(True);
		switch(id = GetFunctionID(*Params))
		{
			case F_ClearAndRedraw:
				ReleaseSaveUnderPixmap();
				ClearAndRedrawOutput();
				break;

			case F_Redraw:
				ReleaseSaveUnderPixmap();
				RedrawOutput();
				break;

			case F_Center:
					/* center cursor and move X pointer too */
				CenterDisplay(TO_SCREEN_X(Crosshair.X),
					TO_SCREEN_Y(Crosshair.Y));
				XtCallActionProc(Output.Output, command[0], NULL,
					&command[1], ENTRIES(command) -1);
				break;

			case F_NameOnPCB:
			case F_CanonicalName:
				ReleaseSaveUnderPixmap();
				ELEMENT_LOOP(PCB->Data, EraseElementName(element););
				if (id == F_NameOnPCB)
					CLEAR_FLAG(CANONICALFLAG, PCB);
				else
					SET_FLAG(CANONICALFLAG, PCB);
				ELEMENT_LOOP(PCB->Data, DrawElementName(element););
				break;

			case F_ToggleAllDirections:
				TOGGLE_FLAG(ALLDIRCETIONFLAG, PCB);
				SetStatusLine();
				break;

			case F_ToggleGrid:
				TOGGLE_FLAG(ABSOLUTEFLAG, PCB);
				SetGrid(PCB->Grid);
				break;

			case F_Grid:
				Settings.DrawGrid = !Settings.DrawGrid;
				if (Settings.DrawGrid &&
					TO_SCREEN(PCB->Grid) < MIN_GRID_DISTANCE)
				{
					Settings.DrawGrid = False;
					break;
				}
				ReleaseSaveUnderPixmap();
				DrawGrid();
				break;

			case F_Pinout:
			{
				ElementTypePtr	element;

				if ((SearchObjectByPosition(ELEMENT_TYPE,
						(void **) &element, (void **) &element,
						Crosshair.X, Crosshair.Y)) != NO_TYPE)
					PinoutWindow(Output.Toplevel, element);
				break;
			}
		}
		RestoreCrosshair(True);
	}
}

/* ---------------------------------------------------------------------------
 * action routine to
 *   set a new mode
 *   save the current one or restore the last saved mode
 *   call an appropriate action for the current mode
 * syntax: Mode(Copy|Line|Move|None|PasteBuffer|Polygon)
 *         Mode(Remove|Rectangle|Text|Via)
 *         Mode(Notify)
 *         Mode(Save|Restore)
 */
void ActionMode(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	static	int		position = 0;
	static	int		stack[MODE_STACK_DEPTH];

	if (*Num == 1)
	{
		HideCrosshair(True);
		switch(GetFunctionID(*Params))
		{
			case F_Copy:		SetMode(COPY_MODE); break;
			case F_Line:		SetMode(LINE_MODE); break;
			case F_Mirror:		SetMode(MIRROR_MODE); break;
			case F_Move:		SetMode(MOVE_MODE); break;
			case F_Notify:		NotifyMode(); break;
			case F_PasteBuffer:	SetMode(PASTEBUFFER_MODE); break;
			case F_Polygon:		SetMode(POLYGON_MODE); break;
			case F_Remove:		SetMode(REMOVE_MODE); break;
			case F_Rectangle:	SetMode(RECTANGLE_MODE); break;
			case F_None:		SetMode(NO_MODE); break;
			case F_Rotate:		SetMode(ROTATE_MODE); break;
			case F_Text:		SetMode(TEXT_MODE); break;
			case F_Via:			SetMode(VIA_MODE); break;

			case F_Restore:		/* restore the last saved mode */
				SetMode(position > 0 ? stack[--position] : NO_MODE);
				break;

			case F_Save:		/* save currently selected mode */
				stack[position] = Settings.Mode;
				if (position < MODE_STACK_DEPTH-1)
					position++;
				break;
		}
		RestoreCrosshair(True);
	}
}

/* ---------------------------------------------------------------------------
 * action routine to remove objects
 * syntax: RemoveSeelected()
 */
void ActionRemoveSelected(Widget W, XEvent *Event,
	String *Params, Cardinal *Num)
{
	if (*Num == 0)
	{
		HideCrosshair(True);
		if (RemoveSelected())
			SetChangedFlag(True);
		RestoreCrosshair(True);
	}
}

/* ---------------------------------------------------------------------------
 * changes the size of objects
 * syntax: ChangeSize(Object, delta)
 *         ChangeSize(SelectedLines|SelectedPins|SelectedVias, delta)
 */
void ActionChangeSize(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (*Num == 2)
	{
		HideCrosshair(True);
		switch(GetFunctionID(*Params))
		{
			case F_Object:
			{
				int		type;
				void	*ptr1, *ptr2;

				if ((type = SearchObjectByPosition(CHANGESIZE_TYPES,
					&ptr1, &ptr2, Crosshair.X, Crosshair.Y)) != NO_TYPE)
					if (ChangeObjectSize(type, ptr1, ptr2, atoi(*(Params+1))))
						SetChangedFlag(True);
				break;
			}

			case F_SelectedVias:
				if (ChangeSelectedViaSize(atoi(*(Params+1))))
					SetChangedFlag(True);
				break;

			case F_SelectedPins:
				if (ChangeSelectedPinSize(atoi(*(Params+1))))
					SetChangedFlag(True);
				break;

			case F_SelectedLines:
				if (ChangeSelectedLineSize(atoi(*(Params+1))))
					SetChangedFlag(True);
				break;
		}
		RestoreCrosshair(True);
	}
}

/* ---------------------------------------------------------------------------
 * changes the 2nd size of objects (drilling hole)
 * syntax: Change2ndSize(Object, delta)
 *         Change2ndSize(SelectedPins|SelectedVias, delta)
 */
void ActionChange2ndSize(Widget W, XEvent *Event,
	String *Params, Cardinal *Num)
{
	if (*Num == 2)
	{
		HideCrosshair(True);
		switch(GetFunctionID(*Params))
		{
			case F_Object:
			{
				int		type;
				void	*ptr1, *ptr2;

				if ((type = SearchObjectByPosition(CHANGE2NDSIZE_TYPES,
					&ptr1, &ptr2, Crosshair.X, Crosshair.Y)) != NO_TYPE)
					if (ChangeObject2ndSize(type,ptr1,ptr2, atoi(*(Params+1))))
						SetChangedFlag(True);
				break;
			}

			case F_SelectedVias:
				if (ChangeSelectedVia2ndSize(atoi(*(Params+1))))
					SetChangedFlag(True);
				break;

			case F_SelectedPins:
				if (ChangeSelectedPin2ndSize(atoi(*(Params+1))))
					SetChangedFlag(True);
				break;
		}
		RestoreCrosshair(True);
	}
}

/* ---------------------------------------------------------------------------
 * sets the name of objects
 * syntax: ChangeName(Object)
 *         ChangeName(Layout|Layer)
 */
void ActionChangeName(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	char	*name;

	if (*Num == 1)
	{
		HideCrosshair(True);
		switch(GetFunctionID(*Params))
		{
				/* change the name of an object */
			case F_Object:
			{
				int		type;
				void	*ptr1, *ptr2;

				if ((type = SearchObjectByPosition(CHANGENAME_TYPES,
					&ptr1, &ptr2, Crosshair.X, Crosshair.Y)) != NO_TYPE)
					if (QueryInputAndChangeObjectName(type, ptr1, ptr2))
						SetChangedFlag(True);
				break;
			}

				/* change the layouts name */
			case F_Layout:
				name= GetUserInput("enter the layouts name:", EMPTY(PCB->Name));
				if (name && ChangeLayoutName(name))
					SetChangedFlag(True);
				break;

				/* change the name of the activ layer */
			case F_Layer:
				name = GetUserInput("enter the layers name:",
					EMPTY(CURRENT->Name));
				if (name && ChangeLayerName(CURRENT, name))
					SetChangedFlag(True);
				break;
		}
		RestoreCrosshair(True);
	}
}

/* ---------------------------------------------------------------------------
 * toggles the selection of the object at the pointer location
 * or sets it if 'All', 'Block' or 'Connection' is passed
 * syntax: Select(ToggleObject)
 *         Select(All|Block|Connection)
 */
void ActionSelect(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (*Num == 1)
	{
		HideCrosshair(True);
		switch(GetFunctionID(*Params))
		{
				/* select a single object */
			case F_ToggleObject:
				if (SelectObject())
					SetChangedFlag(True);
				break;

				/* all objects in block */
			case F_Block:
			{
				BoxType	box;

				box.X1 = MIN(Crosshair.AttachedBox.X1,Crosshair.AttachedBox.X2);
				box.Y1 = MIN(Crosshair.AttachedBox.Y1,Crosshair.AttachedBox.Y2);
				box.X2 = MAX(Crosshair.AttachedBox.X1,Crosshair.AttachedBox.X2);
				box.Y2 = MAX(Crosshair.AttachedBox.Y1,Crosshair.AttachedBox.Y2);
				NotifyBlock();
				if (Crosshair.AttachedBox.State == STATE_THIRD &&
					SelectBlock(&box, True))
				{
					SetChangedFlag(True);
					Crosshair.AttachedBox.State = STATE_FIRST;
				}
				break;
			}

				/* select all visible objects */
			case F_All:
			{
				BoxType	box;

				box.X1 = 0;
				box.Y1 = 0;
				box.X2 = PCB->MaxWidth;
				box.Y2 = PCB->MaxHeight;
				if (SelectBlock(&box, True))
					SetChangedFlag(True);
				break;
			}

				/* all found connections */
			case F_Connection:
				if (SelectConnection(True))
					SetChangedFlag(True);
				break;
		}
		RestoreCrosshair(True);
	}
}

/* ---------------------------------------------------------------------------
 * unselects the object at the pointer location
 * syntax: Unselect(All|Block|Connection)
 */
void ActionUnselect(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (*Num == 1)
	{
		HideCrosshair(True);
		switch(GetFunctionID(*Params))
		{
				/* all objects in block */
			case F_Block:
			{
				BoxType	box;

				box.X1 = MIN(Crosshair.AttachedBox.X1,Crosshair.AttachedBox.X2);
				box.Y1 = MIN(Crosshair.AttachedBox.Y1,Crosshair.AttachedBox.Y2);
				box.X2 = MAX(Crosshair.AttachedBox.X1,Crosshair.AttachedBox.X2);
				box.Y2 = MAX(Crosshair.AttachedBox.Y1,Crosshair.AttachedBox.Y2);
				NotifyBlock();
				if (Crosshair.AttachedBox.State == STATE_THIRD &&
					SelectBlock(&box, False))
				{
					SetChangedFlag(True);
					Crosshair.AttachedBox.State = STATE_FIRST;
				}
				break;
			}

				/* unselect all visible objects */
			case F_All:
			{
				BoxType	box;

				box.X1 = 0;
				box.Y1 = 0;
				box.X2 = PCB->MaxWidth;
				box.Y2 = PCB->MaxHeight;
				if (SelectBlock(&box, False))
					SetChangedFlag(True);
				break;
			}

				/* all found connections */
			case F_Connection:
				if (SelectConnection(False))
					SetChangedFlag(True);
				break;
		}
		RestoreCrosshair(True);
	}
}

/* ---------------------------------------------------------------------------
 * saves data to file
 * syntax: Save(Layout|LayoutAs)
 *         Save(AllConnections|AllUnusedPins|ElementConnections)
 */
void ActionSave(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	char	*name;

	if (*Num == 1)
		switch(GetFunctionID(*Params))
		{
				/* save layout; use its original file name
				 * or 'fall thru' to next routine
				 */
			case F_Layout:
				if (PCB->Filename)
				{
					SavePCB(PCB->Filename);
					break;
				}

				/* save data to any file */
			case F_LayoutAs:
				name = GetUserInput("enter filename:", "");
				if (name)
					SavePCB(name);
				break;

				/* save all connections to file */
			case F_AllConnections:
			{
				FILE	*fp;
				Cursor	oldCursor;

				if ((fp = OpenConnectionDataFile()) != NULL)
				{
					oldCursor = SetOutputXCursor(XC_watch);
					LookupConnectionsToAllElements(fp);
						fclose(fp);
						SetOutputXCursor(oldCursor);
						SetChangedFlag(True);
				}
				break;
			}

				/* save all unused pins to file */
			case F_AllUnusedPins:
			{
				FILE	*fp;
				Cursor	oldCursor;

				if ((fp = OpenConnectionDataFile()) != NULL)
				{
					oldCursor = SetOutputXCursor(XC_watch);
					LookupUnusedPins(fp);
					fclose(fp);
					SetOutputXCursor(oldCursor);
					SetChangedFlag(True);
				}
				break;
			}

				/* save all connections to a file */
			case F_ElementConnections:
			{
				ElementTypePtr	element;
				FILE			*fp;
				Cursor			oldCursor;

				if ((SearchObjectByPosition(ELEMENT_TYPE,
						(void **) &element, (void **) &element,
						Crosshair.X, Crosshair.Y)) != NO_TYPE)
				{
					if ((fp = OpenConnectionDataFile()) != NULL)
					{
						oldCursor = SetOutputXCursor(XC_watch);
						LookupElementConnections(element, fp);
						fclose(fp);
						SetOutputXCursor(oldCursor);
						SetChangedFlag(True);
					}
				}
				break;
			}
		}
}

/* ---------------------------------------------------------------------------
 * load data
 * syntax: Load(ElementToBuffer|Layout|LayoutToBuffer)
 */
void ActionLoad(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	char	*name;

	if (*Num == 1)
	{
		HideCrosshair(True);
		switch(GetFunctionID(*Params))
		{
				/* load element data into buffer */
			case F_ElementToBuffer:
				name = FileSelectBox("load element to buffer:",
					NULL, Settings.ElementPath);
				if (name && LoadElementToBuffer(PASTEBUFFER, name))
					SetMode(PASTEBUFFER_MODE);
				break;

				/* load PCB data into buffer */
			case F_LayoutToBuffer:
				name = FileSelectBox("load file to buffer:",
					NULL, Settings.FilePath);
				if (name && LoadLayoutToBuffer(PASTEBUFFER, name))
					SetMode(PASTEBUFFER_MODE);
				break;

				/* load new data */
			case F_Layout:
				name = FileSelectBox("load file:", NULL, Settings.FilePath);
				if (name)
					if (!PCB->Changed ||
						ConfirmDialog("OK to override layout data?"))
						LoadPCB(name);
				break;
		}
		RestoreCrosshair(True);
	}
}

/* ---------------------------------------------------------------------------
 * print data
 * syntax: Print(Layout|Package, PostScript)
 */
void ActionPrint(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (*Num == 2)
	{
			/* check if layout is empty */
		if (!IsDataEmpty(PCB->Data))
		{
				/* determine type of printer */
			switch(GetFunctionID(*(Params+1)))
			{
				case F_PostScript:
				{
					int		ID;

					switch(ID = GetFunctionID(*Params))
					{
						case F_Layout:
						case F_Package:
							PrintDialog(ID == F_Package);
							break;
					}
					break;
				}
			}
		}
		else
			Message(False, "can't print empty layout");
	}
}

/* ---------------------------------------------------------------------------
 * starts a new layout
 * syntax: New()
 */
void ActionNew(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	char	*name;

	if (*Num == 0)
	{
		HideCrosshair(True);
		if (!PCB->Changed ||
			ConfirmDialog("OK to clear layout data?"))
		{
				/* do emergency saving */
			if (PCB->Changed && Settings.SaveInTMP)
				EmergencySave();
			RemovePCB(PCB);
			PCB = CreateNewPCB();
			ResetStackAndVisibility();
			name = GetUserInput("enter the layouts name:", "");
			if (name)
				PCB->Name = name;
			else
				PCB->Name = MyStrdup("new layout", "ActionNew()");
			CreateDefaultFont();
			UpdateSettingsOnScreen();

				/* reset layout size to resource value */
			XtVaSetValues(Output.Output,
				XtNwidth, TO_SCREEN(PCB->MaxWidth),
				XtNheight, TO_SCREEN(PCB->MaxHeight),
				NULL);

			ClearAndRedrawOutput();
		}
		RestoreCrosshair(True);
	}
}

/* ---------------------------------------------------------------------------
 * no operation, just for testing purposes
 * syntax: Bell()
 */
void ActionBell(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	XBell(Dpy, Settings.Volume);
}

/* ---------------------------------------------------------------------------
 * paste buffer operations
 * syntax: PasteBuffer(AddSelected|Clear|1..MAX_BUFFER)
 *         PasteBuffer(Rotate, 1..3)
 */
void ActionPasteBuffer(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	HideCrosshair(True);
	if (*Num == 1)
	{
		switch(GetFunctionID(*Params))
		{
				/* clear contents of paste buffer */
			case F_Clear:
				ClearBuffer(PASTEBUFFER);
				break;

				/* copies objects to paste buffer */
			case F_AddSelected:
				AddSelectedToBuffer(PASTEBUFFER);
				break;

				/* set number */
			default:
			{
				int	number = atoi(*Params);

					/* correct number */
				if (number)
					SetBufferNumber(number -1);
			}
		}
	}
	if (*Num == 2)
	{
		switch(GetFunctionID(*Params))
		{
			case F_Rotate:
				if (Settings.Mode == PASTEBUFFER_MODE)
				{
					RotateBuffer(PASTEBUFFER, (BYTE) atoi(*(Params+1)));
					SetCrosshairRange(
						PASTEBUFFER->X -PASTEBUFFER->BoundingBox.X1,
						PASTEBUFFER->Y -PASTEBUFFER->BoundingBox.Y1,
						PCB->MaxWidth-
							(PASTEBUFFER->BoundingBox.X2-PASTEBUFFER->X),
						PCB->MaxHeight-
							(PASTEBUFFER->BoundingBox.Y2-PASTEBUFFER->Y));
				}
				break;
		}
	}
	RestoreCrosshair(True);
}

/* ---------------------------------------------------------------------------
 * action routine to 'undo' operations
 * The serial number indicates the operation. This makes it possible
 * to group undo requests.
 * syntax: Undo(ClearList)
 *         Undo()
 */
void ActionUndo(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	HideCrosshair(True);

		/* undo the last destructive operation */
	if (!*Num && Undo())
			SetChangedFlag(True);

	if (*Num == 1)
	{
		switch(GetFunctionID(*Params))
		{
				/* clear 'undo objects' list */
			case F_ClearList:
				if (ConfirmDialog("OK to clear 'undo' buffer ?"))
					ClearUndoList();
				break;
		}
	}
	RestoreCrosshair(True);
}

/* ---------------------------------------------------------------------------
 * some polygon related stuff
 * syntax: Polygon(Close|InsertPoint|PreviousPoint|RemovePoint)
 */
void ActionPolygon(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (*Num == 1 && Settings.Mode == POLYGON_MODE)
	{
		HideCrosshair(True);
		switch(GetFunctionID(*Params))
		{
				/* close open polygon if possible */
			case F_Close:
				ClosePolygon();
				break;

				/* insert a new point */
			case F_InsertPoint:
				InsertPolygonPointAtCrosshairPosition();
				break;

				/* remove point from polygon */
			case F_RemovePoint:
				RemovePolygonPointAtCrosshairPosition();
				break;

				/* go back to the previous point */
			case F_PreviousPoint:
				GoToPreviousPolygonPoint();
				break;
		}
		RestoreCrosshair(True);
	}
}
