/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: move.c,v 2.2 94/09/29 13:53:44 nau Exp $";

/* functions used to move pins, elements ...
 */

#include <stdlib.h>

#include "global.h"

#include "crosshair.h"
#include "data.h"
#include "draw.h"
#include "misc.h"
#include "move.h"
#include "polygon.h"
#include "search.h"
#include "select.h"
#include "undo.h"

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void	*MoveElementName(ElementTypePtr);
static	void	*MoveElement(ElementTypePtr);
static	void	*MoveVia(PinTypePtr);
static	void	*MoveLine(LayerTypePtr, LineTypePtr);
static	void	*MoveText(LayerTypePtr, TextTypePtr);
static	void	*MovePolygon(LayerTypePtr, PolygonTypePtr);
static	void	*MovePolygonPoint(PolygonTypePtr, PolygonPointTypePtr);

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	Position			DeltaX,		/* used by local routines as offset */
							DeltaY;
static	ObjectFunctionType	MoveFunctions = {
	MoveLine,
	MoveText,
	MovePolygon,
	MoveVia,
	MoveElement,
	MoveElementName,
	NULL,
	MovePolygonPoint };

/* ---------------------------------------------------------------------------
 * moves a element by +-X and +-Y
 */
void MoveElementLowLevel(ElementTypePtr Element, Position DX, Position DY)
{
	ELEMENTLINE_LOOP(Element, MOVE_LINE_LOWLEVEL(line, DX, DY););
	PIN_LOOP(Element, MOVE_PIN_LOWLEVEL(pin, DX, DY););
	ARC_LOOP(Element, MOVE_ARC_LOWLEVEL(arc, DX, DY););
	MOVE_BOX_LOWLEVEL(&Element->BoundingBox, DX, DY);
	MOVE(Element->MarkX, Element->MarkY, DX, DY);

		/* move both names */
	MOVE_TEXT_LOWLEVEL(&CANONICAL_TEXT(Element), DX, DY);
	MOVE_TEXT_LOWLEVEL(&NAMEONPCB_TEXT(Element), DX, DY);
}

/* ----------------------------------------------------------------------
 * moves both names of an element to a new position
 */
static void *MoveElementName(ElementTypePtr Element)
{
	if (PCB->ElementOn)
	{
		EraseElementName(Element);
		MOVE_TEXT_LOWLEVEL(&CANONICAL_TEXT(Element), DeltaX, DeltaY);
		MOVE_TEXT_LOWLEVEL(&NAMEONPCB_TEXT(Element), DeltaX, DeltaY);
		DrawElementName(Element);
	}
	else
	{
		MOVE_TEXT_LOWLEVEL(&CANONICAL_TEXT(Element), DeltaX, DeltaY);
		MOVE_TEXT_LOWLEVEL(&NAMEONPCB_TEXT(Element), DeltaX, DeltaY);
	}
	return(Element);
}

/* ---------------------------------------------------------------------------
 * moves an element
 */
static void *MoveElement(ElementTypePtr Element)
{
	if (PCB->ElementOn)
	{
		EraseElement(Element);
		MoveElementLowLevel(Element, DeltaX, DeltaY);
		DrawElementName(Element);
		DrawElementPackage(Element);
	}
	else
		MoveElementLowLevel(Element, DeltaX, DeltaY);
	if (PCB->PinOn)
		DrawElementPins(Element);
	return(Element);
}

/* ---------------------------------------------------------------------------
 * moves a via
 */
static void *MoveVia(PinTypePtr Via)
{
	if (PCB->ViaOn)
	{
		EraseVia(Via);
		MOVE_VIA_LOWLEVEL(Via, DeltaX, DeltaY);
		DrawVia(Via);
	}
	else
		MOVE_VIA_LOWLEVEL(Via, DeltaX, DeltaY);
	return(Via);
}

/* ---------------------------------------------------------------------------
 * moves a line
 */
static void *MoveLine(LayerTypePtr Layer, LineTypePtr Line)
{
	if (Layer->On)
	{
		EraseLine(Line);
		MOVE_LINE_LOWLEVEL(Line, DeltaX, DeltaY);
		DrawLine(Layer, Line);
	}
	else
		MOVE_LINE_LOWLEVEL(Line, DeltaX, DeltaY);
	return(Line);
}

/* ---------------------------------------------------------------------------
 * moves a text object
 */
static void *MoveText(LayerTypePtr Layer, TextTypePtr Text)
{
	if (Layer->On)
	{
		EraseText(Text);
		MOVE_TEXT_LOWLEVEL(Text, DeltaX, DeltaY);
		DrawText(Layer, Text);
	}
	else
		MOVE_TEXT_LOWLEVEL(Text, DeltaX, DeltaY);
	return(Text);
}

/* ---------------------------------------------------------------------------
 * low level routine to move a polygon
 */
void MovePolygonLowLevel(PolygonTypePtr Polygon,
	Position DeltaX, Position DeltaY)
{
	POLYGONPOINT_LOOP(Polygon,
		MOVE(point->X, point->Y, DeltaX, DeltaY);
	);
	MOVE_BOX_LOWLEVEL(&Polygon->BoundingBox, DeltaX, DeltaY);
}

/* ---------------------------------------------------------------------------
 * moves a polygon
 */
static void *MovePolygon(LayerTypePtr Layer, PolygonTypePtr Polygon)
{
	if (Layer->On)
	{
		ErasePolygon(Polygon);
		MovePolygonLowLevel(Polygon, DeltaX, DeltaY);
		DrawPolygon(Layer, Polygon);
	}
	else
		MovePolygonLowLevel(Polygon, DeltaX, DeltaY);
	return(Polygon);
}

/* ---------------------------------------------------------------------------
 * moves a polygon-point
 */
static void *MovePolygonPoint(PolygonTypePtr Polygon, PolygonPointTypePtr Point)
{
	LayerTypePtr	layer;

		/* the layer isn't passed in so we have to lookup it's pointer
		 * to see if it is switched on.
		 * !!! works for PCB only, not available for buffers !!!
		 */
	if ((layer = GetLayerOfPolygon(PCB->Data, Polygon)) != NULL)
	{
		if (layer->On)
		{
			ErasePolygon(Polygon);
			MOVE(Point->X, Point->Y, DeltaX, DeltaY);
			DrawPolygon(layer, Polygon);
		}
		else
			MOVE(Point->X, Point->Y, DeltaX, DeltaY);
		SetPolygonBoundingBox(Polygon);
		return(Point);
	}
	return(NULL);
}

/* ---------------------------------------------------------------------------
 * moves the object identified by its data pointers and the type
 */
void MoveObject(int Type, void *Ptr1, void *Ptr2, Position DX, Position DY)
{
		/* setup offset */
	DeltaX = DX;
	DeltaY = DY;
	ObjectOperation(&MoveFunctions, Type, Ptr1, Ptr2);
	AddObjectToMoveUndoList(Type, Ptr1, Ptr2, DX, DY);
	IncrementUndoSerialNumber();
}
