/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: search.c,v 2.2 94/10/05 08:41:42 nau Exp $";

/* search routines
 */
#include <math.h>

#include "global.h"

#include "data.h"
#include "draw.h"
#include "search.h"

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	Position		PosX,		/* search position for subroutines */
						PosY;

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	Boolean	SearchLineByPosition(LayerTypePtr *, LineTypePtr *);
static	Boolean	SearchTextByPosition(LayerTypePtr *, TextTypePtr *);
static	Boolean	SearchPolygonByPosition(LayerTypePtr *, PolygonTypePtr *);
static	Boolean	SearchPinByPosition(ElementTypePtr *, PinTypePtr *);
static	Boolean	SearchViaByPosition(PinTypePtr *);
static	Boolean	SearchElementNameByPosition(ElementTypePtr *, TextTypePtr *);
static	Boolean	SearchPolygonPointByPosition(PolygonTypePtr *,
					PolygonPointTypePtr *);


/* ---------------------------------------------------------------------------
 * searches a via
 */
static Boolean SearchViaByPosition(PinTypePtr *Via)
{
		/* search only if via-layer is visible */
	if (PCB->ViaOn)
		VIA_LOOP(PCB->Data,
			if (abs(via->X - PosX) <= via->Thickness/3 &&
		    	abs(via->Y - PosY) <= via->Thickness/3 )
			{
				*Via = via;
				return(True);
			}
		);
	return(False);
}

/* ---------------------------------------------------------------------------
 * searches a pin
 * starts with the newest element
 */
static Boolean SearchPinByPosition(ElementTypePtr *Element, PinTypePtr *Pin)
{
		/* search only if pin-layer is visible */
	if (PCB->PinOn)
		ELEMENT_LOOP(PCB->Data,
			PIN_LOOP(element,
				if (abs(pin->X - PosX) <= pin->Thickness/3 &&
					abs(pin->Y - PosY) <= pin->Thickness/3 )
				{
					*Pin = pin;
					return(True);
				}
			);
		);
	return(False);
}

/* ---------------------------------------------------------------------------
 * searches line on all layers that are switched on in layerstack order
 */
static Boolean SearchLineByPosition(LayerTypePtr *Layer, LineTypePtr *Line)
{
	Cardinal		i;

	for (i = 0; i < MAX_LAYER; i++)
	{
		*Layer = LAYER_ON_STACK(i);
		if ((*Layer)->On)
			LINE_LOOP(*Layer,
				if (IsPointOnLine(PosX, PosY, 0, line))
				{
					*Line = line;
					return(True);
				}
			);
	}
	return(False);
}

/* ---------------------------------------------------------------------------
 * searches text on all layers that are switched on in layerstack order
 */
static Boolean SearchTextByPosition(LayerTypePtr *Layer, TextTypePtr *Text)
{
	Cardinal		i;

	for (i = 0; i < MAX_LAYER; i++)
	{
		*Layer = LAYER_ON_STACK(i);
		if ((*Layer)->On)
			TEXT_LOOP(*Layer,
				if (POINT_IN_BOX(PosX, PosY, &text->BoundingBox))
				{
					*Text = text;
					return(True);
				}
			);
	}
	return(False);
}

/* ---------------------------------------------------------------------------
 * searches a polygon on all layers that are switched on in layerstack order
 */
static Boolean SearchPolygonByPosition(LayerTypePtr *Layer,
	PolygonTypePtr *Polygon)
{
	Cardinal		i;

	for (i = 0; i < MAX_LAYER; i++)
	{
		*Layer = LAYER_ON_STACK(i);
		if ((*Layer)->On)
			POLYGON_LOOP(*Layer,
				if (IsPointInPolygon(polygon, PosX, PosY))
				{
					*Polygon = polygon;
					return(True);
				}
			);
	}
	return(False);
}

/* ---------------------------------------------------------------------------
 * searches a polygon-point on all layers that are switched on
 * in layerstack order
 */
static Boolean SearchPolygonPointByPosition(PolygonTypePtr *Polygon,
	PolygonPointTypePtr *Point)
{
	Cardinal		i;

	for (i = 0; i < MAX_LAYER; i++)
	{
		LayerTypePtr	layer = LAYER_ON_STACK(i);

		if (layer->On)
			POLYGON_LOOP(layer,
				POLYGONPOINT_LOOP(polygon,
				{
					float	d;

					d = (point->X - PosX)*(point->X - PosX)+
						(point->Y - PosY)*(point->Y - PosY);
					if (d < MAX_SEARCH_POINT_DISTANCE*MAX_SEARCH_POINT_DISTANCE)
					{
						*Polygon = polygon;
						*Point = point;
						return(True);
					}
				}
				);
			);
	}
	return(False);
}

/* ---------------------------------------------------------------------------
 * searches the name of an element
 * the search starts with the last element and goes back to the beginning
 */
static Boolean SearchElementNameByPosition(ElementTypePtr *Element,
	TextTypePtr *Text)
{
	TextTypePtr		text;

		/* package layer have to be switched on */
	if (PCB->ElementOn)
	{
		ELEMENT_LOOP(PCB->Data,
			text = &ELEMENT_TEXT(PCB, element);
			if (POINT_IN_BOX(PosX, PosY, &text->BoundingBox))
			{
				*Element = element;
				*Text = text;
				return(True);
			}
		);
	}
	return(False);
}

/* ---------------------------------------------------------------------------
 * searches an element
 * the search starts with the last element and goes back to the beginning
 * if more than one element matches, the smallest one is taken
 */
static Boolean SearchElementByPosition(ElementTypePtr *Element)
{
	ElementTypePtr	save = NULL;
	long	 		area = 0;

		/* package layer have to be switched on */
	if (PCB->ElementOn || PCB->PinOn)
	{
			/* the element names bounding box is not necessarily
			 * a part of the elements bounding box;
			 * we have to check all of them
			 */
		ELEMENT_LOOP(PCB->Data,
			if (POINT_IN_BOX(PosX, PosY,&element->BoundingBox) ||
				POINT_IN_BOX(PosX, PosY,&CANONICAL_TEXT(element).BoundingBox) ||
				POINT_IN_BOX(PosX, PosY,&NAMEONPCB_TEXT(element).BoundingBox))
			{
				long	newarea;

					/* use the element with the smallest bounding box */
				newarea = (element->BoundingBox.X2 -element->BoundingBox.X1) *
					(element->BoundingBox.Y2 -element->BoundingBox.Y1);
				if (!save || newarea < area)
				{
					area = newarea;
					save = element;
				}
			}
		);
	}
	*Element = save;
	return(save != NULL);
}

/* ---------------------------------------------------------------------------
 * checks if a line intersects with a PV
 * constant recognition by the optimizer is assumed
 *
 * let the point be (X,Y) and the line (X1,Y1)(X2,Y2)
 * the length of the line is
 *
 *   l = ((X2-X1)^2 + (Y2-Y1)^2)^0.5
 * 
 * let Q be the point of perpendicular projection of (X,Y) onto the line
 *
 *   QX = X1 +r*(X2-X1)
 *   QY = Y1 +r*(Y2-Y1)
 * 
 * with (from vector geometry)
 *
 *       (Y1-Y)(Y1-Y2)+(X1-X)(X1-X2)
 *   r = ---------------------------
 *                   l*l
 *
 *   r < 0     Q is on backward extension of the line
 *   r > 1     Q is on forward extension of the line
 *   else      Q is on the line
 *
 * the signed distance from (X,Y) to Q is
 *
 *       (Y2-Y1)(X-X1)-(X2-X1)(Y-Y1)
 *   d = ----------------------------
 *                    l
 */
Boolean IsPointOnLine(float X, float Y, float Radius, LineTypePtr Line)
{
	float	r, d,
			l,
			dx, dy;

	Radius += ((float) Line->Thickness /2.0 +1.0);
	l = sqrt((Line->X2 -Line->X1)*(Line->X2 -Line->X1) +
		(Line->Y2-Line->Y1)*(Line->Y2-Line->Y1));
	if (l == 0.0)
		return(False);

	d = ((Line->Y2 -Line->Y1)*(X -Line->X1)-
		(Line->X2 -Line->X1)*(Y -Line->Y1))/l;

		/* check distance from PV to line */
	if (fabs(d) > Radius)
		return(False);

		/* they intersect in Q is on line */
	dx = Line->X1 -X;
	dy = Line->Y1 -Y;
	r = (dy*(Line->Y1 -Line->Y2) + dx*(Line->X1 -Line->X2))/l/l;
	if (r >= 0 && r <= 1.0)
		return(True);

		/* we have to check P1 or P2 depending on the sign of r */
	if (r < 0.0)
		return((dx*dx +dy*dy) <= Radius*Radius);
	dx = Line->X2 -X;
	dy = Line->Y2 -Y;
	return((dx*dx +dy*dy) <= Radius*Radius);
}

/* ---------------------------------------------------------------------------
 * checks if a point lies inside a polygon
 * the code assumes that the last point isn't equal to the first one
 * The algorithm fails if the point is equal to a corner
 * from news FAQ:
 *   This code is from Wm. Randolph Franklin, wrf@ecse.rpi.edu, a
 *   professor at RPI.
 */
Boolean IsPointInPolygon(PolygonTypePtr Polygon, Position X, Position Y)
{
	Boolean		inside = False;

	if (POINT_IN_BOX(X, Y, &Polygon->BoundingBox))
	{
		PolygonPointTypePtr	ptr = &Polygon->Points[0];

			/* POLYGONPOINT_LOOP decrements pointers !!! */
		POLYGONPOINT_LOOP(Polygon,
			if ((((ptr->Y <= Y) && (Y < point->Y)) ||
				((point->Y <= Y) && (Y < ptr->Y))) &&
				(X < ((float) (point->X -ptr->X) * (float) (Y -ptr->Y) /
					(float) (point->Y -ptr->Y) +ptr->X)))
				inside = !inside;
			ptr = point;
		);
	}
	return(inside);
}

/* ---------------------------------------------------------------------------
 * searches for any kind of object or for a set of object types
 * the calling routine passes two pointers to allocated memory for storing
 * the results. 
 * A type value is returned too which is NO_TYPE if no objects has been found.
 * A set of object types is passed in.
 * The object is located by it's position.
 *
 * The layout is checked in the following order:
 *   polygon-point, pin, via, line, text, elementname, polygon, element
 */
int SearchObjectByPosition(int Type, void **Result1, void **Result2,
	Position X, Position Y)
{
		/* setup local identifiers */
	PosX = X;
	PosY = Y;

	if (Type & POLYGONPOINT_TYPE &&
		SearchPolygonPointByPosition((PolygonTypePtr *) Result1,
			(PolygonPointTypePtr *) Result2))
		return(POLYGONPOINT_TYPE);
	
	if (Type & PIN_TYPE &&
		SearchPinByPosition((ElementTypePtr *) Result1, (PinTypePtr *) Result2))
		return(PIN_TYPE);
	
	if (Type & VIA_TYPE &&
		SearchViaByPosition((PinTypePtr *) Result1))
	{
		*Result2 = *Result1;
		return(VIA_TYPE);
	}

	if (Type & LINE_TYPE &&
		SearchLineByPosition((LayerTypePtr *) Result1, (LineTypePtr *) Result2))
		return(LINE_TYPE);

	if (Type & TEXT_TYPE &&
		SearchTextByPosition((LayerTypePtr *) Result1, (TextTypePtr *) Result2))
		return(TEXT_TYPE);

	if (Type & ELEMENTNAME_TYPE &&
		SearchElementNameByPosition((ElementTypePtr *) Result1,
			(TextTypePtr *) Result2))
		return(ELEMENTNAME_TYPE);

	if (Type & POLYGON_TYPE &&
		SearchPolygonByPosition((LayerTypePtr *) Result1,
			(PolygonTypePtr *) Result2))
		return(POLYGON_TYPE);

	if (Type & ELEMENT_TYPE &&
		SearchElementByPosition((ElementTypePtr *) Result1))
	{
		*Result2 = *Result1;
		return(ELEMENT_TYPE);
	}

	return(NO_TYPE);
}

/* ---------------------------------------------------------------------------
 * searches for a object by it's unique ID. It doesn't matter if
 * the object is visible or not. The search is performed on a PCB, a
 * buffer or on the remove list.
 * The calling routine passes two pointers to allocated memory for storing
 * the results. 
 * A type value is returned too which is NO_TYPE if no objects has been found.
 */
int SearchObjectByID(DataTypePtr Base, void **Result1, void **Result2, int ID)
{
	ALLLINE_LOOP(Base,
		if (line->ID == ID)
		{
			*Result1 = (void *) layer;
			*Result2 = (void *) line;
			return(LINE_TYPE);
		}
	);

	ALLTEXT_LOOP(Base,
		if (text->ID == ID)
		{
			*Result1 = (void *) layer;
			*Result2 = (void *) text;
			return(TEXT_TYPE);
		}
	);

	ALLPOLYGON_LOOP(Base,
		if (polygon->ID == ID)
		{
			*Result1 = (void *) layer;
			*Result2 = (void *) polygon;
			return(POLYGON_TYPE);
		}
		POLYGONPOINT_LOOP(polygon,
			if (point->ID == ID)
			{
				*Result1 = (void *) polygon;
				*Result2 = (void *) point;
				return(POLYGONPOINT_TYPE);
			}
		);
	);

	VIA_LOOP(Base,
		if (via->ID == ID)
		{
			*Result1 = *Result2 = (void *) via;
			return(VIA_TYPE);
		}
	);

		/* check pins and elementnames too */
	ELEMENT_LOOP(Base,
		*Result1 = (void *) element;
		if (element->ID == ID)
		{
			*Result2 = (void *) element;
			return(ELEMENT_TYPE);
		}
		if (CANONICAL_TEXT(element).ID == ID)
		{
			*Result2 = (void *) &CANONICAL_TEXT(element);
			return(ELEMENTNAME_TYPE);
		}
		if (NAMEONPCB_TEXT(element).ID == ID)
		{
			*Result2 = (void *) &NAMEONPCB_TEXT(element);
			return(ELEMENTNAME_TYPE);
		}
		PIN_LOOP(element,
			if (pin->ID == ID)
			{
				*Result2 = (void *) pin;
				return(PIN_TYPE);
			}
		);
	);

	return(NO_TYPE);
}
