/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: /sda4/users/nau/src/pcb/RCS/select.c,v 2.2 1994/10/03 15:51:30 nau Exp nau $";

/* select routines
 */

#include "global.h"

#include "data.h"
#include "draw.h"
#include "search.h"
#include "select.h"

/* ---------------------------------------------------------------------------
 * toggles the selection of any kind of object
 * the different types are defined by search.h
 */
Boolean SelectObject(void)
{
	void	*ptr1, *ptr2;
	Boolean	changed = True;
	
	switch(SearchObjectByPosition(ALL_TYPES, &ptr1, &ptr2,
		Crosshair.X, Crosshair.Y))
	{
		case VIA_TYPE:
			TOGGLE_FLAG(SELECTEDFLAG, (PinTypePtr) ptr2);
			DrawVia((PinTypePtr) ptr1);
			break;

		case LINE_TYPE:
			TOGGLE_FLAG(SELECTEDFLAG, (LineTypePtr) ptr2);
			DrawLine((LayerTypePtr) ptr1, (LineTypePtr) ptr2);
			break;

		case TEXT_TYPE:
			TOGGLE_FLAG(SELECTEDFLAG, (TextTypePtr) ptr2);
			DrawText((LayerTypePtr) ptr1, (TextTypePtr) ptr2);
			break;

		case POLYGON_TYPE:
			TOGGLE_FLAG(SELECTEDFLAG, (PolygonTypePtr) ptr2);
			DrawPolygon((LayerTypePtr) ptr1, (PolygonTypePtr) ptr2);
			break;

		case PIN_TYPE:
			TOGGLE_FLAG(SELECTEDFLAG, (PinTypePtr) ptr2);
			DrawPin((PinTypePtr) ptr2);
			break;

		case ELEMENTNAME_TYPE:
		{
			ElementTypePtr	element = (ElementTypePtr) ptr1;

				/* select all names of the element */
			TOGGLE_FLAG(SELECTEDFLAG, &CANONICAL_TEXT(element));
			TOGGLE_FLAG(SELECTEDFLAG, &NAMEONPCB_TEXT(element));
			DrawElementName(element);
			break;
		}
			
		case ELEMENT_TYPE:
		{
			ElementTypePtr	element = (ElementTypePtr) ptr2;

				/* select all pins and names of the element */
			PIN_LOOP(element, TOGGLE_FLAG(SELECTEDFLAG, pin););
			TOGGLE_FLAG(SELECTEDFLAG, &CANONICAL_TEXT(element));
			TOGGLE_FLAG(SELECTEDFLAG, &NAMEONPCB_TEXT(element));
			TOGGLE_FLAG(SELECTEDFLAG, element);
			if (PCB->ElementOn)
			{
				DrawElementName(element);
				DrawElementPackage(element);
			}
			if (PCB->PinOn)
				DrawElementPins(element);
			break;
		}
	}
	return(changed);
}

/* ----------------------------------------------------------------------
 * selects/unselects all visible objects within the passed box
 * Flag determines if the block is to be selected or unselected
 * returns True if the state of any object has changed
 */
Boolean SelectBlock(BoxTypePtr Box, Boolean Flag)
{
	Boolean		changed = False;

		/* check layers */
	VISIBLELINE_LOOP(PCB->Data,
		if (LINE_IN_BOX(line, Box))
		{
			ASSIGN_FLAG(SELECTEDFLAG, Flag, line);
			DrawLine(layer, line);
			changed = True;
		}
	);
	VISIBLETEXT_LOOP(PCB->Data,
		if (TEXT_IN_BOX(text, Box))
		{
			ASSIGN_FLAG(SELECTEDFLAG, Flag, text);
			DrawText(layer, text);
			changed = True;
		}
	);
	VISIBLEPOLYGON_LOOP(PCB->Data,
		if (POLYGON_IN_BOX(polygon, Box))
		{
			ASSIGN_FLAG(SELECTEDFLAG, Flag, polygon);
			DrawPolygon(layer, polygon);
			changed = True;
		}
	);

		/* elements */
	if (PCB->ElementOn && PCB->PinOn)
		ELEMENT_LOOP(PCB->Data,
			if (ELEMENT_IN_BOX(element, Box))
			{
					/* select all pins and names of element */
				ASSIGN_FLAG(SELECTEDFLAG, Flag, &CANONICAL_TEXT(element));
				ASSIGN_FLAG(SELECTEDFLAG, Flag, &NAMEONPCB_TEXT(element));
				PIN_LOOP(element,
					ASSIGN_FLAG(SELECTEDFLAG, Flag, pin);
				);
				ASSIGN_FLAG(SELECTEDFLAG, Flag, element);
				DrawElement(element);
				changed = True;
			}
		);

		/* element names */
	if (PCB->ElementOn)
		ELEMENT_LOOP(PCB->Data,
			if (BOX_IN_BOX(&ELEMENT_TEXT(PCB, element).BoundingBox, Box))
			{
					/* select all pins and names of element */
				ASSIGN_FLAG(SELECTEDFLAG, Flag, &CANONICAL_TEXT(element));
				ASSIGN_FLAG(SELECTEDFLAG, Flag, &NAMEONPCB_TEXT(element));
				DrawElementName(element);
				changed = True;
			}
		);

		/* pins */
	if (PCB->PinOn)
		ALLPIN_LOOP(PCB->Data,
			if (VIA_OR_PIN_IN_BOX(pin, Box))
			{
				ASSIGN_FLAG(SELECTEDFLAG, Flag, pin);
				DrawPin(pin);
				changed = True;
			}
		);

		/* end with vias */
	if (PCB->ViaOn)
		VIA_LOOP(PCB->Data,
			if (VIA_OR_PIN_IN_BOX(via, Box))
			{
				ASSIGN_FLAG(SELECTEDFLAG, Flag, via);
				DrawVia(via);
				changed = True;
			}
		);

	return(changed);
}

/* ----------------------------------------------------------------------
 * performs several operations on the passed object
 */
void *ObjectOperation(ObjectFunctionTypePtr F,
	int Type, void *Ptr1, void *Ptr2)
{
	switch(Type)
	{
		case LINE_TYPE:
			if (F->Line)
				return(F->Line((LayerTypePtr) Ptr1, (LineTypePtr) Ptr2));
			break;

		case TEXT_TYPE:
			if (F->Text)
				return(F->Text((LayerTypePtr) Ptr1, (TextTypePtr) Ptr2));
			break;

		case POLYGON_TYPE:
			if (F->Polygon)
				return(F->Polygon((LayerTypePtr) Ptr1, (PolygonTypePtr) Ptr2));
			break;

		case POLYGONPOINT_TYPE:
			if (F->PolygonPoint)
				return(F->PolygonPoint((PolygonTypePtr) Ptr1,
					(PolygonPointTypePtr) Ptr2));
			break;

		case VIA_TYPE:
			if (F->Via)
				return(F->Via((PinTypePtr) Ptr2));
			break;

		case ELEMENT_TYPE:
			if (F->Element)
				return(F->Element((ElementTypePtr) Ptr2));
			break;

		case PIN_TYPE:
			if (F->Pin)
				return(F->Pin((ElementTypePtr) Ptr1, (PinTypePtr) Ptr2));
			break;

		case ELEMENTNAME_TYPE:
			if (F->ElementName)
				return(F->ElementName((ElementTypePtr) Ptr1));
			break;
	}
	return(NULL);
}

/* ----------------------------------------------------------------------
 * performs several operations on selected objects which are also visible
 * The lowlevel procdures are passed together with additional information
 * returns True if anything has changed
 */
Boolean SelectedOperation(ObjectFunctionTypePtr F)
{
	Boolean		changed = False;

		/* check lines */
	if (F->Line)
		VISIBLELINE_LOOP(PCB->Data,
			if (TEST_FLAG(SELECTEDFLAG, line))
			{
				F->Line(layer, line);
				changed = True;
			}
		);

		/* check text */
	if (F->Text)
		VISIBLETEXT_LOOP(PCB->Data,
			if (TEST_FLAG(SELECTEDFLAG, text))
			{
				F->Text(layer, text);
				changed = True;
			}
		);

		/* check polygons */
	if (F->Polygon)
		VISIBLEPOLYGON_LOOP(PCB->Data,
			if (TEST_FLAG(SELECTEDFLAG, polygon))
			{
				F->Polygon(layer, polygon);
				changed = True;
			}
		);

		/* elements; both text objects share one selection flag */
	if (PCB->ElementOn)
		ELEMENT_LOOP(PCB->Data,
			if (F->Element && TEST_FLAG(SELECTEDFLAG, element))
			{
				F->Element(element);
				changed = True;
			}
			if (F->ElementName &&
				TEST_FLAG(SELECTEDFLAG, &ELEMENT_TEXT(PCB, element)))
			{
				F->ElementName(element);
				changed = True;
			}
		);
	if (PCB->PinOn && F->Pin)
		ELEMENT_LOOP(PCB->Data,
			PIN_LOOP(element,
				if (TEST_FLAG(SELECTEDFLAG, pin))
				{
					F->Pin(element, pin);
					changed = True;
				}
			);
		);
		

		/* and vias */
	if (PCB->ViaOn && F->Via)
		VIA_LOOP(PCB->Data,
			if (TEST_FLAG(SELECTEDFLAG, via))
			{
				F->Via(via);
				changed = True;
			}
		);

	return(changed);
}

/* ----------------------------------------------------------------------
 * selects/unselects all objects which were found during a connection scan
 * Flag determines if they are to be selected or unselected
 * returns True if the state of any object has changed
 *
 * text objects and elements cannot be selected by this routine
 */
Boolean SelectConnection(Boolean Flag)
{
	Boolean		changed = False;

	VISIBLELINE_LOOP(PCB->Data,
		if (TEST_FLAG(FOUNDFLAG, line))
		{
			ASSIGN_FLAG(SELECTEDFLAG, Flag, line);
			DrawLine(layer, line);
			changed = True;
		}
	);
	VISIBLEPOLYGON_LOOP(PCB->Data,
		if (TEST_FLAG(FOUNDFLAG, polygon))
		{
			ASSIGN_FLAG(SELECTEDFLAG, Flag, polygon);
			DrawPolygon(layer, polygon);
			changed = True;
		}
	);

	if (PCB->PinOn)
		ALLPIN_LOOP(PCB->Data,
			if (TEST_FLAG(FOUNDFLAG, pin))
			{
				ASSIGN_FLAG(SELECTEDFLAG, Flag, pin);
				DrawPin(pin);
				changed = True;
			}
	);
	if (PCB->ViaOn)
		VIA_LOOP(PCB->Data,
			if (TEST_FLAG(FOUNDFLAG, via))
			{
				ASSIGN_FLAG(SELECTEDFLAG, Flag, via);
				DrawVia(via);
				changed = True;
			}
	);

	return(changed);
}
