/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * Base.h
 *
 * Base classes for parser types.  I seperated the base classes from
 * PTypes.h.  This allows other C++ files to include only the base classes,
 * without including the other classes (15kbytes).
 */

#ifndef BASE_H
#define BASE_H

#include <stdlib.h>
#include <iostream.h>
#include "Bool.h"
#include "String.h"
#include "List.h"
#include "Bitvector.h"

class Stmts;			// Needed for Sim_setup declaration.
class Number;			// Needed for the Sim_evaluate declaration.
class HashValue;		// Needed for Index declaration.
class STnode;			// Needed for Sim_setup declaration.

/* Base classes... */

/* Expressions: add, sub, range id, etc */
class Expression
	{
public:
	// Data for error reporting.
	String filename;
	int lineno;
	Expression(String fn = String("unknown"), int ln = -1)
		: filename(fn), lineno(ln) {}
	Expression(Expression &exp)
		: filename(exp.filename), lineno(exp.lineno) {}
	virtual HashValue *Index(void)
		{ return NULL; }
	virtual int GetRange(unsigned long &, unsigned long &)
		{ return 0; }
	virtual Expression *copy(void)
		{ return new Expression(*this); }
	virtual ostream &display(ostream &s)
		{ return s; }
	virtual void Sim_setup(Stack<int> &)
		{}
	virtual void Sim_trigger(String &)
		{}
	virtual Number Sim_evaluate(void);
	friend ostream &operator<<(ostream &s, Expression &expr)
		{ return expr.display(s); }
	// Required by the List class to display the contents.
	friend ostream &operator<<(ostream &s, Expression *expr)
		{ return expr->display(s); }
	};

/* Delay or Event control */
enum Dec { DECnum, DECid, DECevnt };
class DelayEvntCtl
	{
public:
	Dec flag;		// Delay #/id or Event?
	String filename;	// Data for error reporting.
	int lineno;
	DelayEvntCtl(Dec fl, String fn = String("unknown"), int ln = -1)
		: flag(fl), filename(fn), lineno(ln) {}
	DelayEvntCtl(DelayEvntCtl &dec)
		: flag(dec.flag), filename(dec.filename), lineno(dec.lineno)
		{}
	virtual DelayEvntCtl *copy(void)
		{ return new DelayEvntCtl(*this); }
	virtual ostream &display(ostream &s)
		{ return s; }
	virtual void Sim_setup(Stack<int> &, Stmts *)
		{}
	virtual void Sim_trigger(Stmts *)
		{}
	friend ostream &operator<<(ostream &s, DelayEvntCtl &expr)
		{ return expr.display(s); }
	friend ostream &operator<<(ostream &s, DelayEvntCtl *expr)
		{ return expr->display(s); }
	};

/*
 * Not supported constructs in statements.
 */
enum NotSupported
	{
	NS_ALL		= 0x00000000,
	NS_DELAY	= 0x00000001
	};

/* Statements: if, for, assignment, etc */
class Stmts
	{
public:
	Bool always;		// In always statement?
	DelayEvntCtl *dec;
	String filename;	// Data for error reporting.
	int lineno;
	Stmts(DelayEvntCtl *d = NULL, Bool al = FALSE,
			String fn = String("unknown"), int ln = -1)
		: always(al), filename(fn), lineno(ln)
		{ dec = (d == NULL) ? NULL : d->copy(); }
	Stmts(Stmts &st)
		: always(st.always), filename(st.filename), lineno(st.lineno)
		{ dec = (st.dec == NULL) ? NULL : st.dec->copy(); }
	virtual ~Stmts()
		{ if (dec) delete dec; }
	virtual ostream &display(ostream &s)
		{ return s; }
	virtual Stmts *copy(void)
		{ return new Stmts(*this); }
	virtual void Sim_setup(Stack<int> &, Bool = FALSE, int = NS_ALL)
		{}
	virtual int Sim_trigger(Stmts *)
		{ return TRUE; }
	friend ostream &operator<<(ostream &s, Stmts &st)
		{ return st.display(s); }
	friend ostream &operator<<(ostream &s, Stmts *st)
		{ return st->display(s); }
	};

/*
 * RegDecl will inherit from both TFDecl and ModuleItem.  But we need
 * <filename> for both.  Since the filename will be the same, we can
 * try use a virtual base class.
 */
class ErrorData
	{
public:
	String filename;	// Data for error reporting.
	int lineno;
	ErrorData(String fn = String("unknown"), int ln = -1)
		: filename(fn), lineno(ln) {}
	ErrorData(ErrorData &ed)
		: filename(ed.filename), lineno(ed.lineno) {}
	virtual ~ErrorData()
		{}
	};

/* TF (Task/Function?) declarations:  register declaration  */
class TFDecl : public virtual ErrorData
	{
public:
	TFDecl()
		{}
	TFDecl(TFDecl &tfd)
		: ErrorData(tfd) {}
	virtual ostream &display(ostream &s)
		{ return s; }
	virtual TFDecl *tfdecl_copy(void) = 0;
	TFDecl *copy(void)
		{ return tfdecl_copy(); }
	virtual void Sim_setup(Stack<int>, STnode * = NULL)
		{}
	friend ostream &operator<<(ostream &s, TFDecl &tfd)
		{ return tfd.display(s); }
	friend ostream &operator<<(ostream &s, TFDecl *tfd)
		{ return tfd->display(s); }
	};

/* Base class for module item:  initial, always, etc... */
class ModuleItem : public virtual ErrorData
	{
	// Generic declarations
	//	reg declarations
	//	initial
	//	always
public:
	ModuleItem()
		{}
	ModuleItem(ModuleItem &mi)
		: ErrorData(mi) {}
	virtual ostream &display(ostream &s)
		{ return s; }
	virtual ModuleItem *moditem_copy(void) = 0;
	ModuleItem *copy(void)
		{ return moditem_copy(); }
	virtual void Sim_setup(Stack<int>, STnode * = NULL)
		{}
	friend ostream &operator<<(ostream &s, ModuleItem &itm)
		{ return itm.display(s); }
	friend ostream &operator<<(ostream &s, ModuleItem *itm)
		{ return itm->display(s); }
	};

/*
 * Number is moved here because Symtab.h needs it.  But Number is basic
 * enough to stand by itself.  See Systask.cc for explanation on why
 * QString is here.
 */
/* Number:  1'b0, 6'o47, 8'h4a */
class Number : public Expression
	{
	BitVector value;
public:
	Number(Bits b = UNUSED)
		: value(b) {}
	Number(unsigned long num)
		: value(num) {}
	Number(int ms, int ls)
		: value(ms, ls) {}
	Number(String &str, int base = 10, int len = 0)
		: value(str, base, len) {}
	Number(BitVector &b)
		: value(b) {}
	Number(Number &n)
		: Expression(n), value(n.value) {}
	Bits &operator[](int idx)
		{ return value[idx]; }
	Number &operator=(Number &n)
		{
		value = n.value;
		return *this;
		}
	ostream &display(ostream &s)
		{ s << value; return s; }
	Expression *copy(void)
		{ return new Number(value); }
	void Sim_trigger(String &);
	Number Sim_evaluate(void)
		{ return *this; }
	operator unsigned long()
		{ return (unsigned long)value; }
	operator BitVector()
		{ return value; }
	SubBitVector operator()(int ms, int ls)
		{ return value(ms, ls); }
	friend int operator!(Number &n)
		{
		BitVector tmp(n.value);
		return (!tmp);
		}
	friend Number operator~(Number &n)
		{ return ~n.value; }
	friend Number operator+(Number &l, Number &r)
		{ return (l.value + r.value); }
	friend Number operator-(Number &l, Number &r)
		{ return (l.value - r.value); }
	friend int operator==(Number &l, Number &r)
		{ return (l.value == r.value); }
	friend int operator==(Number &l, unsigned long val)
		{
		BitVector tmp(val);
		return (l.value == tmp);
		}
	friend int operator==(unsigned long val, Number &r)
		{
		BitVector tmp(val);
		return (unsigned long)(r.value == tmp);
		}
	friend int operator!=(Number &l, Number &r)
		{ return (l.value != r.value); }
	friend int operator!=(Number &l, unsigned long val)
		{
		BitVector tmp(val);
		return (l.value != tmp);
		}
	friend int operator!=(unsigned long val, Number &r)
		{
		BitVector tmp(val);
		return (r.value != tmp);
		}
	friend int operator<(Number &l, Number &r)
		{ return (l.value < r.value); }
	friend int operator<=(Number &l, Number &r)
		{ return (l.value <= r.value); }
	friend int operator>(Number &l, Number &r)
		{ return (l.value > r.value); }
	friend int operator>=(Number &l, Number &r)
		{ return (l.value >= r.value); }
	friend Number operator|(Number &l, Number &r)
		{ return (l.value | r.value); }
	friend Number operator&(Number &l, Number &r)
		{ return (l.value & r.value); }
	friend Number operator<<(Number &l, Number &r)
		{ return (l.value << (unsigned long)r.value); }
	friend Number operator>>(Number &l, Number &r)
		{ return (l.value >> (unsigned long)r.value); }
	friend int Size(Number &n)
		{ return Size(n.value); }
	friend int StartBit(Number &n)
		{ return StartBit(n.value); }
	friend int EndBit(Number &n)
		{ return EndBit(n.value); }
	friend String Convert(Number &n, int base = 10, int len = 0)
		{ return Convert(n.value, base, len); }
	friend ostream &operator<<(ostream &s, Number &num)
		{ return num.display(s); }
	};

/* String class to inherit from Expression */
class QString : public Expression
	{
	String str;
public:
	QString()
		: str(NULL, 0) {}
	QString(char *s)
		: str(s) {}
	QString(String &s)
		: str(s) {}
	QString(QString &qstr)
		: Expression(qstr), str(qstr.str) {}
	char &operator[](int i)
		{ return str[i]; }
	void Sim_trigger(String &);
	Number Sim_evaluate(void)
		{
		String tmp("x");
		return Number(tmp);
		}
	ostream &display(ostream &s)
		{ s << str; return s; }
	Expression *copy(void)
		{ return new QString(str); }
	friend int Length(QString &s)
		{ return Length(s.str); }
	friend ostream &operator<<(ostream &s, QString &str)
		{ return str.display(s); }
	};

/*
 * This function finds the base and size of a format string.  It is usually
 * called by the $write system task.  But Number needs it, so it is here.
 * A bad hack, I know.
 */
inline void
get_base_size(String &fmt, int &base, int &size)
	{
	int j = 0;
	String buf(NULL, Length(fmt));
	for (int i = 0; i < Length(fmt); i++)
		{
		// There should be no errors in <fmt>, so we will not
		// do error checking...
		if (fmt[i] >= '0' && fmt[i] <= '9')
			buf[j++] = fmt[i];
		else if (fmt[i] == 'h')
			{ base = 16; break; }
		else if (fmt[i] == 'd')
			{ base = 10; break; }
		else if (fmt[i] == 'o')
			{ base = 8; break; }
		else if (fmt[i] == 'b')
			{ base = 2; break; }	
		// c for ascii, t for current time format.
		}
	buf[j] = '\0';

	// Convert to format.
	size = atoi((char *)buf);
	}

inline Number
Expression::Sim_evaluate(void)
	{ return 0UL; }

inline void
Number::Sim_trigger(String &fmt)
	{
	// Need to find the format of the output.
	int base, s;
	get_base_size(fmt, base, s);
	String tmp = Convert(*this, base, s);
	cout << tmp;
	}

inline void
QString::Sim_trigger(String &)
	{
	// Nothing to do but print out the string.
	int size = Length(str);
	// Skip the first '"' and the last '"'.
	for (int i = 1; i < (size-1); i++)
		cout << str[i];
	}

#endif // BASE_H
