/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * Bitvector.cc
 *
 * Support functions for BitVector class.
 */

#include <string.h>
#include <stdlib.h>
#include "Bool.h"
#include "String.h"
#include "Bitvector.h"

/*
 * Table for Inverter operation
 *             A
 *          0 1 X Z .
 *          ---------
 *    Not   1 0 X X X
 */

Bits NOTTABLE[NUMBITS] = {HI,LO,DC,DC,DC};

/*
 * Table for AND operation
 *             A
 *    AND   0 1 X Z .
 *          ---------
 *       0| 0 0 0 X X
 *       1| 0 1 X X X
 *   B   X| 0 X X X X
 *       Z| X X X X X
 *       .| X X X X X
 */

Bits ANDTABLE[][NUMBITS] =
	{
	{LO,LO,LO,DC,DC},
	{LO,HI,DC,DC,DC},
	{LO,DC,DC,DC,DC},
	{DC,DC,DC,DC,DC},
	{DC,DC,DC,DC,DC}
	};

/*
 * Table for OR operation
 *             A
 *    OR    0 1 X Z .(UNUSED)
 *          ---------
 *       0| 0 1 X X X
 *       1| 1 1 1 X X
 *   B   X| X 1 X X X
 *       Z| X X X X X
 *       .| X X X X X
 */

Bits ORTABLE[][NUMBITS] =
	{
	{LO,HI,DC,DC,DC},
	{HI,HI,HI,DC,DC},
	{DC,HI,DC,DC,DC},
	{DC,DC,DC,DC,DC},
	{DC,DC,DC,DC,DC}
	};

/*
 * Table for XOR operation
 *             A
 *    XOR   0 1 X Z .
 *          ---------
 *       0| 0 1 X X X
 *       1| 1 0 X X X
 *   B   X| X X X X X
 *       Z| X X X X X
 *       .| X X X X X
 */

Bits XORTABLE[][NUMBITS] =
	{
	{LO,HI,DC,DC,DC},
	{HI,LO,DC,DC,DC},
	{DC,DC,DC,DC,DC},
	{DC,DC,DC,DC,DC},
	{DC,DC,DC,DC,DC}
	};

/*
 *  Table for adding two bit vectors:
 *  Ci A B | Sum Co
 *  ---------------
 *   0 0 0 |  0  0
 *   0 0 1 |  1  0
 *   0 1 0 |  1  0
 *   0 1 1 |  0  1
 *   1 0 0 |  1  0
 *   1 0 1 |  0  1
 *   1 1 0 |  0  1
 *   1 1 1 |  1  1
 */

struct bv_at
	{
	Bits sum;
	Bits carry;
	};
struct bv_at ADDTABLE[] =
	{
	{LO,LO},
	{HI,LO},
	{HI,LO},
	{LO,HI},
	{HI,LO},
	{LO,HI},
	{LO,HI},
	{HI,HI}
	};

/*
 * Direct compare table.
 *
 * 0 ... everything else
 * 1 ... l = r
 * 2 ... l < r
 * 3 ... l > r
 *             R
 *   COMP   0 1 X Z .(UNUSED)
 *          ---------
 *       0| 1 2 0 0 0
 *       1| 3 1 0 0 0
 *   L   X| 0 0 0 0 0
 *       Z| 0 0 0 0 0
 *       .| 0 0 0 0 0
 */

enum { COMPERR = 0, COMPEQU = 1, COMPLT = 2, COMPGT = 3 };
int COMPTABLE[][NUMBITS] =
	{
	{1,2,0,0,0},
	{3,1,0,0,0},
	{0,0,0,0,0},
	{0,0,0,0,0},
	{0,0,0,0,0}
	};

int
bv_compare(BitVector &l, BitVector &r)
	{
	/*
	 * Compare the two bit vectors <l> and <r>.  Both vectors must be
	 * the same size.  See COMPTABLE for return values.
	 */
	Bits A, B;
	int li = l.end;
	int ri = r.end;
	for (int i = 0; i < l.size; i++, li--, ri--)
		{
		if (li < l.begin || ri < r.begin)
			return COMPEQU;
		A = l.bits[li];
		B = r.bits[ri];
		if (COMPTABLE[A][B] != COMPEQU)
			return COMPTABLE[A][B];
		}
	return COMPEQU;
	}

inline int
CONCAT(Bits A, Bits B, Bits C, Bits D = LO)
	{
	int ret;
	if (A > HI || B > HI || C > HI || D > HI)
		ret = -1;
	else
		ret = (D << 3) | (C << 2) | (B << 1) | (A);
	return ret;
	}

BitVector
bv_logicops(BitVector &l, BitVector &r, Bits table[][NUMBITS])
	{
	int size = (l.size >= r.size) ? l.size : r.size;
	BitVector res(size-1, 0);
	int li = l.begin;
	int ri = r.begin;
	Bits A, B;
	for (int i = 0; li <= l.end && ri <= r.end; i++, li++, ri++)
		{
		A = l.bits[li];
		B = r.bits[ri];
		res[i] = table[A][B];
		}
	BitSet(res, i, (res.end+1)-i, LO);
	return res;
	}

int
find_hw(dec_t val, dec_t &wt)
	{
	// Find the highest weight of val.  Return the number of digits
	// in weight.
	wt = 1;
	for (int i = 1; (val/wt) >= 10; i++)
		wt *= 10;
	return i;
	}

String
make_x(int size)
	{
	// Create a string with <size> number of X's.
	String ret;
	if (size > 1)
		{
		char *buf = new char[size+1];
		memset(buf, 'x', size);
		buf[size] = '\0';
		ret = String(buf);
		delete buf;
		}
	else
		ret = String("x");
	return ret;
	}

String
bv_bin2dec(BitVector &bv, int len)
	{
	dec_t wt, num = 0;

	// Convert from binary to decimal.
	int j = bv.begin;
	for (int i = 0; j <= bv.end && i < 8*sizeof(dec_t); i++, j++)
		{
		if (bv[j] == DC)
			return make_x(len);
		num += (bv[j]*(1 << i));
		}

	// Convert decimal to string.
	int size;
	if (len > 0)
		size = len;
	else
		size = find_hw(num, wt);
	String res(NULL, size);
	dec_t tmp1 = 10, tmp2;
	for (i = size-1; i >= 0; i--, tmp1 *= 10)
		{
		tmp2 = num % tmp1;
		num -= tmp2;
		// Shift the number down to less than 10.
		tmp2 = tmp2 / (tmp1 / 10);
		res[i] = tmp2 + '0';
		}
	return res;
	}

int
bv_store(BitVector &bv, int &j, Bits A, Bits B, Bits C, Bits D)
	{
	if (j > bv.end) return 0;
	bv.bits[j++] = A;
	if (j > bv.end) return 0;
	bv.bits[j++] = B;
	if (j > bv.end) return 0;
	bv.bits[j++] = C;
	if (j > bv.end) return 0;
	if (D != UNUSED)
		bv.bits[j++] = D;
	return 1;
	}

void
bv_bin2bin(BitVector &bv, String &str)
	{
	int i = Length(str) - 1;
	for (int j = bv.begin; j <= bv.end && i >= 0; j++, i--)
		{
		switch(str[i])
			{
			case '0': bv.bits[j] = LO; break;
			case '1': bv.bits[j] = HI; break;
			case 'x':
			case 'X': bv.bits[j] = DC; break;
			case 'z':
			case 'Z': bv.bits[j] = Z; break;
			case '_': j--; break;
			default: return;
			}
		}
	}

void
bv_oct2bin(BitVector &bv, String &str)
	{
	int i = Length(str) - 1;
	for (int j = bv.begin; j <= bv.end && i >= 0; i--)
		{
		switch(str[i])
			{
			case '0':
				if (!bv_store(bv, j, LO, LO, LO))
					return;
				break;
			case '1':
				if (!bv_store(bv, j, HI, LO, LO))
					return;
				break;
			case '2':
				if (!bv_store(bv, j, LO, HI, LO))
					return;
				break;
			case '3':
				if (!bv_store(bv, j, HI, HI, LO))
					return;
				break;
			case '4':
				if (!bv_store(bv, j, LO, LO, HI))
					return;
				break;
			case '5':
				if (!bv_store(bv, j, HI, LO, HI))
					return;
				break;
			case '6':
				if (!bv_store(bv, j, LO, HI, HI))
					return;
				break;
			case '7':
				if (!bv_store(bv, j, HI, HI, HI))
					return;
				break;
			case 'x':
			case 'X':
				if (!bv_store(bv, j, DC, DC, DC))
					return;
				break;
			case 'z':
			case 'Z':
				if (!bv_store(bv, j, Z, Z, Z))
					return;
				break;
			case '_': j--; break;
			default: return;
			}
		}
	}

void
bv_dec2bin(BitVector &bv, String &str)
	{
	dec_t value;
	if (strpbrk((char *)str, "XxZz") != NULL)
		// Set entire bit vector to don't cares.
		memset(bv.bits, DC, bv.size);
	else
		{
		value = strtoul(str, NULL, 10);
		if (value == 0)
			// We can return if <value> is zero, because <bv>
			// is already set to zero by the caller.
			return;
		}
	// We don't know how big the number is.  So allocate enough to
	// store the largest value.  Then copy only what we need later.
	BitVector tmp((8*sizeof(dec_t))-1, 0);
	dec_t rem = value;
	for(int i = 0; rem > 0 && i < 8*sizeof(dec_t); i++)
		{
		// Now convert integer to binary.
		tmp.bits[i] = ((rem % 2) == 1) ? HI : LO;
		rem /= 2;
		}
	if (bv.bits != NULL)
		delete [] bv.bits;
	bv.size = i;
	bv.end = bv.size - 1;
	bv.bits = new Bits[bv.size];
	memcpy(bv.bits, tmp.bits, bv.size);
	}

void
bv_hex2bin(BitVector &bv, String &str)
	{
	int i = Length(str) - 1;
	for (int j = bv.begin; j <= bv.end && i >= 0; i--)
		{
		switch(str[i])
			{
			case '0':
				if (!bv_store(bv, j, LO, LO, LO, LO))
					return;
				break;
			case '1':
				if (!bv_store(bv, j, HI, LO, LO, LO))
					return;
				break;
			case '2':
				if (!bv_store(bv, j, LO, HI, LO, LO))
					return;
				break;
			case '3':
				if (!bv_store(bv, j, HI, HI, LO, LO))
					return;
				break;
			case '4':
				if (!bv_store(bv, j, LO, LO, HI, LO))
					return;
				break;
			case '5':
				if (!bv_store(bv, j, HI, LO, HI, LO))
					return;
				break;
			case '6':
				if (!bv_store(bv, j, LO, HI, HI, LO))
					return;
				break;
			case '7':
				if (!bv_store(bv, j, HI, HI, HI, LO))
					return;
				break;
			case '8':
				if (!bv_store(bv, j, LO, LO, LO, HI))
					return;
				break;
			case '9':
				if (!bv_store(bv, j, HI, LO, LO, HI))
					return;
				break;
			case 'a':
			case 'A':
				if (!bv_store(bv, j, LO, HI, LO, HI))
					return;
				break;
			case 'b':
			case 'B':
				if (!bv_store(bv, j, HI, HI, LO, HI))
					return;
				break;
			case 'c':
			case 'C':
				if (!bv_store(bv, j, LO, LO, HI, HI))
					return;
				break;
			case 'd':
			case 'D':
				if (!bv_store(bv, j, HI, LO, HI, HI))
					return;
				break;
			case 'e':
			case 'E':
				if (!bv_store(bv, j, LO, HI, HI, HI))
					return;
				break;
			case 'f':
			case 'F':
				if (!bv_store(bv, j, HI, HI, HI, HI))
					return;
				break;
			case 'x':
			case 'X':
				if (!bv_store(bv, j, DC, DC, DC, DC))
					return;
				break;
			case 'z':
			case 'Z':
				if (!bv_store(bv, j, Z, Z, Z, Z))
					return;
				break;
			case '_': j--; break;
			default: return;
			}
		}
	}

/*
 * SubBitVector.  These functions are here instead of in the header file
 * becuase we need to have class BitVector defined first.  The only way
 * to have this was to move these member functions out of the header file.
 */

inline
SubBitVector::SubBitVector(BitVector &bv, int ms, int ls)
	: BV(bv)
	{
	// According to the Verilog LRM, <ms> and <ls> must be non-negative
	// and <ms> must reference a more significant bit than <ls>
	if (ms < 0 || ls < 0)
		{
		begin = -1;
		end = -2;
		size = 0;
		}
	else if (bv.msb_lower == FALSE && ms >= ls)
		{
		begin = ls;
		end = ms;
		size = (end - begin) + 1;
		}
	else if (bv.msb_lower == TRUE && ms <= ls)
		{
		/*
		 * When <msb_lower> is true, it means this register was
		 * declared with [0:7].  Thus, when range selecting, we
		 * must convert to the internal bitvector representation.
		 * This only affects range selection, and not logic or
		 * arithmetic operations.
		 */
		begin = bv.end - ls;
		end = bv.end - ms;
		size = (end - begin) + 1;
		}
	else
		{
		begin = -1;
		end = -2;
		size = 0;
		}
	}

inline
SubBitVector::SubBitVector(SubBitVector &sbv)
	: BV(sbv.BV), begin(sbv.begin), end(sbv.end), size(sbv.size)
	{}

SubBitVector &
SubBitVector::operator=(SubBitVector &sbv)
	{
	int i = begin;
	int j = sbv.begin;
	// Try to copy over what we need, only.
	for (int k = 0; k < size ; i++, j++, k++)
		{
		// Did we go past the end of <sbv>?  I.e sbv.size < size
		if (j > sbv.end)
			break;
		// If we are unused, don't modify it.
		else if (BV.bits[i] == UNUSED)
			continue;
		// Do not copy over unused bits.  Make them DCs in our
		// copy.  Is this the correct behavior?
		if (sbv.BV.bits[j] == UNUSED)
			BV.bits[i] = DC;
		else
			BV.bits[i] = sbv.BV.bits[j];
		}
	// Did we get everything we asked for?
	if (k < size)
		BitSet(BV, k, size-k, LO);
	return *this;
	}

SubBitVector &
SubBitVector::operator=(BitVector &bv)
	{
	// Exactly like the above.
	int i = begin;
	int j = bv.begin;
	for (int k = 0; k < size ; i++, j++, k++)
		{
		if (j > bv.end)
			break;
		else if (BV.bits[i] == UNUSED)
			continue;
		if (bv.bits[j] == UNUSED)
			BV.bits[i] = DC;
		else
			BV.bits[i] = bv.bits[j];
		}
	if (k < size)
		BitSet(BV, k, size-k, LO);
	return *this;
	}

int
Size(SubBitVector &sbv)
	{ return sbv.size; }

ostream &
operator<<(ostream &s, SubBitVector &sbv)
	{
	// Since bits are kept least significant to most significant.
	// We do not need to reverse the order.
	for (int pos = sbv.end; pos >= sbv.begin; pos--)
		{
		switch (sbv.BV[pos])
			{
			case LO: s << '0'; break;
			case HI: s << '1'; break;
			case DC: s << 'x'; break;
			case Z: s << 'z'; break;
			case UNUSED: s << '.'; break;
			default: s << sbv.BV[pos]; break;
			}
		}
	return s;
	}

/*
 * BitVector
 */
 
BitVector::BitVector(Bits b)
	: msb_lower(FALSE)
	{
	if (b == UNUSED)
		{
		begin = -1;
		end = -2;
		size = 0;
		bits = NULL;
		}
	else
		{
		begin = end = 0;
		size = 1;
		bits = new Bits[size];
		bits[0] = b;
		}
	}

BitVector::BitVector(unsigned long value)
	: begin(0)
	{
	if (value == 0)
		{
		// A BitVector of 0,base10, is a bit of value 0.
		msb_lower = 0;
		end = 0;
		size = 1;
		bits = new Bits[size];
		bits[0] = LO;
		return;
		}
	// We don't know how big the number is.  So allocate enough to
	// store the largest value.  Then copy only what we need later.
	int largest = 8 * sizeof(dec_t);
	BitVector tmp(largest-1, 0);
	dec_t rem = value;
	for(int i = 0; rem > 0 && i < largest; i++)
		{
		// Now convert integer to binary.
		tmp.bits[i] = ((rem % 2) == 1) ? HI : LO;
		rem /= 2;
		}
	size = i;
	end = size - 1;
	bits = new Bits[size];
	memcpy(bits, tmp.bits, size);
	}

BitVector::BitVector(int ms, int ls)
	{
	// According to the Verilog LRM, <ls> can be greater than
	// <ms>.  Thus, we must be careful about the <begin>/<end>.
	if (ms < ls)
		{
		msb_lower = TRUE;
		begin = ms;
		end = ls;
		}
	else
		{
		msb_lower = FALSE;
		begin = ls;
		end = ms;
		}
	size = (end - begin) + 1;
	bits = new Bits[end+1];
	// If we begin offset from zero, we need to
	// setup the unused bits.
	if (begin > 0)
		memset(bits, UNUSED, begin);
	// Initialize array to don't cares.
	memset(bits+begin, DC, size);
	}

BitVector::BitVector(String &str, int base, int len)
	: msb_lower(FALSE), begin(0)
	{
	// Determine our length.
	if (len > 0)
		size = len;
	else
		{
		size = Length(str);
		// Convert to bit size.
		switch(base)
			{
			case 2: break;
			case 8: size *= 3; break;
			// We don't know the size for decimal.  Need to calculate.
			// So let the conversion function do it.  But, we
			// still need to allocate something for the memset.
			case 10: size = 1; break;
			case 16: size *= 4; break;
			}
		}
	end = size - 1;
	bits = new Bits[size];
	if (str != NULL)
		{
		// Zero out the bits.
		memset(bits, LO, size);
		// Now, convert to binary.
		switch(base)
			{
			case 2:
				bv_bin2bin(*this, str); break;
			case 8:
				bv_oct2bin(*this, str); break;
			case 10:
				bv_dec2bin(*this, str); break;
			case 16:
				bv_hex2bin(*this, str); break;
			default:
				memset(bits+begin, DC, size);
				break;
			}
		}
	else
		memset(bits+begin, DC, size);
	}

BitVector::BitVector(BitVector &bv)
	: msb_lower(bv.msb_lower), begin(bv.begin), end(bv.end), size(bv.size)
	{
	if (size > 0)
		{
		// Copy constructor copies everything.  Note: <size> might be
		// smaller than the actual length of <bits>.
		bits = new Bits[end+1];
		memcpy(bits, bv.bits, end+1);
		}
	else
		bits = NULL;
	}

BitVector::BitVector(SubBitVector &sbv)
	: msb_lower(FALSE), begin(0), size(sbv.size)
	{
	if (size > 0)
		{
		end = size - 1;
		bits = new Bits[size];
		memcpy(bits, sbv.BV.bits+sbv.begin, size);
		}
	else
		bits = NULL;
	}

BitVector::~BitVector()
	{
	if (bits != NULL)
		delete [] bits;
	}

Bits &
BitVector::operator[](int idx)
	{
	// This is for internal use only.  Thus, <idx> should not exceed
	// limits.  To reference a range of BitVector, use operator().
	return bits[idx];
	}

BitVector &
BitVector::operator=(BitVector &bv)
	{
	if (size == 0)
		{
		// If we were nothing before, copy everything
		// over now.
		msb_lower = bv.msb_lower;
		begin = bv.begin;
		end = bv.end;
		size = bv.size;
		// For
		//	reg [8:1] a;
		// size does not show the actual length.
		bits = new Bits[end+1];
		memcpy(bits, bv.bits, end+1);
		}
	else if (bv.size < size && bv.size != 0)
		{
		memset(bits+begin, LO, size);
		memcpy(bits+begin, bv.bits+bv.begin, bv.size);
		}
	else
		memcpy(bits+begin, bv.bits+bv.begin, size);
	return *this;
	}

BitVector &
BitVector::operator=(SubBitVector &sbv)
	{
	if (size == 0)
		{
		msb_lower = FALSE;
		begin = 0;
		size = sbv.size;
		end = size - 1;
		bits = new Bits[size];
		memcpy(bits, sbv.BV.bits+sbv.begin, size);
		}
	else if (sbv.size < size && sbv.size != 0)
		{
		memset(bits, LO, size);
		memcpy(bits, sbv.BV.bits+sbv.begin, size);
		}
	else
		memcpy(bits, sbv.BV.bits+sbv.begin, size);
	return *this;
	}

Bits &
BitVector::operator()(int idx)
	{
	static Bits errbit = DC;
	// If <idx> is out of range.  We must return a DC.
	if (idx < begin || idx > end)
		{
		if (errbit != DC)
			errbit = DC;
		return errbit;
		}
	if (msb_lower == TRUE)
		return bits[end - idx];
	return bits[idx];
	}

SubBitVector
BitVector::operator()(int ms, int ls)
	{
	// Even though <ms> and <ls> must be non-negative and <ms> greater
	// than <ls>, we must return something.  So we let the constructor
	// do the testing and return whatever the constructor constructed.
	SubBitVector res(*this, ms, ls);
	return res;
	}

BitVector::operator unsigned long(void)
	{
	unsigned long res = 0;
	for (int i = begin; i < 8*sizeof(res) && i <= end; i++)
		res += bits[i] * (1 << i);
	return res;
	}

/*
 * Friends of BitVector...
 */

int
operator==(BitVector &l, BitVector &r)
	{
	// Compare all data?
	if (l.size >= r.size)
		{
		BitVector tmp(l.size-1, 0);
		tmp = r;
		return (memcmp(l.bits+l.begin, tmp.bits+tmp.begin, l.size) == 0);
		}
	else
		{
		BitVector tmp(r.size-1, 0);
		tmp = l;
		return (memcmp(tmp.bits+tmp.begin, r.bits+r.begin, r.size) == 0);
		}
	}

int
operator!=(BitVector &l, BitVector &r)
	{ return !(l == r); }

int
operator<(BitVector &l, BitVector &r)
	{
	int ret = 0;
	if (l.size != r.size)
		{
		if (l.size > r.size)
			{
			BitVector tmp(l.size-1, 0);
			tmp = r;
			ret = bv_compare(l, tmp);
			}
		else
			{
			BitVector tmp(r.size-1, 0);
			tmp = l;
			ret = bv_compare(tmp, r);
			}
		}
	else
		ret = bv_compare(l, r);
	return (ret == COMPLT);
	}

int
operator>(BitVector &l, BitVector &r)
	{ return (r < l); }

int
operator<=(BitVector &l, BitVector &r)
	{ return !(r < l); }

int
operator>=(BitVector &l, BitVector &r)
	{ return !(l < r); }

int
operator!(BitVector &bv)
	{
	// Create bit vector of value zero, and compare it with <bv>.
	BitVector tmp(bv.size-1, 0);
	BitSet(tmp, 0, bv.size, LO);
	return (bv == tmp);
	}

BitVector
operator~(BitVector &bv)
	{
	BitVector res(bv.size-1, 0);
	int j = 0;
	Bits A;
	for (int i = bv.begin; i <= bv.end; i++, j++)
		{
		A = bv.bits[i];
		res.bits[j] = NOTTABLE[A];
		}
	return res;
	}

BitVector
operator|(BitVector &l, BitVector &r)
	{ return bv_logicops(l, r, ORTABLE); }

BitVector
operator&(BitVector &l, BitVector &r)
	{ return bv_logicops(l, r, ANDTABLE); }

BitVector
operator^(BitVector &l, BitVector &r)
	{ return bv_logicops(l, r, XORTABLE); }

BitVector
operator+(BitVector &l, BitVector &r)
	{
	// The result of an addition supports the carryout bit.  But
	// if the user doesn't save this bit, it is destroyed upon
	// call to destructor.  I'm not sure if Verilog has this behavior.
	int size = (l.size > r.size) ? l.size : r.size;
	BitVector res(size, 0);		// Usually size-1 length.
	BitSet(res, 0, size, LO);	// Initialize to zero.
	int pad = 0;
	Bits A, B;
	Bits ci = LO;
	for (int pos = 0; pos <= res.end; pos++)
		{
		A = B = UNUSED;
		// Have we reached the end of one of our vectors?
		if (pos > r.end)
			{
			// If so, are there still something to add?
			if (ci == LO)
				{
				if (l.end > r.end)
					memcpy(res.bits+pos, l.bits+pos,
						(l.end+1)-pos);
				return res;
				}
			else
				B = LO;
			}
		if (pos > l.end)
			{
			if (ci == LO)
				{
				if (r.end > l.end)
					memcpy(res.bits+pos, l.bits+pos,
						(l.end+1)-pos);
				return res;
				}
			else
				A = LO;
			}
		// Now try to add the vectors.
		if (A == UNUSED)
			A = l.bits[pos];
		if (B == UNUSED)
			B = r.bits[pos];
		pad = CONCAT(A, B, ci);
		if (pad < 0)
			{
			// A DC or Z was found.  Set the rest of the
			// result to DC's.
			BitSet(res, pos, (res.end+1)-pos, DC);
			return res;
			}
		res[pos] = ADDTABLE[pad].sum;
		ci = ADDTABLE[pad].carry;
		}
	return res;
	}

BitVector
operator-(BitVector &l, BitVector &r)
	{
	int size = (l.size > r.size) ? l.size : r.size;
	BitVector One(HI), res(size-1, 0);
	BitVector ltmp(size-1, 0), rtmp(size-1, 0);
	ltmp = l;
	rtmp = r;				// Extend our vectors to the correct size.
	rtmp = ~rtmp + One;		// 2's compliment
	SignExtend(ltmp, l.end);
	// Do we need to sign extend <r>?  Of course, we lose nothing from 
	// doing so.
	SignExtend(rtmp, r.end);
	res = ltmp + rtmp;
	return res;
	}

BitVector
operator<<(BitVector &bv, int amt)
	{
	// Shift <bv> to the left by <amt>.  Prepend zeros to the
	// beginning of the result.
	BitVector res(bv.size-1, 0);
	BitSet(res, 0, amt, LO);
	memcpy(res.bits+amt, bv.bits+bv.begin, res.size-amt);
	return res;
	}

BitVector
operator>>(BitVector &bv, int amt)
	{
	// Shift <bv> to the right by <amt>.  Appending zeros to the
	// front.
	BitVector res(bv.size-1, 0);
	int start = bv.begin + amt;
	memcpy(res.bits, bv.bits+start, (bv.end+1)-start);
	BitSet(res, (bv.end+1)-amt, amt, LO);
	return res;
	}

void
BitSet(BitVector &bv, int start, int len, Bits val)
	{
	if (len > 0 && start >= 0)
		memset(bv.bits+start, val, len);
	}

void
SignExtend(BitVector &bv, int end)
	{
	int size = bv.end - end;
	if (size > 0)
		{
		Bits hb = bv.bits[end];
		BitSet(bv, end, size, hb);
		}
	}

String
Convert(BitVector &bv, int base, int len)
	{
	// <Rnd> rounds the size given in <s> to the next higher size of the
	// base given in <b>.  I.e.  if s = 31, and b = 3 (octal), then the
	// next higher size is 33.  Thus, <Rnd> will evaluate to 33.
#define Rnd(s,b)	((s%b)==0 ? s : s + (b-s%b))
	int tmp[17] = { -1,-1,1,-1,-1,-1,-1,-1,3,-1,-1,-1,-1,-1,-1,-1,4 };

	// If the base is decimal (10), call the convert function for it.
	if (base == 10)
		return bv_bin2dec(bv, len);
	else if (base > 16 || tmp[base] < 1)
		return make_x(len);

	// For base 2, 8 and 16, the conversion process is almost the same.
	int size;
	if (len > 0)
		size = len;
	else
		size = Rnd(bv.size,tmp[base]) / tmp[base];
	if (size <= 0)
		return make_x(len);

	String res(NULL, size);
	Strset(res, '0');		// Remove junk in the string.
	int pad, j = bv.begin;
	Bits A,B,C,D;
	for (int i = size-1; j <= bv.end && i >= 0; i--)
		{
		switch(base)
			{
			case 2:
				if (bv[j] > HI)
					{
					pad = -1;
					j++;
					}
				else
					pad = bv[j++];
				break;
			case 8:
				if (j > bv.end)
					{
					pad = -1;
					break;
					}
				A = bv.bits[j++];
				if (j > bv.end)
					{
					pad = CONCAT(A,LO,LO);
					break;
					}
				B = bv.bits[j++];
				if (j > bv.end)
					{
					pad = CONCAT(A,B,LO);
					break;
					}
				C = bv.bits[j++];
				pad = CONCAT(A,B,C);
				break;
			case 16:
				if (j > bv.end)
					{
					pad = -1;
					break;
					}
				A = bv.bits[j++];
				if (j > bv.end)
					{
					pad = CONCAT(A,LO,LO,LO);
					break;
					}
				B = bv.bits[j++];
				if (j > bv.end)
					{
					pad = CONCAT(A,B,LO,LO);
					break;
					}
				C = bv.bits[j++];
				if (j > bv.end)
					{
					pad = CONCAT(A,B,C,LO);
					break;
					}
				D = bv.bits[j++];
				pad = CONCAT(A,B,C,D);
				break;
			}
		// Convert to ASCII.
		if (pad > 9)
			{
			pad -= 10;
			res[i] = pad + 'a';
			}
		else if (pad >= 0)
			res[i] = pad + '0';
		else
			res[i] = 'x';
		}
	return res;
#undef Rnd
	}

int
Size(BitVector &bv)
	{ return bv.size; }

int
StartBit(BitVector &bv)
	{ return bv.begin; }

int
EndBit(BitVector &bv)
	{ return bv.end; }

ostream &
operator<<(ostream &s, BitVector &bv)
	{
	for (int i = bv.end; i >= bv.begin; i--)
		{
		switch(bv.bits[i])
			{
			case LO: s << '0'; break;
			case HI: s << '1'; break;
			case DC: s << 'x'; break;
			case Z: s << 'z'; break;
			case UNUSED: s << '.'; break;
			default: s << bv.bits[i]; break;
			}
		}
	return s;
	}
