/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * Bitvector.h
 *
 * Bit vector class.  Needed to represent don't cares in our bit vector.
 *
 * Things to note:
 * (1)  <begin> and <end> are inclusive indices.  This means they are
 * the actual beginning and ending indices.
 * (2)  <Bits> type is of unsigned char.  This allows the use of memcpy,
 * memcmp, memset, etc, for various array manipulation.
 * (3)  Logic operations are done through table lookup.  This will
 * increase program size, but speed up execution.  Support for non-table
 * lookup will be added later.
 * (4)  Bits are internally kept least significant bit to most significant
 * bit.  Thus, a conversion is needed for declarations where the ls_exp
 * is larger than the ms_exp.
 * (5)  When a decimal string (char *) is passed as input to the
 * constructor, the number must be less than <dec_t> size.  Else, we
 * can't store the temporary number.
 */

#ifndef BITVECTOR_H
#define BITVECTOR_H

#include <string.h>
#include <iostream.h>
#include "String.h"

/*
 * Size of a decimal number.  This should be set to the largest integer
 * size that the machine can handle.
 */
#ifndef DECTYPE
# ifdef __GNUG__
	typedef long long int dec_t;
# else
	typedef long int dec_t;
# endif
#else
	typedef DECTYPE dec_t;
#endif

/*
 * How can one make enum an unsigned char or 8 bits?  Until a way is
 * found, we use this as a temporary fix.
 */
typedef unsigned char Bits;
const Bits LO = 0;
const Bits HI = 1;
const Bits DC = 2;
const Bits  Z = 3;
const Bits UNUSED = 4;
#define NUMBITS 5

/*
 * A partial bit vector representation.  Used for range assignments or
 * references.  This class is based on FSF's G++ implementation of Strings.
 */
class BitVector;
class SubBitVector
	{
	// Give BitVector access to constructor.
	friend class BitVector;

	BitVector &BV;		// The original bit vector we are part of.
	int begin;		// Starting position.
	int end;		// Stopping position.
	int size;		// Size of this sub-bit vector.

	// Only BitVector can create a SubBitVector object.
	SubBitVector(BitVector &, int, int);
	SubBitVector(SubBitVector &);
public:
	SubBitVector &operator=(SubBitVector &);
	SubBitVector &operator=(BitVector &);
	friend int Size(SubBitVector &);
	friend ostream &operator<<(ostream &, SubBitVector &);
	// We don't need any other method, yet!
	};

/*
 * Class representing a vector of bits.
 */
class BitVector
	{
	// Give SubBitVector access to our datas.
	friend class SubBitVector;

	// Verilog allows both:
	//	reg [16:0] a;
	//	reg [0:15] b;
	// Therefore, we must keep track of which was used.  Msb is lower
	// for the second case.  Note:  this is only used in range access.
	// I.e. internal bit storage is from least to most significant bit.
	// So arithmetic and logic operations are not affected.  This is
	// mostly used for error checking.
	int msb_lower;
	int begin;
	int end;
	int size;
	Bits *bits;
	friend BitVector bv_logicops(BitVector &, BitVector &,
		Bits [][NUMBITS]);
	friend int bv_store(BitVector &, int &,
		Bits, Bits, Bits, Bits = UNUSED);
	friend int bv_compare(BitVector &, BitVector &);
	friend String bv_bin2dec(BitVector &, int = 0);
	friend void bv_bin2bin(BitVector &, String &);
	friend void bv_oct2bin(BitVector &, String &);
	friend void bv_dec2bin(BitVector &, String &);
	friend void bv_hex2bin(BitVector &, String &);
public:
	BitVector(Bits = UNUSED);
	BitVector(unsigned long);
	BitVector(int, int);
	BitVector(String &, int = 2, int = 0);
	BitVector(BitVector &);
	BitVector(SubBitVector &);
	~BitVector();
	Bits &operator[](int);
	BitVector &operator=(BitVector &);
	BitVector &operator=(SubBitVector &);
	Bits &operator()(int);
	SubBitVector operator()(int, int);
	operator unsigned long(void);
	friend int operator==(BitVector &, BitVector &);
	friend int operator!=(BitVector &, BitVector &);
	friend int operator<(BitVector &, BitVector &);
	friend int operator<=(BitVector &, BitVector &);
	friend int operator>(BitVector &, BitVector &);
	friend int operator>=(BitVector &, BitVector &);
	friend int operator!(BitVector &);
	friend BitVector operator~(BitVector &);
	friend BitVector operator|(BitVector &, BitVector &);
	friend BitVector operator&(BitVector &, BitVector &);
	friend BitVector operator^(BitVector &, BitVector &);
	friend BitVector operator+(BitVector &, BitVector &);
	friend BitVector operator-(BitVector &, BitVector &);
	friend BitVector operator<<(BitVector &, int);
	friend BitVector operator>>(BitVector &, int);
	friend void BitSet(BitVector &, int, int, Bits);
	friend void SignExtend(BitVector &, int);
	friend int Size(BitVector &);
	friend int StartBit(BitVector &);
	friend int EndBit(BitVector &);
	friend String Convert(BitVector &, int = 10, int = 0);
	friend ostream &operator<<(ostream &, BitVector &);
	};

#endif // BITVECTOR_H
