/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * Error.cc
 *
 * Modularized error messages.  See Error.h on how to add more error
 * detection.
 */

#include <iostream.h>
#include <stdarg.h>
#include <stdlib.h>
#include "String.h"
#include "Error.h"

/*
 * progname: filename[Sim_lineno] -Sim_errno- ErrMsg (arg)
 *
 * Example:
 *
 * vbs: test.v[101] -4- syntax error, unexpected symbol (xxx)
 */

String Sim_programname;			// Our name to the system.
String Sim_filename("Unknown");	// File name with error.
int Sim_lineno = 0;				// Line number of error.
Errno Sim_errno;				// Error value.

String Sim_errmsg[] =
	{
	// General Errors...
	String("no errors or warnings"),
	String("construct not yet supported"),
	String("internal simulator error"),
	String("no top-level module found"),
	String("invalid range"),
	String("order of indices are wrong"),
	String("only one default case is allowed"),
	String("invalid type for this usage"),
	String("not an lvalue"),
	String("repeated operation"),
	String("incomplete type, needs storage area"),

	// Errors related to symbol table.
	String("symbol table not initialized"),
	String("symbol table is full"),
	String("duplicate symbol in symbol table"),
	String("undefined symbol"),

	// Paser errors.
	String("unexpected symbol"),
	String("unexpected keyword"),
	String("unexpected end of file"),

	// Errors related to task and function calls.
	String("not a task or system task"),
	String("not a function or system function"),
	String("Wrong number of arguments"),
	String("missing type specifier in format string"),
	String("unknown type specifier in format string"),
	String("invalid symbol in format string"),
	String("no termination in format string")
	};

void
Sim_perror(int flag, char *fmt, ... )
	{
	va_list ap;

	// Start our list of args.
	va_start(ap, fmt);

	cout << endl << Sim_programname << ": ";
	// Not yet supported.
	cout	<< Sim_filename << '[';
	if (Sim_lineno > 0)
		cout << Sim_lineno;
	cout << "] ";

	if (Sim_errno != SE_NONE)
		cout << "-" << Sim_errno;
	cout << "- " << Sim_errmsg[Sim_errno];

	cout << " (";
	if (strcmp(fmt, "%d") == 0)
		{
		int val;
		val = va_arg(ap, int);
		cout << val;
		}
	else if (strcmp(fmt, "%s") == 0)
		{
		char *val;
		val = va_arg(ap, char *);
		cout << '\'' << val << '\'';
		}
	cout << ")." << endl;

	// Clean up the args.
	va_end(ap);

	cout << endl;
	if (flag & SEF_SIM)
		cout << "End simulation...";
	else if (flag & SEF_SETUP)
		cout << "End setup...";
	else if (flag & SEF_COMP)
		cout << "End compilation...";
	else
		cout << "End unknown state...";
	cout << endl;

	if (flag & SEF_EXIT)
		exit(Sim_errno);
	}

void
Sim_seterror(Errno num, String &fn, int ln)
	{
	Sim_errno = num;
	Sim_filename = fn;
	Sim_lineno = ln;
	}
