/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * Event.h
 *
 * Event queue class to support "always @" statements.  This class uses
 * templates because we want to test it with any type.  Even though we
 * know that the event queue will be instantiated with type "Stmts", we
 * can not test this object if we make "Stmts" a static type.  In other
 * words;  if we remove the templates, we have to use complete statement
 * objects to test this class.  This is not easy since statements are
 * large objects.  As a result, this class will stay a template until a
 * way can be found to test this object without creating an entire
 * statement object.
 */

#ifndef EVENT_H
#define EVENT_H

#include "Bool.h"
#include "List.h"

// Trigger types.
enum TrigType	{
		TRIG_NEG	= 0x00000004,
		TRIG_POS	= 0x00000002,
		TRIG_CHNG	= 0x00000001,
		TRIG_NONE	= 0x00000000
		};

template<class T> class EvntQueue;

template<class T>
class Events
	{
	Bool queued;		// Were we queued already?
	T obj;			// Object to trigger.
public:
	TrigType trigger;	// Trigger method.
	int bit;		// Bit being monitored.
	Events(T t)
		: queued(FALSE), trigger(TRIG_NONE), bit(-1) { obj = t; }
	Events(TrigType type, T t)
		: queued(FALSE), trigger(type), bit(-1) { obj = t; }
	void Sim_trigger(void (*handler)(T))
		{
		handler(obj);
		queued = FALSE;
		}
	void QueueEvnt(EvntQueue<T> &eq, TrigType type)
		{
		// If not queued, queue it if the type is correct and
		// change the status.
		if (!queued && (type & trigger) > 0)
			{
			// Do not allocate a new Events object.
			// We must be able to modify <queued> when
			// this object is triggered.
			eq += this;
			queued = TRUE;
			}
		}
	};

template<class T>
class EvntQueue
	{
	List< Events<T> *> stmts;	// List of events to trigger.
public:
	EvntQueue()
		{}
	EvntQueue &operator+=(Events<T> *ev)
		{ stmts += ev; return *this; }
	void Sim_trigger(void (*handler)(T))
		{
		// Trigger and delete each event in the list
		Events<T> *ev;
		Reset(stmts);
		while (Data(stmts, ev))
			// Trigger this event.  Do not delete this
			// event object.  It is not a copy, see Events
			// class for reason...
			ev->Sim_trigger(handler);

		// Deleting the list does not delete the objects, just the
		// nodes.
		Delete(stmts);
		}
	};

#endif // EVENT_H
