/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * List.h
 *
 * List template class.  We should use STL (standard template library)
 * instead of this.  This is not a good implementation.  But STL is not
 * yet adopted as standard, so we'll stick with this until it becomes
 * standard.
 */

#ifndef LIST_H
#define LIST_H

#include <iostream.h>

template<class T>
class Listnode
	{
	friend class List<T>;
	friend class Stack<T>;
	T data;
	Listnode<T> *next;
public:
	Listnode(T t, Listnode<T> *n = NULL)
		{ data = t; next = n; }
	friend ostream &operator<<(ostream &s, Listnode<T> &n)
		{ s << n.data; return s; }
	};

template<class T>
class List
	{
	int size;
	int curidx;
	Listnode<T> *head;
	Listnode<T> *tail;
	Listnode<T> *curnode;
public:
	List()
		: size(0), curidx(-1)
		{ head = tail = curnode = NULL; }
	List(List<T> &lst)
		: size(0), curidx(-1)
		{ head = tail = curnode = NULL; *this += lst; }
	~List()
		// Need a function to delete the nodes.  The Destructor
		// is passed a pointer.  Delete() is passed a reference.
		{ Delete(*this); }
	T operator[](int idx)
		{
		// Do not use this operator if you expect the size of
		// the list to change while you are referencing the list.
		// I.e. the Size() function is inline.  Thus it does not
		// update if you append another node into the list.
		int i;
		Listnode<T> *tmp;
		if (curidx >= 0 && idx >= curidx)
			{
			tmp = curnode;	// Skip to current node.
			i = curidx;
			}
		else
			{
			tmp = head;	// Must start from beginning.
			i = 0;
			}
		for (; tmp != NULL && i < idx; i++)
			tmp = tmp->next;
		T dummy;
		return (tmp == NULL) ? dummy : tmp->data;
		}
	List<T> &operator=(List<T> &lst)
		{
		// Remove old list.
		Delete(*this);
		// Append new list.
		Listnode<T> *tmp = lst.head;
		while (tmp != NULL)
			{
			*this += tmp->data;
			tmp = tmp->next;
			}
		return *this;
		}
	// Append <lst> to our list.
	List<T> &operator+=(List<T> &lst)
		{
		Listnode<T> *tmp = lst.head;
		while (tmp != NULL)
			{
			*this += tmp->data;
			tmp = tmp->next;
			}
		return *this;
		}
	// Append <data> into our list.
	List<T> &operator+=(T data)
		{
		// Add to the end of the list.
		Listnode<T> *tmp = new Listnode<T>(data);
		if(tail != NULL)
			{
			tail->next = tmp;
			tail = tmp;
			if (curidx >= 0 && curnode == NULL)
				curnode = tmp;
			}
		else			// First Node!!!
			{
			curidx = 0;
			curnode = tmp;
			head = tmp;
			tail = tmp;
			}
		size++;
		return *this;
		}
	// Get the next node, <curnode>.
	//	Return 1 if data available
	//	Return 0 if data is not available
	friend int Data(List<T> &lst, T &data)
		{
		// <flag> needed because this function is inline,
		// can't continue after return in inline function.
		int flag = 0;
		if (lst.curnode != NULL)
			{
			data = lst.curnode->data;
			lst.curnode = lst.curnode->next;
			lst.curidx++;
			flag++;		// Return OK!
			}
		return flag;
		}
	// Free the nodes, but don't touch the data.
	friend void Delete(List<T> &lst)
		{
		Listnode<T> *tmp;
		while (lst.head != NULL)
			{
			tmp = lst.head;
			lst.head = lst.head->next;
			delete tmp;
			}
		lst.head = lst.tail = lst.curnode = NULL;
		lst.size = 0;
		lst.curidx = -1;
		}
	// Delete node index <idx> from list <lst>.
	friend void Delete(List<T> &lst, int idx)
		{
		Listnode<T> *prevnode = NULL;
		Listnode<T> *node = lst.head;
		if (idx == 0)
			{
			// If we want to delete the first node, we have
			// to modify <head>.  <node> has already been
			// set to <lst.head>.
			lst.head = lst.head->next;
			// Make sure we update other information.  The
			// current index should be decremented only if
			// it is greater than the index that was deleted.
			if (lst.curnode == node)
				lst.curnode = node->next;
			if (lst.curidx > idx)
				lst.curidx--;
			delete node;
			lst.size--;
			}
		else
			{
			for (int i = 0; i < idx && node != NULL; i++)
				{
				prevnode = node;
				node = node->next;
				}
			// Did we get our node, or have we past the end?
			// If the user specified an index larger than our
			// size, then of course that node doesn't exist,
			// so no need delete it.
			if (node != NULL)
				{
				prevnode->next = node->next;
				if (lst.curnode == node)
					lst.curnode = node->next;
				if (lst.curidx > idx)
					lst.curidx--;
				delete node;
				lst.size--;
				}
			}
		}
	friend void Reset(List<T> &lst)
		{
		lst.curidx = 0;
		lst.curnode = lst.head;
		}
	friend int Size(List<T> &lst) { return lst.size; }
	friend ostream &operator<<(ostream &s, List<T> &lst)
		{
		Listnode<T> *tmp;
		tmp = lst.head;
		while(tmp != NULL)
			{
			s << *tmp << ' ';
			tmp = tmp->next;
			}
		return s;
		}
	};

/*
 * Stack template class
 */

template<class T>
class Stack
	{
	int size;
	Listnode<T> *top;
public:
	Stack()
		: size(0) { top = NULL; }
	Stack(Stack<T> &stk)
		: size(0)
		{
		top = NULL;
		// Set <top> first.
		Listnode<T> *tmp1 = stk.top;
		if (tmp1 != NULL)
			{
			top = new Listnode<T>(tmp1->data);
			tmp1 = tmp1->next;
			size++;
			}
		// Now copy the rest.
		Listnode<T> *tmp2, *nnode;
		for (tmp2 = top; tmp1 != NULL; size++, tmp1 = tmp1->next)
			{
			nnode = new Listnode<T>(tmp1->data);
			tmp2->next = nnode;
			tmp2 = tmp2->next;
			}
		}
	~Stack(void)
		{
		Listnode<T> *tmp, *tmp2;
		for (tmp = top; tmp != NULL;)
			{
			tmp2 = tmp;
			tmp = tmp->next;
			delete tmp2;
			}
		top = NULL;
		size = 0;
		}
	Stack<T> &operator=(Stack<T> &stk)
		{
		// Delete old nodes.
		for (int i = 0; i < size; i++)
			Pop(*this);
		// Set <top> first.
		Listnode<T> *tmp1 = stk.top;
		if (tmp1 != NULL)
			{
			top = new Listnode<T>(tmp1->data);
			tmp1 = tmp1->next;
			size++;
			}
		// Now copy the rest.
		Listnode<T> *tmp2, *nnode;
		for (tmp2 = top; tmp1 != NULL; size++, tmp1 = tmp1->next)
			{
			nnode = new Listnode<T>(tmp1->data);
			tmp2->next = nnode;
			tmp2 = tmp2->next;
			}
		return *this;
		}
	friend void Push(Stack<T> &stk, T obj)
		{
		// Make new node and add to list.
		Listnode<T> *nnode = new Listnode<T>(obj, stk.top);
		stk.top = nnode;
		stk.size++;
		}
	friend T Pop(Stack<T> &stk)
		{
		T obj;
		if (stk.top != NULL)
			{
			// Take node off list.
			Listnode<T> *n = stk.top;
			stk.top = stk.top->next;
			// Get data and delete node.
			obj = n->data;
			delete n;
			stk.size--;
			}
		return obj;
		}
	friend int Size(Stack<T> &stk)
		{ return stk.size; }
	friend ostream &operator<<(ostream &s, Stack<T> &stk)
		{
		Listnode<T> *tmp;
		for (tmp = stk.top; tmp != NULL; tmp = tmp->next)
			s << *tmp << " ";
		return s;
		}
	};

#endif // LIST_H
