/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * PExpr.h
 *
 * Expression parser types.  This file will reduce the size of PTypes.h
 */

#ifndef PEXPR_H
#define PEXPR_H

#include <iostream.h>
#include "Bool.h"
#include "String.h"
#include "Symtab.h"		// HashValue
#include "Base.h"

/* Constant Expression:  constant number, 14, 55, etc... */
class ConstExpression : public Expression
	{
	Number amt;
public:
	ConstExpression(unsigned long n = 0)
		: amt(n) {}
	ConstExpression(int size, unsigned long n)
		: amt(size,n) {}
	ConstExpression(Number &num)
		: amt(num) {}
	ConstExpression(ConstExpression &ce)
		: Expression(ce), amt(ce.amt) {}
	ConstExpression &operator=(ConstExpression &ce)
		{ amt = ce.amt; return *this; }
	Expression *copy(void)
		{ return new ConstExpression(*this); }
	ostream &display(ostream &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);
	friend ostream &operator<<(ostream &s, ConstExpression &cexpr)
		{ s << Convert(cexpr.amt, 10); return s; }
	};

/* Range:  [1:4], [6:13], etc... */
class Range
	{
	ConstExpression le;	// constant expression
	ConstExpression re;	// constant expression
public:
	Range(unsigned long l = 0, unsigned long r = 0)
		: le(l), re(r) {}
	Range(ConstExpression &l, ConstExpression &r)
		: le(l), re(r) {}
	Range(Range &r)
		: le(r.le), re(r.re) {}
	friend void GetRangeData(Range &r, Number &lnum, Number &rnum)
		{
		lnum = r.le.Sim_evaluate();
		rnum = r.re.Sim_evaluate();
		}
	friend ostream &operator<<(ostream &s, Range &r)
		{ s << '[' << r.le << ':' << r.re << ']'; return s; }
	};

/* Range identifier:  any identifier [optional range] */
class RangeId : public Expression
	{
	// We must keep track of whether we have a range or not.
	// Not having a range doesn't necessarily mean that we
	// have only one bit.  If we are a bitvector, no range means
	// the entire vector.
	Bool flag;
	String name;		// Name of variable
	Range range;		// Optional range
public:
	HashValue idx;		// Index into symbol table.
	RangeId()
		: flag(FALSE) {}
	RangeId(String &s)
		: flag(FALSE), name(s) {}
	RangeId(String &s, Range &r)
		: flag(TRUE), name(s), range(r) {}
	RangeId(RangeId &r)
		: Expression(r), flag(r.flag), name(r.name), range(r.range)
		{}
	operator String()
		{ return name; }
	operator Range()
		{ return range; }
	HashValue *Index(void)
		{ return &idx; }
	int GetRange(unsigned long &ms, unsigned long &ls)
		{
		if (flag)
			{
			Number lnum, rnum;
			GetRangeData(range, lnum, rnum);
			ms = lnum;
			ls = rnum;
			return 1;
			}
		return 0;
		}
	Expression *copy(void)
		{ return new RangeId(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);
	friend Bool HaveRange(RangeId &id)
		{ return id.flag; }
	friend ostream &operator<<(ostream &s, RangeId &r)
		{ return r.display(s); }
	};

/* Unary Expressions:  "!" operator */
class NotOp: public Expression
	{
	Expression *exp;
public:
	NotOp()
		{ exp = NULL; }
	NotOp(Expression &p)
		{ exp = p.copy(); }
	NotOp(NotOp &nop)
		: Expression(nop)
		{ exp = nop.exp->copy(); }
	virtual ~NotOp()
		{ if (exp) delete exp; }
	Expression *copy(void)
		{ return new NotOp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);
	friend ostream &operator<<(ostream &s, NotOp &no)
		{ return no.display(s); }
	};

/* Unary expression:  "~" operaror */
class InvertOp : public Expression
	{
	Expression *exp;
public:
	InvertOp()
		{ exp = NULL; }
	InvertOp(Expression &p)
		{ exp = p.copy(); }
	InvertOp(InvertOp &inv)
		: Expression(inv)
		{ exp = inv.exp->copy(); }
	virtual ~InvertOp()
		{ if (exp) delete exp; }
	Expression *copy()
		{ return new InvertOp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);
	friend ostream &operator<<(ostream &s, InvertOp &io)
		{ return io.display(s); }
	};

/* Binary expression: '+' add operator */
class AddOp : public Expression
	{
	Expression *exp1;
	Expression *exp2;
public:
	AddOp()
		{
		exp1 = NULL;
		exp2 = NULL;
		}
	AddOp(Expression &p1, Expression &p2)
		{
		exp1 = p1.copy();
		exp2 = p2.copy();
		}
	AddOp(AddOp &ad)
		: Expression(ad)
		{
		exp1 = ad.exp1->copy();
		exp2 = ad.exp2->copy();
		}
	virtual ~AddOp()
		{
		if (exp1) delete exp1;
		if (exp2) delete exp2;
		}
	Expression *copy()
		{ return new AddOp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);	
	friend ostream &operator<<(ostream &s, AddOp &ao)
		{ return ao.display(s); }
	};

/* Binary expression: '-' add operator */
class SubOp : public Expression
	{
	Expression *exp1;
	Expression *exp2;
public:
	SubOp()
		{
		exp1 = NULL;
		exp2 = NULL;
		}
	SubOp(Expression &p1, Expression &p2)
		{
		exp1 = p1.copy();
		exp2 = p2.copy();
		}
	SubOp(SubOp &sub)
		: Expression(sub)
		{
		exp1 = sub.exp1->copy();
		exp2 = sub.exp2->copy();
		}
	virtual ~SubOp()
		{
		if (exp1) delete exp1;
		if (exp2) delete exp2;
		}
	Expression *copy()
		{ return new SubOp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);	
	friend ostream &operator<<(ostream &s, SubOp &so)
		{ return so.display(s); }
	};

/* Binary expression: '==' comparator */ 
class EqEqComp : public Expression
	{
	Expression *exp1;
	Expression *exp2;
public:
	EqEqComp()
		{
		exp1 = NULL;
		exp2 = NULL;
		}
	EqEqComp(Expression &p1, Expression &p2)
		{
		exp1 = p1.copy();
		exp2 = p2.copy();
		}
	EqEqComp(EqEqComp &eqeq)
		: Expression(eqeq)
		{
		exp1 = eqeq.exp1->copy();
		exp2 = eqeq.exp2->copy();
		}
	virtual ~EqEqComp()
		{
		if (exp1) delete exp1;
		if (exp2) delete exp2;
		}
	Expression *copy()
		{ return new EqEqComp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);
	friend ostream &operator<<(ostream &s, EqEqComp &eeCmp)
		{ return eeCmp.display(s); }	
	};

/* Binary expression: '!=' Not comparator */
class NotEqComp : public Expression
	{
	Expression *exp1;
	Expression *exp2;
public:
	NotEqComp()
		{
		exp1 = NULL;
		exp2 = NULL;
		}
	NotEqComp(Expression &p1, Expression &p2)
		{
		exp1 = p1.copy();
		exp2 = p2.copy();
		}
	NotEqComp(NotEqComp &notEq)
		: Expression(notEq)
		{
		exp1 = notEq.exp1->copy();
		exp2 = notEq.exp2->copy();
		}
	virtual ~NotEqComp()
		{
		if (exp1) delete exp1;
		if (exp2) delete exp2;
		}
	Expression *copy()
		{ return new NotEqComp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);
	friend ostream &operator<<(ostream &s, NotEqComp &neCmp)
		{ return neCmp.display(s); }
	};

/* Binary expression: '||' Union of expressions */
class OrOrOp : public Expression
	{
	Expression *exp1;
	Expression *exp2;
public:
	OrOrOp()
		{
		exp1 = NULL;
		exp2 = NULL;
		}
	OrOrOp(Expression &p1, Expression &p2)
		{
		exp1 = p1.copy();
		exp2 = p2.copy();
		}
	OrOrOp(OrOrOp &orOr)
		: Expression(orOr)
		{
		exp1 = orOr.exp1->copy();
		exp2 = orOr.exp2->copy();
		}
	virtual ~OrOrOp()
		{
		if (exp1) delete exp1;
		if (exp2) delete exp2;
		}
	Expression *copy()
		{ return new OrOrOp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);	
	friend ostream &operator<<(ostream &s, OrOrOp &ooOp)
		{ return ooOp.display(s); }
	};

/* Binary expression: '&&' Intersection of expressions */
class AndAndOp : public Expression
	{
	Expression *exp1;
	Expression *exp2;
public:
	AndAndOp()
		{
		exp1 = NULL;
		exp2 = NULL;
		}
	AndAndOp(Expression &p1, Expression &p2)
		{
		exp1 = p1.copy();
		exp2 = p2.copy();
		}
	AndAndOp(AndAndOp &andAnd)
		: Expression(andAnd)
		{
		exp1 = andAnd.exp1->copy();
		exp2 = andAnd.exp2->copy();
		}
	virtual ~AndAndOp()
		{
		if (exp1) delete exp1;
		if (exp2) delete exp2;
		}
	Expression *copy()
		{ return new AndAndOp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);	
	friend ostream &operator<<(ostream &s, AndAndOp &aaOp)
		{ return aaOp.display(s); }
	};

/* Binary expression: '|' bitwise or of two expressions */
class LogOrOp : public Expression
	{
	Expression *exp1;
	Expression *exp2;
public:
	LogOrOp()
		{
		exp1 = NULL;
		exp2 = NULL;
		}
	LogOrOp(Expression &p1, Expression &p2)
		{
		exp1 = p1.copy();
		exp2 = p2.copy();
		}
	LogOrOp(LogOrOp &logOr)
		: Expression(logOr)
		{
		exp1 = logOr.exp1->copy();
		exp2 = logOr.exp2->copy();
		}
	virtual ~LogOrOp()
		{
		if (exp1) delete exp1;
		if (exp2) delete exp2;
		}
	Expression *copy()
		{ return new LogOrOp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);	
	friend ostream &operator<<(ostream &s, LogOrOp &logOr)
		{ return logOr.display(s); }
	};

/* Binary expression: '&' bitwise and of two expressions */
class LogAndOp : public Expression
	{
	Expression *exp1;
	Expression *exp2;
public:
	LogAndOp()
		{
		exp1 = NULL;
		exp2 = NULL;
		}
	LogAndOp(Expression &p1, Expression &p2)
		{
		exp1 = p1.copy();
		exp2 = p2.copy();
		}
	LogAndOp(LogAndOp &logAnd)
		: Expression(logAnd)
		{
		exp1 = logAnd.exp1->copy();
		exp2 = logAnd.exp2->copy();
		}
	virtual ~LogAndOp()
		{
		if (exp1) delete exp1;
		if (exp2) delete exp2;
		}
	Expression *copy()
		{ return new LogAndOp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);	
	friend ostream &operator<<(ostream &s, LogAndOp &aaOp)
		{ return aaOp.display(s); }
	};

/* Binary expression: "<<" Logical left shift */
class LogLShiftOp : public Expression
	{
	Expression *exp1;
	Expression *exp2;
public:
	LogLShiftOp()
		{
		exp1 = NULL;
		exp2 = NULL;
		}
	LogLShiftOp(Expression &p1, Expression &p2)
		{
		exp1 = p1.copy();
		exp2 = p2.copy();
		}
	LogLShiftOp(LogLShiftOp &logLS)
		: Expression(logLS)
		{
		exp1 = logLS.exp1->copy();
		exp2 = logLS.exp2->copy();
		}
	virtual ~LogLShiftOp()
		{
		if (exp1) delete exp1;
		if (exp2) delete exp2;
		}
	Expression *copy()
		{ return new LogLShiftOp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);	
	friend ostream &operator<<(ostream &s, LogLShiftOp &lsOp)
		{ return lsOp.display(s); }
	};

/* Binary expression: ">>" Logical right shift */
class LogRShiftOp : public Expression
	{
	Expression *exp1;
	Expression *exp2;
public:
	LogRShiftOp()
		{
		exp1 = NULL;
		exp2 = NULL;
		}
	LogRShiftOp(Expression &p1, Expression &p2)
		{
		exp1 = p1.copy();
		exp2 = p2.copy();
		}
	LogRShiftOp(LogRShiftOp &logRS)
		: Expression(logRS)
		{
		exp1 = logRS.exp1->copy();
		exp2 = logRS.exp2->copy();
		}
	virtual ~LogRShiftOp()
		{
		if (exp1) delete exp1;
		if (exp2) delete exp2;
		}
	Expression *copy()
		{ return new LogRShiftOp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);	
	friend ostream &operator<<(ostream &s, LogRShiftOp &rsOp)
		{ return rsOp.display(s); }
	};

/* Binary expression: '>' comparator greater */
class GrtThanOp : public Expression
	{
	Expression *exp1;
	Expression *exp2;
public:
	GrtThanOp()
		{
		exp1 = NULL;
		exp2 = NULL;
		}
	GrtThanOp(Expression &p1, Expression &p2)
		{
		exp1 = p1.copy();
		exp2 = p2.copy();
		}
	GrtThanOp(GrtThanOp &grtOp)
		: Expression(grtOp)
		{
		exp1 = grtOp.exp1->copy();
		exp2 = grtOp.exp2->copy();
		}
	virtual ~GrtThanOp()
		{
		if (exp1) delete exp1;
		if (exp2) delete exp2;
		}
	Expression *copy()
		{ return new GrtThanOp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);	
	friend ostream &operator<<(ostream &s, GrtThanOp &grtOp)
		{ return grtOp.display(s); }
	};

/* Binary expression: ">=" comparator greater or equal*/
class GrtEqOp : public Expression
	{
	Expression *exp1;
	Expression *exp2;
public:
	GrtEqOp()
		{
		exp1 = NULL;
		exp2 = NULL;
		}
	GrtEqOp(Expression &p1, Expression &p2)
		{
		exp1 = p1.copy();
		exp2 = p2.copy();
		}
	GrtEqOp(GrtEqOp &greOp)
		: Expression(greOp)
		{
		exp1 = greOp.exp1->copy();
		exp2 = greOp.exp2->copy();
		}
	virtual ~GrtEqOp()
		{
		if (exp1) delete exp1;
		if (exp2) delete exp2;
		}
	Expression *copy()
		{ return new GrtEqOp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);	
	friend ostream &operator<<(ostream &s, GrtEqOp &greOp)
		{ return greOp.display(s); }
	};

/* Binary expression: '<' comparator less */
class LesThanOp : public Expression
	{
	Expression *exp1;
	Expression *exp2;
public:
	LesThanOp()
		{
		exp1 = NULL;
		exp2 = NULL;
		}
	LesThanOp(Expression &p1, Expression &p2)
		{
		exp1 = p1.copy();
		exp2 = p2.copy();
		}
	LesThanOp(LesThanOp &lesOp)
		: Expression(lesOp)
		{
		exp1 = lesOp.exp1->copy();
		exp2 = lesOp.exp2->copy();
		}
	virtual ~LesThanOp()
		{
		if (exp1) delete exp1;
		if (exp2) delete exp2;
		}
	Expression *copy()
		{ return new LesThanOp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);	
	friend ostream &operator<<(ostream &s, LesThanOp &lesOp)
		{ return lesOp.display(s); }
	};

/* Binary expression: "<=" comparator less or equal */
class LesEqOp : public Expression
	{
	Expression *exp1;
	Expression *exp2;
public:
	LesEqOp()
		{
		exp1 = NULL;
		exp2 = NULL;
		}
	LesEqOp(Expression &p1, Expression &p2)
		{
		exp1 = p1.copy();
		exp2 = p2.copy();
		}
	LesEqOp(LesEqOp &leqOp)
		: Expression(leqOp)
		{
		exp1 = leqOp.exp1->copy();
		exp2 = leqOp.exp2->copy();
		}
	virtual ~LesEqOp()
		{
		if (exp1) delete exp1;
		if (exp2) delete exp2;
		}
	Expression *copy()
		{ return new LesEqOp(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &);
	void Sim_trigger(String &);
	Number Sim_evaluate(void);	
	friend ostream &operator<<(ostream &s, LesEqOp &leqOp)
		{ return leqOp.display(s); }
	};

#endif // PEXPR_H
