/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * PTypes.cc
 *
 * C++ functions to create parser objects.  These functions are used to
 * interface with the parser functions in intrface.c.
 */

#include "glo.h"		// filename, lineno
#include "intrface.h"
#include "String.h"
#include "List.h"
#include "Base.h"
#include "PExpr.h"
#include "PTypes.h"
#include "Systask.h"

void *
P_CreateNumber(char *str, int base, int len)
	{
	Number *ret;
	String res(str);
	ret = new Number(res, base, len);
	return ret;
	}

void *
P_CreateConstExpr(p_Expression *expr)
	{
	ConstExpression *ret;
	Expression *exp = (Expression *)expr->data;
	ret = new ConstExpression(*(Number *)exp);
	delete exp;
	return ret;
	}

void *
P_CreateExpr(int type, void *p1, void *p2)
	{
	Expression *ret;
	switch(type)
		{
		case EXPR_STR:
			{
			ret = new QString((char *)p1);
			break;
			}
		case EXPR_NUM:
			{
			p_Number *pp = (p_Number *)p1;
			ret = new Number(*(Number *)pp->data);
			delete (Number *)pp->data;
			break;
			}
		case EXPR_ID:
			{
			p_RangeId *pp = (p_RangeId *)p1;
			ret = new RangeId(*(RangeId *)pp->data);
			delete (RangeId *)pp->data;
			break;
			}
		case EXPR_FUNC:
			{
			p_FunctionCall *pp = (p_FunctionCall *)p1;
			ret = new FunctionCall(*(FunctionCall *)pp->data);
			delete (FunctionCall *)pp->data;
			break;
			}
		case EXPR_EXP:
			{
			p_Expression *pp = (p_Expression *)p1;
			Expression *ppp = (Expression *)pp->data;
			ret = ppp->copy();
			delete ppp;
			break;
			}
		case EXPR_NOTOP:
			{
			p_Expression *pp = (p_Expression *)p1;
			ret = new NotOp(*(Expression *)pp->data);
			delete (Expression *)pp->data;
			break;
			}
		case EXPR_INVOP:
			{
			p_Expression *pp = (p_Expression *)p1;
			ret = new InvertOp((*(Expression *)pp->data));
			delete (Expression *)pp->data;
			break;
			}
		case EXPR_ADD:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			ret = new AddOp(*(Expression *)pp->data,
				*(Expression *)pp2->data);
			delete (Expression *)pp->data;
			delete (Expression *)pp2->data;
			break;
			}
		case EXPR_SUB:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			ret = new SubOp(*(Expression *)pp->data,
				*(Expression *)pp2->data);
			delete (Expression *)pp->data;
			delete (Expression *)pp2->data;
			break;
			}
		case EXPR_EQEQ:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			ret = new EqEqComp(*(Expression *)pp->data,
				*(Expression *)pp2->data);
			delete (Expression *)pp->data;
			delete (Expression *)pp2->data;
			break;
			}
		case EXPR_NOTEQ:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			ret = new NotEqComp(*(Expression *)pp->data,
				*(Expression *)pp2->data);
			delete (Expression *)pp->data;
			delete (Expression *)pp2->data;
			break;
			}
		case EXPR_OROR:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			ret = new OrOrOp(*(Expression *)pp->data,
				*(Expression *)pp2->data);
			delete (Expression *)pp->data;
			delete (Expression *)pp2->data;
			break;
			}
		case EXPR_ANDAND:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			ret = new AndAndOp(*(Expression *)pp->data,
				*(Expression *)pp2->data);
			delete (Expression *)pp->data;
			delete (Expression *)pp2->data;
			break;
			}
		case EXPR_LOGOR:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			ret = new LogOrOp(*(Expression *)pp->data,
				*(Expression *)pp2->data);
			delete (Expression *)pp->data;
			delete (Expression *)pp2->data;
			break;
			}
		case EXPR_LOGAND:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			ret = new LogAndOp(*(Expression *)pp->data,
				*(Expression *)pp2->data);
			delete (Expression *)pp->data;
			delete (Expression *)pp2->data;
			break;
			}
		case EXPR_LOGLSHIFT:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			ret = new LogLShiftOp(*(Expression *)pp->data,
				*(Expression *)pp2->data);
			delete (Expression *)pp->data;
			delete (Expression *)pp2->data;
			break;
			}
		case EXPR_LOGRSHIFT:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			ret = new LogRShiftOp(*(Expression *)pp->data,
				*(Expression *)pp2->data);
			delete (Expression *)pp->data;
			delete (Expression *)pp2->data;
			break;
			}
		case EXPR_GREATEREQ:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			ret = new GrtEqOp(*(Expression *)pp->data,
				*(Expression *)pp2->data);
			delete (Expression *)pp->data;
			delete (Expression *)pp2->data;
			break;
			}
		case EXPR_GRTTHAN:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			ret = new GrtThanOp(*(Expression *)pp->data,
				*(Expression *)pp2->data);
			delete (Expression *)pp->data;
			delete (Expression *)pp2->data;
			break;
			}
		case EXPR_LESSEQ:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			ret = new LesEqOp(*(Expression *)pp->data,
				*(Expression *)pp2->data);
			delete (Expression *)pp->data;
			delete (Expression *)pp2->data;
			break;
			}
		case EXPR_LESTHAN:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			ret = new LesThanOp(*(Expression *)pp->data,
				*(Expression *)pp2->data);
			delete (Expression *)pp->data;
			delete (Expression *)pp2->data;
			break;
			}
		}
	return ret;
	}

void *
P_CreateExprLst(p_ExpressionLst *lst, p_Expression *p)
	{
	List<Expression *> *ret;
	if (lst != NULL)
		ret = (List<Expression *> *)lst->data;
	else
		ret = new List<Expression *>;

	p_Expression *pp = (p_Expression *)p;
	Expression *ppp = (Expression *)pp->data;
	*ret += ppp->copy();
	delete ppp;
	return ret;
	}

void *
P_CreateRange(p_ConstExpression *l, p_ConstExpression *r)
	{
	Range *ret;
	ret = new Range(*(ConstExpression *)l->data,
		*(ConstExpression *)r->data);
	delete (ConstExpression *)l->data;
	delete (ConstExpression *)r->data;
	return ret;
	}

void *
P_CreateRangeId(char *str, p_Range *r)
	{
	String s(str);
	RangeId *ret;
	if (r == NULL)
		ret = new RangeId(s);
	else
		ret = new RangeId(s,*(Range *)r->data);
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	if (r != NULL)
		delete (Range *)r->data;
	return ret;
	}

void *
P_CreateRangeIdLst(p_RangeIdLst *lst, p_RangeId *id)
	{
	List<RangeId *> *ret;

	if (lst != NULL)
		ret = (List<RangeId *> *)lst->data;
	else
		ret = new List<RangeId *>;

	*ret += new RangeId(*(RangeId *)id->data);
	delete (RangeId *)id->data;
	return ret;
	}

void *
P_CreateFunctionCall(p_RangeId *ri, p_ExpressionLst *lst)
	{
	FunctionCall *ret;
	if (lst != NULL)
		{
		ret = new FunctionCall(*(RangeId *)ri->data,
			*(List<Expression *> *)lst->data);
		delete (List<Expression *> *)lst->data;
		}
	else
		ret = new FunctionCall(*(RangeId *)ri->data);
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	delete (RangeId *)ri->data;
	return ret;
	}

void *
P_CreateLvalue(p_RangeId *ri)
	{
	Lvalue *ret;
	ret = new Lvalue(*(RangeId *)ri->data);
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	delete (RangeId *)ri->data;
	return ret;
	}

void *
P_CreateDelayEvntCtl(int type, void *p)
	{
	DelayEvntCtl *ret;
	switch(type)
		{
		case DEC_DLAYNUM:
			{
			p_Number *pp = (p_Number *)p;
			ret = new DelayNum(*(Number *)pp->data);
			delete (Number *)pp->data;
			break;
			}
		case DEC_DLAYID:
			{
			p_RangeId *pp = (p_RangeId *)p;
			ret = new DelayRangeId(*(RangeId *)pp->data);
			delete (RangeId *)pp->data;
			break;
			}
		case DEC_EVNTID:
			{
			p_RangeId *pp = (p_RangeId *)p;
			EvntExpression ee(*(RangeId *)pp->data);
			OredEvntExpression *ppp = new OredEvntExpression;
			*ppp += ee;
			ret = ppp;
			delete (RangeId *)pp->data;
			break;
			}
		case DEC_EVNTEXPR:
			{
			p_EvntExpressionLst *pp =
				(p_EvntExpressionLst *)p;
			OredEvntExpression *ppp =
				(OredEvntExpression *)pp->data;
			ret = new OredEvntExpression(*ppp);
			delete ppp;
			break;
			}
		}
	return ret;
	}

void *
P_CreateEvntExpression(int type, p_Expression *p)
	{
	EvntExpression *ret;
	Expression *pp = (Expression *)p->data;
	switch(type)
		{
		case EVNT_NONE:
			{
			ret = new EvntExpression(TRIG_NONE, *pp);
			break;
			}
		case EVNT_POS:
			{
			ret = new EvntExpression(TRIG_POS, *pp);
			break;
			}
		case EVNT_NEG:
			{
			ret = new EvntExpression(TRIG_NEG, *pp);
			break;
			}
		case EVNT_CHNG:
			{
			ret = new EvntExpression(TRIG_CHNG, *pp);
			break;
			}
		}
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	delete pp;
	return ret;
	}

void *
P_CreateEvntExpressionLst(p_EvntExpressionLst *lst,
		p_EvntExpression *p)
	{
	OredEvntExpression *ret;
	if (lst != NULL)
		ret = (OredEvntExpression *)lst->data;
	else
		ret = new OredEvntExpression;
	// OredEvntExpression has an operator+=.
	*ret += *(EvntExpression *)p->data;
	delete (EvntExpression *)p->data;
	return ret;
	}

void *
P_CreateCaseItem(int type, p_ExpressionLst *p, p_Stmts *s)
	{
	CaseItem *ret;
	if (p != NULL)
		{
		ret = new CaseItem(*(List<Expression *> *)p->data,
			*(Stmts *)s->data);
		delete (List<Expression *> *)p->data;
		}
	else
		ret = new CaseItem(*(Stmts *)s->data);
	delete (Stmts *)s->data;
	return ret;
	}

void *
P_CreateCaseItemLst(p_CaseItemLst *cl, p_CaseItem *c)
	{
	List<CaseItem *> *ret;
	if (cl != NULL)
		ret = (List<CaseItem *> *)cl->data;
	else
		ret = new List<CaseItem *>;
	*ret += new CaseItem(*(CaseItem *)c->data);
	delete (CaseItem *)c->data;
	return ret;
	}

void *
P_CreateTaskStmt(p_RangeId *r, p_ExpressionLst *p)
	{
	TaskStmt *ret;
	if (p != NULL)
		{
		ret = new TaskStmt(*(RangeId *)r->data,
			*(List<Expression *> *)p->data);
		delete (List<Expression *> *)p->data;
		}
	else
		ret = new TaskStmt(*(RangeId *)r->data);
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	delete (RangeId *)r->data;
	return ret;
	}

void *
P_CreateAssignStmt(p_Lvalue *lval, p_Expression *expr)
	{
	AssignStmt *ret;
	ret = new AssignStmt(*(Lvalue *)lval->data,
		*(Expression *)expr->data);
	delete (Lvalue *)lval->data;
	delete (Expression *)expr->data;
	return ret;
	}

void *
P_CreateSeqBlkStmt(p_RangeId *id, p_StmtsLst *p)
	{
	SeqBlkStmt *ret;
	if (id != NULL)
		{
		ret = new SeqBlkStmt(*(RangeId *)id->data,
			*(List<Stmts *> *)p->data);
		delete (RangeId *)id->data;
		}
	else
		ret = new SeqBlkStmt(*(List<Stmts *> *)p->data);
	delete (List<Stmts *> *)p->data;
	return ret;
	}

void *
P_CreateStmts(int type, void *p1, void *p2, void *p3, void *p4)
	{
	Stmts *ret;
	switch(type)
		{   
		case STMT_TASK:
			{
			p_TaskStmt *pp = (p_TaskStmt *)p1;
			ret = new TaskStmt(*(TaskStmt *)pp->data);
			delete (TaskStmt *)pp->data;
			break;
			}
		case STMT_ASSIGN:
			{
			p_AssignStmt *pp = (p_AssignStmt *)p1;
			ret = new AssignStmt(*(AssignStmt *)pp->data);
			delete (AssignStmt *)pp->data;
			break;
			}
		case STMT_SEQBLK:
			{
			p_SeqBlkStmt *pp = (p_SeqBlkStmt *)p1;
			ret = new SeqBlkStmt(*(SeqBlkStmt *)pp->data);
			delete (SeqBlkStmt *)pp->data;
			break;
			}
		case STMT_IF:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Stmts *pp2 = (p_Stmts *)p2;
			ret = new IfStmt(*(Expression *)pp->data,
				*(Stmts *)pp2->data);
			delete (Expression *)pp->data;
			delete (Stmts *)pp2->data;
			break;
			}
		case STMT_IF_ELSE:
			{
			p_Expression *pp = (p_Expression *)p1;
			p_Stmts *pp2 = (p_Stmts *)p2;
			p_Stmts *pp3 = (p_Stmts *)p3;
			ret = new IfElseStmt(*(Expression *)pp->data,
				*(Stmts *)pp2->data, *(Stmts *)pp3->data);
			delete (Expression *)pp->data;
			delete (Stmts *)pp2->data;
			delete (Stmts *)pp3->data;
			break;
			}
		case STMT_FOR:
			{
			p_AssignStmt *pp = (p_AssignStmt *)p1;
			p_Expression *pp2 = (p_Expression *)p2;
			p_AssignStmt *pp3 = (p_AssignStmt *)p3;
			p_Stmts *pp4 = (p_Stmts *)p4;
			ret = new ForStmt(*(AssignStmt *)pp->data,
				*(Expression *)pp2->data,
				*(AssignStmt *)pp3->data,
				*(Stmts *)pp4->data);
			delete (AssignStmt *)pp->data;
			delete (Expression *)pp2->data;
			delete (AssignStmt *)pp3->data;
			delete (Stmts *)pp4->data;
			break;
			}
		case STMT_CASE:
			{
			p_Expression *pp1 = (p_Expression *)p1;
			p_CaseItemLst *pp2 = (p_CaseItemLst *)p2;
			
			ret = new CaseStmt(*(Expression *)pp1->data,
				*(List<CaseItem *> *)pp2->data);
			delete (Expression *)pp1->data;
			delete (List<CaseItem *> *)pp2->data;
			break;
			}
		case STMT_STMT:
			{
			p_Stmts *pp = (p_Stmts *)p1;
			Stmts *ppp = (Stmts *)pp->data;
			ret = ppp->copy();
			// Set the delay/event control if available
			if (p2)
				{
				p_DelayEvntCtl *pp2 = (p_DelayEvntCtl *)p2;
				// Should I really copy and delete?
				DelayEvntCtl *decp = (DelayEvntCtl *)pp2->data;
				ret->dec = decp->copy();
				delete decp;
				}
			delete ppp;
			break;
			}
		case STMT_NULL:
			{
			ret = new NullStmt;
			break;
			}
		}
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	return ret;
	}

void *
P_CreateStmtsLst(p_StmtsLst *lst, int type, p_Stmts *p)
	{
	List<Stmts *> *ret;
	if (lst != NULL)
		ret = (List<Stmts *> *)lst->data;
	else
		ret = new List<Stmts *>;
	Stmts *st;
	switch(type)
		{   
		case STMT_STMT:
			{
			Stmts *pp = (Stmts *)p->data;
			st = pp->copy();
			delete (Stmts *)p->data;
			break;
			}
		case STMT_NULL:
			{
			st = new NullStmt;
			break;
			}
		}
	*ret += st;
	return ret;
	}

void *
P_CreateWireDecl(p_Range *r, p_RangeIdLst *ids)
	{
	WireDecl *ret;
	if (r != NULL)
		{
		ret = new WireDecl(*(Range *)r->data,
			*(List<RangeId *> *)ids->data);
		delete (Range *)r->data;
		}
	else
		ret = new WireDecl(*(List<RangeId *> *)ids->data);
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	delete (List<RangeId *> *)ids->data;
	return ret;
	}

void *
P_CreateRegDecl(p_Range *r, p_RangeIdLst *ids)
	{
	RegDecl *ret;
	if (r != NULL)
		{
		ret = new RegDecl(*(Range *)r->data,
			*(List<RangeId *> *)ids->data);
		delete (Range *)r->data;
		}
	else
		ret = new RegDecl(*(List<RangeId *> *)ids->data);
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	delete (List<RangeId *> *)ids->data;
	return ret;
	}

void *
P_CreateInputDecl(p_Range *r, p_RangeIdLst *lst)
	{
	InputDecl *ret;
	if (r != NULL)
		{
		ret = new InputDecl(*(Range *)r->data,
			*(List<RangeId *> *)lst->data);
		delete (Range *)r->data;
		}
	else
		ret = new InputDecl(*(List<RangeId *> *)lst->data);
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	delete (List<RangeId *> *)lst->data;
	return ret;
	}

void *
P_CreateOutputDecl(p_Range *r, p_RangeIdLst *lst)
	{
	OutputDecl *ret;
	if (r != NULL)
		{
		ret = new OutputDecl(*(Range *)r->data,
			*(List<RangeId *> *)lst->data);
		delete (Range *)r->data;
		}
	else
		ret = new OutputDecl(*(List<RangeId *> *)lst->data);
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	delete (List<RangeId *> *)lst->data;
	return ret;
	}

void *
P_CreateInoutDecl(p_Range *r, p_RangeIdLst *lst)
	{
	InoutDecl *ret;
	if (r != NULL)
		{
		ret = new InoutDecl(*(Range *)r->data,
			*(List<RangeId *> *)lst->data);
		delete (Range *)r->data;
		}
	else
		ret = new InoutDecl(*(List<RangeId *> *)lst->data);
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	delete (List<RangeId *> *)lst->data;
	return ret;
	}

void *
P_CreateTFDecl(int type, void *p)
	{
	TFDecl *ret;
	switch(type)
		{
		case DECL_REG:
			{
			p_RegDecl *pp = (p_RegDecl *)p;
			ret = new RegDecl(*(RegDecl *)pp->data);
			delete (RegDecl *)pp->data;
			break;
			}
		case DECL_INPUT:
			{
			p_InputDecl *pp = (p_InputDecl *)p;
			ret = new InputDecl(*(InputDecl *)pp->data);
			delete (InputDecl *)pp->data;
			break;
			}
		case DECL_OUTPUT:
			{
			p_OutputDecl *pp = (p_OutputDecl *)p;
			ret = new OutputDecl(*(OutputDecl *)pp->data);
			delete (OutputDecl *)pp->data;
			break;
			}
		case DECL_INOUT:
			{
			p_InoutDecl *pp = (p_InoutDecl *)p;
			ret = new InoutDecl(*(InoutDecl *)pp->data);
			delete (InoutDecl *)pp->data;
			break;
			}
		}
	return ret;
	}

void *
P_CreateTFDeclLst(p_TFDeclLst *lst, int type, p_TFDecl *p)
	{
	List<TFDecl *> *ret;
	if (lst != NULL)
		ret = (List<TFDecl *> *)lst->data;
	else
		ret = new List<TFDecl *>;
	TFDecl *tfd;
	switch(type)
		{
		case DECL_DECL:
			{
			TFDecl *pp = (TFDecl *)p->data;
			tfd = pp->copy();
			delete (TFDecl *)p->data;
			break;
			}
		case DECL_NULL:
			{
			tfd = new NullTFDecl;
			break;
			}
		}
	*ret += tfd;
	return ret;
	}

void *
P_CreateFunction(p_Range *r, p_RangeId *n, p_TFDeclLst *decl, p_Stmts *st)
	{
	Function *ret;
	if (r != NULL)
		{
		ret = new Function(*(RangeId *)n->data,
			*(List<TFDecl *> *)decl->data,
			*(Stmts *)st->data, *(Range *)r->data);
		delete (Range *)r->data;
		}
	else
		{
		ret = new Function(*(RangeId *)n->data,
			*(List<TFDecl *> *)decl->data,
			*(Stmts *)st->data);
		}
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	delete (RangeId *)n->data;
	delete (List<TFDecl *> *)decl->data;
	delete (Stmts *)st->data;
	return ret;
	}

void *
P_CreateTask(p_RangeId *n, p_TFDeclLst *decl, p_Stmts *st)
	{
	Task *ret;
	ret = new Task(*(RangeId *)n->data,
		*(List<TFDecl *> *)decl->data,
		*(Stmts *)st->data);
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	delete (RangeId *)n->data;
	delete (List<TFDecl *> *)decl->data;
	delete (Stmts *)st->data;
	return ret;
	}

void *
P_CreateInitialStmt(p_Stmts *st)
	{
	InitialStmt *ret;
	ret = new InitialStmt(*(Stmts *)st->data);
	delete (Stmts *)st->data;
	return ret;
	}

void *
P_CreateAlwaysStmt(p_Stmts *st)
	{
	AlwaysStmt *ret;
	ret = new AlwaysStmt(*(Stmts *)st->data);
	delete (Stmts *)st->data;
	return ret;
	}

void *
P_CreatePortConn(p_Expression *p)
	{
	PortConnection *ret;
	if (p != NULL)
		{
		ret = new PortConnection(*(Expression *)p->data);
		delete (Expression *)p->data;
		}
	else
		ret = new PortConnection;
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	return ret;
	}

void *
P_CreatePortConnLst(p_PortConnLst *lst, p_PortConn *p)
	{
	List<PortConnection *> *ret;
	if (lst != NULL)
		ret = (List<PortConnection *> *)lst->data;
	else
		ret = new List<PortConnection *>;
	*ret += new PortConnection(*(PortConnection *)p->data);
	delete (PortConnection *)p->data;
	return ret;
	}

void *
P_CreateModInstance(p_RangeId *id, p_PortConnLst *lst)
	{
	ModuleInstance *ret;
	if (lst != NULL)
		{
		ret = new ModuleInstance(*(RangeId *)id->data,
			*(List<PortConnection *> *)lst->data);
		delete (List<PortConnection *> *)lst->data;
		}
	else
		ret = new ModuleInstance(*(RangeId *)id->data);
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	delete (RangeId *)id->data;
	return ret;
	}

void *
P_CreateModInstanceLst(p_ModInstanceLst *lst, p_ModInstance *mi)
	{
	List<ModuleInstance *> *ret;
	if (lst != NULL)
		ret = (List<ModuleInstance *> *)lst->data;
	else
		ret = new List<ModuleInstance *>;
	*ret += new ModuleInstance(*(ModuleInstance *)mi->data);
	delete (ModuleInstance *)mi->data;
	return ret;
	}

void *
P_CreateModInstan(p_RangeId *id, p_ModInstanceLst *lst)
	{
	ModuleInstan *ret;
	ret = new ModuleInstan(*(RangeId *)id->data,
		*(List<ModuleInstance *> *)lst->data);
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	delete (RangeId *)id->data;
	delete (List<ModuleInstance *> *)lst->data;
	return ret;
	}

void *
P_CreateModItem(int type, void *p)
	{
	ModuleItem *ret;
	switch(type)
		{
		case ITEM_WIREDECL:
			{
			p_WireDecl *pp = (p_WireDecl *)p;
			ret = new WireDecl(*(WireDecl *)pp->data);
			delete (WireDecl *)pp->data;
			break;
			}
		case ITEM_REGDECL:
			{
			p_RegDecl *pp = (p_RegDecl *)p;
			ret = new RegDecl(*(RegDecl *)pp->data);
			delete (RegDecl *)pp->data;
			break;
			}
		case ITEM_INPUTDECL:
			{
			p_InputDecl *pp = (p_InputDecl *)p;
			ret = new InputDecl(*(InputDecl *)pp->data);
			delete (InputDecl *)pp->data;
			break;
			}
		case ITEM_OUTPUTDECL:
			{
			p_OutputDecl *pp = (p_OutputDecl *)p;
			ret = new OutputDecl(*(OutputDecl *)pp->data);
			delete (OutputDecl *)pp->data;
			break;
			}
		case ITEM_INOUTDECL:
			{
			p_InoutDecl *pp = (p_InoutDecl *)p;
			ret = new InoutDecl(*(InoutDecl *)pp->data);
			delete (InoutDecl *)pp->data;
			break;
			}
		case ITEM_FUNC:
			{
			p_Function *pp = (p_Function *)p;
			ret = new Function(*(Function *)pp->data);
			delete (Function *)pp->data;
			break;
			}
		case ITEM_TASK:
			{
			p_Task *pp = (p_Task *)p;
			ret = new Task(*(Task *)pp->data);
			delete (Task *)pp->data;
			break;
			}
		case ITEM_INITIAL:
			{
			p_InitialStmt *pp = (p_InitialStmt *)p;
			ret = new InitialStmt(*(InitialStmt *)pp->data);
			delete (InitialStmt *)pp->data;
			break;
			}
		case ITEM_ALWAYS:
			{
			p_AlwaysStmt *pp = (p_AlwaysStmt *)p;
			ret = new AlwaysStmt(*(AlwaysStmt *)pp->data);
			delete (AlwaysStmt *)pp->data;
			break;
			}
		case ITEM_INSTAN:
			{
			p_ModInstan *pp = (p_ModInstan *)p;
			ret = new ModuleInstan(*(ModuleInstan *)pp->data);
			delete (ModuleInstan *)pp->data;
			break;
			}
		}
	return ret;
	}

void *
P_CreateModItemLst(p_ModuleItemLst *lst, int type, p_ModuleItem *p)
	{
	List<ModuleItem *> *ret;

	if (lst != NULL)
		ret = (List<ModuleItem *> *)lst->data;
	else
		ret = new List<ModuleItem *>;

	switch(type)
		{
		case ITEM_ITEM:
			{
			ModuleItem *pp = (ModuleItem *)p->data;
			*ret += pp->copy();
			delete pp;
			break;
			}
		case ITEM_NULL:
			{
			*ret += new NullModuleItem;
			break;
			}
		}
	return ret;
	}

void *
P_CreatePort(p_RangeId *id)
	{
	Port *ret;
	ret = new Port(*(RangeId *)id->data);
	delete (RangeId *)id->data;
	return ret;
	}

void *
P_CreatePortLst(p_PortLst *lst, p_Port *p)
	{
	List<Port *> *ret;
	if (lst != NULL)
		ret = (List<Port *> *)lst->data;
	else
		ret = new List<Port *>;
	*ret += new Port(*(Port *)p->data);
	delete (Port *)p->data;
	return ret;
	}

void *
P_CreateModule(p_RangeId *id, p_ModuleItemLst *lst, p_PortLst *plst)
	{
	Module *ret;
	if (plst != NULL)
		{
		ret = new Module(*(RangeId *)id->data,
			*(List<ModuleItem *> *)lst->data,
			*(List<Port *> *)plst->data);
		delete (List<Port *> *)plst->data;
		}
	else
		ret = new Module(*(RangeId *)id->data,
			*(List<ModuleItem *> *)lst->data);
	ret->filename = String(cur_filename);
	ret->lineno = cur_lineno;
	delete (RangeId *)id->data;
	delete (List<ModuleItem *> *)lst->data;
	return ret;
	}
