/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * PTypes.h
 *
 * C++ objects for the Yacc parser.
 */

#ifndef PTYPES_H
#define PTYPES_H

#include <iostream.h>
#include "Bool.h"
#include "List.h"
#include "Event.h"		// TrigType
#include "Base.h"
#include "PExpr.h"

/*
 * Normal objects.
 */

/* Lvalue: abc, abc[6:13], etc... */
class Lvalue
	{
	RangeId id;
public:
	String filename;		// Data for error reporting.
	int lineno;
	Lvalue()
		: filename("unknown"), lineno(-1)
		{}
	Lvalue(RangeId &ri)
		: id(ri), filename("unknown"), lineno(-1) {}
	Lvalue(Lvalue &lval)
		: id(lval.id), filename("unknown"), lineno(-1) {}
	void Sim_setup(Stack<int> &);
	void Sim_trigger(Expression &expr);
	friend RangeId &Id(Lvalue &lval)
		{ return lval.id; }
	friend ostream &operator<<(ostream &s, Lvalue &lval)
		{ s << lval.id; return s; }
	};

/* Delay by number */
class DelayNum : public DelayEvntCtl
	{
	Number amt;
public:
	DelayNum()
		: DelayEvntCtl(DECnum), amt(0UL) {}
	DelayNum(Number &num)
		: DelayEvntCtl(DECnum), amt(num) {}
	DelayNum(DelayNum &dn)
		: DelayEvntCtl(dn), amt(dn.amt) {}
	DelayEvntCtl *copy(void)
		{ return new DelayNum(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &, Stmts *);
	void Sim_trigger(Stmts *);
	friend ostream &operator<<(ostream &s, DelayNum &dn)
		{ return dn.display(s); }
	};

/* Delay by variable */
class DelayRangeId : public DelayEvntCtl
	{
	RangeId id;
public:
	DelayRangeId()
		: DelayEvntCtl(DECid) {}
	DelayRangeId(RangeId &ri)
		:  DelayEvntCtl(DECid), id(ri) {}
	DelayRangeId(DelayRangeId &ri)
		:  DelayEvntCtl(ri), id(ri.id) {}
	DelayEvntCtl *copy(void)
		{ return new DelayRangeId(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &, Stmts *);
	void Sim_trigger(Stmts *);
	friend ostream &operator<<(ostream &s, DelayRangeId &dn)
		{ return dn.display(s); }
	};

/*
 * Event expression:  this class and OredEvntExpression must be defined
 * because Sim_setup of EvntExpression requires instantiating Event<Stmts>.
 * Thus, Stmts must be defined before this class.
 */
class EvntExpression
	{
	TrigType edge;			// Edge trigger type;
	Expression *expr;		// Variable name.
public:
	String filename;		// Data for error reporting.
	int lineno;
	EvntExpression()
		: edge(TRIG_NONE), filename("unknown"), lineno(-1) {}
	EvntExpression(Expression &exp)
		: edge(TRIG_CHNG), filename("unknown"), lineno(-1)
		{ expr = exp.copy(); }
	EvntExpression(TrigType type, Expression &exp)
		: edge(type), filename("unknown"), lineno(-1)
		{ expr = exp.copy(); }
	EvntExpression(EvntExpression &ee)
		: edge(ee.edge), filename(ee.filename), lineno(ee.lineno)
		{ expr = ee.expr->copy(); }
	~EvntExpression()
		{ if (expr) delete expr; }
	void Sim_setup(Stack<int> &, Events<Stmts *> *);
	friend ostream &operator<<(ostream &s, EvntExpression &ee)
		{
		if (ee.edge == TRIG_POS)
			s << "posedge ";
		else if (ee.edge == TRIG_NEG)
			s << "negedge ";
		s << *ee.expr << ' ';
		return s;
		}
	};

/* Event expression:  posedge/negedge, etc... */
class OredEvntExpression : public DelayEvntCtl
	{
	List<EvntExpression *> vars;
public:
	OredEvntExpression()
		: DelayEvntCtl(DECevnt) {}
	OredEvntExpression(List<EvntExpression *> &lst)
		: DelayEvntCtl(DECevnt), vars(lst) {}
	OredEvntExpression(OredEvntExpression &oee)
		: DelayEvntCtl(oee), vars(oee.vars) {}
	OredEvntExpression &operator+=(EvntExpression &ee)
		{ vars += new EvntExpression(ee); return *this; }
	DelayEvntCtl *copy(void)
		{ return new OredEvntExpression(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &, Stmts *);
	void Sim_trigger(Stmts *);
	friend ostream &operator<<(ostream &s, OredEvntExpression &ee)
		{ return ee.display(s); }
	};

/* Null statement */
class NullStmt : public Stmts
	{
	// NULL statement
public:
	Stmts *copy(void)
		{ return new NullStmt; }
	ostream &display(ostream &);
	friend ostream &operator<<(ostream &s, NullStmt &ns)
		{ return ns.display(s); }
	};

/* Assignment statement */
class AssignStmt : public Stmts
	{
	Lvalue lval;
	Expression *rval;
public:
	AssignStmt()
          { rval = NULL; }
	AssignStmt(Lvalue &l, Expression &r)
		: lval(l) { rval = r.copy(); }
	AssignStmt(AssignStmt &as)
		: Stmts(as), lval(as.lval)
		{ rval = as.rval->copy(); }
	virtual ~AssignStmt()
		{ if (rval) delete rval; }
	Stmts *copy(void)
		{ return new AssignStmt(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &, Bool = FALSE, int = NS_ALL);
	int Sim_trigger(Stmts *);
	friend ostream &operator<<(ostream &s, AssignStmt &as)
		{ return as.display(s); }
	};

/* Sequential block:  begin ... end */
class SeqBlkStmt : public Stmts
	{
	RangeId name;
	List<Stmts *> stmts;

	// Simulation info.
	Bool pushed;			// Are we on the stack?
	Bool loop;			// Are we part of a loop?
	Stack<SeqBlkStmt *> stk;	// Stack of nested blocks.
	Stack<Stmts *> savestmts;	// Save current statements.
	Stmts *curstmt;			// Current statement being exec'ed.
public:
	SeqBlkStmt(List<Stmts *> &sts)
		: stmts(sts), pushed(FALSE), loop(FALSE)
		{ curstmt = NULL; }
	SeqBlkStmt(RangeId &id, List<Stmts *> &sts)
		: name(id), stmts(sts), pushed(FALSE), loop(FALSE)
		{ curstmt = NULL; }
	SeqBlkStmt(SeqBlkStmt &seq)
		: Stmts(seq), name(seq.name), stmts(seq.stmts),
			pushed(seq.pushed),
			loop(seq.loop), stk(seq.stk),
			savestmts(seq.savestmts)
		{
		// We never create sequential blocks during simulation.
		// So don't copy <curstmt>.
		curstmt = NULL;
		}
	// A destructor is not needed for the <curstmt> because it points
	// to data that is in the list already.  So when the list is
	// destroyed, so will the data pointed to by <curstmt>.
	//virtual ~SeqBlkStmt(void)
	//	{ if (curstmt) delete curstmt; }
	Stmts *copy(void)
		{ return new SeqBlkStmt(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &, Bool = FALSE, int = NS_ALL);
	int Sim_trigger(Stmts *);
	friend ostream &operator<<(ostream &s, SeqBlkStmt &seq)
		{ return seq.display(s); }
	};

/* Class of 'if' statement */
class IfStmt : public Stmts
	{
	Expression *expr;
	Stmts *stmt;
public:
	IfStmt()
		{
		expr = NULL;
		stmt = NULL;
		}
	IfStmt(Expression &exp, Stmts &st)
		{
		expr = exp.copy();
		stmt = st.copy();
		}
	IfStmt(IfStmt &is)
		: Stmts(is)
		{
		expr = is.expr->copy();
		stmt = is.stmt->copy();
		}
	virtual ~IfStmt()
		{
		if (expr) delete expr;
		if (stmt) delete stmt;
		}
	Stmts *copy(void)
		{ return new IfStmt(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &, Bool = FALSE, int = NS_ALL);
	int Sim_trigger(Stmts *);
	friend ostream &operator<<(ostream &s, IfStmt &is)
		{ return is.display(s); }	
	};

/* Class of "if_else" statement */
class IfElseStmt : public Stmts
	{
	Expression *expr;
	Stmts *stmt1;
	Stmts *stmt2;
public:
	IfElseStmt()
		{
		expr = NULL;
		stmt1 = NULL;
		stmt2 = NULL;
		}
	IfElseStmt(Expression &exp, Stmts &st1, Stmts &st2)
		{
		expr = exp.copy();
		stmt1 = st1.copy();
		stmt2 = st2.copy();
		}
	IfElseStmt(IfElseStmt &ies)
		: Stmts(ies)
		{
		expr = ies.expr->copy();
		stmt1 = ies.stmt1->copy();
		stmt2 = ies.stmt2->copy();
		}
	virtual ~IfElseStmt()
		{
		if (expr) delete expr;
		if (stmt1) delete stmt1;
		if (stmt2) delete stmt2;
		}
	Stmts *copy(void)
		{ return new IfElseStmt(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &, Bool = FALSE, int = NS_ALL);
	int Sim_trigger(Stmts *);
	friend ostream &operator<<(ostream &s, IfElseStmt &ies)
		{ return ies.display(s); }	
	};

/* Class of "for" statement */
class ForStmt : public Stmts
	{
	AssignStmt assign1;
	AssignStmt assign2;
	Expression *expr;
	Stmts *stmt;
public:
	ForStmt()
		{
		expr = NULL;
		stmt = NULL;
		}
	ForStmt(AssignStmt &as1, Expression &exp, AssignStmt &as2, Stmts &st)
		: assign1(as1), assign2(as2)
		{
		expr = exp.copy();
		stmt = st.copy();
		}
	ForStmt(ForStmt &fs)
		: Stmts(fs), assign1(fs.assign1), assign2(fs.assign2)
		{
		expr = fs.expr->copy();
		stmt = fs.stmt->copy();
		}
	virtual ~ForStmt()
		{
		if (expr) delete expr;
		if (stmt) delete stmt;
		}
	Stmts *copy(void)
		{ return new ForStmt(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &, Bool = FALSE, int = NS_ALL);
	int Sim_trigger(Stmts *);
	friend ostream &operator<<(ostream &s, ForStmt &fs)
		{ return fs.display(s); }	
	};

/* class case item */
class CaseItem
	{
	Bool def;		// To indicate the default case.
	List<Expression *> expr;
	Stmts *stmt;
public:
	CaseItem()
		: def(FALSE)
		{ stmt = NULL; }
	CaseItem(Stmts &st)
		: def(TRUE)
		{ stmt = st.copy(); }
	CaseItem(List<Expression *> &exp, Stmts &st)
		: def(FALSE), expr(exp)
		{ stmt = st.copy(); }
	CaseItem(CaseItem &ci)
		: def(ci.def), expr(ci.expr)
		{ stmt = ci.stmt->copy(); }
	virtual ~CaseItem()
		{ if (stmt) delete stmt; }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &, Bool = FALSE);
	int Sim_trigger(Number &, Bool &, Stmts *);
	friend Bool IsDefault(CaseItem &ci)
		{ return ci.def; }
	friend ostream &operator<<(ostream &s, CaseItem &ci)
		{ return ci.display(s); }
	friend ostream &operator<<(ostream &s, CaseItem *ci)
		{ return ci->display(s); }
	};

/* class case statement */
class CaseStmt : public Stmts
	{
	List<CaseItem *> ci;
	Expression *expr;
	CaseItem *def;			// Default case, set by setup.
public:
	CaseStmt()
		{ expr = NULL; def = NULL; }
	CaseStmt(Expression &exp, List<CaseItem *> &c)
		: ci(c)
		{ expr = exp.copy(); def = NULL; }
	CaseStmt(CaseStmt &cs)
		: Stmts(cs), ci(cs.ci)
		{ expr = cs.expr->copy(); def = NULL; }
	virtual ~CaseStmt()
		{
		if (expr) delete expr;
		if (def) delete def;
		}
	Stmts *copy(void)
		{ return new CaseStmt(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &, Bool = FALSE, int = NS_ALL);
	int Sim_trigger(Stmts *);
	friend ostream &operator<<(ostream &s, CaseStmt &cs)
		{ return cs.display(s); }
	};

/* Null module item:  empty item */
class NullModuleItem : public ModuleItem
	{
	// Null module item
public:
	ModuleItem *moditem_copy(void)
		{ return new NullModuleItem; }
	ostream &display(ostream &);
	friend ostream &operator<<(ostream &s, NullModuleItem &mi)
		{ return mi.display(s); }
	};

/* Null declaration:  empty declaration */
class NullTFDecl : public TFDecl
	{
	// Null tf declaration
public:
	TFDecl *tfdecl_copy(void)
		{ return new NullTFDecl; }
	ostream &display(ostream &);
	friend ostream &operator<<(ostream &s, NullTFDecl &ntfd)
		{ return ntfd.display(s); }
	};

/* Wire declaration */
class WireDecl : public ModuleItem
	{
	Range range;
	List<RangeId *> ids;
public:
	WireDecl(List<RangeId *> &lst)
		: ids(lst) {}
	WireDecl(Range &r, List<RangeId *> &lst)
		: range(r), ids(lst) {}
	WireDecl(WireDecl &wd)
		: ModuleItem(wd), ErrorData(wd),
			range(wd.range), ids(wd.ids) {}
	ModuleItem *moditem_copy(void)
		{ return new WireDecl(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int>, STnode * = NULL);
	friend ostream &operator<<(ostream &s, WireDecl &wd)
		{ return wd.display(s); }
	};

/* Register declaration */
class RegDecl : public ModuleItem, public TFDecl
	{
	Range range;
	List<RangeId *> ids;
public:
	RegDecl(List<RangeId *> &id)
		: ids(id) {}
	RegDecl(Range &r, List<RangeId *> &id)
		: range(r), ids(id) {}
	RegDecl(RegDecl &rd)
		: ModuleItem(rd), TFDecl(rd), ErrorData(rd),
			range(rd.range), ids(rd.ids)
		{}
	ModuleItem *moditem_copy(void)
		{ return new RegDecl(*this); }
	TFDecl *tfdecl_copy(void)
		{ return new RegDecl(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int>, STnode * = NULL);
	friend ostream &operator<<(ostream &s, RegDecl &rd)
		{ return rd.display(s); }
	};

/* Input declaration */
class InputDecl : public ModuleItem, public TFDecl
	{
	Range range;
	List<RangeId *> ids;
public:
	InputDecl(List<RangeId *> &id)
		: ids(id) {}
	InputDecl(Range &r, List<RangeId *> &id)
		: range(r), ids(id) {}
	InputDecl(InputDecl &rd)
		: ModuleItem(rd), TFDecl(rd), ErrorData(rd),
			range(rd.range), ids(rd.ids)
		{}
	ModuleItem *moditem_copy(void)
		{ return new InputDecl(*this); }
	TFDecl *tfdecl_copy(void)
		{ return new InputDecl(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int>, STnode * = NULL);
	friend ostream &operator<<(ostream &s, InputDecl &ind)
		{ return ind.display(s); }
	};

/* Output declaration */
class OutputDecl : public ModuleItem, public TFDecl
	{
	Range range;
	List<RangeId *> ids;
public:
	OutputDecl(List<RangeId *> &id)
		: ids(id) {}
	OutputDecl(Range &r, List<RangeId *> &id)
		: range(r), ids(id) {}
	OutputDecl(OutputDecl &rd)
		: ModuleItem(rd), TFDecl(rd), ErrorData(rd),
			range(rd.range), ids(rd.ids)
		{}
	ModuleItem *moditem_copy(void)
		{ return new OutputDecl(*this); }
	TFDecl *tfdecl_copy(void)
		{ return new OutputDecl(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int>, STnode * = NULL);
	friend ostream &operator<<(ostream &s, OutputDecl &outd)
		{ return outd.display(s); }
	};

/* Inout declaration */
class InoutDecl : public ModuleItem, public TFDecl
	{
	Range range;
	List<RangeId *> ids;
public:
	InoutDecl(List<RangeId *> &id)
		: ids(id) {}
	InoutDecl(Range &r, List<RangeId *> &id)
		: range(r), ids(id) {}
	InoutDecl(InoutDecl &rd)
		: ModuleItem(rd), TFDecl(rd), ErrorData(rd),
			range(rd.range), ids(rd.ids)
		{}
	ModuleItem *moditem_copy(void)
		{ return new InoutDecl(*this); }
	TFDecl *tfdecl_copy(void)
		{ return new InoutDecl(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int>, STnode * = NULL);
	friend ostream &operator<<(ostream &s, InoutDecl &ind)
		{ return ind.display(s); }
	};

/* Function definition */
class Function : public ModuleItem
	{
	RangeId name;
	Range retsize;
	List<TFDecl *> decl;
	Stmts *stmt;
public:
	Function()
		{ stmt = NULL; }
	Function(RangeId &id, List<TFDecl *> &d, Stmts &st)
		: name(id), decl(d)
		{ stmt = st.copy(); }
	Function(RangeId &id, List<TFDecl *> &d, Stmts &st, Range &r)
		: name(id), retsize(r), decl(d)
		{ stmt = st.copy(); }
	Function(Function &func)
		: ModuleItem(func), ErrorData(func), name(func.name),
			retsize(func.retsize), decl(func.decl)
		{ stmt = func.stmt->copy(); }
	virtual ~Function()
		{ if (stmt) delete stmt; }
	ModuleItem *moditem_copy(void)
		{ return new Function(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int>, STnode * = NULL);
	friend ostream &operator<<(ostream &s, Function &func)
		{ return func.display(s); }
	};

/* Task definition */
class Task : public ModuleItem
	{
	RangeId name;
	List<TFDecl *> decl;
	Stmts *stmt;
public:
	Task()
		{ stmt = NULL; }
	Task(RangeId &id, List<TFDecl *> &d, Stmts &st)
		: name(id), decl(d)
		{ stmt = st.copy(); }
	Task(Task &tsk)
		: ModuleItem(tsk), ErrorData(tsk),
			name(tsk.name), decl(tsk.decl)
		{ stmt = tsk.stmt->copy(); }
	virtual ~Task()
		{ if (stmt) delete stmt; }
	ModuleItem *moditem_copy(void)
		{ return new Task(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int>, STnode * = NULL);
	friend ostream &operator<<(ostream &s, Task &tsk)
		{ return tsk.display(s); }
	};

/* Initial statement */
class InitialStmt : public ModuleItem
	{
	Stmts *stmt;
public:
	InitialStmt(Stmts &st)
		{ stmt = st.copy(); }
	InitialStmt(InitialStmt &st)
		: ModuleItem(st), ErrorData(st)
		{ stmt = st.stmt->copy(); }
	virtual ~InitialStmt()
		{ if (stmt) delete stmt; }
	ModuleItem *moditem_copy(void)
		{ return new InitialStmt(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int>, STnode * = NULL);
	friend ostream &operator<<(ostream &s, InitialStmt &ist)
		{ return ist.display(s); }
	};

/* Always statement */
class AlwaysStmt : public ModuleItem
	{
	Stmts *stmt;
public:
	AlwaysStmt(Stmts &st)
		{ stmt = st.copy(); }
	AlwaysStmt(AlwaysStmt &as)
		: ModuleItem(as), ErrorData(as)
		{ stmt = as.stmt->copy(); }
	virtual ~AlwaysStmt()
		{ if (stmt) delete stmt; }
	ModuleItem *moditem_copy(void)
		{ return new AlwaysStmt(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int>, STnode * = NULL);
	friend ostream &operator<<(ostream &s, AlwaysStmt &as)
		{ return as.display(s); }
	};

/* Port connection for module instantiation. */
class PortConnection
	{
	Expression *exp;
public:
	String filename;
	int lineno;
	PortConnection()
		: filename("unknown"), lineno(-1)
		{ exp = NULL; }
	PortConnection(Expression &expr)
		: filename("unknown"), lineno(-1)
		{ exp = expr.copy(); }
	PortConnection(PortConnection &pc)
		: filename(pc.filename), lineno(pc.lineno)
		{ exp = (pc.exp != NULL) ? pc.exp->copy() : NULL; }
	ostream &display(ostream &s)
		{
		if (exp)
			s << *exp;
		return s;
		}
	HashValue *Index(void)
		{
		HashValue *ret = NULL;
		if (exp != NULL)
			ret = exp->Index();
		return ret;
		}
	int GetRange(unsigned long &ms, unsigned long &ls)
		{
		if (exp != NULL)
			return exp->GetRange(ms, ls);
		return 0;
		}
	void Sim_setup(Stack<int> &scope)
		{ exp->Sim_setup(scope); }
	friend ostream &operator<<(ostream &s, PortConnection &pc)
		{ return pc.display(s); }
	};

/* Module instantiation. */
class ModuleInstance
	{
	RangeId name;
	List<PortConnection *> connections;
public:
	String filename;
	int lineno;
	ModuleInstance(RangeId &id)
		: name(id), filename("unknown"), lineno(-1) {}
	ModuleInstance(RangeId &id, List<PortConnection *> &lst)
		: name(id), connections(lst),
			filename("unknown"), lineno(-1) {}
	ModuleInstance(ModuleInstance &mi)
		: name(mi.name), connections(mi.connections),
			filename(mi.filename), lineno(mi.lineno) {}
	ostream &display(ostream &);
	void Sim_setup(Stack<int> &, STnode *);
	friend ostream &operator<<(ostream &s, ModuleInstance &mi)
		{ return mi.display(s); }
	};

class ModuleInstan : public ModuleItem
	{
	RangeId modname;
	List<ModuleInstance *> instances;
public:
	ModuleInstan(RangeId &id, List<ModuleInstance *> &lst)
		: modname(id), instances(lst) {}
	ModuleInstan(ModuleInstan &mi)
		: ModuleItem(mi), ErrorData(mi),
			modname(mi.modname), instances(mi.instances) {}
	ModuleItem *moditem_copy(void)
		{ return new ModuleInstan(*this); }
	ostream &display(ostream &);
	void Sim_setup(Stack<int>, STnode * = NULL);
	friend ostream &operator<<(ostream &s, ModuleInstan &mi)
		{ return mi.display(s); }
	};

/* Port for module */
class Port
	{
	RangeId name;
public:
	Port(RangeId &id)
		: name(id) {}
	Port(Port &p)
		: name(p.name) {}
	operator String()
		{ return String(name); }
	ostream &display(ostream &s);
	void Sim_setup(void);
	friend ostream &operator<<(ostream &s, Port &p)
		{ return p.display(s); }
	};

/* Top Module */
class Module
	{
	RangeId name;
	List<Port *> portlst;
	List<ModuleItem *> items;
public:
	String filename;
	int lineno;
	Module(RangeId &n, List<ModuleItem *> &itm)
		: name(n), items(itm) {}
	Module(RangeId &n, List<ModuleItem *> &itm, List<Port *> &lst)
		: name(n), portlst(lst), items(itm) {}
	friend RangeId &Id(Module &mod)
		{ return mod.name; }
	friend List<Port *> &PortLst(Module &mod)
		{ return mod.portlst; }
	friend List<ModuleItem *> &Items(Module &mod)
		{ return mod.items; }
	};

#endif // PTYPES_H
